<?php

use App\Exceptions\Security;
use App\Fields\Currency;
use App\Fields\Double;
use App\Purifier;
use App\Validator;

/**
 * Inventory GrossPrice Field Class.
 *
 * @package   InventoryField
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class Vtiger_GrossPrice_InventoryField extends Vtiger_Basic_InventoryField
{
	protected $type = 'GrossPrice';
	protected $defaultLabel = 'LBL_GROSS_PRICE';
	protected $defaultValue = 0;
	protected $columnName = 'gross';
	protected $dbType = 'decimal(28,8) DEFAULT 0';
	protected $summationValue = true;
	protected $maximumLength = '99999999999999999999';
	protected $purifyType = Purifier::NUMBER;
	/** {@inheritdoc} */
	protected $params = ['summary_enabled'];
	protected bool $calculated = true;

	/** {@inheritdoc} */
	public function getDisplayValue($value, array $rowData = [], bool $rawText = false)
	{
		$value = Double::formatToDisplay($value);
		if (isset($rowData['currency']) && $currencySymbol = Currency::getById($rowData['currency'])['currency_symbol'] ?? '') {
			$value = CurrencyField::appendCurrencySymbol($value, $currencySymbol);
		}

		return $value;
	}

	/** {@inheritdoc} */
	public function getEditValue(array $itemData, string $column = '')
	{
		$value = parent::getEditValue($itemData, $column);

		return Double::formatToDisplay($value);
	}

	/** {@inheritdoc} */
	public function getDBValue($value, ?string $name = '')
	{
		if (!isset($this->dbValue["{$value}"])) {
			$this->dbValue["{$value}"] = Double::formatToDb($value);
		}
		return $this->dbValue["{$value}"];
	}

	/** {@inheritdoc} */
	public function validate($value, string $columnName, bool $isUserFormat, $originalValue = null)
	{
		if ($isUserFormat) {
			$value = $this->getDBValue($value, $columnName);
			if (null !== $originalValue) {
				$originalValue = $this->getDBValue($originalValue, $columnName);
			}
		}
		if (!is_numeric($value)) {
			throw new Security("ERR_ILLEGAL_FIELD_VALUE||$columnName||$value", 406);
		}
		if ($this->maximumLength < $value || -$this->maximumLength > $value) {
			throw new Security("ERR_VALUE_IS_TOO_LONG||$columnName||$value", 406);
		}
		if (null !== $originalValue && !Validator::floatIsEqualUserCurrencyDecimals($value, $originalValue)) {
			throw new Security('ERR_ILLEGAL_FIELD_VALUE||' . ($columnName ?? $this->getColumnName()) . "||{$this->getModuleName()}||$value($originalValue)", 406);
		}
	}

	/** {@inheritdoc} */
	public function getValueForSave(array $item, bool $userFormat = false, ?string $column = null)
	{
		return $this->roundDecimal(
			static::getInstance($this->getModuleName(), 'NetPrice')->getValueForSave($item, $userFormat)
			+ static::getInstance($this->getModuleName(), 'Tax')->getValueForSave($item, $userFormat)
		);
	}

	/** {@inheritdoc} */
	public function compare($value, $prevValue, string $column): bool
	{
		return Validator::floatIsEqual((float) $value, (float) $prevValue, 8);
	}
}
