<?php

/**
 * Login password change action file.
 *
 * @package   Action
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\EventHandler;
use App\Exceptions\AppException;
use App\Exceptions\Security;
use App\Language;
use App\Log;
use App\Purifier;
use App\Request;
use App\Session;
use App\Utils\Tokens;

/**
 * Login password change action class.
 */
class Users_LoginPassChange_Action extends Users_Login_Action
{
	/** {@inheritdoc} */
	public function process(Request $request)
	{
		$response = new Vtiger_Response();
		$bruteForceInstance = Settings_BruteForce_Module_Model::getCleanInstance();
		$moduleName = $request->getModule();
		try {
			$bruteForceInstance = Settings_BruteForce_Module_Model::getCleanInstance();
			if ($bruteForceInstance->isActive() && $bruteForceInstance->isBlockedIp()) {
				$bruteForceInstance->incAttempts();
				throw new Security('LBL_IP_IS_BLOCKED', 406);
			}
			if ($request->isEmpty('token')) {
				throw new Security('ERR_NO_TOKEN', 405);
			}
			$token = $request->getByType('token', Purifier::ALNUM);
			$tokenData = Tokens::get($token) ?: Session::get($token);
			if (empty($tokenData)) {
				throw new Security('ERR_TOKEN_DOES_NOT_EXIST', 405);
			}
			$password = $request->getRaw('password');
			if ($password !== $request->getRaw('confirm_password')) {
				$response->setError(406, Language::translate('LBL_PASSWORD_SHOULD_BE_SAME', 'Users'));
			} else {
				$userRecordModel = Users_Record_Model::getInstanceById($tokenData['params'][0], $moduleName);
				$userRecordModel->set('changeUserPassword', true);
				$userRecordModel->set('user_password', $password);
				$userRecordModel->set('date_password_change', date('Y-m-d H:i:s'));
				$userRecordModel->set('force_password_change', 0);

				$eventHandler = new EventHandler();
				$eventHandler->setRecordModel($userRecordModel);
				$eventHandler->setModuleName('Users');
				$eventHandler->setParams(['action' => 'change']);
				$eventHandler->trigger('UsersBeforePasswordChange');

				$userRecordModel->save();
				$eventHandler->trigger('UsersAfterPasswordChange');

				$response->setResult(Language::translate('LBL_PASSWORD_SUCCESSFULLY_CHANGED', 'Users'));
				Session::set('UserLoginMessage', Language::translate('LBL_PASSWORD_SUCCESSFULLY_CHANGED', 'Users'));
				Session::set('UserLoginMessageType', 'success');
				Session::delete($token);
			}
		} catch (Throwable $exc) {
			$message = $exc->getMessage();
			if ($exc instanceof AppException) {
				$message = $exc->getDisplayMessage();
			}
			Log::warning($exc->getMessage() . PHP_EOL . $exc->__toString());
			$response->setError(400, $message);
			$bruteForceInstance->updateBlockedIp();
			if ($bruteForceInstance->isBlockedIp()) {
				$bruteForceInstance->sendNotificationEmail();
			} else {
				Users_Module_Model::getInstance($moduleName)->saveLoginHistory('', 'ERR_PASS_CHANGE_IP_BLOCK');
			}
		}
		$response->emit();
	}
}
