<?php
/**
 * Map driver configuration modal view.
 *
 * @package Settings.View
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

use App\Exceptions\NoPermittedForAdmin;
use App\Language;
use App\Map;
use App\Purifier;

/**
 * Map driver configuration modal view class.
 */
class Settings_Map_ConfigModal_View extends \App\Controller\ModalSettings
{
	/**
	 * Tab.
	 *
	 * @var string
	 */
	private string $tab;

	/**
	 * Map driver.
	 *
	 * @var Map\BaseDriver
	 */
	private Map\BaseDriver $driver;

	/**
	 * Module name.
	 *
	 * @var string
	 */
	private string $module;

	/**
	 * @inheritdoc
	 *
	 * @throws NoPermittedForAdmin
	 */
	public function preProcessAjax(App\Request $request): void
	{
		$this->module = $request->getModule(false);
		$this->tab = $request->getByType('tab', Purifier::STANDARD);
		$connectorClass = match ($this->tab) {
			'TileLayer' => Map\Layer::class,
			'Coordinates' => Map\Coordinates::class,
			'Routing' => Map\Routing::class,
			'default' => throw new NoPermittedForAdmin('ERR_ILLEGAL_VALUE'),
		};

		$this->driver = (new $connectorClass)::getInstance($request->getByType('driver', Purifier::STANDARD));

		$this->pageTitle = Language::translate('LBL_PROVIDER_CONFIG', $this->module) . ': ' . Language::translate($this->driver->getLabel(), $this->module);

		parent::preProcessAjax($request);
	}

	/**
	 * @inheritdoc
	 */
	public function process(App\Request $request): void
	{
		$viewer = $this->getViewer($request);

		$viewer->assign('TAB', $this->tab);
		$viewer->assign('DRIVER', $this->driver);
		$viewer->assign('MODULE_NAME', $request->getModule());
		$viewer->assign('DRIVER_FIELDS', $this->getDriverFields());

		$viewer->view('ConfigModal.tpl', $this->module);
	}

	/**
	 * Get driver fields
	 *
	 * @return array
	 */
	public function getDriverFields(): array
	{
		$formFields = $this->driver->getFormFields();
		if(isset($formFields['api_key'])){
			$driverConfig = $this->driver->getConfig();
			$formFields['api_key']->set('tab', $this->tab);
			$formFields['api_key']->set('driver', $driverConfig['driver']);
		}
		return $formFields;
	}
}
