<?php
/**
 * Currency rate inventory info.
 *
 * @package TextParser
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\TextParser;

use App\Inventory\Currency\Conversion;

/**
 * Currency rate inventory info class.
 */
class CurrencyRateInventory extends Base
{
	/** @var string Class name */
	public $name = 'LBL_CURRENCY_RATE_INVENTORY';

	/** @var mixed Parser type */
	public $type = 'pdf';

	/** @var string Default template */
	public string $default = '$(custom : CurrencyRateInventory|__RATE_OR_DATE__|__CIT_OR_VAT__)$';

	/**
	 * Process.
	 *
	 * @return string
	 */
	public function process(): string
	{
		if (!$this->textParser->recordModel->getModule()->isInventory()) {
			return '';
		}

		$inventoryRows = $this->textParser->recordModel->getInventoryData();
		$row = reset($inventoryRows);
		if (!$inventoryRows || empty($row['currencyparam'])) {
			return '';
		}

		$fieldModel = \Vtiger_Inventory_Model::getInstance($this->textParser->recordModel->getModuleName())
			->getField('currency');
		$templateParams = $this->parseParams();
		$currencyParam = $fieldModel->createCurrencyParamObject($row['currencyparam']);

		/** @var Conversion $conversion */
		$conversion = match ($templateParams['conversionType']) {
			'CIT' => $fieldModel->getParamConfig('cit_exchange') ? $currencyParam->getCitFor($row['currency']) : null,
			'VAT' => $currencyParam->getVatFor($row['currency'])
		};

		if (null === $conversion) {
			return '';
		}

		return match ($templateParams['dataType']) {
			'DATE' => $conversion->getDateUserFormat(),
			'RATE' => $conversion->getValueUserFormat(),
		};
	}

	/**
	 * Parse template params.
	 *
	 * @return string[]
	 */
	private function parseParams(): array
	{
		[$dataType, $conversionType] = array_pad($this->params, 2, '');

		return [
			'dataType' => \in_array($dataType, ['DATE', 'RATE'], true) ? $dataType : 'RATE',
			'conversionType' => \in_array($conversionType, ['CIT', 'VAT'], true) ? $conversionType : 'VAT',
		];
	}
}
