<?php

/**
 * Abstract file for SSO login provider.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

namespace App\Authenticator;

/**
 * Abstract class for SSO provider.
 */
abstract class SSOProvider extends Password
{
	/** @var string The class added to the button in the login page. */
	public $btnClass = '';

	/**
	 * Get redirect URI.
	 * This URI should match the redirect URIs configured in your provider api platform.
	 *
	 * @var string
	 *
	 * @param int|null $serviceId
	 */
	public function getRedirectUri(?int $serviceId = null): string
	{
		$serviceId ??= $this->get('redirect_uri_id');
		$service = \App\Integrations\Services::getById($serviceId);
		$uri = '';
		if ($service && $service['status'] && \App\Integrations\Services::OAUTH === $service['type']) {
			$uri = \App\Config::main('site_URL') . "webservice/OAuth/SSO/{$this->getName()}/" . \App\Encryption::getInstance()->decrypt($service['api_key']);
		}

		return $uri;
	}

	/**
	 * Returns the base host for authorizing a client.
	 *
	 * @return string
	 */
	abstract public function getAuthorizationHost(): string;

	/**
	 * Begin the authorization process by preparing the environment and creating the authorization URL.
	 *
	 * @return string The URL to redirect to.
	 */
	abstract public function startAuthorization(): string;

	/**
	 * Finalize the authorization process by checking the authorization code and getting an access token.
	 * This method should return the email address of the user.
	 *
	 * @param string $code  Authorization code
	 * @param string $state OAuth2 state parameter
	 *
	 * @throws \App\Exceptions\Unauthorized If the authorization is in an invalid state that should not be reachable through normal use.
	 *
	 * @return int User ID
	 */
	abstract public function finishAuthorization(string $code, string $state): int;

	/** {@inheritdoc} */
	public function preProcess()
	{
		\App\Session::set('UserAuthMethod', 'SSO');
	}
}
