<?php namespace Winter\Storm\Halcyon;

use Illuminate\Cache\Repository;

/**
 * Provides a simple request-level cache.
 *
 * @author Alexey Bobkov, Samuel Georges
 */
class MemoryRepository extends Repository
{
    /**
     * Values stored in memory
     *
     * @var array
     */
    protected $cache = [];

    /**
     * Retrieve an item from the cache by key.
     *
     * @param  string|array $key
     * @param  mixed        $default
     */
    public function get($key, $default = null): mixed
    {
        if (is_array($key)) {
            return $this->many($key);
        }

        $value = $this->getFromMemoryCache($key);

        if (!is_null($value)) {
            return $value;
        }

        $value = parent::get($key, $default);
        $this->putInMemoryCache($key, $value);

        return $value;
    }

    /**
     * Store an item in the cache.
     *
     * @param  string|array  $key
     * @param  mixed   $value
     * @param  \DateTimeInterface|\DateInterval|int  $seconds
     * @return bool
     */
    public function put($key, $value, $seconds = null)
    {
        if (is_array($key)) {
            $this->putMany($key, $value);
        }

        $this->putInMemoryCache($key, $value);
        return parent::put($key, $value, $seconds);
    }

    /**
     * Increment the value of an item in the cache.
     *
     * @param  string  $key
     * @param  mixed  $value
     * @return int|bool
     */
    public function increment($key, $value = 1)
    {
        $newValue = parent::increment($key, $value);
        $this->putInMemoryCache($key, $newValue);
        return $newValue;
    }

    /**
     * Decrement the value of an item in the cache.
     *
     * @param  string  $key
     * @param  mixed  $value
     * @return int|bool
     */
    public function decrement($key, $value = 1)
    {
        $newValue = parent::decrement($key, $value);
        $this->putInMemoryCache($key, $newValue);
        return $newValue;
    }

    /**
     * Store an item in the cache indefinitely.
     *
     * @param  string  $key
     * @param  mixed   $value
     * @return bool
     */
    public function forever($key, $value)
    {
        $this->putInMemoryCache($key, $value);
        return parent::forever($key, $value);
    }

    /**
     * Remove an item from the cache.
     *
     * @param  string  $key
     * @return bool
     */
    public function forget($key)
    {
        unset($this->cache[$key]);
        return parent::forget($key);
    }


    /**
     * Remove all items from the cache.
     *
     * @return bool
     */
    public function flush()
    {
        $this->flushInternalCache();
        return parent::flush();
    }

    /**
     * Retrieve an item from the internal memory cache without trying the external driver.
     * Used in testing
     *
     * @param string $key
     * @return mixed|null
     */
    public function getFromMemoryCache($key)
    {
        return $this->cache[$key] ?? null;
    }

    /**
     * Puts an item in the memory cache, but not in the external cache.
     * Used in testing
     *
     * @param string $key
     * @param mixed $value
     */
    public function putInMemoryCache($key, $value)
    {
        $this->cache[$key] = $value;
    }

    /**
     * Flushes the memory cache.
     * Calling this directly is generally only useful in testing.
     * Use flush() otherwise.
     */
    public function flushInternalCache()
    {
        $this->cache = [];
    }

    /**
     * Get the cache key prefix.
     *
     * @return string
     */
    public function getPrefix()
    {
        return $this->store->getPrefix();
    }
}
