<?php

/**
 * This file is part of Webasyst framework.
 *
 * Licensed under the terms of the GNU Lesser General Public License (LGPL).
 * http://www.webasyst.com/framework/license/
 *
 * @link http://www.webasyst.com/
 * @author Webasyst LLC
 * @copyright 2011 Webasyst LLC
 * @package wa-system
 * @subpackage controller
 **/

/**
 * This controller helps to implement potentially long operations when it's impossible to
 * avoid max execution time limit.
 *
 * Each long operation is identified by processId. It is possible to run several processes with
 * different ids at the same time.
 *
 * Controller's entry point for this class expects a 'processId' get or post parameter.
 * It then becomes available as $this->processId. If id is not given in request,
 * a new process is started and $this->info() is responsible for returning id to user
 * for subsequent operations. $this->newProcess indicates whether current call
 * created the process (true) or not (false).
 *
 * Browser must request page from time to time keep the process alive,
 * as well as fetching info about its status.
 *
 * When a correct 'processId' get/post parameter is present, the call can either become a Runner
 * or a Messenger. There can only be one Runner at given time with given processId.
 * Runner is an instance that performs actual work. Runner works until he's done or until
 * he dies exceeding max execution time.
 *
 * While Runner is alive, all other possible instances will automatically become Messengers.
 * Messengers don't do real work but can access (and return to browser) some information about
 * Runner's status.
 * See $this->_obtainLock() for the inner mechanism of how Runner uniqueness is achieved.
 *
 * Runner performs work by repeatedly calling $this->step() in a loop,
 * while $this->isDone() returns false. See the actual loop in $this->execute().
 *
 * step(), isDone() and some other methods of this class run inside a transaction.
 * See comments for each method specifying transaction status.
 * Transactions guarantee data consistency regardless of when (or if) script fails.
 * $this->data is a persistent array and $this->fd is a file descriptor with this guarantee.
 * Only Runner has write access to this persistent data. All data in $this->data must be serializable.
 *
 * Transaction makes the following guarantees:
 * 1) If a script fails inside a transaction then all changes to $this->data and
 *    $this->fd file get reverted and are not visible inside subsequent transactions.
 * 2) If transaction completes successfully then all changes to $this->data and $this->fd
 *    are visible to subsequent transactions.
 * See $this->_save() and $this->_loadData() for the inner mechanism of how guarantees are achieved.
 *
 * Execution time for every function that runs inside a transaction must be reasonably small
 * for this class to be able to progress while keeping it's guarantees.
 * Reasonably small = no more than 10-20% of max execution time for each transaction (i.e. step()).
 *
 * As soon as all work is done and $this->isDone() returns true, all new calls become Messengers
 * and should return something to browser indicating success. For such Messengers $this->finish() is called first.
 * It may return true to make the class clean up all data of current process. Or ir may return false
 * to postpone cleaning-up, allowing more time (and more Messengers). This may be useful, for example,
 * to wait until user downloads data file generated by a long process.
 *
 * Note that for such Messengers, when finish() returned false, infoReady() is called instead of info()
 * to return data to browser.
 *
 * $this->max_exec_time is also available for reading. It contains max execution time for this script (false if unknown).
 *
 * @property-read string $processId
 * @property array $data
 * @property-read resource $fd
 * @property-read boolean $newProcess
 * @property-read int $current_exec_time    since 1.8.2
 * @property-read int $remaining_exec_time  since 1.8.2
 * @property-read int $max_exec_time
 * @property-read int $isRunner
 */
#[\AllowDynamicProperties]
abstract class waLongActionController extends waController
{
    const TYPE_RUNNER = 'runner';
    const TYPE_MESSENGER = 'messenger';
    const TYPE_NONE = 'no process';

    /**
     * @var waLongActionController
     */
    protected static $instance;

    /**
     * Checks if it's ok to initialize a new process.
     * @return boolean true if initialization can start
     */
    protected function preInit()
    {
        return true;
    }

    /**
     * Initializes new process.
     * Runs inside a transaction ($this->data and $this->fd are accessible).
     */
    abstract protected function init();

    /**
     * Checks if there is any more work for $this->step() to do.
     * Runs inside a transaction ($this->data and $this->fd are accessible).
     *
     * $this->getStorage() session is already closed.
     *
     * @return boolean whether all the work is done
     */
    abstract protected function isDone();

    /**
     * Performs a small piece of work.
     * Runs inside a transaction ($this->data and $this->fd are accessible).
     * Should never take longer than 3-5 seconds (10-15% of max_execution_time).
     * It is safe to make very short steps: they are batched into longer packs between saves.
     *
     * $this->getStorage() session is already closed.
     * @return boolean false to end this Runner and call info(); true to continue.
     */
    abstract protected function step();

    /**
     * Called when $this->isDone() is true
     * $this->data is read-only, $this->fd is not available.
     *
     * $this->getStorage() session is already closed.
     *
     * @param $filename string full path to resulting file
     * @return boolean true to delete all process files; false to be able to access process again.
     */
    abstract protected function finish($filename);

    /** Called by a Messenger when the Runner is still alive, or when a Runner
     * exited voluntarily, but isDone() is still false.
     *
     * This function must send $this->processId to browser to allow user to continue.
     *
     * $this->data is read-only. $this->fd is not available.
     */
    abstract protected function info();

    /**
     * Called by a new Runner when the old one dies.
     * Should be used to restore any non-persistent data for $this->step() if needed.
     * Runs inside a transaction ($this->data and $this->fd are accessible).
     * $this->getStorage() session is already closed.
     */
    protected function restore()
    {
    }

    /** Called instead of info() when isDone() is true but finish() returned false,
     * indicating that it's not ready to clean up data yet. This could return meaningful
     * data to browser.
     * $this->data is read-only, $this->fd is not available.
     * $this->getStorage() session is already closed.
     * @param $filename string full path to resulting file
     */
    protected function infoReady($filename)
    {
    }

    /**
     * Called by a Runner when PHP is just about to die of max_execution_time limit.
     * Class can be in any state. You should be careful and not make any assumptions
     * about where exactly the script died.
     *
     * Still, can be useful to send $this->info() to user from a dying runner.
     */
    protected function uncleanShutdown()
    {
    }

    //
    // Implementation details below this point
    //

    // actual source for $this->processId for __get()
    protected $_processId = 0;

    // persistent storage.
    protected $_data = array(
        'data'        => array(), // actual source for $this->data for __get() and __set()
        'avg_time'    => null, // average time in seconds between calls to $this->_save(), total_time/total_saves
        'total_saves' => 0,
        'total_time'  => 0,
        'total_steps' => 0,
        'heartbeat'   => false, // microtime of last save to disk
        'ready'       => false, // true if last isDone() call returned true
    );

    /**
     * actual source for $this->fd for __get()
     * @var resource
     */
    protected $_fd = null;

    // Whether we're currently inside a transaction
    protected $_transaction = false;

    // Whether it's a Runner object
    protected $_runner = false;

    // true if this Runner generated process id himself (didn't get it from request)
    protected $_newProcess = false;

    protected $_max_exec_time;
    protected $_chunk_time;

    // Files used by this class.
    // Actual filenames are filled in by $this->_initDataStructures() and $this->_obtainLock()
    protected $_files = array(
        'new'      => array(
            'data' => '', // file with $this->data serialized.
            'file' => '', // $this->fd points here. File is locked by a live Runner permanently.
        ),
        'old'      => array( // A second pair of data files to ensure persistence.
            'data' => '',
            'file' => '',
        ),
        'flock_ok' => '', // this file exists if we're sure that flock works in this system.
    );

    /**
     * Contains microtime of start of current script run.
     * Used to calculate $this->current_exec_time and $this->remaining_exec_time
     * @since 1.8.2
     * @var float
     */
    private $_start_time = 0.0;

    /**
     * Contains microtime and resets on start of the Runner and on each save() that writes to disk.
     * Used to measure total time of the process.
     * @var float
     */
    protected $_heartbeat = 0.0;

    /**
     *
     *
     * @var int
     */
    protected $_read_attempt_limit = 5;

    /**
     * @throws waException
     */
    public function execute()
    {
        $this->initEnv();

        // Get processId from GET/POST parameters
        foreach (array('processId', 'processid') as $field) {
            if (($this->_processId = waRequest::request($field))) {
                break;
            }
        }
        if (!$this->_processId) {
            if (!$this->preInit()) {
                return;
            }
            $this->_initDataStructures(); // it calls init() too
            $this->getStorage()->close();
        } else {
            $this->_newProcess = false;
            $status = $this->_obtainLock();
            if ($this->_data['ready']) {
                $this->_runner = false;
                if ($this->finish($this->_files['old']['file'])) {
                    $this->_cleanup();
                } else {
                    $this->infoReady($this->_files['old']['file']);
                }
                return;
            }
            switch ($status) {
                case self::TYPE_RUNNER:
                    ignore_user_abort(true);

                    if (self::$instance !== false) {
                        register_shutdown_function(array(__CLASS__, 'shutdown'));
                    }
                    self::$instance =& $this;
                    $this->_transaction = true;
                    $this->restore();
                    $this->_transaction = false;
                    $this->getStorage()->close();
                    break;
                case self::TYPE_MESSENGER:
                    $this->info();
                    return;
                case self::TYPE_NONE:
                    // must be a lost messenger
                    echo json_encode(array(
                        'ready'     => true,
                        'processId' => $this->_processId,
                    ));
                    return;
            }
        }

        // For new processes do not call step() in the first iteration
        // to be able to return processId to browser instantly.
        $continue = !$this->_newProcess;

        $this->_last_save_step = $this->_data['total_steps'];
        $this->_heartbeat = microtime(true);
        $this->_transaction = true;

        $is_done = $this->isDone();
        while ($continue && !$is_done) {
            $this->_data['total_steps']++;
            $continue = $this->step();
            $continue = $this->_save() && $continue;
            $is_done = $this->isDone();
        }

        // Remember the ready flag so that next attempt does not have to start a runner again
        if ($is_done && !$this->_data['ready']) {
            $this->_data['ready'] = true;
            $this->_save(true);
        }
        // Force a save if last call to _save() was not forced
        if ($this->_data['total_steps'] != $this->_last_save_step) {
            $this->_save(true);
        }

        $this->_runner = false;
        $this->_transaction = false;

        // Disable our shutdown handler at this point
        self::$instance = false;

        if ($is_done) {
            if ($this->finish($this->_files['old']['file'])) {
                $this->_cleanup();
            } else {
                $this->infoReady($this->_files['old']['file']);
            }
        } else {
            $this->info();
        }
    }

    /** Close $this->_fd and remove all files we created */
    protected function _cleanup()
    {
        if (is_resource($this->_fd)) {
            fflush($this->_fd);
            flock($this->_fd, LOCK_UN);
            fclose($this->_fd);
        }
        @unlink($this->_files['new']['data']);
        @unlink($this->_files['new']['file']);
        @unlink($this->_files['old']['data']);
        @unlink($this->_files['old']['file']); // could have been moved by finish()
        @unlink($this->_files['flock_ok']);
        rmdir(dirname($this->_files['old']['data']));
    }

    /** Creates private files and $this->... data structures for new process.
     * Initializes $this->_processId, $this->_data, $this->_fd, $this->_runner = true
     * Called once when a process is created. */
    protected function _initDataStructures()
    {
        // Generate new unique id
        $attempts = 3;
        $dir = waSystem::getInstance()->getTempPath('longop').'/';
        do {
            $attempts--;
            $id = uniqid();
        } while ($attempts >= 0 && !@mkdir($dir.$id, 0775));

        if ($attempts <= 0) {
            throw new waException('Unable to create unique dir in '.$dir);
        }

        $this->_newProcess = true;
        $this->_processId = $id;
        $this->_runner = true;

        // Create folder, locked files, unlocked files and data files
        $this->_files = $this->_getFilenames();
        touch($this->_files['new']['file']);
        touch($this->_files['old']['file']);

        // init $this->fd
        if (!($this->_fd = fopen($this->_files['new']['file'], 'a+b'))) {
            throw new waException('Unable to open file: '.$this->_files['new']['file']);
        }

        // $this->data is already fine, but we have to write data files
        $this->put($this->_files['new']['data'], 'garbage');

        // Allowing init() to modify $this->data before we first save it.
        $this->_transaction = true;
        $this->init();
        $this->save();
        $this->_transaction = false;
        $this->put($this->_files['old']['data'], $this->serializeData($this->_data));
    }

    /** Checks if there's a Runner for $this->processId.
     * If there is one then initializes $this->_data, $this->_runner = false
     * If there are no Runner then initializes $this->_data, $this->_fd, $this->_runner = true\
     *
     * @throws waException
     * @return string status of this instance: TYPE_RUNNER, TYPE_MESSENGER, or TYPE_NONE if data files not found.
     */
    protected function _obtainLock()
    {
        $this->_files = $this->_getFilenames();

        if (!file_exists($this->_files['new']['file'])) {
            return self::TYPE_NONE;
        }

        // Main Lock needs stats, so we load $this->data first
        $attempts_limit = max(1, $this->_read_attempt_limit);
        $attempts = $attempts_limit;
        $data = null;
        while ($attempts > 0) {
            if ($attempts < $attempts_limit) {
                usleep(mt_rand(500, 1500));
            }
            --$attempts;

            // Either 'new' data file, or 'old' data file has to be ok.
            // When both of them are corrupt, something went terribly wrong.
            foreach (array($this->_files['new']['data'], $this->_files['old']['data']) as $file) {
                if (!file_exists($file)) {
                    return self::TYPE_NONE;
                }

                if (!($fd = fopen($file, 'rb'))) {
                    continue;
                }
                if (!flock($fd, LOCK_SH)) {
                    fclose($fd);
                    continue;
                }
                $data = $this->unserializeData($this->get($file));
                if (!$data) {
                    fflush($fd);
                    flock($fd, LOCK_UN);
                    fclose($fd);
                    continue;
                }
                $this->_data = $data;
                fflush($fd);
                flock($fd, LOCK_UN);
                fclose($fd);
                break 2;
            }
        }
        if (!$data && $attempts <= 0) {
            throw new waException('Unable to read data from '.$this->_files['old']['data'].', '.$this->_files['new']['data'], 302);
        }

        if (!$this->_mainLock($this->_files['new']['file'], $this->_files['old']['file'])) {
            // A live Runner exists. We're the Messenger. $this->data is already loaded.
            $this->_runner = false;
            return self::TYPE_MESSENGER;
        }

        // We're the new Runner.

        $this->_runner = true;
        $this->_loadData();
        return self::TYPE_RUNNER;
    }

    /**
     *
     * @throws waException
     * @param $fileContents string data file contents
     * @return Array|boolean Unserialized array or false on failure
     **/
    protected function unserializeData($fileContents)
    {
        $arr = explode('#', $fileContents, 2);
        if (count($arr) != 2) {
            return false;
        }
        list($length, $serialized) = $arr;
        if (!$serialized || strlen($serialized) != $length) {
            return false;
        }
        $unserialized = unserialize($serialized);
        if (!$unserialized) {
            throw new waException('Unable to unserialize data with correct length.');
        }
        return $unserialized;
    }

    /** @param $array Array data to serialize
     *
     * @throws waException
     * @return String serialized data
     */
    protected function serializeData($array)
    {
        $serialized = serialize($array);
        if (!$serialized) {
            throw new waException('Unable to serialize '.print_r($array, true));
        }
        return strlen($serialized).'#'.$serialized;
    }

    /** Loads data from files using filenames from $this->_files
     * Makes sure both $this->_data and $this->_fd contain
     * non-corrupt data, restoring it if needed. */
    protected function _loadData()
    {
        if (!$this->_runner) {
            throw new waException('Cannot _loadData() for Messenger.');
        }

        // Invariant of $this->_save() ensures that when new_data unserializes successfully,
        // then new_file is ok; and when unserialization fails, then old_data and old_file
        // represent consistent state.
        $newData = $this->unserializeData($this->get($this->_files['new']['data']));
        if (!$newData) {
            // at this moment $this->_data must be loaded from old data at _obtainLock
            if (!$this->_data) {
                throw new waException('Both sets of data are corrupt in waLongActionController.');
            }

            // Just copy old file into new file, and we're ok.
            ftruncate($this->_fd, 0);
            fseek($this->_fd, 0);
            $fd2 = fopen($this->_files['old']['file'], 'rb');
            while (($c = fread($fd2, 8192))) {
                fwrite($this->_fd, $c);
            }
            fclose($fd2);
            return;
        }

        // Use new set of data. $this->fd is already good.
        $this->_data = $newData;
    }

    /** Return a new $this->_files structure using $this->processId */
    protected function _getFilenames()
    {
        $dir = waSystem::getInstance()->getTempPath('longop/'.$this->processId);
        return array(
            'new'      => array(
                'data' => $dir.'/new_data',
                'file' => $dir.'/new_file',
            ),
            'old'      => array(
                'data' => $dir.'/old_data',
                'file' => $dir.'/old_file',
            ),
            'flock_ok' => $dir.'/flock_ok',
        );
    }

    /** Saves output data chunk */
    protected function save()
    {

    }

    /**
     * Saves current persistent data.
     *
     * Non-forced saves will batch multiple steps into one write to disk, when steps are too short.
     *
     * Saving and loading process must ensure data consistency between transactions
     * no matter how and where transaction fails (including the saving process).
     *
     * Invariant:
     * 1) if new_data unserializes successfully, then new_data and new_file contain
     * consistent (non-corrupt) state of the process.
     * 2) if new_data fails to unserialize, then old_data and old_file contain
     * consistent state.
     */
    protected function _save($force = false)
    {
        $curTime = microtime(true);
        $force = $force || ($curTime - $this->_heartbeat) > $this->_chunk_time;

        if ($force) {

            //
            // At this point 'old' represents good data before current transaction even started;
            // new_data contains garbage; and new_file is in correct state for the end of current transaction.
            //

            fflush($this->_fd);
            $prev_heartbeat = $this->_data['heartbeat'];
            $this->_data['total_saves']++;
            $this->_data['total_time'] += $curTime - $this->_heartbeat;
            $this->_data['avg_time'] = $this->_data['total_time'] / $this->_data['total_saves'];
            $this->_data['heartbeat'] = $curTime;
            $this->_heartbeat = $curTime;

            $failed = (false === $this->put($this->_files['new']['data'], $this->serializeData($this->_data)));

            if ($failed) {
                // We can't save results of current runner to disk. Hopefully the problems are temporary,
                // and the subsequent save call will be able to continue. If problem persists, the new runner
                // will have to do the batch of work again. Oh, well.
                $this->runnerFatalWarning('Unable to save `new` data file: '.$this->_files['new']['data']);
                $this->_data['heartbeat'] = $prev_heartbeat + 2;
                $this->_data['total_saves']--;

                // save() can not continue, but current runner may.
                return @file_exists($this->_files['new']['data']);
            }

            //
            // Now at this time, 'new' represents good data after current transaction.
            // Now we also need to copy good data into 'old'.
            // (It is now safe to spoil 'old' since we have 'new', see invariant.)
            //

            $attempts = 3;
            do {
                $attempts--;
                $failed = !$this->copyLockedFile($this->_fd, $this->_files['new']['file'], $this->_files['old']['file']);
                $failed = !copy($this->_files['new']['data'], $this->_files['old']['data']) || $failed;

                //clearstatcache();
                //$failed = filesize($this->_files['old']['data']) <= 0 || $failed; // too risky to check this on windows

                if ($failed) {
                    clearstatcache();
                    usleep(mt_rand(300, 900));
                } else {
                    break;
                }
            } while ($attempts > 0);

            if ($failed) {
                // Well... We're currently in kinda ok state since at least we managed to
                // write the `new` set of files. Nothing is lost so far. Hopefully,
                // $this->_loadData() will be able to continue from there.
                $this->runnerFatalWarning('Unable to save `old` set of data files: '.$this->_files['old']['file'].' and '.$this->_files['old']['data']);

                // Current runner can not continue without spoiling the only copy of data we have.
                return false;
            }

            //
            // Now both 'new' and 'old' represent good data after current transaction.
            //

            // Putting garbage in new_data ensures that invariant stays true,
            // while step() works on new_file inside a transaction, possibly spoiling it.
            $this->put($this->_files['new']['data'], 'garbage');

            // Reset file position in $this->fd to EOF. Since we don't save file position
            // between different Runner instances, it's more consistent just to reset it every time.
            fseek($this->_fd, -1, SEEK_END);
            if ($this->_runner && $this->_transaction) {
                $this->save();
            }

            $this->_last_save_step = $this->_data['total_steps'];
        }

        // We're ok, current Runner may continue.
        return true;
    }

    /**
     * PHP 8+ on Windows will refuse to copy() a flock()'ed file.
     * This implements a separate logic to copy from file descriptor.
     */
    protected function copyLockedFile($fd, $source, $dest)
    {
        if (PHP_MAJOR_VERSION >= 8 && PHP_OS_FAMILY == 'Windows') {
            // don't have to fseek() back because _save() resets file position anyway
            fseek($fd, 0);
            $fd2 = fopen($dest, 'wb');
            while ( ( $c = fread($fd, 8192))) {
                if (strlen($c) != fwrite($fd2, $c)) {
                    fclose($fd2);
                    return false;
                }
            }
            fclose($fd2);
            return true;
        } else {
            return copy($source, $dest);
        }
    }

    /**
     * Called when something went badly wrong so that current Runner should not continue,
     * (at least should immediately notify a developer if in debug mode).
     * Still, not badly enough yet to break the whole long action process.
     */
    protected function runnerFatalWarning($msg)
    {
        if (waSystemConfig::isDebug()) {
            // In debug mode, better to fail fast.
            throw new waException($msg);
        } else {
            waLog::log('waLongActionController ('.get_class($this).'): '.$msg);
        }
    }

    /** Our own robust file locking mechanism.
     * Best we can afford still being compatible with everything.
     * Never blocks, except on (damned) Windows in rare circumstances.
     *
     * @throws waException
     * @param string $filename
     * @param string $filename2
     * @return boolean true if lock is obtained, false otherwise
     */
    protected function _mainLock($filename, $filename2)
    {
        // On windows, check if file was not modified recently.
        // We need this because flock() on windows always blocks, regardless of LOCK_NB option.
        // Without this all our messengers would hang and JS would receive no data at all.
        $waitTime = $this->getLockWaitTime();
        if (!$this->_newProcess && !file_exists($this->_files['flock_ok']) && strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            if (!$this->_newProcess && !file_exists($this->_files['flock_ok']) && time() < filemtime($filename) + $waitTime) {
                // Recent modification found. Lock failed.
                return false;
            }
        }

        $this->_fd = fopen($filename, 'a+b');

        // Okay. It seems to be safe to try to obtain flock.
        if (!flock($this->_fd, LOCK_EX | LOCK_NB)) { // On windows it's possible to hang here in flock. Have to live with that.
            // flock failed! Now we're sure it works, so we won't need to check
            // file modification time again.
            if (!touch($this->_files['flock_ok'])) {
                $this->runnerFatalWarning('Unable to create file: '.$this->_files['flock_ok']);
            }
            fflush($this->_fd);
            flock($this->_fd, LOCK_UN); // being paranoid
            fclose($this->_fd);
            $this->_fd = null;
            return false;
        }

        // We've got the flock. But this doesn't mean anything yet :(
        // On some combinations of system and HTTP server flock does not
        // work reliably. We're sure we can trust flock only if we've already seen it working
        // OR if we've just created unique processId and nobody else knows it yet.
        if ($this->_newProcess || file_exists($this->_files['flock_ok'])) {
            // flock is ok in this system, so we're sure there is no other Runner.
            return true;
        }

        // Bad luck, we cannot trust the flock.
        // We'll wait some reasonable time making sure nobody writes to the second file.
        $time = filemtime($filename2);
        while (time() <= $time + $waitTime) {
            sleep(1);
            // Since we're holding the flock(), another Messenger may discover that flock() can be trusted.
            // Then we don't have to wait any longer. In real life this never happens because of
            // the default session locking mechanism though...
            clearstatcache();
            if (file_exists($this->_files['flock_ok'])) {
                return true;
            }
        }

        // Fail to lock if the second file was modified recently.
        if (time() < filemtime($filename2) + $waitTime) {
            // Recent modification found. Releasing...
            fflush($this->_fd);
            flock($this->_fd, LOCK_UN); // being paranoid
            fclose($this->_fd);
            $this->_fd = null;
            return false;
        }

        // A race condition is possible here between where we check filemtime()
        // and where we touch() the file. However, since we close session only
        // after obtaining this lock (and default sessions have their own internal not file-based lock),
        // this serves as a pretty safe back up plan. Still not perfect though.
        touch($filename2);

        // So, finally, we're sure there's no other Runner.
        return true;
    }

    /** Helper for _mainLock(), see usage there. */
    protected function getLockWaitTime()
    {
        // This trick considerably speeds up first runner startup time when we can not trust the flock() yet.
        if ($this->_data['total_saves'] < 3) {
            $this->_chunk_time = min($this->_chunk_time, 2);
        }

        return max($this->_data['total_saves'] > 0 ? $this->_data['avg_time'] : 0, $this->_chunk_time) * 2.5;
    }

    public function &__get($field)
    {
        switch ($field) {
            case 'data':
                if ($this->_processId && $this->_runner && !$this->_transaction) {
                    throw new waException('Data is only accessible inside a transaction.');
                }
                return $this->_data['data']; // by reference
            case 'fd':
                if (!$this->_transaction) {
                    throw new waException('File is only accessible inside a transaction.');
                }
                if (!$this->_runner) {
                    throw new waException('File is only accessible by a Runner.');
                }
                $fd = $this->_fd;
                return $fd; // not by reference
            case 'isRunner':
                $is_runner = $this->_runner;
                return $is_runner; // not by reference
            case 'processId':
                $pid = $this->_processId;
                return $pid; // not by reference
            case 'newProcess':
                $np = $this->_newProcess;
                return $np; // not by reference
            case 'max_exec_time':
                $np = $this->_max_exec_time;
                return $np; // not by reference
            case 'current_exec_time':
                $np = microtime(true) - $this->_start_time;
                return $np; // not by reference
            case 'remaining_exec_time':
                $np = $this->_max_exec_time - (microtime(true) - $this->_start_time);
                return $np; // not by reference
            default:
                throw new waException('Unknown property: '.$field);
        }
    }

    public function __set($field, $value)
    {
        switch ($field) {
            case 'data':
                if ($this->_processId && !$this->_transaction) {
                    throw new waException('Data can only be changed inside a transaction.');
                }
                if ($this->_processId && !$this->_runner) {
                    throw new waException('Data can only be changed by a Runner.');
                }
                if (!is_array($value)) {
                    throw new waException('Data must be an array.');
                }
                $this->_data['data'] = $value;
                return;
            case 'isRunner':
                throw new waException('isRunner is read-only.');
            case 'fd':
                throw new waException('File descriptor is read-only.');
            case 'processId':
                throw new waException('processId is read-only.');
            case 'newProcess':
                throw new waException('newProcess is read-only.');
        }

        // Other not-existing fields are ok.
        $this->$field = $value;
    }

    protected function put($filename, $data)
    {
        $retry = 5;
        while (true) {
            $retry--;
            $success = @file_put_contents($filename, $data);
            if ($success !== false) {
                return $success;
            } elseif ($retry <= 0) {
                return false;
            } else {
                sleep(1);
                waFiles::create(dirname($filename), true);
            }
        }
    }

    protected function get($filename)
    {
        $retry = 5;
        while (($res = @file_get_contents($filename)) === false) {
            if (!$retry--) {
                break;
            }
            sleep(1);
        }
        return $res;
    }

    public static function shutdown()
    {
        if (!empty(self::$instance) && (self::$instance instanceof waLongActionController)) {
            self::$instance->uncleanShutdown();
            self::$instance = null;
        }
    }

    protected function initEnv()
    {
        $this->_start_time = microtime(true);

        // We'll try to increase execution time limit.
        // It doesn't always work, but it doesn't hurt either.
        // (Less than 5 minutes - common timeout for nginx)
        if (function_exists('set_time_limit')) {
            @set_time_limit(287);
        }

        // How much time we can safely run?
        $this->_max_exec_time = ini_get('max_execution_time');
        if ($this->_max_exec_time <= 0) {
            $this->_max_exec_time = 300;
        }

        // Depending on total time limit, determine how often we'd like to write data to file.
        $this->_chunk_time = min(9, $this->_max_exec_time / 6);
    }
}

// EOF
