<?php

class siteCustomAdvantagesBlockType extends siteBlockType {
    /**
     * Элементы блока
     */
    protected array $elements = [
        'main'    => 'site-block-columns',
        'wrapper' => 'site-block-columns-wrapper',
    ];

    public function __construct(array $options = []) {
        if (!isset($options['columns']) || !wa_is_int($options['columns'])) {
            $options['columns'] = 1;
        }
        $options['type'] = 'site.CustomAdvantages';
        parent::__construct($options);
    }

    /**
     * Возвращает данные для примера блока
     *
     * @return siteBlockData
     */
    public function getExampleBlockData(): siteBlockData {
        try {
            // Создаём основной блок
            $result = $this->getEmptyBlockData();

            // Создаём горизонтальную последовательность для колонок
            $hseq = $this->createHorizontalSequence();

            // Добавляем преимущества в горизонтальную последовательность
            $advantages = $this->getDefaultAdvantages();
            foreach ($advantages as $advantage) {
                $hseq->addChild($this->getAdvantage($advantage['heading'], $advantage['paragraph'], $advantage['icon']));
            }

            // Добавляем горизонтальную последовательность в основной блок
            $result->addChild($hseq, '');

            // Настраиваем свойства основного блока
            $block_props = [];
            $block_props[$this->elements['main']] = [
                'padding-top'    => 'p-t-18',
                'padding-bottom' => 'p-b-18',
                'padding-left' => 'p-l-blc',
                'padding-right' => 'p-r-blc',
            ];
            $block_props[$this->elements['wrapper']] = [
                'padding-top'    => 'p-t-12',
                'padding-bottom' => 'p-b-12',
                'flex-align'     => 'y-c',
                'max-width'      => 'cnt',
            ];

            $result->data = [
                'block_props'   => $block_props,
                'wrapper_props' => ['justify-align' => 'y-j-cnt'],
                'elements'      => $this->elements,
            ];

            return $result;
        } catch (Exception $e) {
            waLog::log($e->getMessage());
            return $this->getEmptyBlockData();
        }
    }

    /**
     * Рендеринг блока
     *
     * @param siteBlockData $data
     * @param bool          $is_backend
     * @param array         $tmpl_vars
     * @return string
     */
    public function render(siteBlockData $data, bool $is_backend, array $tmpl_vars = []): string {
        return parent::render($data, $is_backend, $tmpl_vars + [
                'children' => array_reduce($data->getRenderedChildren($is_backend), 'array_merge', []),
            ]);
    }

    /**
     * Возвращает конфигурацию формы настроек блока
     *
     * @return array
     */
    public function getRawBlockSettingsFormConfig(): array {
        return [
                'type_name'    => _w('Block'),
                'type_name_original'    => _w('Advantages'),
                'sections'     => [
                    [
                        'type' => 'ColumnsGroup',
                        'name' => _w('Columns'),
                    ],
                    [
                        'type' => 'RowsAlignGroup',
                        'name' => _w('Columns alignment'),
                    ],
                    [
                        'type' => 'RowsWrapGroup',
                        'name' => _w('Wrap line'),
                    ],
                    [
                        'type' => 'TabsWrapperGroup',
                        'name' => _w('Tabs'),
                    ],
                    [   'type' => 'CommonLinkGroup',
                        'name' => _w('Link or action'),
                        'is_hidden' => true,
                    ],
                    [
                        'type' => 'MaxWidthToggleGroup',
                        'name' => _w('Max width'),
                    ],
                    [
                        'type' => 'BackgroundColorGroup',
                        'name' => _w('Background'),
                    ],
                    [   'type' => 'HeightGroup',
                        'name' => _w('Height'),
                    ],
                    [
                        'type' => 'PaddingGroup',
                        'name' => _w('Padding'),
                    ],
                    [
                        'type' => 'MarginGroup',
                        'name' => _w('Margin'),
                    ],
                    [
                        'type' => 'BorderGroup',
                        'name' => _w('Border'),
                    ],
                    [
                        'type' => 'BorderRadiusGroup',
                        'name' => _w('Angle'),
                    ],
                    [
                        'type' => 'ShadowsGroup',
                        'name' => _w('Shadows'),
                    ],
                    [
                        'type' => 'IdGroup',
                        'name' => _w('Identifier (ID)'),
                    ],
                ],
                'elements'     => $this->elements,
                'semi_headers' => [
                    'main'    => _w('Whole block'),
                    'wrapper' => _w('Container'),
                ],
            ] + parent::getRawBlockSettingsFormConfig();
    }

    /**
     * Создаёт блок с преимуществом
     *
     * @param string $heading
     * @param string $paragraph
     * @return siteBlockData
     */
    private function getAdvantage(string $heading, string $paragraph, string $icon): siteBlockData {
        // Создаём вертикальную последовательность для содержимого колонки
        $vseq = $this->createVerticalSequence();

        // Добавляем заголовок и параграф в вертикальную последовательность
        $vseq->addChild($this->getIcon($icon));
        $vseq->addChild($this->getHeading($heading));
        $vseq->addChild($this->getParagraph($paragraph));

        // Создаём колонку и добавляем в неё вертикальную последовательность
        $column = $this->getColumn();
        $column->addChild($vseq, '');

        return $column;
    }

    /**
     * Создаёт колонку
     *
     * @return siteBlockData
     */
    private function getColumn(): siteBlockData {
        $column = (new siteColumnBlockType())->getEmptyBlockData();

        $column_elements = [
            'main'    => 'site-block-column',
            'wrapper' => 'site-block-column-wrapper',
        ];

        $column->data = [
            'elements'       => $column_elements,
            'column'         => "st-3 st-3-lp st-6-tb st-12-mb",
            'indestructible' => false,
            'block_props'    => [
                $column_elements['main']    => [
                    'padding-top'    => "p-t-12",
                    'padding-bottom' => "p-b-12",
                    'padding-left' => 'p-l-clm',
                    'padding-right' => 'p-r-clm',
                ],
                $column_elements['wrapper'] => [
                    'flex-align'     => "y-c",
                    'padding-top'    => "p-t-12",
                    'padding-bottom' => "p-b-12",
                ],
            ],
        ];

        return $column;
    }

    /**
     * Создаёт параграф с текстом
     *
     * @param string $content
     * @return siteBlockData
     */
    private function getParagraph(string $content): siteBlockData {
        $paragraph = (new siteParagraphBlockType())->getEmptyBlockData();

        $paragraph->data = [
            'html'        => $content,
            'tag'         => 'p',
            'block_props' => [
                'font-header'   => 't-rgl',
                'font-size'     => ['name' => 'Size #6', 'value' => 't-6', 'unit' => 'px', 'type' => 'library'],
                'margin-top'    => 'm-t-0',
                'margin-bottom' => 'm-b-12',
                'align'         => 't-l',
            ],
        ];

        return $paragraph;
    }

    /**
     * Создаёт заголовок с текстом
     *
     * @param string $content
     * @return siteBlockData
     */
    private function getHeading(string $content): siteBlockData {
        $header = (new siteHeadingBlockType())->getEmptyBlockData();

        $header->data = [
            'html'        => $content,
            'tag'         => 'h1',
            'block_props' => [
                'font-header'   => 't-hdn',
                'font-size'     => ['name' => 'Size #5', 'value' => 't-5', 'unit' => 'px', 'type' => 'library'],
                'margin-top'    => 'm-t-0',
                'margin-bottom' => 'm-b-8',
                'align'         => 't-l',
            ],
        ];

        return $header;
    }

    /**
     * Создаёт иконку
     *
     * @param string $content
     * @return siteBlockData
     */
    private function getIcon(string $svg_html): siteBlockData {
        $icon = (new siteImageBlockType())->getEmptyBlockData();

        $icon->data = [
            'image'       => [
                'type'     => 'svg',
                'svg_html' => $svg_html,
            ],
            'block_props' => [
                'margin-bottom' => 'm-b-14',
            ],
        ];

        return $icon;
    }

    /**
     * Создаёт горизонтальную последовательность
     *
     * @return siteBlockData
     */
    private function createHorizontalSequence(): siteBlockData {
        $hseq = (new siteVerticalSequenceBlockType())->getEmptyBlockData();
        $hseq->data['is_horizontal'] = true;
        $hseq->data['is_complex'] = 'only_columns';
        $hseq->data['indestructible'] = true;

        return $hseq;
    }

    /**
     * Создаёт вертикальную последовательность
     *
     * @return siteBlockData
     */
    private function createVerticalSequence(): siteBlockData {
        $vseq = (new siteVerticalSequenceBlockType())->getEmptyBlockData();
        $vseq->data['is_complex'] = 'with_row';

        return $vseq;
    }

    /**
     * Возвращает массив с данными преимуществ по умолчанию
     *
     * @return array
     */
    private function getDefaultAdvantages(): array {
        return [
            [
                'heading'   => 'Sed do eiusmod tempor incididunt',
                'paragraph' => 'Enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat.',
                'icon'      => '<svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="64" height="64" fill="white"/>
<path d="M31.9286 6.87433C32.3149 6.80548 32.9224 7.04888 33.2922 7.1827C34.2114 7.51532 35.0849 7.98953 35.9856 8.36967C38.1946 9.30353 40.4424 10.2073 42.6611 11.1082C43.126 11.2973 43.6948 11.2614 44.1354 11.4884C46.9132 12.9187 50.3577 13.3512 53.154 14.5052C53.5296 14.6604 53.9343 14.8737 54.2196 15.1685C54.6729 15.6369 54.7069 16.5252 54.7146 17.142C54.8796 28.9631 54.4108 39.7806 45.6 48.7672C44.1946 50.2014 42.568 51.3409 40.9413 52.4997C40.6123 52.7334 40.3143 53.0341 39.9688 53.2396C39.6815 53.4113 39.3534 53.4957 39.0613 53.6528C36.9949 54.7641 34.4472 56.6657 32.0295 56.6725C31.7976 56.6696 31.5433 56.6764 31.3152 56.6269C30.1126 56.368 28.882 55.697 27.7794 55.1588C22.4364 52.5531 17.7428 48.8225 14.5031 43.7856C10.9275 38.2261 9.59162 31.5795 9.32306 25.057C9.27123 23.7992 9.31957 22.5298 9.31501 21.2701C9.30918 19.6546 9.06955 17.4542 9.44526 15.9094C9.55309 15.4653 9.71255 15.1937 10.0845 14.9261C11.1153 14.1852 15.8856 13.0138 17.3717 12.5628C18.0535 12.3553 18.8467 12.1934 19.4534 11.8132C20.0614 11.4321 20.7923 11.5116 21.4241 11.1722C22.9518 10.3518 25.3899 9.59542 27.1039 8.90011C28.7276 8.24166 30.2573 7.39411 31.9286 6.87433ZM32.1334 53.9766L32.2819 53.9049C32.4236 53.837 32.5624 53.7856 32.7109 53.7352C33.4757 53.4753 34.2201 53.0505 34.9548 52.7102C40.82 49.9939 45.7262 45.5573 48.5661 39.7059C51.3526 33.9631 51.835 27.6239 51.8214 21.3409C51.8185 19.9959 51.9475 18.5558 51.7641 17.2302C51.5807 17.0227 51.3642 16.9577 51.1109 16.8646C49.4231 16.2478 47.6266 15.8076 45.9096 15.2539C44.8672 14.9164 43.8996 14.4723 42.8679 14.1164C41.7294 13.7236 40.4997 13.3774 39.4252 12.8334C37.1153 11.6619 34.4724 10.9977 32.2062 9.71858L31.9451 9.76318C27.0204 12.2157 21.7764 14.1784 16.5403 15.8435C15.328 16.2285 14.0795 16.5048 12.8649 16.8811C12.683 16.9373 12.3121 17.0392 12.2561 17.2409C12.0166 18.1068 12.1807 20.4769 12.183 21.4883C12.1991 28.5917 12.8429 35.8046 16.7402 41.9538C19.8082 46.7938 24.1766 50.3653 29.344 52.7945C30.1505 53.1737 31.1677 53.7681 32.0266 53.9572C32.0615 53.965 32.0975 53.9699 32.1334 53.9766Z" fill="black"/>
<path d="M41.8013 24.5044C42.0372 24.4782 42.3079 24.4598 42.5419 24.5141C42.9078 24.5985 43.2979 25.0223 43.4629 25.3481C43.5794 25.5799 43.5377 25.8844 43.4717 26.1249C43.2455 26.9462 41.0666 29.3483 40.3057 29.7672C40.2639 29.7895 40.2164 29.8021 40.1814 29.8341C40.1504 29.8642 40.1416 29.9117 40.1145 29.9456C40.0834 29.9835 40.0397 30.0106 40.0077 30.0475C39.9359 30.1318 39.8825 30.2298 39.8126 30.3142C39.2604 30.9736 38.5654 31.5777 37.9569 32.1867L33.9446 36.1995C33.0245 37.1179 31.5288 38.8236 30.6126 39.5151C30.3729 39.6964 30.0574 39.7459 29.7624 39.7604C29.544 39.7711 29.2227 39.775 29.0199 39.6828C27.9707 39.2057 24.566 35.0688 23.4595 34.0816L23.344 33.9788C22.9315 33.4367 22.223 32.9955 21.9687 32.3477C21.8542 32.0548 21.8794 31.7678 22.0192 31.4875C22.2084 31.1093 22.6083 30.6865 23.0247 30.5663C23.1907 30.5178 23.445 30.478 23.6148 30.5285C24.3175 30.7389 27.4864 34.2057 28.1939 34.9728C28.5414 35.35 29.1218 36.2121 29.5819 36.407C29.7886 36.4953 30.5 35.607 30.6475 35.448C32.3606 33.5899 34.2571 31.8483 36.042 30.0523C37.7375 28.3466 39.3895 26.5128 41.1685 24.902C41.3587 24.7294 41.5645 24.6024 41.8013 24.5044Z" fill="black"/>
</svg>',
            ],
            [
                'heading'   => 'Aenean quis finibus arcu, in hendrerit purus',
                'paragraph' => 'Praesent ac aliquet lorem. Morbi feugiat aliquam ligula, et vestibulum ligula hendrerit vitae. Sed ex lorem',
                'icon'      => '<svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="64" height="64" fill="white"/>
<path d="M38.0796 6.9107C38.7773 6.85102 39.4208 6.85466 40.099 7.05341C43.2 7.96222 43.2577 12.1171 44.6309 14.5354C44.8878 14.988 44.8931 15.4394 45.0851 15.9038C45.1802 16.1339 45.3464 16.3145 45.4281 16.5554C45.5543 16.9261 45.621 17.3307 45.7721 17.6918C45.8583 17.8989 45.981 18.0052 46.2041 18.0543C46.4965 18.1188 46.9328 18.062 47.2386 18.0618C48.6926 18.061 49.969 18.0312 51.3199 18.6415C52.1092 18.9981 52.7446 19.4929 53.3348 20.1221C55.4661 22.3979 54.9008 25.4183 54.8822 28.294C54.8786 28.8893 54.7746 29.7435 54.9808 30.2972C55.4892 30.3447 55.8207 30.451 56.1843 30.8244C56.3398 30.9845 56.8037 31.4576 56.8473 31.6663C57.0277 32.5222 56.8802 34.1874 56.8802 35.1241C56.8811 37.8788 56.9504 40.655 56.8544 43.407C56.2971 44.1766 55.7932 44.5269 54.94 44.9093C54.7613 45.5208 54.8964 46.3545 54.9008 46.9991C54.9239 49.8743 55.331 52.9056 53.1019 55.1243C52.2505 55.9714 51.2399 56.5257 50.0641 56.7614C48.0749 57.1604 44.1305 56.938 41.9805 56.9378L25.5593 56.934L16.5025 56.9414C14.9658 56.942 13.3509 57.0512 11.8284 56.8833C11.1005 56.803 10.2934 56.59 9.64463 56.2449C8.64741 55.7148 7.48577 54.2474 7.21114 53.1651C6.68142 51.0866 6.93561 48.3137 6.93827 46.1484L6.94005 36.84L6.93738 27.5588C6.93738 25.8905 6.85028 24.1665 6.97649 22.5046C7.01915 21.9407 7.2058 20.8481 7.5151 20.3767C7.92838 19.7454 8.70786 18.7458 9.36556 18.3866C10.4428 17.7984 11.9564 17.3777 13.1171 16.9252C15.5835 15.9644 18.0303 14.9504 20.487 13.9644C24.5997 12.2785 28.7228 10.6186 32.8562 8.98461C34.4054 8.36538 36.2167 7.38517 37.821 7.0542C37.8743 7.0431 37.9285 7.03334 37.9828 7.02295L38.0796 6.9107ZM52.1438 44.8971C50.6293 44.8889 49.1151 44.8897 47.6012 44.8994C46.3391 44.9015 43.5235 45.1385 42.5067 44.5836C42.0552 44.3373 41.5415 44.2841 41.0953 44.0568C40.8029 43.9073 40.5069 43.6045 40.2501 43.3902C38.9542 42.3114 38.0779 41.2617 37.581 39.6262C37.0362 37.8328 37.2451 35.6592 38.1347 34.0097C39.1035 32.2142 40.8686 30.843 42.84 30.313C44.3483 29.9076 50.1823 30.1382 52.1136 30.1436C52.4007 29.6272 52.3651 23.8516 52.1598 23.1313C51.9643 22.4455 51.1804 21.4675 50.52 21.1473C50.0943 20.9414 49.5699 20.7623 49.0988 20.728C46.4947 20.5384 43.5164 20.6976 40.8687 20.6995L24.8554 20.7067L15.5293 20.7024C14.1321 20.6992 12.4461 20.5238 11.0747 20.7639C11.0569 20.7816 11.0409 20.8 11.0223 20.8169C10.5041 21.3006 9.78594 21.5429 9.75128 22.3702C9.65973 24.5918 9.74329 26.8466 9.74329 29.0723L9.74418 42.3273L9.74596 49.3005C9.74596 50.2909 9.62776 51.4266 9.76996 52.3946C9.83751 52.855 10.217 53.2698 10.529 53.592C10.7707 53.8414 10.92 53.9814 11.2693 54.0306C11.3218 54.038 11.3733 54.0474 11.4249 54.0555C12.9065 54.2845 14.5925 54.1395 16.0928 54.1383L25.2242 54.1308L41.7334 54.136C43.9643 54.1371 48.3585 54.379 50.3502 53.9158C50.3893 53.9069 50.4285 53.9013 50.4667 53.8922C50.9502 53.7773 51.6461 52.8951 51.8959 52.4631C52.0496 52.1972 52.166 51.9139 52.2105 51.6086C52.3394 50.7271 52.3945 45.8242 52.2176 45.1057C52.1971 45.0245 52.1749 44.978 52.1438 44.8971ZM41.7006 41.3424C42.4329 41.7701 43.1315 42.1035 43.9687 42.275C45.2308 42.5337 46.6449 42.4144 47.931 42.4122L54.2342 42.3969C54.316 42.2528 54.3658 42.1509 54.3782 41.9825C54.4876 40.4666 54.3978 38.876 54.3978 37.3532C54.3987 35.8811 54.5009 34.3254 54.3631 32.8625C54.2031 32.6977 54.1098 32.667 53.8787 32.6509C53.7712 32.6434 53.6574 32.6562 53.5499 32.6638C53.3401 32.8398 52.534 32.759 52.2318 32.7599L47.5763 32.7551C46.3143 32.7509 44.846 32.6027 43.607 32.8578C41.7352 33.2436 40.1283 35.0289 39.9461 36.9155C39.7897 38.5338 40.3087 39.6893 41.3299 40.913C41.4401 41.0441 41.5415 41.1538 41.6819 41.2524L41.7006 41.3424ZM22.2708 18.0673L36.7918 18.0681L40.7656 18.0677C41.4721 18.0676 42.264 18.133 42.9493 18.0148C42.9911 18.0076 43.0835 18.0262 43.1279 18.0339C43.0417 17.82 42.9715 17.6164 42.9155 17.3925C42.7636 16.7864 42.4187 16.2135 42.1956 15.629C41.7725 14.524 40.5816 10.1778 39.7034 9.68287C39.331 9.47267 38.8644 9.51609 38.4609 9.57435C38.2272 9.79502 22.0006 16.3335 20.4212 16.8955C19.8826 17.0872 19.344 17.3604 18.8418 17.6317C18.5618 17.7825 18.2081 17.7486 17.9664 17.9533L17.993 18.0391L18.0597 17.9846L18.3183 17.9481L18.3796 18.0907L22.2708 18.0673Z" fill="black"/>
<path d="M45.2419 35.6834C45.5343 35.6519 45.8925 35.7692 46.1627 35.876C46.6515 36.0689 47.063 36.4272 47.2541 36.9247C47.2648 36.9519 47.2745 36.9793 47.2834 37.0071C47.2929 37.0348 47.3015 37.0627 47.3092 37.0908C47.3169 37.119 47.324 37.1473 47.3305 37.1758C47.3371 37.2043 47.3427 37.2329 47.3474 37.2617C47.3522 37.2905 47.3563 37.3194 47.3599 37.3483C47.3628 37.3773 47.3655 37.4064 47.3679 37.4355C47.3697 37.4647 47.3708 37.4938 47.3714 37.523C47.3714 37.5522 47.3711 37.5813 47.3705 37.6105C47.3693 37.6397 47.3673 37.6688 47.3643 37.6978C47.362 37.7269 47.3587 37.7559 47.3545 37.7849C47.3504 37.8138 47.3457 37.8425 47.3403 37.8712C47.335 37.8998 47.3288 37.9284 47.3217 37.9567C47.3146 37.9851 47.3069 38.0132 47.2986 38.0412C47.2903 38.0691 47.2814 38.0968 47.2719 38.1244C47.2618 38.1518 47.2511 38.1791 47.2399 38.2061C47.2286 38.233 47.2168 38.2596 47.2043 38.286C46.9306 38.8588 46.5484 39.1349 45.9663 39.349C45.6134 39.3764 45.2357 39.4166 44.8953 39.293C44.3798 39.1056 43.9407 38.6472 43.7425 38.1404C43.5834 37.7338 43.5772 37.3108 43.7603 36.9111C43.9674 36.4582 44.4775 36.0128 44.9486 35.8556C45.0961 35.8063 45.0748 35.8309 45.1859 35.7391C45.2063 35.7223 45.2232 35.702 45.2419 35.6834Z" fill="black"/>
</svg>',
            ],
            [
                'heading'   => 'Quisque viverra nunc nec vestibulum dapibus',
                'paragraph' => 'Duis felis ante, varius in neque eu, tempor suscipit sem. Maecenas ullamcorper gravida sem sit amet cursus.',
                'icon'      => '<svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="64" height="64" fill="white"/>
<path d="M49.4069 6.62739C49.764 6.57551 50.055 6.59636 50.367 6.78982C51.1738 7.29002 51.3682 8.87797 51.5374 9.74016C53.4714 19.59 54.0448 29.9795 48.1258 38.6924C41.9571 47.7728 32.039 51.0294 21.4565 49.1112C20.1362 48.8719 18.7999 48.5658 17.5043 48.2129C17.057 48.0911 16.7436 47.7091 16.2712 47.8013C15.6034 48.4413 14.3932 52.7318 14.1666 53.772C13.954 54.7476 13.9461 56.0882 13.479 56.9562C13.4657 56.981 13.4509 57.0047 13.4368 57.029C13.2748 57.1903 13.1056 57.372 12.9019 57.4798C12.5716 57.6546 12.1282 57.7643 11.7712 57.6274C11.3119 57.4511 11.0494 57.047 10.8728 56.6163C10.8057 56.4525 10.7356 56.2906 10.666 56.1278L10.9042 56.0108L10.7052 56.2261L10.8875 56.0455L10.6794 56.0066C11.1662 53.5639 12.2831 48.6035 13.4908 46.5073C13.5044 46.4835 13.5202 46.4608 13.5349 46.4376C13.563 45.4853 12.4915 43.8356 12.1239 42.8823C10.2882 38.1205 10.5971 30.8471 12.7002 26.2651C18.5525 13.5152 31.4216 14.2625 42.7675 10.4791C44.4332 9.92364 45.9036 9.21621 47.3391 8.18839C47.9992 7.71583 48.6581 6.9027 49.4069 6.62739ZM40.5775 24.8335C39.8939 25.5548 37.5349 26.1626 36.5153 26.6391C29.3386 29.9929 23.9277 33.9719 19.6661 40.7524C19.227 41.451 17.4666 44.3251 17.4371 44.9616C17.4343 45.0211 17.4416 45.0808 17.4439 45.1404C24.5254 47.1025 31.6736 47.7248 38.3259 43.9517C41.5539 42.1208 43.989 39.4562 46.1495 36.4958C46.8593 35.5234 47.2434 34.1971 47.6762 33.0803C48.4371 31.1172 49.0843 29.2842 49.4942 27.2024C50.3023 23.0987 49.9519 17.9823 49.2949 13.8823C49.2178 13.4008 49.2082 12.8913 49.1026 12.4176C49.0473 12.17 48.8532 11.1672 48.7254 11.0421L48.4094 11.0617C48.3847 11.081 48.3605 11.1007 48.3354 11.1195C47.754 11.5575 47.0397 11.8688 46.3847 12.1866C40.3164 15.1307 33.3744 15.3116 27.0874 17.5929C26.7667 17.7093 26.4834 17.8865 26.1713 18.0179C25.1549 18.4462 24.0933 18.7376 23.1288 19.2944C22.2142 19.8223 21.3649 20.4706 20.4502 20.9858C19.9619 21.2609 19.4891 21.8422 19.095 22.2431C16.8047 24.5744 15.1841 27.3391 14.3989 30.5249C13.5964 33.781 13.7252 37.3765 14.5646 40.6103C14.6929 41.1045 14.8052 41.7552 15.0694 42.1986C15.128 42.297 15.1983 42.3695 15.2784 42.4503L15.4548 42.4766C15.4915 42.3832 15.527 42.2894 15.5599 42.1945C15.8457 41.3715 16.4371 40.5857 16.8805 39.8334C19.2442 35.8234 21.9948 32.6925 25.6399 29.7831C26.1396 29.3843 26.7184 28.7872 27.2839 28.5186C27.3117 28.5054 27.34 28.4938 27.3675 28.48C28.2726 28.0248 29.1552 27.2751 30.0428 26.7456C32.1543 25.4858 34.3439 24.4826 36.6039 23.5201C37.3169 23.2165 38.0936 22.8139 38.8506 22.6422C39.2614 22.549 39.6984 22.5188 40.1077 22.6371C40.3729 22.7136 40.6545 22.8819 40.762 23.1497C40.9247 23.5544 40.9172 24.2329 40.6981 24.6144C40.6665 24.6695 40.6307 24.7225 40.5999 24.7782C40.5903 24.7957 40.585 24.8151 40.5775 24.8335Z" fill="black"/>
</svg>',
            ],
            [
                'heading'   => 'Curabitur vel bibendum lorem nisl arcu.',
                'paragraph' => 'Ut euismod nisl arcu, sed placerat nulla volutpat aliquet. sed auctor sit amet, molestie a nibh ut euismod ',
                'icon'      => '<svg width="64" height="64" viewBox="0 0 64 64" fill="none" xmlns="http://www.w3.org/2000/svg">
<rect width="64" height="64" fill="white"/>
<path d="M3.51028 16.1208C3.8777 16.0609 4.29276 16.0809 4.66441 16.073C12.576 15.9065 18.4944 16.0772 26.4091 16.0746L34.8787 16.0611C36.3017 16.0589 37.8275 15.9487 39.2411 16.101C39.9971 16.1827 40.5826 16.2913 41.1332 16.8526C42.6526 18.4036 42.2206 20.8002 42.1888 22.8054C44.2228 22.7759 53.783 22.4244 54.9445 23.3022C54.9805 23.3294 55.0155 23.3586 55.0515 23.3864C55.6052 23.8199 56.0891 24.2621 56.4967 24.8491C57.6953 26.5754 58.553 28.7498 59.4869 30.6482C60.137 31.971 60.8284 33.2758 61.4753 34.5988C61.7453 35.1519 61.9592 35.7398 62.2483 36.2821C62.4177 36.601 62.675 36.8648 62.8359 37.1819C63.125 37.7545 63.2827 38.3786 63.6036 38.9435C64.1637 39.93 64.0017 47.2395 63.8842 48.7327C63.7952 49.8538 63.5633 51.2658 62.6263 52.0171C62.5924 52.0439 62.5575 52.0684 62.5236 52.0943C60.4049 53.6977 58.391 52.7176 57.528 53.2191C56.9319 53.5652 56.4269 56.6798 54.3346 57.6126C54.2944 57.6707 54.2658 57.738 54.216 57.7869C53.622 58.3664 51.6473 58.8521 50.8225 58.924C50.1353 58.9838 48.5322 59.1354 48.0113 58.6453C47.754 58.404 47.2405 58.3982 46.9027 58.2404C45.0127 57.3565 43.74 55.4976 43.0719 53.5653L42.9459 53.4768L43.0137 53.3603L42.9416 53.3015L42.9851 53.2571L42.9448 53.1437C41.1935 53.005 25.3909 53.0426 24.8827 53.1978C24.8382 53.2114 24.7323 53.3413 24.7207 53.3598C24.4634 53.7432 24.3204 54.3281 24.114 54.7565C23.3188 56.411 22.0376 57.6546 20.3859 58.3994C18.7182 59.1511 16.5391 59.2023 14.8429 58.5241C13.1678 57.8542 11.5764 56.5239 10.8109 54.8354C10.5758 54.3155 10.4699 53.7409 10.2455 53.2103C8.12464 52.714 5.79946 53.6231 3.71357 52.8672C3.02216 52.6167 1.82463 51.8144 1.52286 51.0899C1.23275 50.3949 1.10357 49.5698 1.06122 48.8194C0.970159 47.2164 1.03687 45.5758 1.03687 43.9693L1.04321 34.7003L1.05063 25.5244C1.0411 23.7161 0.956389 21.8793 1.02945 20.0735C1.05592 19.4141 1.13639 18.7485 1.38839 18.1348C1.81404 17.0981 2.52663 16.5551 3.51028 16.1208ZM4.37745 49.8748C5.22981 49.8744 9.77112 50.0357 10.0898 49.7152C10.4647 49.3392 10.6912 48.3814 10.9221 47.8757C12.0476 45.4047 14.3887 43.9764 17.0198 43.8517C19.0147 43.7571 20.9089 44.2534 22.3997 45.669C23.0456 46.2827 23.6184 47.02 24.006 47.8312C24.2527 48.3502 24.4994 49.3087 24.8763 49.7015C25.0267 49.8589 25.2713 49.8508 25.4799 49.8622C27.2957 49.9616 27.1355 49.868 28.9556 49.8623C30.2971 49.8581 31.7393 49.9521 33.0692 49.8523C33.1962 49.8427 33.3084 49.8035 33.4281 49.762C34.7866 49.7927 36.1376 49.7778 37.4961 49.7924C38.0012 49.7978 38.5136 49.901 39.0145 49.8694V19.2964L16.8547 19.2956C12.703 19.2956 8.52805 19.2043 4.37745 19.3062V49.8748ZM60.498 49.8711C60.7405 48.8791 60.7405 40.1117 60.4599 39.6309L60.2397 39.5139C60.0925 39.3536 60.1772 39.3832 60.0015 39.3773C58.4344 39.3252 47.9202 39.5622 47.4141 39.3091C47.3135 39.2584 47.1325 39.0782 47.052 38.9908C46.8699 38.792 46.5798 38.4768 46.4718 38.2283C46.2388 37.6944 46.3998 34.2519 46.4548 33.488C46.4781 33.1614 46.4898 32.3906 46.727 32.1842C47.1981 31.7743 47.646 31.5838 48.2771 31.6312C48.6212 31.657 48.9304 31.7997 49.1559 32.0694C49.8907 32.9492 49.59 35.0476 49.5688 36.1353C50.6721 36.1375 57.9929 36.3079 58.4058 36.0292C58.0871 35.2093 57.6551 34.5486 57.293 33.7685C56.8779 32.8763 56.5634 31.9513 56.0383 31.1172C55.4601 30.1998 54.3187 26.9655 53.7078 26.4805C53.6231 26.4137 53.3849 26.3428 53.2779 26.3144C52.343 26.0672 43.38 26.0621 42.192 26.2213L42.1931 42.1159C42.1931 44.2464 41.9612 47.7413 42.3127 49.6975L42.4334 49.77L42.4069 49.9008C42.6801 49.8339 42.7563 49.7173 42.8961 49.4797C43.3461 48.7186 43.5843 47.8056 44.0079 47.0205C44.7469 45.6488 46.5533 44.4912 47.988 44.023C49.7563 43.4456 51.8548 43.6265 53.5024 44.502C55.0387 45.3191 56.7456 47.1975 57.2718 48.9008C57.3565 49.1738 57.3925 49.4602 57.4158 49.7443C57.6911 50.0081 59.8987 49.9055 60.4027 49.881C60.4345 49.8794 60.4663 49.8744 60.498 49.8711ZM17.58 55.8712C18.0829 55.7682 18.6663 55.7275 19.1513 55.5695C19.7971 55.3592 20.7977 54.3174 21.1662 53.7461C21.7761 52.8027 21.9529 51.6041 21.7104 50.5094C21.6935 50.4346 21.6748 50.3602 21.6543 50.2864C21.6339 50.2125 21.612 50.1391 21.5887 50.0663C21.5647 49.9935 21.5389 49.9214 21.5114 49.8498C21.4839 49.7783 21.4546 49.7076 21.4235 49.6374C21.3931 49.5674 21.361 49.4982 21.3272 49.4297C21.2926 49.3613 21.2566 49.2937 21.2191 49.2271C21.1817 49.1604 21.1429 49.0947 21.1027 49.0301C21.0617 48.9654 21.0194 48.9017 20.9756 48.8391C20.9326 48.7764 20.8877 48.7149 20.8411 48.6546C20.7946 48.5943 20.7466 48.5351 20.6971 48.4771C20.6477 48.4192 20.5969 48.3625 20.5447 48.307C20.4924 48.2516 20.4388 48.1975 20.3837 48.1448C20.3294 48.092 20.2733 48.0406 20.2154 47.9907C20.1582 47.9407 20.1 47.8922 20.0407 47.8451C19.9814 47.7981 19.9207 47.7525 19.8586 47.7085C19.7964 47.6645 19.7336 47.622 19.6701 47.5811C18.9723 47.1419 17.8902 46.8185 17.0654 46.9434C16.8483 47.0063 16.6429 47.0623 16.4205 47.1068C15.9261 47.2059 15.3003 47.7786 14.8492 47.7861C14.071 48.2885 13.4114 49.6007 13.2218 50.4801C12.9614 51.6915 13.2388 52.9307 13.9058 53.9606C14.4935 54.8651 15.4602 55.5738 16.5211 55.7561C16.7933 55.8028 17.3121 55.9166 17.58 55.8712ZM51.1677 55.7462C51.1952 55.7179 51.2206 55.6869 51.2502 55.6616C51.4557 55.4897 51.7881 55.4485 52.0348 55.3227C52.8353 54.9143 53.333 54.3651 53.909 53.6906C54.1144 53.4508 54.5729 51.4215 54.5443 51.0505C54.4532 49.8702 53.7597 48.6292 52.866 47.8856C52.8067 47.8353 52.746 47.7865 52.6839 47.7394C52.6218 47.6923 52.5586 47.6469 52.4944 47.6032C52.4301 47.5596 52.3649 47.5176 52.2985 47.4775C52.2321 47.4373 52.1647 47.399 52.0962 47.3625C52.0278 47.326 51.9583 47.2914 51.8877 47.2587C51.8178 47.2259 51.7468 47.1951 51.6748 47.1662C51.6028 47.1373 51.5301 47.1103 51.4567 47.0854C51.3833 47.0604 51.3095 47.0374 51.2354 47.0164C51.1606 46.9955 51.0854 46.9765 51.0099 46.9596C50.9344 46.9426 50.8585 46.9277 50.7823 46.915C50.706 46.9021 50.6294 46.8913 50.5525 46.8826C50.4763 46.8739 50.3993 46.8673 50.3217 46.8628C50.2447 46.8583 50.1674 46.8559 50.0898 46.8556C50.0128 46.8552 49.9355 46.8569 49.8579 46.8607C49.7809 46.8646 49.704 46.8705 49.6271 46.8785C49.5501 46.8865 49.4735 46.8966 49.3973 46.9087C49.3602 46.9146 49.3242 46.9274 49.2882 46.9368C47.9192 47.3772 47.0192 47.9533 46.3521 49.3077C45.8216 50.3822 45.6798 51.7163 46.0609 52.8636C46.3828 53.8306 47.6354 55.1387 48.5577 55.5571C48.6741 55.6101 48.8033 55.6167 48.9272 55.6327C49.6874 55.7307 50.4011 55.7636 51.1677 55.7462Z" fill="black"/>
</svg>',
            ],
        ];
    }
}
