# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
"""Replace columns username to uid

Revision ID: 15fed017ca0b
Revises: 48efb1674139
Create Date: 2016-07-22 17:19:55.478609

"""

import logging
import os
import collections
from alembic import op
from alembic import context
from sqlalchemy import Column, Integer, String, Float, Index, MetaData, Table

from lvestats.lib import uidconverter

# revision identifiers, used by Alembic.
revision = '15fed017ca0b'
down_revision = '48efb1674139'
branch_labels = None
depends_on = None

HISTORY_GOV_TABLE_NAME = 'lve_stats2_history_gov'
metadata = MetaData()

logger = logging.getLogger(f'alembic.script.{os.path.basename(__name__)}')

history_gov_old_items = (
    Column('ts', Integer, index=True, primary_key=True, autoincrement=False),
    Column('username', String(64), primary_key=True, autoincrement=False),
    Column('sum_cpu', Float),
    Column('sum_write', Float),
    Column('sum_read', Float),
    Column('limit_cpu_on_period_end', Integer),
    Column('limit_read_on_period_end', Integer),
    Column('limit_write_on_period_end', Integer),
    Column('cause_of_restrict', Integer),
    Column('server_id', String(255), primary_key=True, autoincrement=False)
)

history_gov_new_items = (
    Column('ts', Integer, primary_key=True, autoincrement=False),
    Index(f'ix_{HISTORY_GOV_TABLE_NAME}_ts_2', 'ts'),
    Column('uid', Integer, primary_key=True, autoincrement=False),
    Column('sum_cpu', Float),
    Column('sum_write', Float),
    Column('sum_read', Float),
    Column('limit_cpu_on_period_end', Integer),
    Column('limit_read_on_period_end', Integer),
    Column('limit_write_on_period_end', Integer),
    Column('cause_of_restrict', Integer),
    Column('server_id', String(255), primary_key=True, autoincrement=False)
)


def upgrade():
    connection = op.get_bind()
    local_server_id = context.config.attributes.get('lve-stats')['server_id']
    username_to_uid_cache = collections.defaultdict(dict)
    tablename_renamed = HISTORY_GOV_TABLE_NAME + '_' + down_revision

    # upgrade table structure
    op.rename_table(HISTORY_GOV_TABLE_NAME, tablename_renamed)
    history_gov = op.create_table(
        HISTORY_GOV_TABLE_NAME, *history_gov_new_items)

    # upgrade data
    history_gov_old = Table(tablename_renamed, metadata, *history_gov_old_items)
    trans = connection.begin()
    data_history_gov_old = connection.execute(history_gov_old.select())
    for history_gov_old_row in data_history_gov_old.fetchall():
        history_gov_new_row = dict(history_gov_old_row)
        server_id = history_gov_new_row['server_id']
        username = history_gov_new_row.pop('username')
        if username in username_to_uid_cache[server_id]:
            uid = username_to_uid_cache[server_id][username]
        else:
            uid = uidconverter.username_to_uid(
                username, local_server_id, server_id, connection) or -1
            if uid < 0:
                logger.warning(
                    'Can\'t obtain uid for user "%s"; server_id "%s"',
                    username, server_id,
                )
            username_to_uid_cache[server_id][username] = uid
        if uid > 0:
            history_gov_new_row['uid'] = uid
            connection.execute(history_gov.insert(history_gov_new_row))
    trans.commit()
    # delete old table
    op.drop_table(tablename_renamed)


def downgrade():
    connection = op.get_bind()
    local_server_id = context.config.attributes.get('lve-stats')['server_id']
    uid_to_username_cache = collections.defaultdict(dict)
    tablename_renamed = HISTORY_GOV_TABLE_NAME + '_' + revision

    # downgrade table structure history_gov_new=>history_gov_old
    op.rename_table(HISTORY_GOV_TABLE_NAME, tablename_renamed)
    history_gov_old = op.create_table(
        HISTORY_GOV_TABLE_NAME, *history_gov_old_items)

    # downgrade data
    history_gov = Table(tablename_renamed, metadata, *history_gov_new_items)
    trans = connection.begin()
    data_history_gov_new = connection.execute(history_gov.select())
    for history_gov_new_row in data_history_gov_new.fetchall():
        history_gov_old_dict = dict(history_gov_new_row)
        server_id = history_gov_old_dict['server_id']
        uid = history_gov_old_dict.pop('uid')
        if uid in uid_to_username_cache[server_id]:
            username = uid_to_username_cache[server_id][uid]
        else:
            username = uidconverter.uid_to_username(
                uid, local_server_id, server_id, connection)
            if username is None:
                logger.warning(
                    'Can\'t obtain user name for uid "%s"; server_id "%s"',
                    uid, server_id,
                )
            uid_to_username_cache[server_id][uid] = username
        if username:
            history_gov_old_dict['username'] = username
            connection.execute(history_gov_old.insert(history_gov_old_dict))
    trans.commit()

    # delete old table
    op.drop_table(tablename_renamed)
