# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
"""
Wrapper over integration config which provides
typing.
"""

import os
import shlex
from configparser import ConfigParser
from collections import namedtuple

from vendors_api.exceptions import NotImplementedByVendor

CONFIG_PATH = '/opt/cpvendor/etc/integration.ini'
integration_scripts_cache = None
config_last_modify_time = None


class ScriptsCMD(namedtuple('VendorScriptsPath', [
    'panel_info',
    'db_info',
    'packages',
    'users',
    'domains',
    'resellers',
    'admins',
    'php',
])):
    """
    Wrapper over namedtuple that raises exception
    when we access property that has None value
    """

    def __getattribute__(self, item):
        """
        Raise error if command is not implemented,
        but we try to get it in code
        """
        obj = super().__getattribute__(item)
        if obj is None:
            raise NotImplementedByVendor(item)
        return obj


def _split_path_or_none(cmd):
    """
    Taking command line as input, convert it into
    subprocess args
    If cmd is None -> return None
    """
    if cmd is None:
        return None
    return tuple(shlex.split(cmd))


def integration_scripts():
    global config_last_modify_time
    global integration_scripts_cache

    modify_time = os.path.getmtime(CONFIG_PATH)
    if integration_scripts_cache is None or \
            config_last_modify_time != modify_time:
        config = _read_config_file()

        scripts = config['integration_scripts']
        integration_scripts_cache = ScriptsCMD(
            panel_info=_split_path_or_none(scripts.get('panel_info')),
            db_info=_split_path_or_none(scripts.get('db_info')),
            packages=_split_path_or_none(scripts.get('packages')),
            users=_split_path_or_none(scripts.get('users')),
            domains=_split_path_or_none(scripts.get('domains')),
            resellers=_split_path_or_none(scripts.get('resellers')),
            admins=_split_path_or_none(scripts.get('admins')),
            php=_split_path_or_none(scripts.get('php')),
        )
        config_last_modify_time = modify_time

    return integration_scripts_cache


def _read_config_file():
    """
    Parse config file and return result as dictionary.
    """
    parser = ConfigParser(interpolation=None, strict=False)
    parser.read(CONFIG_PATH)

    return {s: dict(parser.items(s)) for s in parser.sections()}
