# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENCE.TXT
#
"""
Module contains various exceptions raised by this api.
"""

from clcommon import FormattedException


class ErrorMessage:
    """
    Documented constants that can be
    received from vendors scripts
    """
    INTERNAL_ERROR = "InternalAPIError"
    PERMISSION_DENIED = "PermissionDenied"
    BAD_REQUEST = "BadRequest"
    NOT_FOUND = "NotFound"


class VendorScriptError(FormattedException):
    """
    Any errors related to vendors scripts.
    """
    def __init__(self, message, **context):
        super().__init__({
            'message': message,
            'context': context
        })


class UnexpectedResult(VendorScriptError):
    """
    An error that is raised in case of
    any unexpected results returned by vendors script.
    """
    pass


class MalformedError(VendorScriptError):
    """
    An error that is raised in case of
    malformed output that we cannot parse
    """
    def __init__(self, output):
        super().__init__(
            'vendor api returned invalid response that '
            'cannot be parsed with json, output: %(output)s',
            output=output
        )


class BadScriptError(VendorScriptError):
    """
    An error that is raised when we cannot
    run vendor's script for some reason.
    """
    pass


class NotImplementedByVendor(VendorScriptError):
    """
    Raised when we try to start script that is
    not implemented by vendor
    """
    def __init__(self, command):
        super().__init__(
            "Command `%(command)s` is not implemented "
            "in integration scripts. Please, implement it and try again.",
            command=command
        )


class _VendorApiFailed(VendorScriptError):
    """
    Bunch of api errors returned by vendors scripts
    """
    def __init__(self, message):
        super().__init__(message=message)


class InternalError(_VendorApiFailed):
    """
    n case when data is temporarily unavailable
    due to internal errors in the integration script
    (database unavailable or something like that)
    """
    pass


class PermissionDenied(_VendorApiFailed):
    """
    In case when data is unavailable due to restricted
    access of a user that called the script
    """
    pass


class BadRequest(_VendorApiFailed):
    """
    In case when known to be false arguments are passed
    to the utility.
    """
    pass


class NotFound(_VendorApiFailed):
    """
    In case when during data filtering the target entity
    doesn't exist in the control panel
    """
    pass


class VendorApiMalformedData(VendorScriptError):
    """
    Raised when vendor script returns bad data.
    """
    def __init__(self, error):
        super().__init__(
            "Integration script returned invalid response "
            "which is not accepted by internal schema check. See error below.\n"
            "%(error)s",
            error=error
        )
