<?php

/**
 * Class Tools_PaypalMailWatchdog
 */
class Tools_PaypalMailWatchdog implements Interfaces_Observer
{

    const DEFAULT_USER_TIMEZONE = 'America/New_York';
    /**
     * Recurring payment (regular)
     */
    const TRIGGER_SUBSCRIPTION_RECURRING_PAYMENT = 'paypal_recurringpayment';

    /**
     * Recurring profile canceled
     */
    const TRIGGER_SUBSCRIPTION_RECURRING_CANCELED = 'paypal_recurringcanceled';

    /**
     * Recurring profile suspended
     */
    const TRIGGER_SUBSCRIPTION_RECURRING_SUSPENDED = 'paypal_recurringsuspended';

    /**
     * Recurring profile failed
     */
    const TRIGGER_SUBSCRIPTION_RECURRING_FAILED = 'paypal_recurringfailed';

    /**
     * Recurring profile reactivated
     */
    const TRIGGER_SUBSCRIPTION_RECURRING_REACTIVATED = 'paypal_recurringreactivated';

    /**
     * @var array
     */
    private $_options;

    /**
     * @var Tools_Mail_Mailer instance of mailer
     */
    private $_mailer;

    /**
     * @var Helpers_Action_Config
     */
    private $_configHelper;

    /**
     * @var Helpers_Action_Website
     */
    private $_websiteHelper;

    /**
     * @var instance of watched object
     */
    private $_object;

    /**
     * Email template
     *
     * @var null
     */
    private $_template = null;

    /**
     * Email entity parser
     *
     * @var null|Tools_Content_EntityParser
     */
    private $_entityParser = null;

    /**
     * Tools_PaypalMailWatchdog constructor.
     * @param array $options
     * @throws Zend_Controller_Action_Exception
     */
    public function __construct($options = array())
    {
        $this->_configHelper = Zend_Controller_Action_HelperBroker::getExistingHelper('config');
        $this->_websiteHelper = Zend_Controller_Action_HelperBroker::getExistingHelper('website');
        $this->_options = $options;
        $this->_initMailer();
        $this->_entityParser = new Tools_Content_EntityParser();
        $this->_initMailMessage();
    }

    /**
     * init mailer
     */
    private function _initMailer()
    {
        $config = $this->_configHelper->getConfig();
        $this->_mailer = new Tools_Mail_Mailer();

        if ((bool)$config['useSmtp']) {
            $smtpConfig = array(
                'host' => $config['smtpHost'],
                'username' => $config['smtpLogin'],
                'password' => $config['smtpPassword']
            );
            if ((bool)$config['smtpSsl']) {
                $smtpConfig['ssl'] = $config['smtpSsl'];
            }
            if (!empty($config['smtpPort'])) {
                $smtpConfig['port'] = $config['smtpPort'];
            }
            $this->_mailer->setSmtpConfig($smtpConfig);
            $this->_mailer->setTransport(Tools_Mail_Mailer::MAIL_TYPE_SMTP);
        } else {
            $this->_mailer->setTransport(Tools_Mail_Mailer::MAIL_TYPE_MAIL);
        }
    }

    /**
     *
     */
    protected function _initMailMessage()
    {
        $this->_options['message'] = (isset($this->_options['mailMessage']) ? $this->_options['mailMessage'] : $this->_options['message']);
        unset($this->_options['mailMessage']);
    }

    /**
     * @param $object
     * @return bool
     * @throws Exceptions_SeotoasterException
     * @throws Zend_Exception
     */
    public function notify($object)
    {
        if (!$object) {
            return false;
        }

        $this->_object = $object;

        if (isset($this->_options['template']) && !empty($this->_options['template'])) {
            $this->_template = $this->_preparseEmailTemplate();
        } else {
            throw new Exceptions_SeotoasterException('Missing template for action email trigger');
        }

        $this->_mailer->setMailFromLabel('')
            ->setSubject($this->_options['subject']);

        $wicEmail = $this->_configHelper->getConfig('wicEmail');
        $this->_entityParser->addToDictionary(array('widcard:BizEmail' => !empty($wicEmail) ? $wicEmail : $this->_configHelper->getConfig('adminEmail')));

        if (!empty($this->_options['from'])) {
            $this->_mailer->setMailFrom($this->_entityParser->parse($this->_options['from']));
        } else {
            $this->_mailer->setMailFrom($this->_configHelper->getConfig('adminEmail'));
        }

        if (isset($this->_options['trigger'])) {
            $methodName = str_replace('paypal_', '', $this->_options['trigger']);
            $methodName = '_send' . ucfirst(strtolower(preg_replace('/\s*/', '', $methodName))) . 'Mail';
            if (method_exists($this, $methodName)) {
                $this->$methodName();
            }
        }
    }

    /**
     * @return bool
     * @throws Exceptions_SeotoasterException
     */
    protected function _send()
    {
        if (!$this->_mailer->getMailFrom() || !$this->_mailer->getMailTo()) {
            throw new Exceptions_SeotoasterException('Missing required "from" and "to" fields');
        }
        $this->_mailer->setBody($this->_entityParser->parse($this->_template));

        return ($this->_mailer->send() !== false);
    }

    /**
     *
     * Send regular recurring payment email
     *
     * @return bool
     * @throws Exceptions_SeotoasterException
     */
    private function _sendRecurringpaymentMail()
    {
        $userEmail = $this->_options['userEmail'];
        $fullName = $this->_options['fullName'];
        $systemConfig = $this->_configHelper->getConfig();
        $userMapper = Application_Model_Mappers_UserMapper::getInstance();
        $adminBccArray = array();
        $adminEmail = isset($systemConfig['adminEmail']) ? $systemConfig['adminEmail'] : 'admin@localhost';
        switch ($this->_options['recipient']) {
            case Tools_Security_Acl::ROLE_ADMIN:
                $this->_mailer->setMailToLabel('Admin')
                    ->setMailTo($adminEmail);
                $where = $userMapper->getDbTable()->getAdapter()->quoteInto("role_id = ?",
                    Tools_Security_Acl::ROLE_ADMIN);
                $adminUsers = $userMapper->fetchAll($where);
                if (!empty($adminUsers)) {
                    foreach ($adminUsers as $admin) {
                        array_push($adminBccArray, $admin->getEmail());
                    }
                    if (!empty($adminBccArray)) {
                        $this->_mailer->setMailBcc($adminBccArray);
                    }
                }
                break;
            default:
                $this->_mailer->setMailToLabel($fullName)->setMailTo($userEmail);
                break;
        }
        $timeZoneTo = self::DEFAULT_USER_TIMEZONE;
        if ($this->_options['recipient'] !== Tools_Security_Acl::ROLE_ADMIN) {
            $userId = $this->_object->getUserId();
            $userModel = Application_Model_Mappers_UserMapper::getInstance()->find($userId);
            if ($userModel instanceof Application_Model_Models_User) {
                $timeZoneTo = $userModel->getTimezone();
            }
        }
        $additionalLexems['user:email'] = $userEmail;
        $additionalLexems['user:fullName'] = $fullName;
        $additionalLexems['profile:startDate'] = Tools_System_Tools::convertDateFromTimezone($this->_object->getProfileStartDate(), 'UTC', $timeZoneTo, 'd M Y');
        $additionalLexems['profile:nextPaymentDate'] = Tools_System_Tools::convertDateFromTimezone($this->_object->getNextBillingDate(), 'UTC', $timeZoneTo, 'd M Y');
        $additionalLexems['profile:lastPaymentDate'] = Tools_System_Tools::convertDateFromTimezone($this->_object->getLastPaymentDate(), 'UTC', $timeZoneTo, 'd M Y');
        $additionalLexems['profile:finalPaymentDate'] = Tools_System_Tools::convertDateFromTimezone($this->_object->getFinalPaymentDueDate(), 'UTC', $timeZoneTo, 'd M Y');
        $this->_entityParser->objectToDictionary($this->_object, 'profile');
        $this->_entityParser->addToDictionary($additionalLexems);

        return $this->_send();
    }


    /**
     *
     * Send canceled recurring payment email
     *
     * @return bool
     * @throws Exceptions_SeotoasterException
     */
    private function _sendRecurringcanceledMail()
    {
        $userEmail = $this->_options['userEmail'];
        $fullName = $this->_options['fullName'];
        $systemConfig = $this->_configHelper->getConfig();
        $userMapper = Application_Model_Mappers_UserMapper::getInstance();
        $adminBccArray = array();
        $adminEmail = isset($systemConfig['adminEmail']) ? $systemConfig['adminEmail'] : 'admin@localhost';
        switch ($this->_options['recipient']) {
            case Tools_Security_Acl::ROLE_ADMIN:
                $this->_mailer->setMailToLabel('Admin')
                    ->setMailTo($adminEmail);
                $where = $userMapper->getDbTable()->getAdapter()->quoteInto("role_id = ?",
                    Tools_Security_Acl::ROLE_ADMIN);
                $adminUsers = $userMapper->fetchAll($where);
                if (!empty($adminUsers)) {
                    foreach ($adminUsers as $admin) {
                        array_push($adminBccArray, $admin->getEmail());
                    }
                    if (!empty($adminBccArray)) {
                        $this->_mailer->setMailBcc($adminBccArray);
                    }
                }
                break;
            default:
                $this->_mailer->setMailToLabel($fullName)->setMailTo($userEmail);
                break;
        }
        $timeZoneTo = self::DEFAULT_USER_TIMEZONE;
        if ($this->_options['recipient'] !== Tools_Security_Acl::ROLE_ADMIN) {
            $userId = $this->_options['userId'];
            $userModel = Application_Model_Mappers_UserMapper::getInstance()->find($userId);
            if ($userModel instanceof Application_Model_Models_User) {
                $timeZoneTo = $userModel->getTimezone();
            }
        }
        $additionalLexems['user:email'] = $userEmail;
        $additionalLexems['user:fullName'] = $fullName;
        $this->_entityParser->objectToDictionary($this->_object, 'profile');
        $this->_entityParser->addToDictionary($additionalLexems);

        return $this->_send();
    }

    /**
     *
     * Send failed recurring payment
     *
     * @return bool
     * @throws Exceptions_SeotoasterException
     */
    private function _sendRecurringfailedMail()
    {
        $userEmail = $this->_options['userEmail'];
        $fullName = $this->_options['fullName'];
        $systemConfig = $this->_configHelper->getConfig();
        $userMapper = Application_Model_Mappers_UserMapper::getInstance();
        $adminBccArray = array();
        $adminEmail = isset($systemConfig['adminEmail']) ? $systemConfig['adminEmail'] : 'admin@localhost';
        switch ($this->_options['recipient']) {
            case Tools_Security_Acl::ROLE_ADMIN:
                $this->_mailer->setMailToLabel('Admin')
                    ->setMailTo($adminEmail);
                $where = $userMapper->getDbTable()->getAdapter()->quoteInto("role_id = ?",
                    Tools_Security_Acl::ROLE_ADMIN);
                $adminUsers = $userMapper->fetchAll($where);
                if (!empty($adminUsers)) {
                    foreach ($adminUsers as $admin) {
                        array_push($adminBccArray, $admin->getEmail());
                    }
                    if (!empty($adminBccArray)) {
                        $this->_mailer->setMailBcc($adminBccArray);
                    }
                }
                break;
            default:
                $this->_mailer->setMailToLabel($fullName)->setMailTo($userEmail);
                break;
        }

        $timeZoneTo = self::DEFAULT_USER_TIMEZONE;
        if ($this->_options['recipient'] !== Tools_Security_Acl::ROLE_ADMIN) {
            $userId = $this->_object->getUserId();
            $userModel = Application_Model_Mappers_UserMapper::getInstance()->find($userId);
            if ($userModel instanceof Application_Model_Models_User) {
                $timeZoneTo = $userModel->getTimezone();
            }
        }
        $additionalLexems['user:email'] = $userEmail;
        $additionalLexems['user:fullName'] = $fullName;
        $this->_entityParser->objectToDictionary($this->_object, 'profile');
        $this->_entityParser->addToDictionary($additionalLexems);

        return $this->_send();
    }

    /**
     *
     * Send suspended recurring payment
     *
     * @return bool
     * @throws Exceptions_SeotoasterException
     */
    private function _sendRecurringsuspendedMail()
    {
        $userEmail = $this->_options['userEmail'];
        $fullName = $this->_options['fullName'];
        $systemConfig = $this->_configHelper->getConfig();
        $userMapper = Application_Model_Mappers_UserMapper::getInstance();
        $adminBccArray = array();
        $adminEmail = isset($systemConfig['adminEmail']) ? $systemConfig['adminEmail'] : 'admin@localhost';
        switch ($this->_options['recipient']) {
            case Tools_Security_Acl::ROLE_ADMIN:
                $this->_mailer->setMailToLabel('Admin')
                    ->setMailTo($adminEmail);
                $where = $userMapper->getDbTable()->getAdapter()->quoteInto("role_id = ?",
                    Tools_Security_Acl::ROLE_ADMIN);
                $adminUsers = $userMapper->fetchAll($where);
                if (!empty($adminUsers)) {
                    foreach ($adminUsers as $admin) {
                        array_push($adminBccArray, $admin->getEmail());
                    }
                    if (!empty($adminBccArray)) {
                        $this->_mailer->setMailBcc($adminBccArray);
                    }
                }
                break;
            default:
                $this->_mailer->setMailToLabel($fullName)->setMailTo($userEmail);
                break;
        }
        $additionalLexems['user:email'] = $userEmail;
        $additionalLexems['user:fullName'] = $fullName;
        $this->_entityParser->objectToDictionary($this->_object, 'profile');
        $this->_entityParser->addToDictionary($additionalLexems);

        return $this->_send();
    }

    /**
     *
     * Send canceled recurring payment email
     *
     * @return bool
     * @throws Exceptions_SeotoasterException
     */
    private function _sendRecurringreactivatedMail()
    {
        $userEmail = $this->_options['userEmail'];
        $fullName = $this->_options['fullName'];
        $systemConfig = $this->_configHelper->getConfig();
        $userMapper = Application_Model_Mappers_UserMapper::getInstance();
        $adminBccArray = array();
        $adminEmail = isset($systemConfig['adminEmail']) ? $systemConfig['adminEmail'] : 'admin@localhost';
        switch ($this->_options['recipient']) {
            case Tools_Security_Acl::ROLE_ADMIN:
                $this->_mailer->setMailToLabel('Admin')
                    ->setMailTo($adminEmail);
                $where = $userMapper->getDbTable()->getAdapter()->quoteInto("role_id = ?",
                    Tools_Security_Acl::ROLE_ADMIN);
                $adminUsers = $userMapper->fetchAll($where);
                if (!empty($adminUsers)) {
                    foreach ($adminUsers as $admin) {
                        array_push($adminBccArray, $admin->getEmail());
                    }
                    if (!empty($adminBccArray)) {
                        $this->_mailer->setMailBcc($adminBccArray);
                    }
                }
                break;
            default:
                $this->_mailer->setMailToLabel($fullName)->setMailTo($userEmail);
                break;
        }
        $timeZoneTo = self::DEFAULT_USER_TIMEZONE;
        if ($this->_options['recipient'] !== Tools_Security_Acl::ROLE_ADMIN) {
            $userId = $this->_options['userId'];
            $userModel = Application_Model_Mappers_UserMapper::getInstance()->find($userId);
            if ($userModel instanceof Application_Model_Models_User) {
                $timeZoneTo = $userModel->getTimezone();
            }
        }
        $additionalLexems['user:email'] = $userEmail;
        $additionalLexems['user:fullName'] = $fullName;
        $this->_entityParser->objectToDictionary($this->_object, 'profile');
        $this->_entityParser->addToDictionary($additionalLexems);

        return $this->_send();
    }


    /**
     * @return bool|null
     * @throws Zend_Exception
     */
    private function _preparseEmailTemplate()
    {
        $tmplName = $this->_options['template'];
        $tmplMessage = $this->_options['message'];
        $mailTemplate = Application_Model_Mappers_TemplateMapper::getInstance()->find($tmplName);
        if (!empty($mailTemplate)) {
            $this->_entityParser->setDictionary(
                array(
                    'emailmessage' => !empty($tmplMessage) ? $tmplMessage : ''
                )
            );
            //pushing message template to email template and cleaning dictionary
            $mailTemplate = $this->_entityParser->parse($mailTemplate->getContent());
            $this->_entityParser->setDictionary(array());

            $mailTemplate = $this->_entityParser->parse($mailTemplate);

            $themeData = Zend_Registry::get('theme');
            $extConfig = Zend_Registry::get('extConfig');
            $parserOptions = array(
                'websiteUrl' => $this->_websiteHelper->getUrl(),
                'websitePath' => $this->_websiteHelper->getPath(),
                'currentTheme' => $extConfig['currentTheme'],
                'themePath' => $themeData['path'],
            );
            $parser = new Tools_Content_Parser($mailTemplate, array(), $parserOptions);

            return $parser->parseSimple();
        }

        return false;
    }

}
