<?php
/**
 * NewsMapper
 * @author: iamne <eugene@seotoaster.com> Seotoaster core team
 * Date: 7/25/12
 * Time: 5:54 PM
 */
class Newslog_Models_Mapper_NewsMapper extends Application_Model_Mappers_Abstract {

    protected $_dbTable = 'Newslog_Models_DbTable_News';

    protected $_model   = 'Newslog_Models_Model_News';

    /**
     * Save the Newslog_Models_Model_News to the database
     *
     * @param Newslog_Models_Model_News $model
     * @return mixed
     * @throws Exceptions_NewslogException
     */
    public function save($model) {
        if(!$model instanceof $this->_model) {
            $model = new $this->_model($model);
        }

        //get news page id
        $pageId = $model->getPageId();

        $data   = array(
            'title'       => $model->getTitle(),
            'metaData'    => $model->getMetaData(),
            'teaser'      => $model->getTeaser(),
            'content'     => $model->getContent(),
            'broadcast'   => (boolean)$model->getBroadcast(),
            'published'   => (boolean)$model->getPublished(),
            'featured'    => (boolean)$model->getFeatured(),
            'archived'    => (boolean)$model->getArchived(),
            'type'        => $model->getType(),
            'event'         => (int) $model->getEvent(),
            'event_date'    => $model->getEventDate(),
            'event_location' => $model->getEventLocation(),
            'page_id'     => ($pageId) ? $pageId : null,
            'external_id' => $model->getExternalId(),
            'user_id'     => $model->getUserId(),
            'created_at'  => $model->getCreatedAt()
        );
        if($model->getId()) {
            if (empty($data['created_at'])) {
                unset($data['created_at']);
            }
            $where = $this->getDbTable()->getAdapter()->quoteInto('id = ?', $model->getId());
            $this->getDbTable()->update($data, $where);
        } else {
            $createdAt          = $model->getCreatedAt();
            $data['created_at'] = $createdAt ? $createdAt : date(Tools_System_Tools::DATE_MYSQL);
            $newsId             = $this->getDbTable()->insert($data);
            if($newsId) {
                $model->setId($newsId);
            } else {
                throw new Exceptions_NewslogException('Can not save news item!');
            }
        }
        if (!is_null($model->getTags())) {
            $newsHasTagDbTable = new Newslog_Models_DbTable_NewsHasTag();
            $newsHasTagDbTable->getAdapter()->beginTransaction();
            $newsHasTagDbTable->delete($newsHasTagDbTable->getAdapter()->quoteInto('news_id = ?', $model->getId()));
            foreach ($model->getTags() as $tag) {
                try {
                    $newsHasTagDbTable->insert(array(
                        'news_id' => $model->getId(),
                        'tag_id'  => $tag['id']
                    ));
                } catch (Exception $e) {
                    Tools_System_Tools::debugMode() && error_log($e->getMessage());
                    continue;
                }
            }
            $newsHasTagDbTable->getAdapter()->commit();
        }

        $model->notifyObservers();

        return $model;
    }

    public function delete(Newslog_Models_Model_News $news) {
        $where        = $this->getDbTable()->getAdapter()->quoteInto('id = ?', $news->getId());
        $deleteResult = $this->getDbTable()->delete($where);
        $news->notifyObservers();
        return $deleteResult;
    }

    public function find($id) {
        $newsItem = parent::find($id);
        if(!$newsItem instanceof Newslog_Models_Model_News){
            return null;
        }
        return $newsItem->setTags($this->_fetchRelatedTags($newsItem->getId()));
    }

    public function findByPageId($pageId) {
        $where = $this->getDbTable()->getAdapter()->quoteInto('page_id = ?', $pageId);
        $row   = $this->getDbTable()->fetchAll($where)->current();
        if(!$row) {
            return null;
        }
        return $this->_toModel($row);
    }

    public function findByTitle($title) {
        $where = $this->getDbTable()->getAdapter()->quoteInto('title = ?', $title);
        $row   = $this->getDbTable()->fetchAll($where)->current();
        if(!$row) {
            return null;
        }
        return $this->_toModel($row);
    }

    public function findByExternalId($id) {
        $where = $this->getDbTable()->getAdapter()->quoteInto('external_id = ?', $id);
        $row   = $this->getDbTable()->fetchAll($where)->current();
        if(!$row) {
            return null;
        }
        return $this->_toModel($row);
    }

    /**
     * @return array|null
     */
    public function fetchNews()
    {
        $select = $this->getDbTable()
            ->select(Zend_Db_Table::SELECT_WITHOUT_FROM_PART)
            ->from(array('n' => 'plugin_newslog_news'),
                array('n.id', 'n.title', 'p.url'))
            ->setIntegrityCheck(false)
            ->joinLeft(array('p' => 'page'), 'n.page_id = p.id', array())
            ->group('n.id');

            $resultSet = $this->getDbTable()->fetchAll($select);
            if (null === $resultSet || empty($resultSet->toArray()) ) {
                return null;
            }
            $countData = count($resultSet->toArray());

            return array('select' => $select, 'count' => $countData);
    }

    /**
     * Optimized Method for news/api getAction
     *
     * @param null $where
     * @param array $order
     * @param null $count
     * @param null $offset
     * @param array $tags
     * @param null $search
     * @param bool $includeCount
     * @return array|null
     */
    public function getAllNews($where = null, $order = array(), $count = null, $offset = null, $tags = array(), $search = null, $includeCount = false)
    {
        $entries = array();

        if ($search !== null) {
            $where = ($where === null) ? 'title LIKE "%' . $search . '%"' : ($where . ' AND title LIKE "%' . $search . '%"');
        }

        $select = $this->getDbTable()
            ->select(Zend_Db_Table::SELECT_WITHOUT_FROM_PART)
            ->from(array('n' => 'plugin_newslog_news'), array(
                'pageId' => 'n.page_id',
                'n.title',
                'n.teaser',
                'n.content',
                'n.broadcast',
                'n.published',
                'n.featured',
                'n.archived',
                'n.metaData',
                'n.type',
                'n.event',
                'n.event_date',
                'eventLocation' => 'n.event_location',
                'createdAt'     => 'n.created_at',
                'updatedAt'     => 'n.updated_at',
                'externalId'    => 'n.external_id',
                'userId'        => 'n.user_id',
                'n.id'
            ))
            ->setIntegrityCheck(false)
            ->group('n.id');

        if (!empty($tags)) {
            $select->from(array('t' => 'plugin_newslog_tag'), null)
                ->join(array('nht' => 'plugin_newslog_news_has_tag'), 'nht.tag_id = t.id AND nht.news_id = n.id', array())
                ->where('t.name IN (?)', $tags);
        }

        if (!empty($order)) {
            $select->join(array('p' => 'page'), 'p.id = n.page_id', array())
                ->order($order);
        }

        if ($where) {
            $select->where($where);
        }

        if ($includeCount) {
            $resultSet = $this->getDbTable()->fetchAll($select)->toArray();

            if(null === $resultSet) {
                return null;
            }

            return array(
                'total' => sizeof($resultSet),
                'data' => array_slice($resultSet, $offset, $count),
                'offset' => $offset,
                'limit' => $count
            );
        } else {
            if (!is_null($count)) {
                $select->limit($count, $offset);
            }
        }


        $resultSet = $this->getDbTable()->fetchAll($select);
        if (null === $resultSet) {
            return null;
        }
        foreach ($resultSet as $row) {
            $entries[] = $this->_toModel($row);
        }
        return $entries;
    }

    public function fetchAll($where = null, $order = array(), $count = null, $offset = null, $tags = array(), $search = null, $includeCount = false, $excludedNewsIds = array()) {
        $entries = array();

        if ($search !== null) {
            $where = ($where === null) ? 'title LIKE "%' . $search . '%"' : ($where . ' AND title LIKE "%' . $search . '%"');
        }

        $select = $this->getDbTable()->select(Zend_Db_Table::SELECT_WITHOUT_FROM_PART);
        $select->from(array('n' => 'plugin_newslog_news'));

        if(!empty($order)) {
            $select->join(array('p' => 'page'), 'p.id = n.page_id', array())
                ->order($order);
        }

        $select->setIntegrityCheck(false);
        $select->group('n.id');

        if(!empty($tags)) {
            $select->from(array('t' => 'plugin_newslog_tag'), null)
                ->join(array('nht' => 'plugin_newslog_news_has_tag'), 'nht.tag_id = t.id AND nht.news_id = n.id', array())
                ->where('t.name IN (?)', $tags);
        }

        if($where) {
            if(!empty($excludedNewsIds)) {
                $where .= ' AND ' . $this->getDbTable()->getAdapter()->quoteInto('n.id NOT IN(?)', $excludedNewsIds);
            }

            $select->where($where);
        }

        if($includeCount) {
            $resultSet = $this->getDbTable()->fetchAll($select)->toArray();
            $data      = array();
            foreach ($resultSet as $resultRow) {
                $data[] = $this->_toModel($resultRow)->toArray();
            }
            return array(
                'total'  => sizeof($resultSet),
                'data'   => array_slice($data, $offset, $count),
                'offset' => $offset,
                'limit'  => $count
            );
        } else {
            if(!is_null($count)) {
                $select->limit($count, $offset);
            }
        }


        $resultSet = $this->getDbTable()->fetchAll($select);
        if(null === $resultSet) {
            return null;
        }
        foreach ($resultSet as $row) {
            $entries[]   = $this->_toModel($row);
        }
        return $entries;
    }

    private function _toModel($row, $fetchTags = true) {
        if(!is_array($row)) {
            $row = $row->toArray();
        }
        if($fetchTags) {
            $row['tags'] = $this->_fetchRelatedTags($row['id']);
        }
        return new $this->_model($row);
    }

    private function _fetchRelatedTags($newsId) {
        $tagMapper = Newslog_Models_Mapper_TagMapper::getInstance();
        $tags      = $tagMapper->findByNewsId($newsId);
        if(is_array($tags) && !empty($tags)) {
            return array_map(function($tag) {
                return array(
                    'id'   => $tag->getId(),
                    'name' => $tag->getName()
                );
            }, $tags);
        }
        return null;
    }

    /**
     * @param $news
     * @return mixed
     */
    public function fetchPublishAtByPageId($news){
        $dbTable = new Application_Model_DbTable_Page();
        $pageIds = array();
        foreach ($news as $new)
        {
            array_push($pageIds, $new->getPageId());
        }

        $select = $dbTable->select(Zend_Db_Table::SELECT_WITHOUT_FROM_PART);
        $select->from(array('p' => 'page'), array(
            'p.id',
            'p.publish_at'
        ))
            ->where('p.id IN (?)', $pageIds);
        $publishAtPairsArr = $this->getDbTable()->getAdapter()->fetchPairs($select);

        return  $publishAtPairsArr;

    }

}
