<?php

// (c) Copyright by authors of the Tiki Wiki CMS Groupware Project
//
// All Rights Reserved. See copyright.txt for details and a complete list of authors.
// Licensed under the GNU LESSER GENERAL PUBLIC LICENSE. See license.txt for details.

/**
 *
 * This class represents a structured view (per word) on a document. Feeding it with additional references, it can be used to generate a
 * complete view of the document including changes made over time (like the "Track changes" in some word processing programs). A statistics
 * of the different authors contributions can be generated as well
 *
 * @author cdrwhite
 * @since 6.0
 */

namespace Tiki\Lib\Wiki;

use Tiki\Lib\Diff\TextDiff;
use Tiki\Lib\Diff\Op\Add;
use Tiki\Lib\Diff\Op\Copy;
use Tiki\Lib\Diff\Op\Delete;
use TikiLib;

class Document
{
    /**
     * @var bool
     */
    private $history;
    private $showpopups;
    /**
     * @var array   a list of words and whitespaces represented by an array(word,author,deleted,diffid,[deleted_by])
     */
    private $document;

    /**
     * @var array   array of statistical data grouped by author each represented by an array(words,deleted_words,whitespaces,deleted_whitespaces,characters,deleted_characters,printables,deleted_printables)
     * @see getStatistics
     */
    private $statistics;

    /**
     * @var array   sum of all statistics for all authors, generated by getStatistics, retrieved by getTotal()
     * @see getTotal;
     */
    private $total;

    /**
     * @var string  filter used in getStatistics to distinguish between characters and printable characters
     * @see getStatistics
     */
    private $filter;

    /**
     * @var int processing settings
     */
    private $process = 1;

    /**
     * @var bool    should the page contents be parsed (HTML instead of WIKI text)
     */
    private $parsed;

    /**
     * @var bool    should the html tags be stripped from the parsed contents
     */
    private $nohtml;

    /**
     * @var string  start marker. If set, text before this marker (including the marker itself) will be removed
     */
    public $startmarker = '';

    /**
     * @var string  end marker. If set, text after this marker (including the marker itself) will be removed
     */
    public $endmarker = '';

    /**
     * @var string  regex for splitting page text into an array of words;
     */
    private $search = "#(\[[^\[].*?\]|\(\(.*?\)\)|(~np~\{.*?\}~/np~)|<[^>]+>|[,\"':\s]+|[^\s,\"':<]+|</[^>]+>)#";

    /**
     * @var array   Page info
     */
    private $info;

    /**
     * @var array   complete page history
     */
    private $data;

    /**
     *
     * Initializing Internal variables for getStatistics and getTotals and adding the first page to the document
     * @param string    $page       Name of the page to include
     * @param int       $lastversion    >0 uses the version specified (or last page, if this is greater than the version of the last page) =0 uses the latest(current) version, <0 means a timestamp (lastModif has to be before that)
     * @param int       $process    0 = don't parse (take original wiki text and count wiki tags/plugins), 1 = parse (take html as base), 2 = parse and strip html tags
     * @param string    $start      start marker (all text will be skipped, including this marker which must be at the beginning of a line)
     * @param string    $end        end marker (all text will be skipped from this marker on, including this marker which must be at the beginning of a line)
     */
    public function __construct($page, $lastversion = 0, $process = 1, $showpopups = true, $startmarker = '', $endmarker = '')
    {
        $histlib = TikiLib::lib('hist');

        $this->document = [];
        $this->history = false;
        $this->filter = '/([[:blank:]]|[[:cntrl:]]|[[:punct:]]|[[:space:]])/';
        $this->parsed = true;
        $this->nohtml = false;
        $this->showpopups = $showpopups;
        switch ($process) {
            case 0:
                $this->parsed = false;
                    $this->process = 0;
                break;
            case 2:
                $this->nohtml = true;
                    $this->process = 2;
                break;
        }
        $this->startmarker = $startmarker;
        $this->endmarker = $endmarker;

        $this->info = $histlib->get_hist_page_info($page, true);
        if ($lastversion == 0) {
            $lastversion = $this->info['version'];
        }
        $this->data = [];
        $this->data = [[
                'version'       => $this->info['version'],
                'lastModif'     => $this->info['lastModif'],
                'user'          => $this->info['user'],
                'ip'            => $this->info['ip'],
                'pageName'      => $page,
                'description'   => $this->info['description'],
                'comment'       => $this->info['comment'],
                'data'          => $this->info['data'],
            ]];
        $this->data = array_merge($this->data, $histlib->get_page_history($page, true, 0, -1));
        $next = count($this->data) - 1;
        $author = $this->data[$next]['user'];
        $next = $this->getLastAuthorText($author, $next, $lastversion);
        if ($next == -1) {  // all pages from the same author, no need to diff
            $index = $this->getIndex($lastversion);
        } else {
            $index = $next;
        }
        $source = $this->removeText($this->data[$index]['data']);
        $source = preg_replace(['/\{AUTHOR\(.+?\)\}/','/{AUTHOR\}/','/\{INCLUDE\(.+?\)\}\{INCLUDE\}/'], ' ~np~$0~/np~', $source);
        if ($this->parsed) {
            $source = TikiLib::lib('parser')->parse_data($source, ['suppress_icons' => true]);
        }
        if ($this->nohtml) {
            $source = strip_tags($source);
        }
        preg_match_all($this->search, $source, $out, PREG_PATTERN_ORDER);
        $words = $out[0];
        $this->document = $this->addWords($this->document, $words, $author);
        if ($next == -1) {
            return;
        }
        do {
            $author = $this->data[$next - 1]['user'];
            $next = $this->getLastAuthorText($author, $next - 1, $lastversion);
            if ($next == -1) {
                $index = $this->getIndex($lastversion);
            } else {
                $index = $next;
            }
            $newpage = $this->removeText($this->data[$index]['data']);
            $this->mergeDiff($newpage, $author);
        } while ($next > 0);
        $this->parseAuthorAndInclude();
    }

    /**
     *
     * Removes all text before the first occurrence of start marker and after the last occurrence of the end marker
     * This copies the original behaviour of the wikiplugin_include even though it could be done with a regex in fewer lines
     * @param   string $text    contains the whole text
     * @return  string          returns the text inside the markers
     */
    private function removeText($text)
    {
        $start = ($this->startmarker != '');
        $stop = ($this->endmarker != '');
        if ($start || $stop) {
            $explText = explode("\n", $text);
            if ($start && $stop) {
                $state = 0;
                foreach ($explText as $i => $line) {
                    if ($state == 0) {
                        // Searching for start marker, dropping lines until found
                        unset($explText[$i]);   // Drop the line
                        if (0 == strcmp($this->startmarker, trim($line))) {
                            $state = 1; // Start retaining lines and searching for stop marker
                        }
                    } else {
                        // Searching for stop marker, retaining lines until found
                        if (0 == strcmp($this->endmarker, trim($line))) {
                            unset($explText[$i]);   // Stop marker, drop the line
                            $state = 0;         // Go back to looking for start marker
                        }
                    }
                }
            } elseif ($start) {
                // Only start marker is set. Search for it, dropping all lines until it is found.
                foreach ($explText as $i => $line) {
                    unset($explText[$i]); // Drop the line
                    if (0 == strcmp($this->startmarker, trim($line))) {
                        break;
                    }
                }
            } else {
                // Only stop marker is set. Search for it, dropping all lines after it is found.
                $state = 1;
                foreach ($explText as $i => $line) {
                    if ($state == 0) {
                        // Dropping lines
                        unset($explText[$i]);
                    } else {
                        // Searching for stop marker, retaining lines until found
                        if (0 == strcmp($this->endmarker, trim($line))) {
                            unset($explText[$i]);   // Stop marker, drop the line
                            $state = 0;         // Start dropping lines
                        }
                    }
                }
            }
            $text = implode("\n", $explText);
        }
        return $text;
    }

    /**
     *
     * get the id of the last text of the given author
     * @param string    $author     name of the current author
     * @param int       $start      start index
     * @param int       $lastversion    last version to check, assuming all versions, if none is provided
     * @return  int                 id of the first text of a different author or -1 if there is none
     * @see get_page_history_all
     */
    private function getLastAuthorText($author, $start = -1, $lastversion = -1)
    {
        if ($start == -1) {
            return $start;
        }
        if ($start < 0) {
            $start = count($this->data) - 1;
        }
        if ($lastversion == -1) {
            $lastversion = $this->data[0]['version'];
        }
        $i = $start;
        while ($i >= 0 and $this->data[$i]['user'] == $author and $this->data[$i]['version'] <= $lastversion) {
            $i--;
        }
        $i++;
        if ($this->data[$i]['version'] >= $lastversion) {
            $i = -1;
        }
        return $i;
    }

    /**
     *
     * gets the index position of the requested version in the data array
     * @param int   $version
     */
    private function getIndex($version)
    {
        for ($i = count($this->data) - 1; $i >= 0; $i--) {
            if ($this->data[$i]['version'] == $version) {
                return $i;
            }
        }
        return -1;
    }

    /**
     *
     * returns the history (identical to $histlib->get_page_history, but saves another fetch from database as we already have the info
     */
    public function getHistory()
    {
        return array_slice($this->data, 1);
    }

    /**
     *
     * returns the page info history (identical to $tikilib->get_page_info, but saves another fetch from database as we already have the info
     */
    public function getInfo()
    {
        return $this->info;
    }


    /**
     *
     * Generates an array of words from the internal document structure, which can be used by the diff class.
     * The internal document structure will be modified to allow mergeDiff to integrate a new page with the current page without losing any information
     * @see mergeDiff
     * @return  array   list of words in the document (no author etc.)
     */
    public function getDiffArray()
    {
        $diffarray = [];
        foreach ($this->document as &$word) {
            if (! $word['deleted']) {
                $word['diffid'] = count($diffarray);
                $diffarray[] = $word['word'];
            } else {
                $word['diffid'] = -1;
            }
        }
        return $diffarray;
    }

    /**
     *
     * Generates a statistics per author, the totals can be retrieved via getTotal
     * @see     getTotal
     * @param   string  $filter     regex to filter out non printable characters (difference between characters and printables)
     * @return  array               array indexed by author containing arrays with statistics (words, deleted_words, whitespaces, deleted_whitespaces, characters, deleted_characters, printables, deleted_printables)
     */
    public function getStatistics($filter = '/([[:blank:]]|[[:cntrl:]]|[[:punct:]]|[[:space:]])/')
    {
        $style = 0;
        if ($this->filter != $filter) { //a new filter invalidates the statistics
            $this->statistics = false;
            $this->filter = $filter;
        }
        if ($this->statistics != false) {
            return $this->statistics; //there is already a history for the current state
        }
        $this->statistics = [];
        $this->total = [
                    'words' => 0,
                    'deleted_words' => 0,
                    'whitespaces' => 0,
                    'deleted_whitespaces' => 0,
                    'characters'    => 0,
                    'deleted_characters' => 0,
                    'printables' => 0,
                    'deleted_printables' => 0,
                ];

        foreach ($this->document as $word) {
            $author = $word['author'];
            if (! isset($this->statistics[$author])) {
                $this->statistics[$author] = [
                    'words' => 0,
                    'words_percent' => 0,
                    'deleted_words' => 0,
                    'deleted_words_percent' => 0,
                    'whitespaces' => 0,
                    'whitespaces_percent' => 0,
                    'deleted_whitespaces' => 0,
                    'deleted_whitespaces_percent' => 0,
                    'characters'    => 0,
                    'characters_percent' => 0,
                    'deleted_characters' => 0,
                    'deleted_characters_percent' => 0,
                    'printables' => 0,
                    'printables_percent' => 0,
                    'deleted_printables' => 0,
                    'deleted_printables_percent' => 0,
                    'style' => "author$style",
                ];
                $style++;
                if ($style > 15) {
                    $style = 0;
                }
            } //isset author
            if ($word['deleted']) {
                $prefix = 'deleted_';
            } else {
                $prefix = '';
            }
            $w = $word['word'];
            if ($this->nohtml) {
                $w = strip_tags($w);
            }
            if (trim($w) == '') {
                $this->statistics[$author][$prefix . 'whitespaces']++;
                $this->total[$prefix . 'whitespaces']++;
            } else {
                $this->statistics[$author][$prefix . 'words']++;
                $this->total[$prefix . 'words']++;
            }
            $l = mb_strlen($w);
            $this->statistics[$author][$prefix . 'characters'] += $l;
            $this->total[$prefix . 'characters'] += $l;
            $l = mb_strlen(preg_replace($this->filter, '', $w));
            $this->statistics[$author][$prefix . 'printables'] += $l;
            $this->total[$prefix . 'printables'] += $l;
        } //foreach
        //calculate percentages
        foreach ($this->statistics as &$author) {
            $author['words_percent'] = $author['words'] / $this->total['words'];
            $author['deleted_words_percent'] = ($this->total['deleted_words'] != 0 ? $author['deleted_words'] / $this->total['deleted_words'] : 0);
            $author['whitespaces_percent'] = $author['whitespaces'] / $this->total['whitespaces'];
            $author['deleted_whitespaces_percent'] = ($this->total['deleted_whitespaces'] != 0 ? $author['deleted_whitespaces'] / $this->total['deleted_whitespaces'] : 0);
            $author['characters_percent'] = $author['characters'] / $this->total['characters'];
            $author['deleted_characters_percent'] = ($this->total['deleted_characters'] != 0 ? $author['deleted_characters'] / $this->total['deleted_characters'] : 0);
            $author['printables_percent'] = $author['printables'] / $this->total['printables'];
            $author['deleted_printables_percent'] = ($this->total['deleted_printables'] != 0 ? $author['deleted_printables'] / $this->total['deleted_printables'] : 0);
        }
        return $this->statistics;
    }

    /**
     *
     * gets the totals from a previous getStatistics call
     * @see     getStatistics
     * @return  array with statistics (words, deleted_words, whitespaces, deleted_whitespaces, characters, deleted_characters, printables, deleted_printables)
     */
    public function getTotal()
    {
        return $this->total;
    }

    /**
     *
     * Retrieves the document data in different formats,
     * @param string $type      can be one of 'words' (array of words/whitespaces), 'text' (unformatted string), 'wiki' (string with wikiplugin AUTHOR tags to show the authors) or the default empty string '' (returns the internal document structure)
     * @param array  $options   array containing the filter specific options:
     * <table>
     * <tr><th>Type</th><th>Name</th><th>Applicable for</th><th>Purpose</th></tr>
     * <tr><td>bool</td><td>showpopups</td><td>wiki</td><td>renders popups, defaults to true</td></tr>
     * <tr><td>bool</td><td>escape</td><td>text/wiki</td><td>Escapes brackets and htmlspecialchars</td></tr>
     * </table>
     * @return  array|string    depending on the parameter $type, a string or array containing the documents words
     */
    public function get($type = '', $options = [])
    {
        switch ($type) {
            case 'words':
                $words = [];
                foreach ($this->document as $word) {
                    $words[] = $word['word'];
                }
                return $words;
                break;
            case 'text':
                $text = '';
                foreach ($this->document as $word) {
                    $text .= $word['word'];
                }
                return $text;
            if ($options['escape']) {
                if (! $this->parsed) {
                    $text = '~np~' .
                          preg_replace(['/\~np\~/', '//\~\/np\~/'], ['&#126;np&#126;','&#126;/np&#126;;'], $text) .
                          '~/np~';
                }
                $text = preg_replace(['/</','/>/'], ['&lt;','&gt;'], $text);
            }
                break;
            case 'wiki':
                $text = '';
                $author = '';
                $deleted = 0;
                $deleted_by = '';
                if (isset($options['showpopups'])) {
                    $showpopups = $options['showpopups'];
                } else {
                    $showpopups = true;
                }
                foreach ($this->document as $word) {
                    $skip = false;
                    $d = isset($word['deleted_by']) ? $word['deleted_by'] : '';
                    $w = $word['word'];
                    if ($author != $word['author'] or $deleted != $word['deleted'] or $deleted_by != $d) {
                        if ($text != '') {
                            if ($options['escape']) {
                                $text .= '~/np~';
                            }
                            $text .= '{AUTHOR}';
                        }
                        $author = $word['author'];
                        $deleted = $word['deleted'];
                        $deleted_by = $d;
                        $text .= "{AUTHOR(author=\"$author\"" .
                                ($deleted ? ",deleted_by=\"$deleted_by\"" : '') .
                                ',visible="1"' .
                                ($showpopups ? ', popup="1"' : '') .
                                ')}';
                        if ($options['escape']) {
                            $text .= "~np~";
                        }
                    }
                    if (! $options['escape']) {
                        if ($this->parsed and ! $this->nohtml) { // skipping popups for links
                            if (substr($w, 0, 3) == '<a ') {
                                $text .= '{AUTHOR}';
                            }
                            if (substr($w, -4) == '</a>') {
                                $text .= $w . "{AUTHOR(author=\"$author\"" .
                                       ($deleted ? ",deleted_by=\"$deleted_by\"" : '') .
                                       ',visible="1", ' .
                                       ($showpopups ? ', popup="1"' : '') .
                                       ')}';
                                $skip = true;
                            }
                        }
                    } else { //escape existing tags
                        if (! $this->parsed) {
                              $w = preg_replace(['/\~np\~/', '/\~\/np\~/'], ['&#126;np&#126;','&#126;/np&#126;'], $w);
                        }
                        $w = preg_replace(['/</','/>/'], ['&amp;lt;','&amp;gt;'], $w); //double encode!
                    }
                    if (strlen($w) == 0 and ! $this->parsed) {
                        $text .= "\n";
                    } else {
                        if (! $skip) {
                            $text .= $w;
                        }
                    }
                } // foreach
                if ($options['escape']) {
                    $text .= "~/np~";
                }
                $text .= "{AUTHOR}";
                return $text;
                break;
            default:
                return $this->document;
        }
    }

    /**
     *
     * Adds the supplied list of words to the provided document structure
     * @param array     $doc        a list of words (arrays containing word, author, deleted, diffid, optionally deleted_by and statistical data) where the new words will be added to
     * @param array     $list       array of words/whitespaces to add to the document
     * @param string    $author     name of the author to credit
     * @return                      provided document structure $doc with the words from $list appended
     */
    private function addWords($doc, $list, $author, $deleted = false, $deleted_by = '')
    {
        $newdoc = $doc;
        foreach ($list as $word) {
            $newword = [
                            'word'      => $word,
                            'author'    => $author,
                            'deleted'   => $deleted,
                            'diffid'    => -1,
                            ];
            if ($deleted) {
                $newword['deleted_by'] = $deleted_by;
            }
            $newdoc[] = $newword;
        }
        return $newdoc;
    }

    /**
     *
     * moves a nuber of words from the b eginning of this document to the provided document structure
     * @param array     $doc        a list of words (arrays containing word, author, deleted, diffid, optionally deleted_by and statistical data) where the new words will be appended to
     * @param int       $pos        number of characters to move from the current documents beginning to the new list, deleted words which have a negative diff id wille be moved but not counted
     * @param array     $list       list of words to move
     * @param bool      $setDeleted mark the moved words as deleted, if not already deleted
     * @param string    $deletedBy  name of the author who deleted the words
     */
    private function moveWords(&$doc, &$pos, $list, $deleted = false, $deleted_by = '')
    {
        $pos += count($list);
        // get the words from the old document
        $i = 0;
        while ($i < count($this->document) and $this->document[$i]['diffid'] < $pos) {
            $word = $this->document[$i];
            if ($deleted) {
                if (! $word['deleted']) {
                    $word['deleted'] = true;
                    $word['deleted_by'] = $deleted_by;
                }
            }
            $doc[] = $word;
            $i++;
        }
        //take care of deleted words
        while ($i < count($this->document) and $this->document[$i]['diffid'] < 0) {
            $word = $this->document[$i];
            $doc[] = $word;
            $i++;
        }
        $this->document = array_slice($this->document, $i);
    }

    /**
     *
     * Returns an indexed array containing the plugins parameters indexed by key name
     * @param string    $pluginstr      Complete Plugin tag including brackets () containing the parameters
     * @return  array|bool              Array containing the parameters or false if none are given
     */
    public function retrieveParams($pluginstr)
    {
        $params = [];
        $start = strpos($pluginstr, '(');
        if ($start === false) {
            return false;
        }
        $end = strrpos($pluginstr, ')');
        if ($end === false) {
            return false;
        }
        $pstr = substr($pluginstr, $start + 1, $end - $start - 1);
        $plist = explode(',', $pstr);
        foreach ($plist as $paramstr) {
            $p = explode('=', trim($paramstr));
            $params[strtolower(trim($p[0]))] = preg_replace('/^"|^\&quot;|"$|\&quot;$/', '', trim($p[1]));
        }
        return $params;
    }

    /**
     *
     * merges a newer version of a page into the current document
     * @param string    $newpage    a string with a later version of the page
     * @param string    $newauthor  name of the author of the new version
     */
    public function mergeDiff($newpage, $newauthor)
    {
        $this->history = false;
        $author = $newauthor;
        $deleted = false;
        $deleted_by = '';
        $newdoc = [];
        $page = preg_replace(['/\{AUTHOR\(.+?\)\}/','/{AUTHOR\}/','/\{INCLUDE\(.+?\)\}\{INCLUDE\}/'], ' ~np~$0~/np~', $newpage);
        if ($this->parsed) {
            $page = TikiLib::lib('parser')->parse_data($page, ['suppress_icons' => true]);
            $page = preg_replace(['/\{AUTHOR\(.+?\)\}/','/{AUTHOR\}/','/\{INCLUDE\(.+?\)\}\{INCLUDE\}/'], ' ~np~$0~/np~', $page);
        }
        if ($this->nohtml) {
            $page = strip_tags($page);
        }
        preg_match_all($this->search, $page, $out, PREG_PATTERN_ORDER);
        $new = $out[0];
        $z = new TextDiff($this->getDiffArray(), $new);
        $pos = 0;
        foreach ($z->getDiff() as $element) {
            if (is_a($element, Copy::class)) {
                $this->moveWords($newdoc, $pos, $element->orig, $deleted, $deleted_by);
            } else {
                if (is_a($element, Add::class)) {
                    $newdoc = $this->addWords($newdoc, $element->final, $author, $deleted, $deleted_by);
                } else {
                    if (is_a($element, Delete::class)) {
                        $this->moveWords($newdoc, $pos, $element->orig, $deleted, $author);
                    } else { //change
                        $newdoc = $this->addWords($newdoc, $element->final, $author, $deleted, $deleted_by);
                        $this->moveWords($newdoc, $pos, $element->orig, true, $author);
                    } //delete
                } // add
            } // copy
        } // foreach diff
        $this->document = $newdoc;
    }

    /**
     *
     * Kills double whitespaces in parseAuthor before/after {author} tags
     * @param array $newdoc array containing the new document
     * @param int   $index  position in the old document
     */
    private function killDoubleWhitespaces(&$newdoc, &$index)
    {
        if (count($newdoc) > 2) {
            $w1 = $newdoc[count($newdoc) - 1]['word'];
            $w2 = $newdoc[count($newdoc) - 2]['word'];
            if ($this->nohtml) {
                $w1 = strip_tags($w1);
                $w2 = strip_tags($w2);
            }
            if (trim($w1) == '' and trim($w2) == '') {
                array_pop($newdoc); // kill one of the whitespaces
            }
        }
        if ($index < count($this->document) - 2) {
            $w1 = $this->document[$index + 1]['word'];
            $w2 = $this->document[$index + 2]['word'];
            if ($this->nohtml) {
                $w1 = strip_tags($w1);
                $w2 = strip_tags($w2);
            }
            if (trim($w1) == '' and trim($w2) == '') {
                $index++; // jump over one of the whitespaces
            }
        }
    }

    /**
     *
     * parses the left over author/include tags and sets the author accordingly
     */
    public function parseAuthorAndInclude()
    {
        $newdoc = [];
        $author = '';
        $deleted_by = '';
        for ($index = 0, $cdoc = count($this->document); $index < $cdoc; $index++) {
            $word = $this->document[$index];
            if (preg_match('/\{AUTHOR\(.+?\)\}/', $word['word'])) {
                $params = $this->retrieveParams($word['word']);
                $author = $params['author'];
                if (isset($params['deleted_by'])) {
                    $deleted_by = $params['deleted_by'];
                }
                // manage double whitespace before and after
                $this->killDoubleWhitespaces($newdoc, $index);
            } elseif (preg_match('/\{AUTHOR\}/', $word['word'])) {
                $author = '';
                $deleted_by = '';
                $this->killDoubleWhitespaces($newdoc, $index);
            } elseif (preg_match('/\{INCLUDE\(.+?\)\}\{INCLUDE\}/', $word['word'])) {
                $params = $this->retrieveParams($word['word']);
                $start = '';
                $stop = '';
                if (isset($params['start'])) {
                    $start = $params['start'];
                }
                if (isset($params['stop'])) {
                    $stop = $params['stop'];
                }
                $subdoc = new Document($params['page'], 0, $this->process, $this->showpopups, $start, $stop);
                $newdoc = array_merge($newdoc, $subdoc->get());
            } else { //normal word
                if ($author != '') {
                    $word['author'] = $author;
                }
                if ($deleted_by != '') {
                    $word['deleted'] = true;
                    $word['deleted_by'] = $deleted_by;
                }
                $newdoc[] = $word;
            }
        } //foreach
        $this->document = $newdoc;
    }
}
