# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

"""
This module contains an argparse command line parser
for cloudlinux-ssa-manager and cloudlinux-autotracing
"""
import argparse
import logging



def cmd_parser_manager() -> 'argparse.ArgumentParser':
    """
    Command line parser for SSA manager utility using built-in argparse module
    :return: parser
    """
    parser = argparse.ArgumentParser(prog='cloudlinux-ssa-manager',
                                     description='Utility to manage SSA')
    subparsers = parser.add_subparsers(title='Commands', dest='command',
                                       required=True)

    set_config_parser = subparsers.add_parser('set-config',
                                              help='Set SSA configuration',
                                              argument_default=argparse.SUPPRESS)
    set_config_parser.add_argument('--domains-number', metavar='DOMAINS_NUMBER',
                                   help='Size of TOP list for slow domains')
    set_config_parser.add_argument('--request-number', metavar='REQUEST_NUMBER',
                                   help='The threshold value of slow requests number in the period of time to mark URL as a slow one')
    set_config_parser.add_argument('--time', metavar='TIME',
                                   help='Period of time in hours required to analyze these requests')
    set_config_parser.add_argument('--correlation', metavar='CORRELATION',
                                   help='Flag to enable or disable correlation')
    set_config_parser.add_argument('--correlation-coefficient', metavar='CORRELATION_COEFFICIENT',
                                   help='The threshold value of correlation coefficient')
    set_config_parser.add_argument('--urls-number', metavar='URLS_NUMBER',
                                   help='Size of TOP list for slow urls')
    set_config_parser.add_argument('--requests-duration', metavar='REQUESTS_DURATION',
                                   help='The threshold value of request duration in seconds')
    set_config_parser.add_argument('--ignore-list', metavar='IGNORE_LIST',
                                   help='List of URLs or domains that should not be included in the daily report')
    set_config_parser.add_argument('--summary-notification-enabled', metavar='SUMMARY_NOTIFICATION',
                                   help='Turn on|off sending the daily report by e-mail')

    subparsers.add_parser('get-config', help='Get SSA configuration',
                          argument_default=argparse.SUPPRESS)
    subparsers.add_parser('get-ssa-status', help='Get current status of SSA',
                          argument_default=argparse.SUPPRESS)
    subparsers.add_parser('enable-ssa', help='Enable SSA',
                          argument_default=argparse.SUPPRESS)
    subparsers.add_parser('disable-ssa', help='Disable SSA',
                          argument_default=argparse.SUPPRESS)
    subparsers.add_parser('get-report', help='Get the latest report',
                          argument_default=argparse.SUPPRESS)
    subparsers.add_parser('get-stat', help='Get the SSA statistics',
                          argument_default=argparse.SUPPRESS)
    return parser


def cmd_parser_autotracing_util() -> 'argparse.ArgumentParser':
    """
    Command line parser for Autotracing using built-in argparse module
    :return: parser
    """
    parser = argparse.ArgumentParser(prog='cloudlinux-autotracing',
                                     description='Utility to manage Auto Tracing')
    subparsers = parser.add_subparsers(title='Commands', dest='command',
                                       required=True)

    e_parser = subparsers.add_parser('enable',
                                     help='Enable Auto Tracing feature')
    e_parser.add_argument('--all', action='store_true', default=False,
                          help='enable for all users')
    e_parser.add_argument('user', nargs='?', default=None,
                          help='enable for a particular user')

    d_parser = subparsers.add_parser('disable',
                                     help='Disable Auto Tracing feature')
    d_parser.add_argument('--all', action='store_true', default=False,
                          help='disable for all users')
    d_parser.add_argument('user', nargs='?', default=None,
                          help='disable for a particular user')

    s_parser = subparsers.add_parser('status',
                                     help='Get status of Auto Tracing feature')
    s_parser.add_argument('--list-disabled', action='store_true',
                          default=False, help='list disabled users')
    s_parser.add_argument('user', nargs='?', default=None,
                          help='status of a particular user')

    return parser


def parse_cmd_arguments(
        parser: 'argparse.ArgumentParser') -> 'argparse.Namespace':
    """
    Parse arguments with given parser
    :param parser: an instance of ArgumentParser
    :return: arguments Namespace
    """
    try:
        args = parser.parse_args()
        return args
    except SystemExit as e:
        logging.error('Invalid utility invocation', extra={'err': str(e)})
        raise
