# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2021 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

"""
This module contains Autotracing utility main run function
"""
import json
import os
import sys

from .cmdline_parser import (
    cmd_parser_autotracing_util,
    parse_cmd_arguments
)
from .validations import validate_autotracing
from ..autotracing import enable, disable, status, disabled_users
from ..internal.constants import autotracing_log
from ..internal.exceptions import SSAError
from ..internal.utils import configure_logging


def response(msg: str, **kwargs) -> None:
    """Pack response"""
    initial = {'result': msg}
    initial.update({k: v for k, v in kwargs.items() if v is not None})
    print(json.dumps(initial))


def run() -> None:
    """
    Main run function
    """
    current_uid = os.getuid()

    configure_logging(autotracing_log)
    args = parse_cmd_arguments(cmd_parser_autotracing_util())
    validated_args = validate_autotracing(args.__dict__)

    if validated_args.command in ('enable', 'disable'):
        if validated_args.all and validated_args.user:
            response("--all and username are not allowed together")
            sys.exit(1)

        if current_uid == 0:
            # Admin is to specify --all or username
            if not (validated_args.all or validated_args.user):
                response("either --all or username must be specified")
                sys.exit(1)
        else:
            # User cannot specify neither --all nor username
            if validated_args.all or validated_args.user:
                response("neither --all nor username are allowed")
                sys.exit(1)

    if validated_args.command == 'status':
        if validated_args.list_disabled and validated_args.user:
            response("--list_disabled and username are not allowed together")
            sys.exit(1)

    try:
        if validated_args.command == 'enable':
            issue = enable(validated_args.user, validated_args.all)
            response("success", malfunction=issue)
        elif validated_args.command == 'disable':
            disable(validated_args.user, validated_args.all)
            response("success")
        elif validated_args.command == 'status':
            current_status, issue = status(validated_args.user)
            if validated_args.list_disabled:
                if current_uid != 0:
                    response("--list-disabled is not allowed")
                    sys.exit(1)
                response("success", status=current_status,
                         disabled_users=disabled_users(),
                         malfunction=issue)
            else:
                response("success", status=current_status,
                         malfunction=issue)
    except SSAError as e:
        print(e)
        sys.exit(1)
