/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { BlockControls, InspectorControls } from '@wordpress/block-editor';
import { ServerSideRender } from '@wordpress/editor';
import {
	Button,
	Disabled,
	PanelBody,
	Placeholder,
	RangeControl,
	Toolbar,
	withSpokenMessages,
	ToggleControl,
} from '@wordpress/components';
import { Component, Fragment } from '@wordpress/element';
import PropTypes from 'prop-types';
import { MAX_COLUMNS, MIN_COLUMNS } from '@kkart/block-settings';
import GridContentControl from '@kkart/editor-components/grid-content-control';
import ProductsControl from '@kkart/editor-components/products-control';
import ProductOrderbyControl from '@kkart/editor-components/product-orderby-control';
import { gridBlockPreview } from '@kkart/resource-previews';
import { Icon, widgets } from '@kkart/icons';

/**
 * Component to handle edit mode of "Hand-picked Products".
 */
class ProductsBlock extends Component {
	getInspectorControls() {
		const { attributes, setAttributes } = this.props;
		const {
			columns,
			contentVisibility,
			orderby,
			alignButtons,
		} = attributes;

		return (
			<InspectorControls key="inspector">
				<PanelBody
					title={ __( 'Layout', 'kkart' ) }
					initialOpen
				>
					<RangeControl
						label={ __(
							'Columns',
							'kkart'
						) }
						value={ columns }
						onChange={ ( value ) =>
							setAttributes( { columns: value } )
						}
						min={ MIN_COLUMNS }
						max={ MAX_COLUMNS }
					/>
					<ToggleControl
						label={ __(
							'Align Buttons',
							'kkart'
						) }
						help={
							alignButtons
								? __(
										'Buttons are aligned vertically.',
										'kkart'
								  )
								: __(
										'Buttons follow content.',
										'kkart'
								  )
						}
						checked={ alignButtons }
						onChange={ () =>
							setAttributes( { alignButtons: ! alignButtons } )
						}
					/>
				</PanelBody>
				<PanelBody
					title={ __( 'Content', 'kkart' ) }
					initialOpen
				>
					<GridContentControl
						settings={ contentVisibility }
						onChange={ ( value ) =>
							setAttributes( { contentVisibility: value } )
						}
					/>
				</PanelBody>
				<PanelBody
					title={ __( 'Order By', 'kkart' ) }
					initialOpen={ false }
				>
					<ProductOrderbyControl
						setAttributes={ setAttributes }
						value={ orderby }
					/>
				</PanelBody>
				<PanelBody
					title={ __( 'Products', 'kkart' ) }
					initialOpen={ false }
				>
					<ProductsControl
						selected={ attributes.products }
						onChange={ ( value = [] ) => {
							const ids = value.map( ( { id } ) => id );
							setAttributes( { products: ids } );
						} }
					/>
				</PanelBody>
			</InspectorControls>
		);
	}

	renderEditMode() {
		const { attributes, debouncedSpeak, setAttributes } = this.props;
		const onDone = () => {
			setAttributes( { editMode: false } );
			debouncedSpeak(
				__(
					'Showing Hand-picked Products block preview.',
					'kkart'
				)
			);
		};

		return (
			<Placeholder
				icon={ <Icon srcElement={ widgets } /> }
				label={ __(
					'Hand-picked Products',
					'kkart'
				) }
				className="kkart-block-products-grid kkart-block-handpicked-products"
			>
				{ __(
					'Display a selection of hand-picked products in a grid.',
					'kkart'
				) }
				<div className="kkart-block-handpicked-products__selection">
					<ProductsControl
						selected={ attributes.products }
						onChange={ ( value = [] ) => {
							const ids = value.map( ( { id } ) => id );
							setAttributes( { products: ids } );
						} }
					/>
					<Button isPrimary onClick={ onDone }>
						{ __( 'Done', 'kkart' ) }
					</Button>
				</div>
			</Placeholder>
		);
	}

	render() {
		const { attributes, name, setAttributes } = this.props;
		const { editMode } = attributes;

		if ( attributes.isPreview ) {
			return gridBlockPreview;
		}

		return (
			<Fragment>
				<BlockControls>
					<Toolbar
						controls={ [
							{
								icon: 'edit',
								title: __( 'Edit' ),
								onClick: () =>
									setAttributes( { editMode: ! editMode } ),
								isActive: editMode,
							},
						] }
					/>
				</BlockControls>
				{ this.getInspectorControls() }
				{ editMode ? (
					this.renderEditMode()
				) : (
					<Disabled>
						<ServerSideRender
							block={ name }
							attributes={ attributes }
						/>
					</Disabled>
				) }
			</Fragment>
		);
	}
}

ProductsBlock.propTypes = {
	/**
	 * The attributes for this block
	 */
	attributes: PropTypes.object.isRequired,
	/**
	 * The register block name.
	 */
	name: PropTypes.string.isRequired,
	/**
	 * A callback to update attributes
	 */
	setAttributes: PropTypes.func.isRequired,
	// from withSpokenMessages
	debouncedSpeak: PropTypes.func.isRequired,
};

export default withSpokenMessages( ProductsBlock );
