<?php
/**
 * REST API bootstrap.
 */

namespace Automattic\Kkart\Admin\API;

defined( 'ABSPATH' ) || exit;

use \Automattic\Kkart\Admin\Loader;

/**
 * Init class.
 */
class Init {
	/**
	 * The single instance of the class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return object Instance.
	 */
	final public static function instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}
		return static::$instance;
	}

	/**
	 * Boostrap REST API.
	 */
	public function __construct() {
		// Hook in data stores.
		add_filter( 'kkart_data_stores', array( __CLASS__, 'add_data_stores' ) );
		// REST API extensions init.
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );

		// Add currency symbol to orders endpoint response.
		add_filter( 'kkart_rest_prepare_shop_order_object', array( __CLASS__, 'add_currency_symbol_to_order_response' ) );
	}

	/**
	 * Init REST API.
	 */
	public function rest_api_init() {
		$controllers = array(
			'Automattic\Kkart\Admin\API\Notes',
			'Automattic\Kkart\Admin\API\NoteActions',
			'Automattic\Kkart\Admin\API\Coupons',
			'Automattic\Kkart\Admin\API\Customers',
			'Automattic\Kkart\Admin\API\Data',
			'Automattic\Kkart\Admin\API\DataCountries',
			'Automattic\Kkart\Admin\API\DataDownloadIPs',
			'Automattic\Kkart\Admin\API\Leaderboards',
			'Automattic\Kkart\Admin\API\Marketing',
			'Automattic\Kkart\Admin\API\MarketingOverview',
			'Automattic\Kkart\Admin\API\Options',
			'Automattic\Kkart\Admin\API\Orders',
			'Automattic\Kkart\Admin\API\Products',
			'Automattic\Kkart\Admin\API\ProductCategories',
			'Automattic\Kkart\Admin\API\ProductVariations',
			'Automattic\Kkart\Admin\API\ProductReviews',
			'Automattic\Kkart\Admin\API\ProductVariations',
			'Automattic\Kkart\Admin\API\Reports\Controller',
			'Automattic\Kkart\Admin\API\SettingOptions',
			'Automattic\Kkart\Admin\API\Reports\Import\Controller',
			'Automattic\Kkart\Admin\API\Reports\Export\Controller',
			'Automattic\Kkart\Admin\API\Reports\Products\Controller',
			'Automattic\Kkart\Admin\API\Reports\Variations\Controller',
			'Automattic\Kkart\Admin\API\Reports\Products\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Variations\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Revenue\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Orders\Controller',
			'Automattic\Kkart\Admin\API\Reports\Orders\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Categories\Controller',
			'Automattic\Kkart\Admin\API\Reports\Taxes\Controller',
			'Automattic\Kkart\Admin\API\Reports\Taxes\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Coupons\Controller',
			'Automattic\Kkart\Admin\API\Reports\Coupons\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Stock\Controller',
			'Automattic\Kkart\Admin\API\Reports\Stock\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Downloads\Controller',
			'Automattic\Kkart\Admin\API\Reports\Downloads\Stats\Controller',
			'Automattic\Kkart\Admin\API\Reports\Customers\Controller',
			'Automattic\Kkart\Admin\API\Reports\Customers\Stats\Controller',
			'Automattic\Kkart\Admin\API\Taxes',
			'Automattic\Kkart\Admin\API\Themes',
			'Automattic\Kkart\Admin\API\Plugins',
			'Automattic\Kkart\Admin\API\OnboardingProfile',
			'Automattic\Kkart\Admin\API\OnboardingTasks',
			'Automattic\Kkart\Admin\API\OnboardingThemes',
		);

		// The performance indicators controller must be registered last, after other /stats endpoints have been registered.
		$controllers[] = 'Automattic\Kkart\Admin\API\Reports\PerformanceIndicators\Controller';

		$controllers = apply_filters( 'kkart_admin_rest_controllers', $controllers );

		foreach ( $controllers as $controller ) {
			$this->$controller = new $controller();
			$this->$controller->register_routes();
		}
	}

	/**
	 * Adds data stores.
	 *
	 * @param array $data_stores List of data stores.
	 * @return array
	 */
	public static function add_data_stores( $data_stores ) {
		return array_merge(
			$data_stores,
			array(
				'report-revenue-stats'    => 'Automattic\Kkart\Admin\API\Reports\Orders\Stats\DataStore',
				'report-orders'           => 'Automattic\Kkart\Admin\API\Reports\Orders\DataStore',
				'report-orders-stats'     => 'Automattic\Kkart\Admin\API\Reports\Orders\Stats\DataStore',
				'report-products'         => 'Automattic\Kkart\Admin\API\Reports\Products\DataStore',
				'report-variations'       => 'Automattic\Kkart\Admin\API\Reports\Variations\DataStore',
				'report-products-stats'   => 'Automattic\Kkart\Admin\API\Reports\Products\Stats\DataStore',
				'report-variations-stats' => 'Automattic\Kkart\Admin\API\Reports\Variations\Stats\DataStore',
				'report-categories'       => 'Automattic\Kkart\Admin\API\Reports\Categories\DataStore',
				'report-taxes'            => 'Automattic\Kkart\Admin\API\Reports\Taxes\DataStore',
				'report-taxes-stats'      => 'Automattic\Kkart\Admin\API\Reports\Taxes\Stats\DataStore',
				'report-coupons'          => 'Automattic\Kkart\Admin\API\Reports\Coupons\DataStore',
				'report-coupons-stats'    => 'Automattic\Kkart\Admin\API\Reports\Coupons\Stats\DataStore',
				'report-downloads'        => 'Automattic\Kkart\Admin\API\Reports\Downloads\DataStore',
				'report-downloads-stats'  => 'Automattic\Kkart\Admin\API\Reports\Downloads\Stats\DataStore',
				'admin-note'              => 'Automattic\Kkart\Admin\Notes\DataStore',
				'report-customers'        => 'Automattic\Kkart\Admin\API\Reports\Customers\DataStore',
				'report-customers-stats'  => 'Automattic\Kkart\Admin\API\Reports\Customers\Stats\DataStore',
				'report-stock-stats'      => 'Automattic\Kkart\Admin\API\Reports\Stock\Stats\DataStore',
			)
		);
	}

	/**
	 * Add the currency symbol (in addition to currency code) to each Order
	 * object in REST API responses. For use in formatAmount().
	 *
	 * @param {WP_REST_Response} $response REST response object.
	 * @returns {WP_REST_Response}
	 */
	public static function add_currency_symbol_to_order_response( $response ) {
		$response_data                    = $response->get_data();
		$currency_code                    = $response_data['currency'];
		$currency_symbol                  = get_kkart_currency_symbol( $currency_code );
		$response_data['currency_symbol'] = html_entity_decode( $currency_symbol );
		$response->set_data( $response_data );

		return $response;
	}
}
