import template from './sw-dashboard-statistics.html.twig';
import './sw-dashboard-statistics.scss';

const { Criteria } = Shopware.Data;

type OrderEntity = EntitySchema.order;

type HistoryDateRange = {
    label: string;
    range: number;
    interval: 'hour' | 'day';
    aggregate: 'hour' | 'day';
};

type BucketData = {
    key: string;
    count: number;
    totalAmount: {
        sum: number;
    };
};

type HistoryOrderDataCount = {
    apiAlias: 'order_sum_bucket_aggregation';
    buckets: Array<BucketData>;
    name: 'order_sum_bucket';
};

type HistoryOrderDataSum = {
    apiAlias: 'order_sum_bucket_aggregation';
    buckets: Array<BucketData>;
    name: 'order_sum_bucket';
};

type HistoryOrderData = HistoryOrderDataCount | HistoryOrderDataSum | null;

interface ComponentData {
    historyOrderDataCount: HistoryOrderDataCount | null;
    historyOrderDataSum: HistoryOrderDataSum | null;
    todayOrderData: EntityCollection<'order'> | null;
    todayOrderDataLoaded: boolean;
    todayOrderDataSortBy: 'orderDateTime';
    todayOrderDataSortDirection: 'DESC' | 'ASC';
    ordersDateRange: HistoryDateRange;
    turnoverDateRange: HistoryDateRange;
    isLoading: boolean;
}

/**
 * @sw-package after-sales
 *
 * @private might get removed with any update (even minor!) as it likely gets replaced by shopware analytics
 */
export default Shopware.Component.wrapComponentConfig({
    template,

    inject: [
        'repositoryFactory',
        'stateStyleDataProviderService',
        'acl',
    ],

    data(): ComponentData {
        return {
            historyOrderDataCount: null,
            historyOrderDataSum: null,
            todayOrderData: null,
            todayOrderDataLoaded: false,
            todayOrderDataSortBy: 'orderDateTime',
            todayOrderDataSortDirection: 'DESC',
            ordersDateRange: {
                label: '30Days',
                range: 30,
                interval: 'day',
                aggregate: 'day',
            },
            turnoverDateRange: {
                label: '30Days',
                range: 30,
                interval: 'day',
                aggregate: 'day',
            },
            isLoading: true,
        };
    },

    computed: {
        rangesValueMap(): Array<HistoryDateRange> {
            return [
                {
                    label: '30Days',
                    range: 30,
                    interval: 'day',
                    aggregate: 'day',
                },
                {
                    label: '14Days',
                    range: 14,
                    interval: 'day',
                    aggregate: 'day',
                },
                {
                    label: '7Days',
                    range: 7,
                    interval: 'day',
                    aggregate: 'day',
                },
                {
                    label: '24Hours',
                    range: 24,
                    interval: 'hour',
                    aggregate: 'hour',
                },
                {
                    label: 'yesterday',
                    range: 1,
                    interval: 'day',
                    aggregate: 'hour',
                },
            ];
        },

        availableRanges(): string[] {
            return this.rangesValueMap.map((range) => range.label);
        },

        chartOptionsOrderCount() {
            return {
                xaxis: {
                    type: 'datetime',
                    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment,@typescript-eslint/no-unsafe-call
                    min: this.getDateAgo(this.ordersDateRange).getTime(),
                    labels: {
                        datetimeUTC: false,
                    },
                },
                yaxis: {
                    min: 0,
                    tickAmount: 3,
                    labels: {
                        formatter: (value: string) => {
                            return parseInt(value, 10);
                        },
                    },
                },
            };
        },

        chartOptionsOrderSum() {
            return {
                xaxis: {
                    type: 'datetime',
                    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment,@typescript-eslint/no-unsafe-call
                    min: this.getDateAgo(this.turnoverDateRange).getTime(),
                    labels: {
                        datetimeUTC: false,
                    },
                },
                yaxis: {
                    min: 0,
                    tickAmount: 5,
                    labels: {
                        // price aggregations do not support currencies yet, see NEXT-5069
                        formatter: (value: string) =>
                            Shopware.Utils.format.currency(
                                Number.parseFloat(value),
                                Shopware.Context.app.systemCurrencyISOCode as string,
                                2,
                            ),
                    },
                },
            };
        },

        orderRepository() {
            return this.repositoryFactory.create('order');
        },

        orderCountSeries() {
            if (!this.historyOrderDataCount) {
                return [];
            }

            // format data for chart
            const seriesData = this.historyOrderDataCount.buckets.map((data: BucketData) => {
                return { x: this.parseDate(data.key), y: data.count };
            });

            // add empty value for today if there isn't any order, otherwise today would be missing
            if (!this.todayBucketCount) {
                seriesData.push({ x: this.today.getTime(), y: 0 });
            }

            return [
                {
                    name: this.$tc('sw-dashboard.monthStats.numberOfOrders'),
                    data: seriesData,
                },
            ];
        },

        orderCountToday() {
            if (this.todayBucketCount) {
                return this.todayBucketCount.count;
            }
            return 0;
        },

        orderSumMonthSeries() {
            return this.orderSumSeries;
        },

        orderSumSeries() {
            if (!this.historyOrderDataSum) {
                return [];
            }

            // format data for chart
            const seriesData = this.historyOrderDataSum.buckets.map((data: BucketData) => {
                return {
                    x: this.parseDate(data.key),
                    y: data.totalAmount.sum,
                };
            });

            // add empty value for today if there isn't any order, otherwise today would be missing
            if (!this.todayBucketSum) {
                seriesData.push({ x: this.today.getTime(), y: 0 });
            }

            return [
                {
                    name: this.$tc('sw-dashboard.monthStats.totalTurnover'),
                    data: seriesData,
                },
            ];
        },

        orderSumToday() {
            if (this.todayBucketCount) {
                return this.todayBucketCount.totalAmount.sum;
            }
            return 0;
        },

        hasOrderToday() {
            return this.todayOrderData && this.todayOrderData.length > 0;
        },

        hasOrderInMonth() {
            return !!this.historyOrderDataCount && !!this.historyOrderDataSum;
        },

        today() {
            const today = Shopware.Utils.format.dateWithUserTimezone();
            today.setHours(0, 0, 0, 0);
            return today;
        },

        todayBucketCount(): BucketData | null {
            return this.calculateTodayBucket(this.historyOrderDataCount);
        },

        todayBucketSum(): BucketData | null {
            return this.calculateTodayBucket(this.historyOrderDataSum);
        },

        systemCurrencyISOCode() {
            return Shopware.Context.app.systemCurrencyISOCode;
        },

        isSessionLoaded() {
            return !Shopware.Store.get('session')?.userPending;
        },

        currencyFilter() {
            return Shopware.Filter.getByName('currency');
        },

        /**
         * @deprecated tag:v6.8.0 - Will be removed, because the filter is unused
         */
        dateFilter() {
            return Shopware.Filter.getByName('date');
        },
    },

    watch: {
        isSessionLoaded: {
            immediate: true,
            async handler() {
                if (this.isSessionLoaded) {
                    await this.initializeOrderData();
                }
            },
        },
    },

    methods: {
        calculateTodayBucket(aggregation: HistoryOrderData): BucketData | null {
            const buckets = aggregation?.buckets;

            if (!buckets) {
                return null;
            }

            const today = this.today;
            // search for stats with same timestamp as today
            const findDateStats = buckets.find((dateCount) => {
                // when date exists
                if (dateCount.key) {
                    // if time is today
                    const date = new Date(dateCount.key);

                    return date.setHours(0, 0, 0, 0) === today.setHours(0, 0, 0, 0);
                }

                return false;
            });

            if (findDateStats) {
                return findDateStats;
            }
            return null;
        },

        async initializeOrderData() {
            if (!this.acl.can('order.viewer')) {
                this.isLoading = false;

                return;
            }

            this.todayOrderDataLoaded = false;

            await this.getHistoryOrderData();
            this.todayOrderData = await this.fetchTodayData();
            this.todayOrderDataLoaded = true;
            this.isLoading = false;
        },

        getHistoryOrderData() {
            return Promise.all([
                this.fetchHistoryOrderDataCount().then((response) => {
                    this.historyOrderDataCount = response;
                }),
                this.fetchHistoryOrderDataSum().then((response) => {
                    this.historyOrderDataSum = response;
                }),
            ]);
        },

        fetchHistoryOrderDataCount() {
            return this.fetchHistory(false, this.formatDateToISO(this.getDateAgo(this.ordersDateRange)));
        },

        fetchHistoryOrderDataSum() {
            return this.fetchHistory(true, this.formatDateToISO(this.getDateAgo(this.turnoverDateRange)));
        },

        fetchHistory(paid: boolean, since: string) {
            const headers = this.orderRepository.buildHeaders();

            const initContainer = Shopware.Application.getContainer('init');
            const httpClient = initContainer.httpClient;
            const timezone = Shopware.Store.get('session').currentUser?.timeZone ?? 'UTC';

            return httpClient
                .get<
                    undefined,
                    {
                        data: {
                            statistic: Array<{
                                date: string;
                                count: number;
                                amount: number;
                            }>;
                        };
                    }
                >(`/_admin/dashboard/order-amount/${since}?timezone=${timezone}&paid=${paid.toString()}`, { headers })
                .then((response) => {
                    const buckets = response.data.statistic.map((bucket) => {
                        return {
                            key: bucket.date,
                            count: bucket.count,
                            apiAlias: 'aggregation_bucket',
                            totalAmount: {
                                sum: bucket.amount,
                                name: 'totalAmount',
                            },
                        };
                    });

                    return {
                        name: 'order_sum_bucket',
                        buckets: buckets,
                        apiAlias: 'order_sum_bucket_aggregation',
                    } as const;
                });
        },

        fetchTodayData() {
            const criteria = new Criteria(1, 10);

            criteria.addAssociation('currency');
            criteria.addAssociation('stateMachineState');

            criteria.addFilter(Criteria.equals('orderDate', this.formatDateToISO(new Date())));
            criteria.addSorting(Criteria.sort(this.todayOrderDataSortBy, this.todayOrderDataSortDirection));

            return this.orderRepository.search(criteria);
        },

        formatDateToISO(date: Date) {
            return Shopware.Utils.format.toISODate(date, false);
        },

        formatChartHeadlineDate(date: Date) {
            const lastKnownLang = Shopware.Application.getContainer('factory').locale.getLastKnownLocale();

            return date.toLocaleDateString(lastKnownLang, {
                day: 'numeric',
                month: 'short',
            });
        },

        orderGridColumns() {
            return [
                {
                    property: 'orderNumber',
                    label: 'sw-order.list.columnOrderNumber',
                    routerLink: 'sw.order.detail',
                    allowResize: true,
                    primary: true,
                },
                {
                    property: 'orderDateTime',
                    dataIndex: 'orderDateTime',
                    label: 'sw-dashboard.todayStats.orderTime',
                    allowResize: true,
                    primary: false,
                },
                {
                    property: 'orderCustomer.firstName',
                    dataIndex: 'orderCustomer.firstName,orderCustomer.lastName',
                    label: 'sw-order.list.columnCustomerName',
                    allowResize: true,
                },
                {
                    property: 'stateMachineState.name',
                    label: 'sw-order.list.columnState',
                    allowResize: true,
                },
                {
                    property: 'amountTotal',
                    label: 'sw-order.list.columnAmount',
                    align: 'right',
                    allowResize: true,
                },
            ];
        },

        getVariantFromOrderState(order: OrderEntity): string {
            const state = order.stateMachineState?.technicalName;
            if (!state) {
                return '';
            }

            return this.stateStyleDataProviderService.getStyle('order.state', state).variant;
        },

        parseDate(date: string): number {
            const parsedDate = new Date(
                date
                    .replace(/-/g, '/')
                    .replace('T', ' ')
                    .replace(/\..*|\+.*/, ''),
            );
            return parsedDate.valueOf();
        },

        async onOrdersRangeUpdate(range: string): Promise<void> {
            const ordersDateRange = this.rangesValueMap.find((item: HistoryDateRange) => item.label === range);

            if (!ordersDateRange) {
                throw Error('Range not found');
            }

            this.ordersDateRange = ordersDateRange;

            this.historyOrderDataCount = await this.fetchHistoryOrderDataCount();
        },

        async onTurnoverRangeUpdate(range: string): Promise<void> {
            const turnoverDateRange = this.rangesValueMap.find((item: HistoryDateRange) => item.label === range);

            if (!turnoverDateRange) {
                throw Error('Range not found');
            }

            this.turnoverDateRange = turnoverDateRange;
            this.historyOrderDataSum = await this.fetchHistoryOrderDataSum();
        },

        getCardSubtitle(range: HistoryDateRange): string {
            return `${this.formatChartHeadlineDate(this.getDateAgo(range))} - ${this.formatChartHeadlineDate(this.today)}`;
        },

        getDateAgo(range: HistoryDateRange): Date {
            const date = Shopware.Utils.format.dateWithUserTimezone();

            if (range.interval === 'hour') {
                date.setHours(date.getHours() - range.range);

                return date;
            }

            date.setDate(date.getDate() - range.range);
            date.setHours(0, 0, 0, 0);

            return date;
        },
    },
});

/**
 * @private might get removed with any update (even minor!) as it likely gets replaced by shopware analytics
 */
export type { HistoryDateRange };
