# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2024 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT
import os
import pwd

import secureio
from clcagefslib.const import BASEDIR, ETC_CL_ALT_PATH
from clcagefslib.fs import get_user_prefix
from clcommon import clcaptain, clcagefs
from clcommon.utils import ExternalProgramFailed
from cldetectlib import is_da


PHP_INI_MASK = '/usr/local/directadmin/data/users/%s/php/php.ini'
ALT_PHP_INI_MASK = '/etc/cl.php.d/alt-php%s/alt_php.ini'


# Change user php.ini
# Call from cl-selector
def da_change_user_php_ini(username, version, is_write_log=False):
    if not is_da():
        return

    link_name = ETC_CL_ALT_PATH + '/' + username + '/php/php.ini'
    if not clcagefs.in_cagefs():
        # userdir = /var/cagefs/[prefix]/[user]/etc
        prefix = get_user_prefix(username)
        link_name = BASEDIR + '/' + prefix + '/' + username + link_name

    # drop permissions if we are running as root
    uid = None
    gid = None
    drop_perm = False
    # check if we work under the specified user
    user_data = pwd.getpwnam(username)
    if os.geteuid() != user_data.pw_uid:
        uid = user_data.pw_uid
        gid = user_data.pw_gid
        drop_perm = True

    if drop_perm:
        secureio.set_user_perm(uid, gid)

    # Create user link/file
    create_user_php_ini(username, version, link_name, is_write_log)

    if drop_perm:
        secureio.set_root_perm()


# Create user php.ini link or file dependently of PHP version
# Works only under user, not root!
def create_user_php_ini(user, php_vers, link_name, is_write_log):
    if php_vers == 'native':
        # '/usr/local/directadmin/data/users/%s/php/php.ini'
        php_ini_path = PHP_INI_MASK % user
        if os.path.isfile(php_ini_path):
            # user's php.ini present
            try:
                content = secureio.read_file_secure(php_ini_path, None, None, False, is_write_log)
            except (OSError, IOError):
                return

            if os.path.islink(link_name):
                try:
                    os.unlink(link_name)
                except OSError:
                    pass
            try:
                clcaptain.write(link_name, ''.join(content))
            except (OSError, IOError, ExternalProgramFailed) as e:
                secureio.logging("Error: Failed to write file " + link_name + ": " + str(e), secureio.SILENT_FLAG, 1, is_write_log)
        else:
            # user's php.ini absent, delete link to alt php.ini
            try:
                os.unlink(link_name)
            except OSError:
                pass
    else:
        # create symlink to php.ini for alternative php version
        link_to = ALT_PHP_INI_MASK % php_vers.replace(".", "")
        # Delete symlink if it points to wrong location, or if it is not symlink (regular file)
        if not (os.path.islink(link_name) and os.readlink(link_name) == link_to):
            try:
                os.unlink(link_name)      # delete symlink if previous version differs current
            except OSError:
                pass
            try:
                clcaptain.symlink(link_to, link_name)
            except (OSError, ExternalProgramFailed) as e:
                secureio.logging("Error: Failed to create link " + link_name + ": " + str(e), secureio.SILENT_FLAG, 1, is_write_log)
