"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import asyncio
import logging
import os
from contextlib import suppress

from defence360agent.contracts.config import Malware, MalwareSignatures
from imav.malwarelib.scan.mds import MDS_PATH
from imav.malwarelib.scan.mds.detached import MDSDetachedCleanupDir
from imav.malwarelib.scan.mds.restore import storage_path

logger = logging.getLogger(__name__)


class MalwareDatabaseCleaner:
    def __init__(self, cleanup_id, path, app_name):
        if not path:
            raise ValueError("Cannot instantiate cleaner without path")
        if not app_name:
            raise ValueError("Cannot instantiate cleaner without app names")
        self.cleanup_id = cleanup_id
        self.path = path
        self.app_name = app_name
        self.work_dir = MDSDetachedCleanupDir(self.cleanup_id)

    @property
    def storage_path(self):
        return storage_path(self.path, self.app_name)

    @property
    def cmd(self):
        return [
            "/opt/ai-bolit/wrapper",
            MDS_PATH,
            "--clean",
            "--path",
            self.path,
            "--avdb",
            MalwareSignatures.MDS_AI_BOLIT_HOSTER,
            "--report-file",
            str(self.work_dir.report_file),
            "--detached",
            self.cleanup_id,
            "--progress",
            str(self.work_dir.progress_file),
            "--backup-file",
            self.storage_path,
            "--app-name",
            self.app_name,
            "--procudb",
            MalwareSignatures.MDS_PROCU_DB,
            "--db-timeout",
            str(Malware.MDS_DB_TIMEOUT),
        ]

    async def start(self):
        with self.work_dir:
            with self.work_dir.log_file.open(
                mode="w"
            ) as l_f, self.work_dir.err_file.open(mode="w") as e_f:
                logger.info("Running MDS Cleaner with: %s", self.cmd)

                with suppress(FileNotFoundError):
                    os.unlink(self.storage_path)

                await asyncio.create_subprocess_exec(
                    *self.cmd,
                    start_new_session=True,
                    stdout=l_f,
                    stderr=e_f,
                    cwd=str(self.work_dir),
                )
