"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
from pathlib import Path


def trim(message: str, *, max_length: int) -> str:
    """
    Remove the middle part of the string,
    so that the overall length is no more than specified.

    Returned string's length should not exceed max_length.
    """
    # We don't want to handle corner cases.
    # If you need to trim strings that much use another function.
    assert (
        max_length > 50
    ), "Specify a bigger max_length or don't use this function"

    if len(message) <= max_length:
        return message

    filler = "\n...\n"
    left_cutoff = max_length // 2
    right_cutoff = max(1, max_length - left_cutoff - len(filler))
    return message[:left_cutoff] + filler + message[-right_cutoff:]


def trim_file_content(log_path: Path) -> str:
    # 4000 is 50 (~ screen height worth of text) lines of 80 characters
    try:
        log = trim(
            log_path.read_text(encoding="utf-8", errors="backslashreplace"),
            max_length=4000,
        )
    except FileNotFoundError:
        log = ""
    return log
