"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import os
import shutil
from datetime import datetime, timedelta
from logging import getLogger

from imav.malwarelib.scan.ai_bolit.detached import (
    AiBolitDetachedDir,
)
from defence360agent.contracts.plugins import Scope
from defence360agent.mr_proper import BaseCleaner
from defence360agent.subsys.persistent_state import register_lock_file
from defence360agent.utils import antivirus_mode

logger = getLogger(__name__)


class MalwareDetachedDirCleaner(BaseCleaner):
    PERIOD = timedelta(days=1).total_seconds()
    CLEANUP_LIMIT_DELTA = timedelta(days=30)
    LOCK_FILE = register_lock_file(
        "malware-detached-dir-cleaner", Scope.AV_IM360
    )

    if antivirus_mode.enabled:
        _DETACHED_DIRS = [
            AiBolitDetachedDir,
        ]
    else:
        from imav.malwarelib.scan.mds.detached import (
            MDSDetachedCleanupDir,
            MDSDetachedRestoreDir,
            MDSDetachedScanDir,
        )

        _DETACHED_DIRS = [
            AiBolitDetachedDir,
            MDSDetachedCleanupDir,
            MDSDetachedRestoreDir,
            MDSDetachedScanDir,
        ]

    @classmethod
    async def cleanup(cls):
        now = datetime.now()
        time_limit = (now - cls.CLEANUP_LIMIT_DELTA).timestamp()
        for dir_cls in cls._DETACHED_DIRS:
            detached_directory = dir_cls.DETACHED_DIR
            if os.path.exists(detached_directory):
                entry: os.DirEntry
                with os.scandir(detached_directory) as it:
                    for entry in it:
                        if (
                            entry.is_dir()
                            and entry.stat().st_mtime < time_limit
                        ):
                            shutil.rmtree(entry.path)
                            logger.info(
                                "Cleaned %s outdated operation dir",
                                entry.name,
                            )
