import glob
import json
import logging
import os
import os.path
import pwd
import shutil
import sys
from collections import OrderedDict, defaultdict
from contextlib import suppress
from pathlib import Path
from typing import Dict, List, Set
from urllib.parse import urlparse

from packaging.version import Version

from defence360agent.application.determine_hosting_panel import (
    is_cpanel_installed,
)
from defence360agent.contracts import config
from defence360agent.utils import (
    CheckRunError,
    antivirus_mode,
    async_lru_cache,
    check_run,
    run,
)
from defence360agent.utils.ipecho import IPEchoAPI
from defence360agent.utils.kwconfig import KWConfig

from .. import base
from ..base import DomainData, forbid_dns_only
from . import packages
from .whm import WHMAPIException, whmapi1

CPANEL_PACKAGE_EXTENSIONS_PATH = Path("/var/cpanel/packages/extensions")
CPANEL_HOOKS_PATH = Path("/usr/local/cpanel")
PREINSTALL_PACKAGE_EXTENSIONS_PATH = (
    Path(config.Packaging.DATADIR) / "cpanel/packages/extensions"
)
CPANEL_USERPLANS_PATH = "/etc/userplans"
CPANEL_USERDATADOMAINS_PATH = (
    "/etc/userdatadomains;/var/cpanel/userdata/{user}/cache"
)
AV_PLUGIN_NAME = "imunify-antivirus"
IM360_PLUGIN_NAME = "imunify360"
PLUGIN_NAME = AV_PLUGIN_NAME if antivirus_mode.enabled else IM360_PLUGIN_NAME
PLUGIN_INSTALL_SCRIPT = "/usr/local/cpanel/scripts/install_plugin"
PLUGIN_UNINSTALL_SCRIPT = "/usr/local/cpanel/scripts/uninstall_plugin"
CONFIG_PATH = "/etc/sysconfig/imunify360/cpanel/"

logger = logging.getLogger(__name__)

CONFIG_FILE_TEMPLATE = "/etc/sysconfig/imunify360/cpanel/{name}.conf"
TCP_PORTS_CPANEL = base.TCP_PORTS_COMMON + ["2086-2087"]

BASE_DIR = "/home"
WWWACT_CONF = "/etc/wwwacct.conf"

_CACHE = {"userplans": {}, "userdatadomains": {}}


class cPanelException(base.PanelException):
    pass


class AccountConfig(KWConfig):
    SEARCH_PATTERN = r"^{}\s+(.*)?$"
    WRITE_PATTERN = "{} {}"
    DEFAULT_FILENAME = WWWACT_CONF


class cPanel(base.AbstractPanel):
    NAME = "cPanel"
    OPEN_PORTS = {
        "tcp": {
            "in": ["143", "465", "2077-2080", "2082-2083", "2095", "2096"]
            + TCP_PORTS_CPANEL,
            "out": [
                "37",
                "43",
                "113",
                "873",
                "2073",
                "2089",
                "2195",
                "2703",
                "6277",
                "24441",
            ]
            + TCP_PORTS_CPANEL,
        },
        "udp": {
            "in": ["20", "21", "53", "443"],
            "out": ["20", "21", "53", "113", "123", "873", "6277", "24441"],
        },
    }
    exception = cPanelException
    smtp_allow_users = ["cpanel"]  # type: List[str]
    USER_INFO_DIR = "/var/cpanel/users.cache/"
    RESELLERS_INFO = "/var/cpanel/resellers"

    @staticmethod
    def _is_dns_only():
        return os.path.isfile("/var/cpanel/dnsonly")

    @classmethod
    def get_server_ip(cls):
        ip_conf = "/var/cpanel/mainip"
        # fallback: in case there is not ip file
        if not os.path.exists(ip_conf):
            return IPEchoAPI.get_ip()
        with open(ip_conf) as f:
            return f.read().strip()

    @classmethod
    def is_installed(cls):
        return is_cpanel_installed()

    @classmethod
    async def version(cls):
        _, data, _ = await run(["/usr/local/cpanel/cpanel", "-V"])
        version = data.decode().split()
        return version[0] if version else "unknown"

    @base.ensure_valid_panel()
    async def enable_imunify_plugin(self, name=None):
        plugin_name = name or PLUGIN_NAME
        assert plugin_name in [AV_PLUGIN_NAME, IM360_PLUGIN_NAME]
        config_filename = CONFIG_FILE_TEMPLATE.format(name=plugin_name)
        new_conf = config_filename + ".rpmnew"
        if os.path.exists(new_conf):
            shutil.move(new_conf, config_filename)
        if Version(await self.version()) > Version("65.0"):
            os.system(
                '/bin/sed -i -e "s@^target=.*@target=%s@g" ' % "_self"
                + config_filename
            )
        # (re-) register plugin
        sys.stdout.write("cPanel: register_appconfig...\n")
        os.system(
            "/usr/local/cpanel/bin/register_appconfig " + config_filename
        )

    @base.ensure_valid_panel()
    async def disable_imunify_plugin(self, plugin_name=None):
        plugin = plugin_name or PLUGIN_NAME

        config_filename = CONFIG_FILE_TEMPLATE.format(name=plugin)
        config_created = False
        if not os.path.exists(config_filename):
            logger.info(
                "Warning: cpanel "
                f"{plugin}.conf missing, "
                "creating temporary config for uninstall"
            )
            os.makedirs(CONFIG_PATH, exist_ok=True)
            with open(config_filename, "w") as f:
                f.write("# Temporary config for uninstall\n")
            config_created = True

        sys.stderr.write("cPanel: unregister_appconfig...\n")
        os.system(
            "/usr/local/cpanel/bin/unregister_appconfig " + config_filename
        )

        if config_created:
            try:
                os.remove(config_filename)
            except Exception as e:
                logger.error(f"Failed to remove temporary config: {e}")

    @forbid_dns_only
    async def get_user_domains(self):
        """
        :return: list: domains hosted on server via cpanel
        """
        return [
            domain
            for user in await self.get_users()
            for domain, user_path in self._userdomains(user)
        ]

    @classmethod
    async def get_user_domains_details(
        cls, username, _path=CPANEL_USERDATADOMAINS_PATH, quiet=True
    ) -> list[DomainData]:
        domains = []

        def parser(path, d, domain_data):
            user_ = domain_data[0]
            if user_ != username:
                return
            doc_type = domain_data[2]
            docroot = domain_data[4]
            domains.append(
                DomainData(
                    docroot=docroot, domain=d, type=doc_type, username=username
                )
            )

        cls._parse_userdatadomains(_path, parser, quiet=quiet)
        return domains

    async def _do_get_users(self, userplans_path: str) -> List[str]:
        if not os.path.isfile(userplans_path):
            return []
        _cached_mtime = _CACHE["userplans"].get("mtime", 0)
        if _cached_mtime == os.path.getmtime(userplans_path):
            return _CACHE["userplans"]["users"]

        with open(
            userplans_path, encoding="utf-8", errors="surrogateescape"
        ) as f:
            users = []
            for line in f:
                if (
                    not line.startswith("#")
                    and line.count(":") == 1
                    and len(line.strip()) > 3
                ):
                    users.append(line.split(":")[0].strip())
        _CACHE["userplans"]["mtime"] = os.path.getmtime(userplans_path)
        _CACHE["userplans"]["users"] = users
        return users

    async def get_users(
        self,
    ) -> List[str]:
        return await self._do_get_users(CPANEL_USERPLANS_PATH)

    async def get_domain_to_owner(self) -> Dict[str, List[str]]:
        """
        Returns dict with domain to list of users pairs
        :return: dict domain to list of users:
        """
        domain_to_users = defaultdict(list)  # type: Dict[str, List[str]]
        for user in await self.get_users():
            for domain, _ in self._userdomains(user):
                domain_to_users[domain].append(user)
        return domain_to_users

    async def get_domains_per_user(self):
        """
        Returns dict with users to list of domains pairs
        :return: dict user to list of domains
        """
        user_to_domains = defaultdict(list)
        for user in await self.get_users():
            for domain, _ in self._userdomains(user):
                user_to_domains[user].append(domain)
        return user_to_domains

    @async_lru_cache(maxsize=128)
    async def panel_user_link(self, username) -> str:
        """
        Returns panel url
        :return: str
        """
        link = (
            await whmapi1(
                "create_user_session", user=username, service="cpaneld"
            )
        )["url"]
        if len(link) == 0:
            return ""

        parsed = urlparse(link)
        return f"{parsed.scheme}://{parsed.netloc}/cpsess0000000000/frontend/jupiter/imunify/imunify.live.pl"

    async def switch_ui_config(self, myimunify_enabled: bool) -> None:
        """
        Switch UI panel configuration between Im360 and MyImunify
        """
        if antivirus_mode.enabled:
            return None

        if not Path("/var/imunify360/i360-userside-plugin.installed").exists():
            return None

        config_to_enable = "myimunify_conf" if myimunify_enabled else "conf"
        config_to_disable = "conf" if myimunify_enabled else "myimunify_conf"

        for theme_path in glob.glob("/usr/local/cpanel/base/frontend/*"):
            if Path(theme_path).is_dir():
                theme_name = os.path.basename(theme_path)
                await self.disable_config(config_to_disable, theme_name)
                await self.enable_config(config_to_enable, theme_name)

    async def enable_config(self, config: str, theme: str) -> None:
        try:
            await check_run(
                [
                    PLUGIN_INSTALL_SCRIPT,
                    f"{CONFIG_PATH}{config}",
                    "--theme",
                    theme,
                ]
            )
        except CheckRunError as e:
            logger.warning("Error in enabling config '%s': %s", config, e)

    async def disable_config(self, config: str, theme: str) -> None:
        try:
            await check_run(
                [
                    PLUGIN_UNINSTALL_SCRIPT,
                    f"{CONFIG_PATH}{config}",
                    "--theme",
                    theme,
                ]
            )
        except CheckRunError as e:
            logger.warning("Error in disabling config '%s': %s", config, e)

    async def get_user_details(self) -> Dict[str, Dict[str, str]]:
        """
        Returns dict with user to email and locale pairs
        """

        user_details = {}

        # noinspection PyBroadException
        try:
            resellers = {
                line.split(":", 1)[0]
                for line in Path(self.RESELLERS_INFO).read_text().splitlines()
            }
        except Exception:
            resellers = None

        for user in await self.get_users():
            level = base.UserLevel.REGULAR_USER
            if user == "root":
                try:
                    with open("/etc/wwwacct.conf.cache") as f:
                        user_info = json.load(f)
                    email = user_info.get("CONTACTEMAIL", "")
                except (FileNotFoundError, json.JSONDecodeError):
                    email = ""
                locale = "en"
                parent = "root"
                suspended = False
                level = base.UserLevel.ADMIN
            else:
                try:
                    with open(os.path.join(self.USER_INFO_DIR, user)) as f:
                        user_info = json.load(f)
                    email = user_info.get("CONTACTEMAIL", "")
                    locale = user_info.get("LOCALE", "")
                    parent = user_info.get("OWNER", "")
                    suspended = user_info.get("SUSPENDED", "") == "1"
                except (FileNotFoundError, json.JSONDecodeError):
                    email = ""
                    locale = ""
                    parent = ""
                    suspended = False

                if resellers and user in resellers:
                    # 1 for the root (see above)
                    # 2 for a reseller
                    # 3 for a regular customer
                    level = base.UserLevel.RESSELER

            user_details[user] = {
                "email": email,
                "locale": locale,
                "parent": parent,
                "suspended": suspended,
                "level": int(level),
            }

        return user_details

    @classmethod
    def _get_max_mtime(cls, _path):
        """checks mtime of userdatadomains files (including cache)
        returns max mtime of all files"""

        _mtimes = []
        if "{user}" in _path:
            call_as_user = pwd.getpwuid(os.getuid()).pw_name
            _path = _path.replace("{user}", call_as_user)
        path_list = _path.split(";")
        for path_ in path_list:
            if os.path.exists(path_):
                _mtimes.append(os.path.getmtime(path_))
        return max(_mtimes) if _mtimes else 0

    @classmethod
    def _get_from_cache(cls, cpuser, _path):
        """check and invalidate cache if needed"""

        _cached_mtime = (
            _CACHE["userdatadomains"].get(cpuser, {}).get("mtime", 0)
        )

        if _cached_mtime < cls._get_max_mtime(_path):
            _CACHE["userdatadomains"][cpuser] = {}
            return None
        return _CACHE["userdatadomains"].get(cpuser, {}).get("domains", [])

    @classmethod
    def _userdomains(
        cls, cpuser, _path=CPANEL_USERDATADOMAINS_PATH, quiet=True
    ):
        cached_data = cls._get_from_cache(cpuser, _path)
        if cached_data is not None:
            return cached_data
        # use dict to avoid duplicates
        domains_tmp = OrderedDict()
        domains = OrderedDict()

        def parser(path, d, domain_data):
            user_ = domain_data[0]
            if user_ == cpuser:
                document_root = domain_data[4]
                if "main" == domain_data[2]:
                    # main domain must be first in list
                    domains.update({d: document_root})
                else:
                    domains_tmp.update({d: document_root})

        cls._parse_userdatadomains(_path, parser, quiet=quiet)
        domains.update(domains_tmp)
        _CACHE["userdatadomains"][cpuser] = {
            "mtime": cls._get_max_mtime(_path),
            "domains": domains.items(),
        }
        return domains.items()

    @staticmethod
    def _parse_userdatadomains(_path, parser, quiet=True):
        if "{user}" in _path:
            call_as_user = pwd.getpwuid(os.getuid()).pw_name
            _path = _path.replace("{user}", call_as_user)
        path_list = _path.split(";")
        for path_ in path_list:
            try:
                file_ = open(path_, "rb")
            except Exception as e:
                if not quiet:
                    logger.warning("Can't open file %s [%s]", path_, e)
                continue
            try:
                # example line:
                # test.russianguns.ru: russianguns==root==sub==russianguns.ru==
                # /home/russianguns/fla==192.168.122.40:80======0
                for i, line in enumerate(file_):
                    try:
                        line = line.decode()
                    except UnicodeDecodeError:
                        logger.warning(
                            'Broken %s line in file "%s"; line was ignored',
                            i,
                            path_,
                        )
                        continue
                    if not line.strip():  # ignore the empty string
                        continue
                    if line.count(": ") != 1:
                        if not quiet:
                            logger.warning(
                                "Can't parse %s line in file '%s'; "
                                "line was ignored",
                                i,
                                path_,
                            )
                        continue
                    domain, domain_raw_data = line.split(": ")
                    domain_data = domain_raw_data.strip().split("==")
                    parser(path_, domain, domain_data)
            finally:
                file_.close()

    @classmethod
    def is_extension_installed(cls, pkgs):
        return all(
            CPANEL_PACKAGE_EXTENSIONS_PATH.joinpath(file).is_file()
            for file in pkgs
        )

    @classmethod
    async def is_hook_installed(cls):
        try:
            hooks = await whmapi1("list_hooks")
            category = next(
                cat
                for cat in hooks["categories"]
                if cat["category"] == "Whostmgr"
            )

            for event_name in (
                "Accounts::change_package",
                "Accounts::Create",
                "Accounts::Modify",
            ):
                event = next(
                    ev
                    for ev in category["events"]
                    if ev["event"] == event_name
                )
                stage = next(
                    st for st in event["stages"] if st["stage"] == "post"
                )
                if not any(
                    action["hook"] == "ImunifyHook::hook_processing"
                    for action in stage["actions"]
                ):
                    return False
        except (StopIteration, WHMAPIException):
            return False

        return True

    @classmethod
    async def install_extension(
        cls,
        extention_name: str,
        extention_files,
        **kwargs,
    ) -> None:
        # copy cpanel's package extension files
        CPANEL_PACKAGE_EXTENSIONS_PATH.mkdir(
            mode=0o700, parents=True, exist_ok=True
        )
        for filename in extention_files:
            shutil.copy2(
                PREINSTALL_PACKAGE_EXTENSIONS_PATH / filename,
                CPANEL_PACKAGE_EXTENSIONS_PATH,
            )

        # enable extension for all packages
        await packages.add_extension_for_all(extention_name, **kwargs)

        # add hooks for native feature management
        os.makedirs(config.Core.INBOX_HOOKS_DIR, mode=0o700, exist_ok=True)
        shutil.copy2(
            PREINSTALL_PACKAGE_EXTENSIONS_PATH / "ImunifyHook.pm",
            CPANEL_HOOKS_PATH,
        )
        await check_run(
            [
                "/usr/local/cpanel/bin/manage_hooks",
                "add",
                "module",
                "ImunifyHook",
            ]
        )

    @classmethod
    async def uninstall_extension(cls, extension_name: str, extention_files):
        # remove the hook
        await check_run(
            [
                "/usr/local/cpanel/bin/manage_hooks",
                "del",
                "module",
                "ImunifyHook",
            ]
        )
        with suppress(FileNotFoundError):
            (CPANEL_HOOKS_PATH / "ImunifyHook.pm").unlink()

        # remove the package extension from all packages
        await packages.remove_extension_from_all(extension_name)
        # remove cpanel's package extension files
        for filename in extention_files:
            with suppress(FileNotFoundError):
                (CPANEL_PACKAGE_EXTENSIONS_PATH / filename).unlink()

    @staticmethod
    def mounts():
        mounts = []
        with open("/proc/mounts", "r") as f:
            for line in f:
                values = line.strip().split()
                if len(values) > 1:
                    mounts.append(values[1])
        return mounts

    def basedirs(self) -> Set[str]:
        """Fetch list of basedirs.

        On cPanel, basedir is configured as HOMEDIR variable in
        /etc/wwwacct.conf.  Also, there is a way to specify additional mount
        points as containing user folders, through HOMEMATCH variable. If
        value from HOMEMATCH variable is contained within a mount point path,
        cPanel uses this directory too."""
        homedir = AccountConfig("HOMEDIR").get()
        homematch = AccountConfig("HOMEMATCH").get()
        homedir = BASE_DIR if homedir is None else homedir
        basedirs = {BASE_DIR, homedir}
        if homematch is None:
            return basedirs

        for mount in self.mounts():
            # exclude virtfs from basedirs (DEF-14266)
            if homematch in mount and not mount.startswith("/home/virtfs/"):
                basedirs.add(mount)

        return basedirs

    @classmethod
    async def notify(cls, *, message_type, params, user=None):
        """
        Notify a customer using cPanel iContact Notifications
        """
        if not config.AdminContacts.ENABLE_ICONTACT_NOTIFICATIONS:
            return False
        if not config.should_send_user_notifications(username=user):
            return False

        data = {"message_type": message_type, "params": params, "user": user}

        logger.info(f"{cls.__name__}.notify(%s)", data)

        cmd = (
            "/usr/local/cpanel/whostmgr/docroot/cgi"
            "/imunify/handlers/notify.cgi"
        )
        stdin = json.dumps(data)
        out = await check_run([cmd], input=stdin.encode())

        return json.loads(out.decode(errors="surrogateescape"))

    async def list_docroots(self) -> Dict[str, str]:
        result = dict()

        def parser(path, d, domain_data):
            result[domain_data[4]] = domain_data[3]

        self._parse_userdatadomains(
            CPANEL_USERDATADOMAINS_PATH, parser, quiet=True
        )

        return result

    async def get_domain_paths(self) -> Dict[str, List[str]]:
        result = {}

        def parser(_, d, domain_data):
            result[d] = [domain_data[4]]

        self._parse_userdatadomains(
            CPANEL_USERDATADOMAINS_PATH, parser, quiet=True
        )

        return result
