"""
This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License,
or (at your option) any later version.


This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.


You should have received a copy of the GNU General Public License
 along with this program.  If not, see <https://www.gnu.org/licenses/>.

Copyright © 2019 Cloud Linux Software Inc.

This software is also available under ImunifyAV commercial license,
see <https://www.imunify360.com/legal/eula>
"""
import pwd
from dataclasses import dataclass
from pathlib import Path
from typing import List, Set

from defence360agent.utils import is_cloudways


@dataclass
class CloudwaysUser:
    """
    Helper to override a `user` for Cloudways environment where a file (db)
    in a customer's directory may belong to another user but should be
    visible for the customer and at the same time it must be cleanable
    (the true `owner` of a file is used for cleaning up)
    """

    name: str = None
    uid: int = None

    @classmethod
    def get_user_from_path(
        cls,
        path: Path,
        fallback: "CloudwaysUser",
        users_from_panel: Set[str],
        pw_all: List[pwd.struct_passwd],
    ) -> "CloudwaysUser":
        if is_cloudways():
            return next(
                (
                    cls(name=pw.pw_name, uid=pw.pw_uid)
                    for pw in pw_all
                    if pw.pw_name in users_from_panel
                    and Path(pw.pw_dir) in path.parents
                ),
                fallback,
            )
        return fallback

    @classmethod
    def override_name_by_path(
        cls,
        path: Path,
        fallback_name: str,
        users_from_panel: Set[str],
        pw_all: List[pwd.struct_passwd],
    ) -> str:
        fallback = cls(name=fallback_name)
        user = cls.get_user_from_path(path, fallback, users_from_panel, pw_all)
        return user.name

    @classmethod
    def override_uid_by_path(
        cls,
        path: Path,
        fallback_uid: int,
        users_from_panel: Set[str],
        pw_all: List[pwd.struct_passwd],
    ) -> int:
        fallback = cls(uid=fallback_uid)
        user = cls.get_user_from_path(path, fallback, users_from_panel, pw_all)
        return user.uid
