# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import argparse

import lvestats
from lvestats.lib.commons.argparse_utils import SmartFormatter, check_from_zero_to_max_int


class ArgParseClTop(argparse.ArgumentParser):
    def error(self, message):
        raise ValueError(message)


def cloudlinux_top_parser(name="cloudlinux_top"):
    def check_order(value):
        # type: (str) -> str
        value_lowered = value.lower()
        if value_lowered in ["cpu", "mysql_cpu", "io", "mysql_io", "iops", "ep", "nproc", "pmem"]:
            return value_lowered
        else:
            raise argparse.ArgumentTypeError(
                f'"{value}" is an invalid; must be one of the available options: \n'
                '"cpu", "mysql_cpu", "io", "mysql_io", "iops", "ep", "nproc", "pmem"'
            )

    cltop_parser = ArgParseClTop(
        formatter_class=SmartFormatter,
        description="%(prog)s - Utility provides information about current MySQL and LVE usage of a running system "
                    "in JSON format",
        prog=name
    )
    cltop_parser.add_argument(
        '-v', '--version',
        version=lvestats.__version__,
        help='show program\'s version number and exit',
        action='version')
    cltop_parser.add_argument(
        '-j', '--json', action='store_true', default=False,
        help='return data in JSON format'
    )
    cltop_parser.add_argument(
        '--hide-mysql', action='store_true', default=False,
        help="don't show MySQL related info"
    )
    exclusive_group = cltop_parser.add_mutually_exclusive_group()
    exclusive_group.add_argument(
        '-u', '--username', type=str,
        help='show data only for specific user, can be used to filter the output, '
             'returns users where username like "%%USERNAME%%"'
    )
    exclusive_group.add_argument(
        '-r', '--for-reseller',
        help='get information only about specified reseller and his users'
    )
    cltop_parser.add_argument(
        '-d', '--domain', type=str,
        help='show data only for specific domain, can be used to filter the output, '
             'returns users where domain like "%%DOMAIN%%"'
    )
    cltop_parser.add_argument(
        '-m', '--max', type=check_from_zero_to_max_int, default=25,
        help='show up to N records. If --max key is omitted, by default will show top 25 users'
    )
    cltop_parser.add_argument(
        '-o', '--order-by', type=check_order, default="cpu",
        help='sort output by resourse usage, available options: '
             '"cpu", "mysql_cpu", "io", "mysql_io", "iops", "ep", "nproc", "pmem"'
    )
    return cltop_parser
