# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import logging
from lvestats.lib import dbengine
from lvestats.lib.commons.logsetup import setup_logging
from lvestats.lib.server_id import UpdateOrCreateServerID


def configure_db(configuration, opts):
    log = setup_logging(configuration, 'create_db', console_level=logging.INFO)
    log.info('Configuring connection to db')
    try:
        engine = dbengine.make_db_engine(configuration, just_print=opts.print_sql)
    except dbengine.MakeDbException as e:
        log.error(str(e))
    else:
        _configure_db(configuration, engine, log, opts)


def _configure_db(configuration, engine, log, opts):
    if opts.validate:
        validation = dbengine.validate_database(engine)
        if not (validation['column_error'] or validation['table_error']):
            log.info("Database structure is correct")
    elif opts.fix:
        log.info('Try fix errors in database')
        dbengine.fix_db(engine, configuration, log_=log)
    # Re-create DB
    elif opts.recreate:
        log.info('Recreate DB')
        dbengine.recreate_db(engine)
    # Create DB if no params
    else:
        log.info('Creating tables')
        dbengine.setup_db(engine, cfg=configuration)
        log.info('Done')
    if opts.create_missing_tables:
        dbengine.setup_db(engine, create_missing_tables=True, cfg=configuration)

    # Update server_id manually
    if opts.update_serverid_prompt:
        log.info('Updating server ID')
        UpdateOrCreateServerID(engine).prompt()
        log.info('Done')
    # Update server_id (if server_id in restricted list) automatically
    elif opts.update_serverid_auto:
        log.info('Updating server ID with uuid')
        UpdateOrCreateServerID(engine).auto()
        log.info('Done')
