# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import datetime
import os
import platform

from lvestat import LVEStat
from lvestats.core.plugin import LveStatsPlugin


class LVEStatsData(LveStatsPlugin):
    """lve_stats_data encapsulates LVE data gathered from /proc/lve/list."""
    order = 1750

    def __init__(self, dump_dir="/var/lve/json-stats"):
        self.period = 60
        self.attributes = set(slot for slot in LVEStat.__slots__ if not slot.startswith('_')) - {'id'}
        self.timestamp = ''
        self.dump_dir = dump_dir
        if not os.path.exists(self.dump_dir):
            os.makedirs(self.dump_dir)

    def _set_timestamp(self, tstmp=None):
        if tstmp:
            self.timestamp = tstmp
        else:
            self.timestamp = datetime.datetime.utcnow().isoformat()

    def format_json(self, lve_usages):
        """Returns LVE usage data contained in this object in JSON format."""
        lve_json_set = []

        for lve_id, lve in lve_usages.items():

            lve_json_element = f'{{ "id": "{lve_id}"'

            for lve_attr in dir(lve):
                if lve_attr in self.attributes:
                    lve_json_element += f', "{lve_attr}": "{getattr(lve, lve_attr)}"'

            lve_json_element += ' }'
            lve_json_set.append(lve_json_element)

        result = '{ '

        result += f'"hostname": "{platform.node()}", '
        result += f'"timestamp": "{self.timestamp}", '
        result += '"data": [ ' + ', '.join(lve_json_set) + ' ]'

        result += ' }'

        return result

    def execute(self, lve_data):
        usages = lve_data.get('stats', {})
        self._set_timestamp()
        name = self.dump_dir + "/stat." + self.timestamp
        with open(name, 'w', encoding='utf-8') as f:
            f.write(self.format_json(usages))
