# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import logging
import os

import psutil
from guppy import hpy

from lvestats.core.plugin import LveStatsPlugin


class LVEStatsMemoryUsage(LveStatsPlugin):
    def __init__(self):
        self.log = logging.getLogger('plugin.LVEStatsMemoryUsage')
        self.now = 0  # This changes in MainLoop
        self.period = 60 * 60
        self.threshold = 512
        try:
            self._process = psutil.Process(os.getpid())
        except AttributeError:
            self._process = None
            self.log.error("Process attribute in psutil is not available. psutil.__version__=%s", psutil.__version__)
        self._hpy = hpy()
        self._hpy.setrelheap()

    @staticmethod
    def split(value):
        return str(value).split("\n")

    def execute(self, lve_data):
        if self._process is None:
            return
        rss = self._process.memory_info()[0] / float(1024 * 1024)
        self.log.debug(str(rss))
        if rss >= self.threshold:
            heap = self._hpy.heap()[:10]
            result = {
                'heap': self.split(heap),
                'byid': self.split(heap.byid),
                'byvia': self.split(heap.byvia),
                'byrcs': self.split(heap.byrcs),
            }
            self.log.error("LVEStats uses too much memory", extra=result)
        elif rss < self.threshold / 2:
            self._hpy.setrelheap()
