# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from clcommon import clpwd
from clcommon.clproc import LIMIT_LVP_ID, ProcLve
from lvestat import LVEStat
from lvestats.core.plugin import LveStatsPlugin
from lvestats.lib.commons.func import serialize_lve_id

MEGAHERZ = 1000000
GIGAHERZ = 1000000000

__author__ = 'iseletsk'


class LVECollector(LveStatsPlugin):
    """
    Collects data about LVE and LVP usage;
    Uses /proc/lve/list and /proc/lve/resellers/lvpXXX/list
    """

    def __init__(self):
        self.min_uid = clpwd.ClPwd().get_sys_min_uid(500)
        self._proc_lve = ProcLve()

    def __parse_stats(self, lvp_id=0):
        """
        Get information about LVE usage for given lvp_id (container id);
        :type lvp_id: int
        :rtype: dict[int, LVEStat]
        """
        stats = {}
        for line in self._proc_lve.lines(lvp_id, without_limits=False):
            stat = LVEStat(line, self._proc_lve.version())
            if stat.id == 0 or stat.id >= self.min_uid:
                serialized_id = serialize_lve_id(stat.id, stat.reseller_id)
                stats[serialized_id] = stat
        return stats

    def execute(self, lve_data):
        lve_data['LVE_VERSION'] = self._proc_lve.version()
        stats = {}
        if self._proc_lve.resellers_supported():
            for lvp_id in self._proc_lve.lvp_id_list():
                stats.update(self.__parse_stats(lvp_id=lvp_id))
        stats.update(self.__parse_stats())
        stats.pop(LIMIT_LVP_ID, None)  # remove information about global limits
        lve_data['old_stats'] = lve_data.get('stats', {})
        lve_data['stats'] = stats


class LVEUsernamesCollector(LveStatsPlugin):
    def __init__(self):
        self.server_id = 'localhost'
        self.period = 60 * 60  # once an hour
        self.enabled = True

    def set_config(self, config):
        self.enabled = config.get('collect_usernames', 'false').lower() == 'true'
        self.server_id = config.get('server_id', self.server_id)

    def execute(self, lve_data):
        if self.enabled:
            db = clpwd.ClPwd()
            users = db.get_user_dict()
            lve_data['users'] = [(u.pw_uid, u.pw_name, self.server_id) for u in list(users.values())]


class CPUInfoCollector(LveStatsPlugin):
    """
    Collects information on number of cores, and total # of Hz
    """

    def get_cpuinfo_file(self):
        """
        This should make it trivial to mock up that file
        """
        return open('/proc/cpuinfo', 'r', encoding='utf-8')

    @staticmethod
    def count(f):
        """
        Method that counts number of cores and cpu herz for version > 4.

        In modern lve, cpu usage is always calculated as number of virtual ticks
        on a processor that has 1Ghz frequency on each core.

        :return: procs, total_hz
        """
        procs = 0

        line = f.readline()
        while line:
            if line.lower().find('processor') >= 0:
                procs += 1

            line = f.readline()
        total_hz = GIGAHERZ * procs

        return procs, total_hz

    def execute(self, lve_data):
        f = self.get_cpuinfo_file()
        lve_data['procs'], lve_data['totalHz'] = self.count(f)
        f.close()
