# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import os
import configparser
import sys

from clcommon import clconfpars
from lvestats.lib.commons import logsetup


GLOBAL_CONFIG_LOCATION = '/etc/sysconfig/lvestats2'
CL_CONFIG = '/etc/sysconfig/cloudlinux'
CONFIG_PATH_DIR = '/etc/sysconfig/lvestats.config/'
log = logsetup.setup_logging(None)

HIDE_MAX_UID_LVE_PARAMETER = 'hide_lve_more_than_maxuid'
USER_NOTIFICATIONS_OFF_MARKER = 'lvestats_notifications_off'


class ConfigError(Exception):
    def __init__(self, location, ex):
        self.value = location
        self.ex = ex

    def log_and_exit(self, log_=None, exit_code=1):
        if log_ is None:
            log_ = log
        log_.error(str(self))
        sys.exit(exit_code)

    def __str__(self):
        if self.ex is IOError:
            return f"Unable to read config {self.value} ({self.ex})"
        else:
            return f"Unable to parse config {self.value} ({self.ex})"


def _load_config_and_process_errors(location):
    try:
        return clconfpars.load(location)
    except IOError as ie:
        if ie.errno == 13 and not location.endswith('.readonly'):  # permission denied:
            return _load_config_and_process_errors(location + '.readonly')
        log.error('Config %s not loadable: %s', location, str(ie))
        raise ConfigError(location, ie) from ie
    except configparser.ParsingError as pe:
        log.error(str(pe))
        raise ConfigError(location, pe) from pe


def read_plugin_config(config, plugin):
    plugin = plugin.strip()
    plugin_cfg_path = CONFIG_PATH_DIR + plugin + '.cfg'
    if os.path.isfile(plugin_cfg_path):
        plugin_config = _load_config_and_process_errors(plugin_cfg_path)
        config.update(plugin_config)
    return config


def read_config():
    """
    reads lvestats config. Throws ConfigError exception -- if any errors
    :return: config
    """
    return _load_config_and_process_errors(GLOBAL_CONFIG_LOCATION)


def is_normalized_user_cpu():
    """
    Check global config parameter for normalize user CPU values
    """
    config = configparser.ConfigParser(strict=False)
    config.read(CL_CONFIG)

    try:
        return config.get("license_check", "normalized_user_cpu").upper() == "Y"
    except IOError:
        # no file
        pass
    except configparser.NoOptionError:
        # no config parameter
        pass
    except configparser.NoSectionError:
        # no section/invalid file (or m.b no file)
        pass
    return True


def str_to_bool(value: str) -> bool:
    """
    Convert string to bool
    """
    if value.lower() in ['y', 'yes', 't', 'true', 'on', '1']:
        return True
    elif value.lower() in ['n', 'no', 'f', 'false', 'off', '0']:
        return False
    else:
        raise ValueError(f'Invalid truth value {value}')


def get_max_lve_id() -> int:
    """
    The function returns max lve id according to option `use_big_uids`
    """
    lvestats_config = read_config()
    use_big_uids = lvestats_config.get('use_big_uids', 'false')
    if str_to_bool(use_big_uids):
        return 2 ** 31 - 1
    else:
        return 10 ** 9
