# coding=utf-8
#
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import mmap

__author__ = 'shaman'

# bytes count in big values (kilobytes, megabytes, etc.)
BYTE_SIZES = [
    {"value": 1099511627776, "title": "TB"},
    {"value": 1073741824, "title": "GB"},
    {"value": 1048576, "title": "MB"},
    {"value": 1024, "title": "KB"}
]

POWERS_OF_1000_SIZES = [
    {"value": 1000000000000, "title": "T"},
    {"value": 1000000000, "title": "G"},
    {"value": 1000000, "title": "M"},
    {"value": 1000, "title": "K"}
]

PAGE_SIZE = mmap.PAGESIZE  # Memory page size; as usualy 4096


def convert(value, powers_of_1000, for_graph):
    """
    Convert bytes value to human readable format (e.g., 4K 347M 7G)
    for write it on graphics.
    Return 2-digit value - float or digital. Depends on original value.
    The rules are:
    For graphs:
      print int if it is an int or it is more then 10
    Not for graphs:
      if it is > 100, or units are B or no units (KB/MB/GB/K/M/G will print as float)
    >>> convert(12345678, False, True)
    '11MB'
    >>> convert(1234567, False, True)
    '1.2MB'
    """
    result = float(value)
    if powers_of_1000:
        sizes_list = POWERS_OF_1000_SIZES
        title = ''
    else:
        sizes_list = BYTE_SIZES
        title = 'B'
    for item in sizes_list:
        if value >= item["value"]:
            result = float(value) / item["value"]
            title = item["title"]
            break
    if (for_graph and (result >= 10 or result.is_integer())) or result >= 100 or \
            ((title == 'B' or title == '') and not for_graph):
        return f"{int(result)}{title}"
    else:
        return f"{result:.1f}{title}"


def convert_bytes(s_bytes):
    return convert(s_bytes, False, False)


def convert_bytes_for_graph(s_bytes):
    return convert(s_bytes, False, True)


def convert_powers_of_1000(value):
    return convert(value, True, False)


def convert_powers_of_1000_for_graph(value):
    return convert(value, True, True)


def mempages_to_bytes(pages):
    """
    Convert memory pages to bytes
    :param int pages: memory pages
    :return int: bytes
    """
    return int(round(pages * PAGE_SIZE))


def dbgov_io_bytes_value(io_read_mb, io_write_mb):
    """
    Convert two (read and write) io values (in MB) from MySQL Governor to single value in bytes
    :param io_read_mb: IO read value in MB
    :param io_write_mb: IO write value in MB
    :return: IO value in bytes
    """
    return int((io_read_mb + io_write_mb) * 1024 * 1024)
