#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

# Implementation of Ramer-Douglas-Peucker algorithm

__all__ = ['ramerdouglas']


def ramerdouglas(line, dist):
    """Does Ramer-Douglas-Peucker simplification of a curve with `dist`
    threshold.

    `line` is a list-of-tuples, where each tuple is a 2D coordinate

    Usage is like so:

    >>> myline = [(0.0, 0.0), (1.0, 2.0), (2.0, 1.0)]
    >>> simplified = ramerdouglas(myline, dist = 1.0)
    """

    if len(line) < 3:
        return line

    (begin, end) = (line[0], line[-1]) if line[0] != line[-1] else (line[0], line[-2])

    begin_end_x = end[0] - begin[0]
    begin_end_y = end[1] - begin[1]

    dist_sq = []
    dist_append = dist_sq.append
    for curr in line[1:-1]:
        begin_curr_x = curr[0] - begin[0]
        begin_curr_y = curr[1] - begin[1]
        dist_append((begin_curr_y * begin_end_x - begin_curr_x * begin_end_y)**2 /
                    (begin_end_x**2 + begin_end_y**2))

    maxdist = max(dist_sq)
    if maxdist < dist ** 2:
        return [begin, end]

    if len(dist_sq) > 500:
        pos = len(dist_sq) // 2
    else:
        pos = dist_sq.index(maxdist)

    return (ramerdouglas(line[:pos + 2], dist) +
            ramerdouglas(line[pos + 1:], dist)[1:])
