#!/usr/bin/python
# coding=utf-8
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from functools import wraps

import cldetectlib
from clcommon.public_hooks.lib import ModifyPackageHook
from clcontrollib import detect_panelclass
from clveconfig.ve_config import get_xml_config, save_xml


def is_plesk_or_exit(f):
    """
    Check that function runs on plesk, or return None
    """
    @wraps(f)
    def inner(*args, **kwargs):
        if not cldetectlib.is_plesk():
            return None
        return f(*args, **kwargs)
    return inner


class PleskRenameHook(ModifyPackageHook):
    """
    Sync plesk DB package id with ve.cfg package name
    """
    @is_plesk_or_exit
    def post_modify_package(self, name, **kwargs):
        """
        Synchronize plesk DB and ve.cfg package names by plesk DB id
        """
        panel = detect_panelclass()
        new_package_name = panel.get_package_name_by_id_from_plesk_db(int(name))
        xml, _ = get_xml_config(False)
        packages = xml.getElementsByTagName('package')
        try:
            el_to_rename = next(filter(
                lambda x: x.getAttribute('plesk_id') == name,
                packages
            ))
        except StopIteration:
            return
        el_to_rename.setAttribute('id', new_package_name)
        save_xml(xml)
