#!/usr/bin/python
# coding=utf-8
# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

import pwd
import subprocess

from clcommon.public_hooks.lib import ModifyUserHook
from clcommon.lib.cledition import lve_supported_or_exit


class LveLifecycleHook(ModifyUserHook):
    """
    Apply limits on user creation and remove
    limits on user destroy
    """
    LVECTL = '/usr/sbin/lvectl'

    @classmethod
    def exec_lvectl_cmd(cls, cmd, uid):
        """
        This function doesn't return! Should be last call in code
        :param cmd: apply|destroy
        :param uid: lve id
        """
        return subprocess.call([cls.LVECTL, cmd, str(uid)])

    @lve_supported_or_exit
    def post_create_user(self, username, owner, **kwargs):
        uid = pwd.getpwnam(username).pw_uid

        self.exec_lvectl_cmd('apply', uid)

    @lve_supported_or_exit
    def pre_delete_user(self, username, **kwargs):
        uid = pwd.getpwnam(username).pw_uid

        self.exec_lvectl_cmd('delete', uid)
        self.exec_lvectl_cmd('destroy', uid)
