<?php
/**
 * Copyright 2017 Facebook, Inc.
 *
 * You are hereby granted a non-exclusive, worldwide, royalty-free license to
 * use, copy, modify, and distribute this software in source code or binary
 * form for use in connection with the web services and APIs provided by
 * Facebook.
 *
 * As with any software that integrates with the Facebook platform, your use
 * of this software is subject to the Facebook Developer Principles and
 * Policies [http://developers.facebook.com/policy/]. This copyright notice
 * shall be included in all copies or substantial portions of the software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */

namespace Facebook\HttpClients;

use Facebook\Http\GraphRawResponse;
use Facebook\Exceptions\FacebookSDKException;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Http\Message\ResponseInterface;

class FacebookGuzzleHttpClient implements FacebookHttpClientInterface
{
    /**
     * @var Client The Guzzle client.
     */
    protected Client $guzzleClient;

    /**
     * @param Client|null $guzzleClient The Guzzle client.
     */
    public function __construct(?Client $guzzleClient = null)
    {
        $this->guzzleClient = $guzzleClient ?: new Client();
    }

    /**
     * @inheritdoc
     */
    public function send(string $url, string $method, ?string $body, array $headers, int $timeOut): GraphRawResponse
    {
        $options = [
            'headers'         => $headers,
            'timeout'         => $timeOut,
            'http_errors'     => false,
            'connect_timeout' => 10,
        ];

        if ($body) {
            $options['body'] = $body;
        }

        try {

            $rawResponse = $this->guzzleClient->request($method, $url, $options);

            return new GraphRawResponse(
                $this->getHeadersAsString($rawResponse),
                $rawResponse->getBody(),
                $rawResponse->getStatusCode()
            );

        } catch (GuzzleException $e) {
            throw new FacebookSDKException($e->getMessage(), $e->getCode());
        }

    }

    /**
     * Returns the Guzzle array of headers as a string.
     *
     * @param ResponseInterface $response The Guzzle response.
     */
    public function getHeadersAsString(ResponseInterface $response): string
    {
        $headers = $response->getHeaders();
        $rawHeaders = [];
        foreach ($headers as $name => $values) {
            $rawHeaders[] = $name . ": " . implode(", ", $values);
        }

        return implode("\r\n", $rawHeaders);
    }
}
