<?php
/**
 * Kkart Webhook functions
 *
 * @package Kkart\Functions
 * @version 3.3.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Process the web hooks at the end of the request.
 *
 * @since 4.4.0
 */
function kkart_webhook_execute_queue() {
	global $kkart_queued_webhooks;
	if ( empty( $kkart_queued_webhooks ) ) {
		return;
	}

	foreach ( $kkart_queued_webhooks as $data ) {
		// Webhooks are processed in the background by default
		// so as to avoid delays or failures in delivery from affecting the
		// user who triggered it.
		if ( apply_filters( 'kkart_webhook_deliver_async', true, $data['webhook'], $data['arg'] ) ) {

			$queue_args = array(
				'webhook_id' => $data['webhook']->get_id(),
				'arg'        => $data['arg'],
			);

			$next_scheduled_date = KKART()->queue()->get_next( 'kkart_deliver_webhook_async', $queue_args, 'kkart-webhooks' );

			// Make webhooks unique - only schedule one webhook every 10 minutes to maintain backward compatibility with WP Cron behaviour seen in KKART < 3.5.0.
			if ( is_null( $next_scheduled_date ) || $next_scheduled_date->getTimestamp() >= ( 600 + gmdate( 'U' ) ) ) {
				KKART()->queue()->add( 'kkart_deliver_webhook_async', $queue_args, 'kkart-webhooks' );
			}
		} else {
			// Deliver immediately.
			$data['webhook']->deliver( $data['arg'] );
		}
	}
}
add_action( 'shutdown', 'kkart_webhook_execute_queue' );

/**
 * Process webhook delivery.
 *
 * @since 3.3.0
 * @param KKART_Webhook $webhook Webhook instance.
 * @param array      $arg     Delivery arguments.
 */
function kkart_webhook_process_delivery( $webhook, $arg ) {
	// We need to queue the webhook so that it can be ran after the request has finished processing.
	global $kkart_queued_webhooks;
	if ( ! isset( $kkart_queued_webhooks ) ) {
		$kkart_queued_webhooks = array();
	}
	$kkart_queued_webhooks[] = array(
		'webhook' => $webhook,
		'arg'     => $arg,
	);
}
add_action( 'kkart_webhook_process_delivery', 'kkart_webhook_process_delivery', 10, 2 );

/**
 * Wrapper function to execute the `kkart_deliver_webhook_async` cron.
 * hook, see KKART_Webhook::process().
 *
 * @since 2.2.0
 * @param int   $webhook_id Webhook ID to deliver.
 * @throws Exception        If webhook cannot be read/found and $data parameter of KKART_Webhook class constructor is set.
 * @param mixed $arg        Hook argument.
 */
function kkart_deliver_webhook_async( $webhook_id, $arg ) {
	$webhook = new KKART_Webhook( $webhook_id );
	$webhook->deliver( $arg );
}
add_action( 'kkart_deliver_webhook_async', 'kkart_deliver_webhook_async', 10, 2 );

/**
 * Check if the given topic is a valid webhook topic, a topic is valid if:
 *
 * + starts with `action.kkart_` or `action.kkart_`.
 * + it has a valid resource & event.
 *
 * @since  2.2.0
 * @param  string $topic Webhook topic.
 * @return bool
 */
function kkart_is_webhook_valid_topic( $topic ) {
	$invalid_topics = array(
		'action.kkart_login_credentials',
		'action.kkart_product_csv_importer_check_import_file_path',
		'action.kkart_webhook_should_deliver',
	);

	if ( in_array( $topic, $invalid_topics, true ) ) {
		return false;
	}

	// Custom topics are prefixed with kkart_ or kkart_ are valid.
	if ( 0 === strpos( $topic, 'action.kkart_' ) || 0 === strpos( $topic, 'action.kkart_' ) ) {
		return true;
	}

	$data = explode( '.', $topic );

	if ( ! isset( $data[0] ) || ! isset( $data[1] ) ) {
		return false;
	}

	$valid_resources = apply_filters( 'kkart_valid_webhook_resources', array( 'coupon', 'customer', 'order', 'product' ) );
	$valid_events    = apply_filters( 'kkart_valid_webhook_events', array( 'created', 'updated', 'deleted', 'restored' ) );

	if ( in_array( $data[0], $valid_resources, true ) && in_array( $data[1], $valid_events, true ) ) {
		return true;
	}

	return false;
}

/**
 * Check if given status is a valid webhook status.
 *
 * @since 3.5.3
 * @param string $status Status to check.
 * @return bool
 */
function kkart_is_webhook_valid_status( $status ) {
	return in_array( $status, array_keys( kkart_get_webhook_statuses() ), true );
}

/**
 * Get Webhook statuses.
 *
 * @since  2.3.0
 * @return array
 */
function kkart_get_webhook_statuses() {
	return apply_filters(
		'kkart_webhook_statuses',
		array(
			'active'   => __( 'Active', 'kkart' ),
			'paused'   => __( 'Paused', 'kkart' ),
			'disabled' => __( 'Disabled', 'kkart' ),
		)
	);
}

/**
 * Load webhooks.
 *
 * @since  3.3.0
 * @throws Exception If webhook cannot be read/found and $data parameter of KKART_Webhook class constructor is set.
 * @param  string   $status Optional - status to filter results by. Must be a key in return value of @see kkart_get_webhook_statuses(). @since 3.5.0.
 * @param  null|int $limit Limit number of webhooks loaded. @since 3.6.0.
 * @return bool
 */
function kkart_load_webhooks( $status = '', $limit = null ) {
	$data_store = KKART_Data_Store::load( 'webhook' );
	$webhooks   = $data_store->get_webhooks_ids( $status );
	$loaded     = 0;

	foreach ( $webhooks as $webhook_id ) {
		$webhook = new KKART_Webhook( $webhook_id );
		$webhook->enqueue();
		$loaded ++;

		if ( ! is_null( $limit ) && $loaded >= $limit ) {
			break;
		}
	}

	return 0 < $loaded;
}

/**
 * Get webhook.
 *
 * @param  int|KKART_Webhook $id Webhook ID or object.
 * @throws Exception          If webhook cannot be read/found and $data parameter of KKART_Webhook class constructor is set.
 * @return KKART_Webhook|null
 */
function kkart_get_webhook( $id ) {
	$webhook = new KKART_Webhook( $id );

	return 0 !== $webhook->get_id() ? $webhook : null;
}

/**
 * Get webhoook REST API versions.
 *
 * @since 3.5.1
 * @return array
 */
function kkart_get_webhook_rest_api_versions() {
	return array(
		'wp_api_v1',
		'wp_api_v2',
		'wp_api_v3',
	);
}
