<?php
/**
 * Marketplace suggestions updater
 *
 * Uses KKART_Queue to ensure marketplace suggestions data is up to date and cached locally.
 *
 * @package Kkart\Classes
 * @since   3.6.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Marketplace Suggestions Updater
 */
class KKART_Marketplace_Updater {

	/**
	 * Setup.
	 */
	public static function load() {
		add_action( 'init', array( __CLASS__, 'init' ) );
	}

	/**
	 * Schedule events and hook appropriate actions.
	 */
	public static function init() {
		add_action( 'kkart_update_marketplace_suggestions', array( __CLASS__, 'update_marketplace_suggestions' ) );
	}

	/**
	 * Fetches new marketplace data, updates kkart_marketplace_suggestions.
	 */
	public static function update_marketplace_suggestions() {
		$data = get_option(
			'kkart_marketplace_suggestions',
			array(
				'suggestions' => array(),
				'updated'     => time(),
			)
		);

		$data['updated'] = time();

		$url     = 'https://kkart.com/sp-json/wccom/marketplace-suggestions/1.0/suggestions.json';
		$request = wp_safe_remote_get( $url );

		if ( is_wp_error( $request ) ) {
			self::retry();
			return update_option( 'kkart_marketplace_suggestions', $data, false );
		}

		$body = wp_remote_retrieve_body( $request );
		if ( empty( $body ) ) {
			self::retry();
			return update_option( 'kkart_marketplace_suggestions', $data, false );
		}

		$body = json_decode( $body, true );
		if ( empty( $body ) || ! is_array( $body ) ) {
			self::retry();
			return update_option( 'kkart_marketplace_suggestions', $data, false );
		}

		$data['suggestions'] = $body;
		return update_option( 'kkart_marketplace_suggestions', $data, false );
	}

	/**
	 * Used when an error has occured when fetching suggestions.
	 * Re-schedules the job earlier than the main weekly one.
	 */
	public static function retry() {
		KKART()->queue()->cancel_all( 'kkart_update_marketplace_suggestions' );
		KKART()->queue()->schedule_single( time() + DAY_IN_SECONDS, 'kkart_update_marketplace_suggestions' );
	}
}

KKART_Marketplace_Updater::load();
