<?php
namespace Automattic\Kkart\Blocks\StoreApi\Schemas;

use Automattic\Kkart\Blocks\RestApi\Routes;

/**
 * BillingAddressSchema class.
 *
 * Provides a generic billing address schema for composition in other schemas.
 *
 * @internal This API is used internally by Blocks--it is still in flux and may be subject to revisions.
 */
class BillingAddressSchema extends AbstractSchema {
	/**
	 * The schema item name.
	 *
	 * @var string
	 */
	protected $title = 'billing_address';

	/**
	 * Term properties.
	 *
	 * @return array
	 */
	public function get_properties() {
		return [
			'first_name' => [
				'description' => __( 'First name', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'last_name'  => [
				'description' => __( 'Last name', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'company'    => [
				'description' => __( 'Company', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'address_1'  => [
				'description' => __( 'Address', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'address_2'  => [
				'description' => __( 'Apartment, suite, etc.', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'city'       => [
				'description' => __( 'City', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'state'      => [
				'description' => __( 'State/County code, or name of the state, county, province, or district.', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'postcode'   => [
				'description' => __( 'Postal code', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'country'    => [
				'description' => __( 'Country/Region code in ISO 3166-1 alpha-2 format.', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
			'email'      => [
				'description' => __( 'Email', 'kkart' ),
				'type'        => 'string',
				'format'      => 'email',
				'context'     => [ 'view', 'edit' ],
			],
			'phone'      => [
				'description' => __( 'Phone', 'kkart' ),
				'type'        => 'string',
				'context'     => [ 'view', 'edit' ],
			],
		];
	}

	/**
	 * Convert a term object into an object suitable for the response.
	 *
	 * @param \KKART_Order|\KKART_Customer $address An object with billing address.
	 *
	 * @throws RouteException When the invalid object types are provided.
	 * @return stdClass
	 */
	public function get_item_response( $address ) {
		if ( ( $address instanceof \KKART_Customer || $address instanceof \KKART_Order ) ) {
			return (object) $this->prepare_html_response(
				[
					'first_name' => $address->get_billing_first_name(),
					'last_name'  => $address->get_billing_last_name(),
					'company'    => $address->get_billing_company(),
					'address_1'  => $address->get_billing_address_1(),
					'address_2'  => $address->get_billing_address_2(),
					'city'       => $address->get_billing_city(),
					'state'      => $address->get_billing_state(),
					'postcode'   => $address->get_billing_postcode(),
					'country'    => $address->get_billing_country(),
					'email'      => $address->get_billing_email(),
					'phone'      => $address->get_billing_phone(),
				]
			);
		}
		throw new RouteException(
			'invalid_object_type',
			sprintf(
				/* translators: Placeholders are class and method names */
				__( '%1$s requires an instance of %2$s or %3$s for the address', 'kkart' ),
				'BillingAddressSchema::get_item_response',
				'KKART_Customer',
				'KKART_Order'
			),
			500
		);
	}
}
