/** @typedef { import('@kkart/type-defs/address-fields').CountryAddressFields } CountryAddressFields */
/** @typedef { import('@kkart/type-defs/address-fields').AddressFieldKey } AddressFieldKey */
/** @typedef { import('@kkart/type-defs/address-fields').AddressField } AddressField */

/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Used to render postcode before the city field.
 *
 * @type {Object <AddressFieldKey, AddressField>}
 */
const postcodeBeforeCity = {
	city: {
		index: 9,
	},
	postcode: {
		index: 7,
	},
};

/**
 * Used to make the state field optional.
 *
 * @type {Object <AddressFieldKey, AddressField>}
 */
const optionalState = {
	state: {
		required: false,
	},
};

/**
 * Used to hide the state field.
 *
 * @type {Object <AddressFieldKey, AddressField>}
 */
const hiddenState = {
	state: {
		required: false,
		hidden: true,
	},
};

/**
 * Used to hide the postcode field.
 *
 * @type {Object <AddressFieldKey, AddressField>}
 */
const hiddenPostcode = {
	postcode: {
		required: false,
		hidden: true,
	},
};

/**
 * Country specific address field properties.
 *
 * @type {CountryAddressFields}
 */
const countryAddressFields = {
	AE: {
		...hiddenPostcode,
		...optionalState,
	},
	AF: hiddenState,
	AO: {
		...hiddenPostcode,
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	AT: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	AU: {
		city: {
			label: __( 'Suburb', 'kkart' ),
			optionalLabel: __(
				'Suburb (optional)',
				'kkart'
			),
		},
		postcode: {
			label: __( 'Postcode', 'kkart' ),
			optionalLabel: __(
				'Postcode (optional)',
				'kkart'
			),
		},
		state: {
			label: __( 'State', 'kkart' ),
			optionalLabel: __(
				'State (optional)',
				'kkart'
			),
		},
	},
	AX: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	BD: {
		postcode: {
			required: false,
		},
		state: {
			label: __( 'District', 'kkart' ),
			optionalLabel: __(
				'District (optional)',
				'kkart'
			),
		},
	},
	BE: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	BH: {
		postcode: {
			required: false,
		},
		...hiddenState,
	},
	BI: hiddenState,
	BO: hiddenPostcode,
	BS: hiddenPostcode,
	CA: {
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	CH: {
		...postcodeBeforeCity,
		state: {
			label: __( 'Canton', 'kkart' ),
			optionalLabel: __(
				'Canton (optional)',
				'kkart'
			),
			required: false,
		},
	},
	CL: {
		city: {
			require: true,
		},
		postcode: {
			required: false,
		},
		state: {
			label: __( 'Region', 'kkart' ),
			optionalLabel: __(
				'Region (optional)',
				'kkart'
			),
		},
	},
	CN: {
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	CO: {
		postcode: {
			required: false,
		},
	},
	CZ: hiddenState,
	DE: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	DK: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	EE: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	ES: {
		...postcodeBeforeCity,
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	FI: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	FR: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	GB: {
		postcode: {
			label: __( 'Postcode', 'kkart' ),
			optionalLabel: __(
				'Postcode (optional)',
				'kkart'
			),
		},
		state: {
			label: __( 'County', 'kkart' ),
			optionalLabel: __(
				'County (optional)',
				'kkart'
			),
		},
	},
	GP: hiddenState,
	GF: hiddenState,
	GR: optionalState,
	HK: {
		postcode: {
			required: false,
		},
		city: {
			label: __( 'Town/District', 'kkart' ),
			optionalLabel: __(
				'Town/District (optional)',
				'kkart'
			),
		},
		state: {
			label: __( 'Region', 'kkart' ),
			optionalLabel: __(
				'Region (optional)',
				'kkart'
			),
		},
	},
	HU: {
		state: {
			label: __( 'County', 'kkart' ),
			optionalLabel: __(
				'County (optional)',
				'kkart'
			),
		},
	},
	ID: {
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	IE: {
		postcode: {
			label: __( 'Eircode', 'kkart' ),
			optionalLabel: __(
				'Eircode (optional)',
				'kkart'
			),
			required: false,
		},
		state: {
			label: __( 'County', 'kkart' ),
			optionalLabel: __(
				'County (optional)',
				'kkart'
			),
		},
	},
	IS: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	IL: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	IM: hiddenState,
	IT: {
		...postcodeBeforeCity,
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	JP: {
		first_name: {
			index: 2,
		},
		last_name: {
			index: 1,
		},
		address_1: {
			index: 7,
		},
		address_2: {
			index: 8,
		},
		postcode: {
			index: 4,
		},
		city: {
			index: 6,
		},
		state: {
			label: __( 'Prefecture', 'kkart' ),
			optionalLabel: __(
				'Prefecture (optional)',
				'kkart'
			),
			index: 5,
		},
	},
	KR: hiddenState,
	KW: hiddenState,
	LB: hiddenState,
	LI: {
		...postcodeBeforeCity,
		state: {
			label: __( 'Municipality', 'kkart' ),
			optionalLabel: __(
				'Municipality (optional)',
				'kkart'
			),
			required: false,
		},
	},
	LK: hiddenState,
	LU: hiddenState,
	LV: {
		state: {
			label: __( 'Municipality', 'kkart' ),
			optionalLabel: __(
				'Municipality (optional)',
				'kkart'
			),
			required: false,
		},
	},
	MQ: hiddenState,
	MT: hiddenState,
	MZ: {
		...hiddenPostcode,
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	NL: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	NG: {
		...hiddenPostcode,
		state: {
			label: __( 'State', 'kkart' ),
			optionalLabel: __(
				'State (optional)',
				'kkart'
			),
		},
	},
	NO: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	NP: {
		postcode: {
			required: false,
		},
		state: {
			label: __( 'State', 'kkart' ),
			optionalLabel: __(
				'State (optional)',
				'kkart'
			),
		},
	},
	NZ: {
		postcode: {
			label: __( 'Postcode', 'kkart' ),
			optionalLabel: __(
				'Postcode (optional)',
				'kkart'
			),
		},
		state: {
			label: __( 'Region', 'kkart' ),
			optionalLabel: __(
				'Region (optional)',
				'kkart'
			),
		},
	},
	PL: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	PT: hiddenState,
	RE: hiddenState,
	RO: {
		state: {
			label: __( 'County', 'kkart' ),
			optionalLabel: __(
				'County (optional)',
				'kkart'
			),
		},
	},
	RS: hiddenState,
	SE: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	SG: {
		city: {
			required: false,
		},
		...hiddenState,
	},
	SK: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	SI: {
		...postcodeBeforeCity,
		...hiddenState,
	},
	SR: {
		...hiddenPostcode,
	},
	ST: {
		...hiddenPostcode,
		state: {
			label: __( 'District', 'kkart' ),
			optionalLabel: __(
				'District (optional)',
				'kkart'
			),
		},
	},
	MD: {
		state: {
			label: __(
				'Municipality/District',
				'kkart'
			),
			optionalLabel: __(
				'Municipality/District (optional)',
				'kkart'
			),
		},
	},
	TR: {
		...postcodeBeforeCity,
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	UG: {
		...hiddenPostcode,
		city: {
			label: __( 'Town/Village', 'kkart' ),
			optionalLabel: __(
				'Town/Village (optional)',
				'kkart'
			),
		},
		state: {
			label: __( 'District', 'kkart' ),
			optionalLabel: __(
				'District (optional)',
				'kkart'
			),
		},
	},
	US: {
		postcode: {
			label: __( 'ZIP', 'kkart' ),
			optionalLabel: __(
				'ZIP (optional)',
				'kkart'
			),
		},
		state: {
			label: __( 'State', 'kkart' ),
			optionalLabel: __(
				'State (optional)',
				'kkart'
			),
		},
	},
	VN: {
		city: {
			index: 8,
		},
		postcode: {
			index: 7,
			required: false,
		},
		...hiddenState,
	},
	WS: hiddenPostcode,
	YT: hiddenState,
	ZA: {
		state: {
			label: __( 'Province', 'kkart' ),
			optionalLabel: __(
				'Province (optional)',
				'kkart'
			),
		},
	},
	ZW: hiddenPostcode,
};

export default countryAddressFields;
