/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { Disabled, PanelBody, ToggleControl } from '@wordpress/components';
import { InspectorControls } from '@wordpress/block-editor';
import { createInterpolateElement } from 'wordpress-element';
import ToggleButtonControl from '@kkart/editor-components/toggle-button-control';
import { getAdminLink } from '@kkart/settings';

/**
 * Internal dependencies
 */
import Block from './block';
import withProductSelector from '../shared/with-product-selector';
import { BLOCK_TITLE, BLOCK_ICON } from './constants';

const Edit = ( { attributes, setAttributes } ) => {
	const {
		productLink,
		imageSizing,
		showSaleBadge,
		saleBadgeAlign,
	} = attributes;

	return (
		<>
			<InspectorControls>
				<PanelBody
					title={ __( 'Content', 'kkart' ) }
				>
					<ToggleControl
						label={ __(
							'Link to Product Page',
							'kkart'
						) }
						help={ __(
							'Links the image to the single product listing.',
							'kkart'
						) }
						checked={ productLink }
						onChange={ () =>
							setAttributes( {
								productLink: ! productLink,
							} )
						}
					/>
					<ToggleControl
						label={ __(
							'Show On-Sale Badge',
							'kkart'
						) }
						help={ __(
							'Overlay a "sale" badge if the product is on-sale.',
							'kkart'
						) }
						checked={ showSaleBadge }
						onChange={ () =>
							setAttributes( {
								showSaleBadge: ! showSaleBadge,
							} )
						}
					/>
					{ showSaleBadge && (
						<ToggleButtonControl
							label={ __(
								'Sale Badge Alignment',
								'kkart'
							) }
							value={ saleBadgeAlign }
							options={ [
								{
									label: __(
										'Left',
										'kkart'
									),
									value: 'left',
								},
								{
									label: __(
										'Center',
										'kkart'
									),
									value: 'center',
								},
								{
									label: __(
										'Right',
										'kkart'
									),
									value: 'right',
								},
							] }
							onChange={ ( value ) =>
								setAttributes( { saleBadgeAlign: value } )
							}
						/>
					) }
					<ToggleButtonControl
						label={ __(
							'Image Sizing',
							'kkart'
						) }
						help={ createInterpolateElement(
							__(
								'Product image cropping can be modified in the <a>Customizer</a>.',
								'kkart'
							),
							{
								a: (
									// eslint-disable-next-line jsx-a11y/anchor-has-content
									<a
										href={ `${ getAdminLink(
											'customize.php'
										) }?autofocus[panel]=kkart&autofocus[section]=kkart_product_images` }
										target="_blank"
										rel="noopener noreferrer"
									/>
								),
							}
						) }
						value={ imageSizing }
						options={ [
							{
								label: __(
									'Full Size',
									'kkart'
								),
								value: 'full-size',
							},
							{
								label: __(
									'Cropped',
									'kkart'
								),
								value: 'cropped',
							},
						] }
						onChange={ ( value ) =>
							setAttributes( { imageSizing: value } )
						}
					/>
				</PanelBody>
			</InspectorControls>
			<Disabled>
				<Block { ...attributes } />
			</Disabled>
		</>
	);
};

export default withProductSelector( {
	icon: BLOCK_ICON,
	label: BLOCK_TITLE,
	description: __(
		'Choose a product to display its image.',
		'kkart'
	),
} )( Edit );
