<?php
/**
 * Connect existing Kkart pages to Kkart Admin.
 *
 * @package Kkart\Admin
 */

use Automattic\Kkart\Admin\Loader;
use Automattic\Kkart\Admin\PageController;

/**
 * Returns core KKART pages to connect to KKART-Admin.
 *
 * @return array
 */
function kkart_admin_get_core_pages_to_connect() {
	$all_reports = KKART_Admin_Reports::get_reports();
	$report_tabs = array();

	foreach ( $all_reports as $report_id => $report_data ) {
		$report_tabs[ $report_id ] = $report_data['title'];
	}

	return array(
		'kkart-addons'   => array(
			'title' => __( 'Extensions', 'kkart' ),
			'tabs'  => array(),
		),
		'kkart-reports'  => array(
			'title' => __( 'Reports', 'kkart' ),
			'tabs'  => $report_tabs,
		),
		'kkart-settings' => array(
			'title' => __( 'Settings', 'kkart' ),
			'tabs'  => apply_filters( 'kkart_settings_tabs_array', array() ),
		),
		'kkart-status'   => array(
			'title' => __( 'Status', 'kkart' ),
			'tabs'  => apply_filters(
				'kkart_admin_status_tabs',
				array(
					'status' => __( 'System status', 'kkart' ),
					'tools'  => __( 'Tools', 'kkart' ),
					'logs'   => __( 'Logs', 'kkart' ),
				)
			),
		),
	);
}

/**
 * Filter breadcrumbs for core pages that aren't explicitly connected.
 *
 * @param array $breadcrumbs Breadcrumb pieces.
 * @return array Filtered breadcrumb pieces.
 */
function kkart_admin_filter_core_page_breadcrumbs( $breadcrumbs ) {
	$screen_id              = PageController::get_instance()->get_current_screen_id();
	$pages_to_connect       = kkart_admin_get_core_pages_to_connect();
	$kkart_breadcrumb = array(
		'admin.php?page=kkart-admin',
		__( 'Kkart', 'kkart' ),
	);

	foreach ( $pages_to_connect as $page_id => $page_data ) {
		if ( preg_match( "/^kkart_page_{$page_id}\-/", $screen_id ) ) {
			if ( empty( $page_data['tabs'] ) ) {
				$new_breadcrumbs = array(
					$kkart_breadcrumb,
					$page_data['title'],
				);
			} else {
				$new_breadcrumbs = array(
					$kkart_breadcrumb,
					array(
						add_query_arg( 'page', $page_id, 'admin.php' ),
						$page_data['title'],
					),
				);

				if ( isset( $_GET['tab'] ) ) {
					$current_tab = kkart_clean( wp_unslash( $_GET['tab'] ) );
				} else {
					$current_tab = key( $page_data['tabs'] );
				}

				$new_breadcrumbs[] = $page_data['tabs'][ $current_tab ];
			}

			return $new_breadcrumbs;
		}
	}

	return $breadcrumbs;
}

/**
 * Render the KKART-Admin header bar on all Kkart core pages.
 *
 * @param bool $is_connected Whether the current page is connected.
 * @param bool $current_page The current page, if connected.
 * @return bool Whether to connect the page.
 */
function kkart_admin_connect_core_pages( $is_connected, $current_page ) {
	if ( false === $is_connected && false === $current_page ) {
		$screen_id        = PageController::get_instance()->get_current_screen_id();
		$pages_to_connect = kkart_admin_get_core_pages_to_connect();

		foreach ( $pages_to_connect as $page_id => $page_data ) {
			if ( preg_match( "/^kkart_page_{$page_id}\-/", $screen_id ) ) {
				add_filter( 'kkart_navigation_get_breadcrumbs', 'kkart_admin_filter_core_page_breadcrumbs' );

				return true;
			}
		}
	}

	return $is_connected;
}

add_filter( 'kkart_navigation_is_connected_page', 'kkart_admin_connect_core_pages', 10, 2 );

$posttype_list_base = 'edit.php';

// Kkart > Orders.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-orders',
		'screen_id' => 'edit-shop_order',
		'title'     => __( 'Orders', 'kkart' ),
		'path'      => add_query_arg( 'post_type', 'shop_order', $posttype_list_base ),
	)
);

// Kkart > Orders > Add New.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-add-order',
		'parent'    => 'kkart-orders',
		'screen_id' => 'shop_order-add',
		'title'     => __( 'Add New', 'kkart' ),
	)
);

// Kkart > Orders > Edit Order.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-edit-order',
		'parent'    => 'kkart-orders',
		'screen_id' => 'shop_order',
		'title'     => __( 'Edit Order', 'kkart' ),
	)
);

// Kkart > Coupons.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-coupons',
		'parent'    => Loader::is_feature_enabled( 'coupons' ) ? 'kkart-marketing' : null,
		'screen_id' => 'edit-shop_coupon',
		'title'     => __( 'Coupons', 'kkart' ),
		'path'      => add_query_arg( 'post_type', 'shop_coupon', $posttype_list_base ),
	)
);

// Kkart > Coupons > Add New.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-add-coupon',
		'parent'    => 'kkart-coupons',
		'screen_id' => 'shop_coupon-add',
		'title'     => __( 'Add New', 'kkart' ),
	)
);

// Kkart > Coupons > Edit Coupon.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-edit-coupon',
		'parent'    => 'kkart-coupons',
		'screen_id' => 'shop_coupon',
		'title'     => __( 'Edit Coupon', 'kkart' ),
	)
);

// Kkart > Products.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-products',
		'screen_id' => 'edit-product',
		'title'     => __( 'Products', 'kkart' ),
		'path'      => add_query_arg( 'post_type', 'product', $posttype_list_base ),
	)
);

// Kkart > Products > Add New.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-add-product',
		'parent'    => 'kkart-products',
		'screen_id' => 'product-add',
		'title'     => __( 'Add New', 'kkart' ),
	)
);

// Kkart > Products > Edit Order.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-edit-product',
		'parent'    => 'kkart-products',
		'screen_id' => 'product',
		'title'     => __( 'Edit Product', 'kkart' ),
	)
);

// Kkart > Products > Import Products.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-import-products',
		'parent'    => 'kkart-products',
		'screen_id' => 'product_page_product_importer',
		'title'     => __( 'Import Products', 'kkart' ),
	)
);

// Kkart > Products > Export Products.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-export-products',
		'parent'    => 'kkart-products',
		'screen_id' => 'product_page_product_exporter',
		'title'     => __( 'Export Products', 'kkart' ),
	)
);

// Kkart > Products > Product categories.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-product-categories',
		'parent'    => 'kkart-products',
		'screen_id' => 'edit-product_cat',
		'title'     => __( 'Product categories', 'kkart' ),
	)
);

// Kkart > Products > Edit category.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-product-edit-category',
		'parent'    => 'kkart-products',
		'screen_id' => 'product_cat',
		'title'     => __( 'Edit category', 'kkart' ),
	)
);

// Kkart > Products > Product tags.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-product-tags',
		'parent'    => 'kkart-products',
		'screen_id' => 'edit-product_tag',
		'title'     => __( 'Product tags', 'kkart' ),
	)
);

// Kkart > Products > Edit tag.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-product-edit-tag',
		'parent'    => 'kkart-products',
		'screen_id' => 'product_tag',
		'title'     => __( 'Edit tag', 'kkart' ),
	)
);

// Kkart > Products > Attributes.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-product-attributes',
		'parent'    => 'kkart-products',
		'screen_id' => 'product_page_product_attributes',
		'title'     => __( 'Attributes', 'kkart' ),
	)
);

// Kkart > Products > Edit attribute.
kkart_admin_connect_page(
	array(
		'id'        => 'kkart-product-edit-attribute',
		'parent'    => 'kkart-products',
		'screen_id' => 'product_page_product_attribute-edit',
		'title'     => __( 'Edit attribute', 'kkart' ),
	)
);
