<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_admin
 *
 * @copyright   (C) 2011 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt

 * @phpcs:disable PSR1.Classes.ClassDeclaration.MissingNamespace
 */

use Joomla\CMS\Extension\ExtensionHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Table\Table;
use Joomla\Component\Fields\Administrator\Model\FieldModel;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Script file of Joomla CMS
 *
 * @since  1.6.4
 */
class JoomlaInstallerScript
{
    /**
     * The Joomla Version we are updating from
     *
     * @var    string
     * @since  3.7
     */
    protected $fromVersion = null;

    /**
     * Callback for collecting errors. Like function(string $context, \Throwable $error){};
     *
     * @var callable
     *
     * @since  4.4.0
     */
    protected $errorCollector;

    /**
     * Set the callback for collecting errors.
     *
     * @param   callable  $callback  The callback Like function(string $context, \Throwable $error){};
     *
     * @return  void
     *
     * @since  4.4.0
     */
    public function setErrorCollector(callable $callback)
    {
        $this->errorCollector = $callback;
    }

    /**
     * Collect errors.
     *
     * @param  string      $context  A context/place where error happened
     * @param  \Throwable  $error    The error that occurred
     *
     * @return  void
     *
     * @since  4.4.0
     */
    protected function collectError(string $context, \Throwable $error)
    {
        // The errorCollector are required
        // However when someone already running the script manually the code may fail.
        if ($this->errorCollector) {
            call_user_func($this->errorCollector, $context, $error);
        } else {
            Log::add($error->getMessage(), Log::ERROR, 'Update');
        }
    }

    /**
     * Function to act prior to installation process begins
     *
     * @param   string     $action     Which action is happening (install|uninstall|discover_install|update)
     * @param   Installer  $installer  The class calling this method
     *
     * @return  boolean  True on success
     *
     * @since   3.7.0
     */
    public function preflight($action, $installer)
    {
        if ($action === 'update') {
            // Get the version we are updating from
            if (!empty($installer->extension->manifest_cache)) {
                $manifestValues = json_decode($installer->extension->manifest_cache, true);

                if (array_key_exists('version', $manifestValues)) {
                    $this->fromVersion = $manifestValues['version'];

                    // Ensure templates are moved to the correct mode
                    $this->fixTemplateMode();

                    return true;
                }
            }

            return false;
        }

        return true;
    }

    /**
     * Method to update Joomla!
     *
     * @param   Installer  $installer  The class calling this method
     *
     * @return  void
     */
    public function update($installer)
    {
        // Uninstall plugins before removing their files and folders
        try {
            $this->uninstallRepeatableFieldsPlugin();
        } catch (\Throwable $e) {
            $this->collectError('uninstallRepeatableFieldsPlugin', $e);
        }

        try {
            $this->uninstallEosPlugin();
        } catch (\Throwable $e) {
            $this->collectError('uninstallEosPlugin', $e);
        }

        // Remove old files
        try {
            Log::add(Text::_('COM_JOOMLAUPDATE_UPDATE_LOG_DELETE_FILES'), Log::INFO, 'Update');
            $this->deleteUnexistingFiles();
        } catch (\Throwable $e) {
            $this->collectError('deleteUnexistingFiles', $e);
        }

        // Further update
        try {
            $this->updateManifestCaches();
            $this->updateDatabase();
            $this->updateAssets($installer);
            $this->clearStatsCache();
            $this->convertTablesToUtf8mb4(true);
            $this->addUserAuthProviderColumn();
        } catch (\Throwable $e) {
            $this->collectError('Further update', $e);
        }

        // Clean cache
        try {
            $this->cleanJoomlaCache();
        } catch (\Throwable $e) {
            $this->collectError('cleanJoomlaCache', $e);
        }
    }

    /**
     * Method to clear our stats plugin cache to ensure we get fresh data on Joomla Update
     *
     * @return  void
     *
     * @since   3.5
     */
    protected function clearStatsCache()
    {
        $db = Factory::getDbo();

        try {
            // Get the params for the stats plugin
            $params = $db->setQuery(
                $db->getQuery(true)
                    ->select($db->quoteName('params'))
                    ->from($db->quoteName('#__extensions'))
                    ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                    ->where($db->quoteName('folder') . ' = ' . $db->quote('system'))
                    ->where($db->quoteName('element') . ' = ' . $db->quote('stats'))
            )->loadResult();
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            return;
        }

        $params = json_decode($params, true);

        // Reset the last run parameter
        if (isset($params['lastrun'])) {
            $params['lastrun'] = '';
        }

        $params = json_encode($params);

        $query = $db->getQuery(true)
            ->update($db->quoteName('#__extensions'))
            ->set($db->quoteName('params') . ' = ' . $db->quote($params))
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('folder') . ' = ' . $db->quote('system'))
            ->where($db->quoteName('element') . ' = ' . $db->quote('stats'));

        try {
            $db->setQuery($query)->execute();
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            return;
        }
    }

    /**
     * Method to update Database
     *
     * @return  void
     */
    protected function updateDatabase()
    {
        if (Factory::getDbo()->getServerType() === 'mysql') {
            $this->updateDatabaseMysql();
        }
    }

    /**
     * Method to update MySQL Database
     *
     * @return  void
     */
    protected function updateDatabaseMysql()
    {
        $db = Factory::getDbo();

        $db->setQuery('SHOW ENGINES');

        try {
            $results = $db->loadObjectList();
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            return;
        }

        foreach ($results as $result) {
            if ($result->Support != 'DEFAULT') {
                continue;
            }

            $db->setQuery('ALTER TABLE #__update_sites_extensions ENGINE = ' . $result->Engine);

            try {
                $db->execute();
            } catch (Exception $e) {
                $this->collectError(__METHOD__, $e);

                return;
            }

            break;
        }
    }

    /**
     * Uninstalls the plg_fields_repeatable plugin and transforms its custom field instances
     * to instances of the plg_fields_subfields plugin.
     *
     * @return  void
     *
     * @since   4.0.0
     */
    protected function uninstallRepeatableFieldsPlugin()
    {
        $app = Factory::getApplication();
        $db  = Factory::getDbo();

        // Check if the plg_fields_repeatable plugin is present
        $extensionId = $db->setQuery(
            $db->getQuery(true)
                ->select('extension_id')
                ->from('#__extensions')
                ->where('name = ' . $db->quote('plg_fields_repeatable'))
        )->loadResult();

        // Skip uninstalling when it doesn't exist
        if (!$extensionId) {
            return;
        }

        // Ensure the FieldsHelper class is loaded for the Repeatable fields plugin we're about to remove
        \JLoader::register('FieldsHelper', JPATH_ADMINISTRATOR . '/components/com_fields/helpers/fields.php');

        // Get the FieldsModelField, we need it in a sec
        $fieldModel = $app->bootComponent('com_fields')->getMVCFactory()->createModel('Field', 'Administrator', ['ignore_request' => true]);
        /** @var FieldModel $fieldModel */

        // Now get a list of all `repeatable` custom field instances
        $db->setQuery(
            $db->getQuery(true)
                ->select('*')
                ->from('#__fields')
                ->where($db->quoteName('type') . ' = ' . $db->quote('repeatable'))
        );

        // Execute the query and iterate over the `repeatable` instances
        foreach ($db->loadObjectList() as $row) {
            // Skip broken rows - just a security measure, should not happen
            if (!isset($row->fieldparams) || !($oldFieldparams = json_decode($row->fieldparams)) || !is_object($oldFieldparams)) {
                continue;
            }

            // First get this field's values for later data migration, so if this fails it happens before saving new subfields
            $query = $db->getQuery(true)
                ->select('*')
                ->from($db->quoteName('#__fields_values'))
                ->where($db->quoteName('field_id') . ' = ' . $row->id);
            $db->setQuery($query);
            $rowFieldValues = $db->loadObjectList();

            /**
             * We basically want to transform this `repeatable` type into a `subfields` type. While $oldFieldparams
             * holds the `fieldparams` of the `repeatable` type, $newFieldparams shall hold the `fieldparams`
             * of the `subfields` type.
             */
            $newFieldparams = [
                'repeat'  => '1',
                'options' => [],
            ];

            /**
             * This array is used to store the mapping between the name of form fields from Repeatable field
             * with ID of the child-fields. It will then be used to migrate data later
             */
            $mapping = [];

            /**
             * Store name of media fields which we need to convert data from old format (string) to new
             * format (json) during the migration
             */
            $mediaFields = [];

            // If this repeatable fields actually had child-fields (normally this is always the case)
            if (isset($oldFieldparams->fields) && is_object($oldFieldparams->fields)) {
                // Small counter for the child-fields (aka sub fields)
                $newFieldCount = 0;

                // Iterate over the sub fields
                foreach (get_object_vars($oldFieldparams->fields) as $oldField) {
                    // Used for field name collision prevention
                    $fieldname_prefix = '';
                    $fieldname_suffix = 0;

                    // Try to save the new sub field in a loop because of field name collisions
                    while (true) {
                        /**
                         * We basically want to create a completely new custom fields instance for every sub field
                         * of the `repeatable` instance. This is what we use $data for, we create a new custom field
                         * for each of the sub fields of the `repeatable` instance.
                         */
                        $data = [
                            'context'  => $row->context,
                            'group_id' => $row->group_id,
                            'title'    => $oldField->fieldname,
                            'name'     => (
                                $fieldname_prefix
                                . $oldField->fieldname
                                . ($fieldname_suffix > 0 ? ('_' . $fieldname_suffix) : '')
                            ),
                            'label'               => $oldField->fieldname,
                            'default_value'       => $row->default_value,
                            'type'                => $oldField->fieldtype,
                            'description'         => $row->description,
                            'state'               => '1',
                            'params'              => $row->params,
                            'language'            => '*',
                            'assigned_cat_ids'    => [-1],
                            'only_use_in_subform' => 1,
                        ];

                        // `number` is not a valid custom field type, so use `text` instead.
                        if ($data['type'] == 'number') {
                            $data['type'] = 'text';
                        }

                        if ($data['type'] == 'media') {
                            $mediaFields[] = $oldField->fieldname;
                        }

                        // Reset the state because else \Joomla\CMS\MVC\Model\AdminModel will take an already
                        // existing value (e.g. from previous save) and do an UPDATE instead of INSERT.
                        $fieldModel->setState('field.id', 0);

                        // If an error occurred when trying to save this.
                        if (!$fieldModel->save($data)) {
                            // If the error is, that the name collided, increase the collision prevention
                            $error = $fieldModel->getError();

                            if ($error == 'COM_FIELDS_ERROR_UNIQUE_NAME') {
                                // If this is the first time this error occurs, set only the prefix
                                if ($fieldname_prefix == '') {
                                    $fieldname_prefix = ($row->name . '_');
                                } else {
                                    // Else increase the suffix
                                    $fieldname_suffix++;
                                }

                                // And start again with the while loop.
                                continue 1;
                            }

                            // Else bail out with the error. Something is totally wrong.
                            throw new \Exception($error);
                        }

                        // Break out of the while loop, saving was successful.
                        break 1;
                    }

                    // Get the newly created id
                    $subfield_id = $fieldModel->getState('field.id');

                    // Really check that it is valid
                    if (!is_numeric($subfield_id) || $subfield_id < 1) {
                        throw new \Exception('Something went wrong.');
                    }

                    // And tell our new `subfields` field about his child
                    $newFieldparams['options'][('option' . $newFieldCount)] = [
                        'customfield'   => $subfield_id,
                        'render_values' => '1',
                    ];

                    $newFieldCount++;

                    $mapping[$oldField->fieldname] = 'field' . $subfield_id;
                }
            }

            try {
                $db->transactionStart();

                // Write back the changed stuff to the database
                $db->setQuery(
                    $db->getQuery(true)
                        ->update('#__fields')
                        ->set($db->quoteName('type') . ' = ' . $db->quote('subform'))
                        ->set($db->quoteName('fieldparams') . ' = ' . $db->quote(json_encode($newFieldparams)))
                        ->where($db->quoteName('id') . ' = ' . $db->quote($row->id))
                )->execute();

                // Migrate field values for this field
                foreach ($rowFieldValues as $rowFieldValue) {
                    // Do not do the version if no data is entered for the custom field this item
                    if (!$rowFieldValue->value) {
                        continue;
                    }

                    /**
                     * Here we will have to update the stored value of the field to new format
                     * The key for each row changes from repeatable to row, for example repeatable0 to row0, and so on
                     * The key for each sub-field change from name of field to field + ID of the new sub-field
                     * Example data format stored in J3: {"repeatable0":{"id":"1","username":"admin"}}
                     * Example data format stored in J4: {"row0":{"field1":"1","field2":"admin"}}
                     */
                    $newFieldValue = [];

                    // Convert to array to change key
                    $fieldValue = json_decode($rowFieldValue->value, true);

                    // If data could not be decoded for some reason, ignore
                    if (!$fieldValue) {
                        continue;
                    }

                    $rowIndex = 0;

                    foreach ($fieldValue as $rowKey => $rowValue) {
                        $rowKey                 = 'row' . ($rowIndex++);
                        $newFieldValue[$rowKey] = [];

                        foreach ($rowValue as $subFieldName => $subFieldValue) {
                            // This is a media field, so we need to convert data to new format required in Joomla! 4
                            if (in_array($subFieldName, $mediaFields)) {
                                $subFieldValue = ['imagefile' => $subFieldValue, 'alt_text' => ''];
                            }

                            if (isset($mapping[$subFieldName])) {
                                $newFieldValue[$rowKey][$mapping[$subFieldName]] = $subFieldValue;
                            } else {
                                // Not found, use the old key to avoid data lost
                                $newFieldValue[$subFieldName] = $subFieldValue;
                            }
                        }
                    }

                    $query->clear()
                        ->update($db->quoteName('#__fields_values'))
                        ->set($db->quoteName('value') . ' = ' . $db->quote(json_encode($newFieldValue)))
                        ->where($db->quoteName('field_id') . ' = ' . $rowFieldValue->field_id)
                        ->where($db->quoteName('item_id') . ' = ' . $db->quote($rowFieldValue->item_id));
                    $db->setQuery($query)
                        ->execute();
                }

                $db->transactionCommit();
            } catch (\Exception $e) {
                $db->transactionRollback();
                throw $e;
            }
        }

        try {
            $db->transactionStart();

            // Now, unprotect the plugin so we can uninstall it
            $db->setQuery(
                $db->getQuery(true)
                    ->update('#__extensions')
                    ->set('protected = 0')
                    ->where($db->quoteName('extension_id') . ' = ' . $extensionId)
            )->execute();

            // And now uninstall the plugin
            $installer = new Installer();
            $installer->setDatabase($db);
            $installer->uninstall('plugin', $extensionId);

            $db->transactionCommit();
        } catch (\Exception $e) {
            $db->transactionRollback();
            throw $e;
        }
    }

    /**
     * Uninstall the 3.10 EOS plugin
     *
     * @return  void
     *
     * @since   4.0.0
     */
    protected function uninstallEosPlugin()
    {
        $db = Factory::getDbo();

        // Check if the plg_quickicon_eos310 plugin is present
        $extensionId = $db->setQuery(
            $db->getQuery(true)
                ->select('extension_id')
                ->from('#__extensions')
                ->where('name = ' . $db->quote('plg_quickicon_eos310'))
        )->loadResult();

        // Skip uninstalling if it doesn't exist
        if (!$extensionId) {
            return;
        }

        try {
            $db->transactionStart();

            // Unprotect the plugin so we can uninstall it
            $db->setQuery(
                $db->getQuery(true)
                    ->update('#__extensions')
                    ->set('protected = 0')
                    ->where($db->quoteName('extension_id') . ' = ' . $extensionId)
            )->execute();

            // Uninstall the plugin
            $installer = new Installer();
            $installer->setDatabase($db);
            $installer->uninstall('plugin', $extensionId);

            $db->transactionCommit();
        } catch (\Exception $e) {
            $db->transactionRollback();
            throw $e;
        }
    }

    /**
     * Update the manifest caches
     *
     * @return  void
     */
    protected function updateManifestCaches()
    {
        $extensions = ExtensionHelper::getCoreExtensions();

        // If we have the search package around, it may not have a manifest cache entry after upgrades from 3.x, so add it to the list
        if (is_file(JPATH_ROOT . '/administrator/manifests/packages/pkg_search.xml')) {
            $extensions[] = ['package', 'pkg_search', '', 0];
        }

        // Attempt to refresh manifest caches
        $db    = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__extensions');

        foreach ($extensions as $extension) {
            $query->where(
                'type=' . $db->quote($extension[0])
                . ' AND element=' . $db->quote($extension[1])
                . ' AND folder=' . $db->quote($extension[2])
                . ' AND client_id=' . $extension[3],
                'OR'
            );
        }

        $db->setQuery($query);

        try {
            $extensions = $db->loadObjectList();
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            return;
        }

        $installer = new Installer();
        $installer->setDatabase($db);

        foreach ($extensions as $extension) {
            if (!$installer->refreshManifestCache($extension->extension_id)) {
                $this->collectError(
                    __METHOD__,
                    new \Exception(sprintf(
                        'Error on updating manifest cache: (type, element, folder, client) = (%s, %s, %s, %s)',
                        $extension->type,
                        $extension->element,
                        $extension->name,
                        $extension->client_id
                    ))
                );
            }
        }
    }

    /**
     * Delete files that should not exist
     *
     * @param bool  $dryRun          If set to true, will not actually delete files, but just report their status for use in CLI
     * @param bool  $suppressOutput   Set to true to suppress echoing any errors, and just return the $status array
     *
     * @return  array
     */
    public function deleteUnexistingFiles($dryRun = false, $suppressOutput = false)
    {
        $status = [
            'files_exist'     => [],
            'folders_exist'   => [],
            'files_deleted'   => [],
            'folders_deleted' => [],
            'files_errors'    => [],
            'folders_errors'  => [],
            'folders_checked' => [],
            'files_checked'   => [],
        ];

        $files = [
            // From 3.10 to 4.3
            '/administrator/components/com_actionlogs/actionlogs.php',
            '/administrator/components/com_actionlogs/controller.php',
            '/administrator/components/com_actionlogs/controllers/actionlogs.php',
            '/administrator/components/com_actionlogs/helpers/actionlogs.php',
            '/administrator/components/com_actionlogs/helpers/actionlogsphp55.php',
            '/administrator/components/com_actionlogs/layouts/logstable.php',
            '/administrator/components/com_actionlogs/libraries/actionlogplugin.php',
            '/administrator/components/com_actionlogs/models/actionlog.php',
            '/administrator/components/com_actionlogs/models/actionlogs.php',
            '/administrator/components/com_actionlogs/models/fields/extension.php',
            '/administrator/components/com_actionlogs/models/fields/logcreator.php',
            '/administrator/components/com_actionlogs/models/fields/logsdaterange.php',
            '/administrator/components/com_actionlogs/models/fields/logtype.php',
            '/administrator/components/com_actionlogs/models/fields/plugininfo.php',
            '/administrator/components/com_actionlogs/models/forms/filter_actionlogs.xml',
            '/administrator/components/com_actionlogs/views/actionlogs/tmpl/default.php',
            '/administrator/components/com_actionlogs/views/actionlogs/tmpl/default.xml',
            '/administrator/components/com_actionlogs/views/actionlogs/view.html.php',
            '/administrator/components/com_admin/admin.php',
            '/administrator/components/com_admin/controller.php',
            '/administrator/components/com_admin/controllers/profile.php',
            '/administrator/components/com_admin/helpers/html/directory.php',
            '/administrator/components/com_admin/helpers/html/phpsetting.php',
            '/administrator/components/com_admin/helpers/html/system.php',
            '/administrator/components/com_admin/models/forms/profile.xml',
            '/administrator/components/com_admin/models/help.php',
            '/administrator/components/com_admin/models/profile.php',
            '/administrator/components/com_admin/models/sysinfo.php',
            '/administrator/components/com_admin/postinstall/eaccelerator.php',
            '/administrator/components/com_admin/postinstall/htaccess.php',
            '/administrator/components/com_admin/postinstall/joomla40checks.php',
            '/administrator/components/com_admin/postinstall/updatedefaultsettings.php',
            '/administrator/components/com_admin/sql/others/mysql/utf8mb4-conversion-01.sql',
            '/administrator/components/com_admin/sql/others/mysql/utf8mb4-conversion-02.sql',
            '/administrator/components/com_admin/sql/others/mysql/utf8mb4-conversion-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-06.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-21-1.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-21-2.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-22.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-23.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2011-12-24.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2012-01-10.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.0-2012-01-14.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.1-2012-01-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.2-2012-03-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.3-2012-03-13.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.4-2012-03-18.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.4-2012-03-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.5.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.6.sql',
            '/administrator/components/com_admin/sql/updates/mysql/2.5.7.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.0.0.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.0.1.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.0.2.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.0.3.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.1.0.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.1.1.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.1.2.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.1.3.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.1.4.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.1.5.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.10.0-2020-08-10.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.10.0-2021-05-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.10.7-2022-02-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.10.7-2022-03-18.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.0.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.1.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.2-2013-12-22.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.2-2013-12-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.2-2014-01-08.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.2-2014-01-15.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.2-2014-01-18.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.2-2014-01-23.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.2.3-2014-02-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.3.0-2014-02-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.3.0-2014-04-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.3.4-2014-08-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.3.6-2014-09-30.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2014-08-24.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2014-09-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2014-09-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2014-10-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2014-12-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2015-01-21.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.4.0-2015-02-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2015-07-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2015-10-13.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2015-10-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2015-10-30.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2015-11-04.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2015-11-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2016-02-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.0-2016-03-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.1-2016-03-25.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.5.1-2016-03-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-04-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-04-06.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-04-08.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-04-09.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-05-06.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-06-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.0-2016-06-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.3-2016-08-15.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.6.3-2016-08-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-08-06.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-08-22.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-08-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-09-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-10-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-10-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-11-04.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-11-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-11-21.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-11-24.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2016-11-27.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-01-08.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-01-09.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-01-15.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-01-17.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-01-31.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-02-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-02-15.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-02-17.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-03-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-03-09.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-03-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-04-10.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.0-2017-04-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.3-2017-06-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.7.4-2017-07-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.0-2017-07-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.0-2017-07-31.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.2-2017-10-14.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.4-2018-01-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.6-2018-02-14.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.8-2018-05-18.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.8.9-2018-06-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-24.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-05-27.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-06-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-06-12.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-06-13.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-06-14.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-06-17.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-07-09.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-07-10.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-07-11.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-08-12.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-08-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-08-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-09-04.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-10-15.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-10-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.0-2018-10-21.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.10-2019-07-09.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.16-2020-02-15.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.16-2020-03-04.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.19-2020-05-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.19-2020-06-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.21-2020-08-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.22-2020-09-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.26-2021-04-07.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.27-2021-04-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.3-2019-01-12.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.3-2019-02-07.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.7-2019-04-23.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.7-2019-04-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.7-2019-05-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.8-2019-06-11.sql',
            '/administrator/components/com_admin/sql/updates/mysql/3.9.8-2019-06-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.0.0.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.0.1.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.0.2.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.0.3.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.1.0.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.1.1.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.1.2.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.1.3.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.1.4.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.1.5.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.10.0-2020-08-10.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.10.0-2021-05-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.10.7-2022-02-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.10.7-2022-02-20.sql.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.10.7-2022-03-18.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.0.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.1.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.2-2013-12-22.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.2-2013-12-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.2-2014-01-08.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.2-2014-01-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.2-2014-01-18.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.2-2014-01-23.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.2.3-2014-02-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.3.0-2013-12-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.3.0-2014-02-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.3.0-2014-04-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.3.4-2014-08-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.3.6-2014-09-30.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2014-08-24.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2014-09-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2014-09-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2014-10-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2014-12-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2015-01-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.0-2015-02-26.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.4.4-2015-07-11.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.5.0-2015-10-13.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.5.0-2015-10-26.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.5.0-2015-10-30.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.5.0-2015-11-04.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.5.0-2015-11-05.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.5.0-2016-03-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.0-2016-04-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.0-2016-04-08.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.0-2016-04-09.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.0-2016-05-06.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.0-2016-06-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.0-2016-06-05.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.3-2016-08-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.3-2016-08-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.6.3-2016-10-04.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-08-06.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-08-22.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-08-29.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-09-29.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-10-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-10-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-11-04.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-11-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-11-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2016-11-24.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-01-08.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-01-09.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-01-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-01-17.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-01-31.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-02-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-02-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-02-17.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-03-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-03-09.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-04-10.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.0-2017-04-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.7.4-2017-07-05.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.0-2017-07-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.0-2017-07-31.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.2-2017-10-14.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.4-2018-01-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.6-2018-02-14.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.8-2018-05-18.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.8.9-2018-06-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-05.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-24.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-05-27.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-06-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-06-12.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-06-13.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-06-14.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-06-17.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-07-09.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-07-10.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-07-11.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-08-12.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-08-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-08-29.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-09-04.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-10-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-10-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.0-2018-10-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.10-2019-07-09.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.15-2020-01-08.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.16-2020-02-15.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.16-2020-03-04.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.19-2020-06-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.21-2020-08-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.22-2020-09-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.26-2021-04-07.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.27-2021-04-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.3-2019-01-12.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.3-2019-02-07.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.7-2019-04-23.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.7-2019-04-26.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.7-2019-05-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.8-2019-06-11.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/3.9.8-2019-06-15.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.2-2012-03-05.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.3-2012-03-13.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.4-2012-03-18.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.4-2012-03-19.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.5.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.6.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/2.5.7.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.0.0.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.0.1.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.0.2.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.0.3.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.1.0.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.1.1.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.1.2.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.1.3.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.1.4.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.1.5.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.10.0-2021-05-28.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.10.1-2021-08-17.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.10.7-2022-02-20.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.10.7-2022-02-20.sql.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.10.7-2022-03-18.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.0.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.1.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.2-2013-12-22.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.2-2013-12-28.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.2-2014-01-08.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.2-2014-01-15.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.2-2014-01-18.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.2-2014-01-23.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.2.3-2014-02-20.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.3.0-2014-02-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.3.0-2014-04-02.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.3.4-2014-08-03.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.3.6-2014-09-30.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2014-08-24.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2014-09-01.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2014-09-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2014-10-20.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2014-12-03.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2015-01-21.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.0-2015-02-26.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.4.4-2015-07-11.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.5.0-2015-10-13.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.5.0-2015-10-26.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.5.0-2015-10-30.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.5.0-2015-11-04.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.5.0-2015-11-05.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.5.0-2016-03-01.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-04-01.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-04-06.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-04-08.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-04-09.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-05-06.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-06-01.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.0-2016-06-05.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.3-2016-08-15.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.6.3-2016-08-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-08-06.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-08-22.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-08-29.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-09-29.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-10-01.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-10-02.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-11-04.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-11-19.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2016-11-24.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-01-08.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-01-09.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-01-15.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-01-17.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-01-31.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-02-02.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-02-15.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-02-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-02-17.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-03-03.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-03-09.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-04-10.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.0-2017-04-19.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.7.4-2017-07-05.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.0-2017-07-28.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.0-2017-07-31.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.2-2017-10-14.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.4-2018-01-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.6-2018-02-14.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.8-2018-05-18.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.8.9-2018-06-19.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-02.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-03.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-05.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-19.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-20.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-24.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-05-27.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-06-02.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-06-12.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-06-13.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-06-14.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-06-17.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-07-09.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-07-10.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-07-11.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-08-12.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-08-28.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-08-29.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-09-04.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-10-15.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-10-20.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.0-2018-10-21.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.10-2019-07-09.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.16-2020-03-04.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.19-2020-06-01.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.21-2020-08-02.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.22-2020-09-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.26-2021-04-07.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.27-2021-04-20.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.3-2019-01-12.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.3-2019-02-07.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.4-2019-03-06.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.7-2019-04-23.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.7-2019-04-26.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.7-2019-05-16.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.8-2019-06-11.sql',
            '/administrator/components/com_admin/sql/updates/sqlazure/3.9.8-2019-06-15.sql',
            '/administrator/components/com_admin/views/help/tmpl/default.php',
            '/administrator/components/com_admin/views/help/tmpl/default.xml',
            '/administrator/components/com_admin/views/help/tmpl/langforum.php',
            '/administrator/components/com_admin/views/help/view.html.php',
            '/administrator/components/com_admin/views/profile/tmpl/edit.php',
            '/administrator/components/com_admin/views/profile/view.html.php',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default.php',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default.xml',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default_config.php',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default_directory.php',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default_phpinfo.php',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default_phpsettings.php',
            '/administrator/components/com_admin/views/sysinfo/tmpl/default_system.php',
            '/administrator/components/com_admin/views/sysinfo/view.html.php',
            '/administrator/components/com_admin/views/sysinfo/view.json.php',
            '/administrator/components/com_admin/views/sysinfo/view.text.php',
            '/administrator/components/com_associations/associations.php',
            '/administrator/components/com_associations/controller.php',
            '/administrator/components/com_associations/controllers/association.php',
            '/administrator/components/com_associations/controllers/associations.php',
            '/administrator/components/com_associations/helpers/associations.php',
            '/administrator/components/com_associations/layouts/joomla/searchtools/default/bar.php',
            '/administrator/components/com_associations/models/association.php',
            '/administrator/components/com_associations/models/associations.php',
            '/administrator/components/com_associations/models/fields/itemlanguage.php',
            '/administrator/components/com_associations/models/fields/itemtype.php',
            '/administrator/components/com_associations/models/fields/modalassociation.php',
            '/administrator/components/com_associations/models/forms/association.xml',
            '/administrator/components/com_associations/models/forms/filter_associations.xml',
            '/administrator/components/com_associations/views/association/tmpl/edit.php',
            '/administrator/components/com_associations/views/association/view.html.php',
            '/administrator/components/com_associations/views/associations/tmpl/default.php',
            '/administrator/components/com_associations/views/associations/tmpl/default.xml',
            '/administrator/components/com_associations/views/associations/tmpl/modal.php',
            '/administrator/components/com_associations/views/associations/view.html.php',
            '/administrator/components/com_banners/banners.php',
            '/administrator/components/com_banners/controller.php',
            '/administrator/components/com_banners/controllers/banner.php',
            '/administrator/components/com_banners/controllers/banners.php',
            '/administrator/components/com_banners/controllers/client.php',
            '/administrator/components/com_banners/controllers/clients.php',
            '/administrator/components/com_banners/controllers/tracks.php',
            '/administrator/components/com_banners/controllers/tracks.raw.php',
            '/administrator/components/com_banners/helpers/html/banner.php',
            '/administrator/components/com_banners/models/banner.php',
            '/administrator/components/com_banners/models/banners.php',
            '/administrator/components/com_banners/models/client.php',
            '/administrator/components/com_banners/models/clients.php',
            '/administrator/components/com_banners/models/download.php',
            '/administrator/components/com_banners/models/fields/bannerclient.php',
            '/administrator/components/com_banners/models/fields/clicks.php',
            '/administrator/components/com_banners/models/fields/impmade.php',
            '/administrator/components/com_banners/models/fields/imptotal.php',
            '/administrator/components/com_banners/models/forms/banner.xml',
            '/administrator/components/com_banners/models/forms/client.xml',
            '/administrator/components/com_banners/models/forms/download.xml',
            '/administrator/components/com_banners/models/forms/filter_banners.xml',
            '/administrator/components/com_banners/models/forms/filter_clients.xml',
            '/administrator/components/com_banners/models/forms/filter_tracks.xml',
            '/administrator/components/com_banners/models/tracks.php',
            '/administrator/components/com_banners/tables/banner.php',
            '/administrator/components/com_banners/tables/client.php',
            '/administrator/components/com_banners/views/banner/tmpl/edit.php',
            '/administrator/components/com_banners/views/banner/view.html.php',
            '/administrator/components/com_banners/views/banners/tmpl/default.php',
            '/administrator/components/com_banners/views/banners/tmpl/default_batch_body.php',
            '/administrator/components/com_banners/views/banners/tmpl/default_batch_footer.php',
            '/administrator/components/com_banners/views/banners/view.html.php',
            '/administrator/components/com_banners/views/client/tmpl/edit.php',
            '/administrator/components/com_banners/views/client/view.html.php',
            '/administrator/components/com_banners/views/clients/tmpl/default.php',
            '/administrator/components/com_banners/views/clients/view.html.php',
            '/administrator/components/com_banners/views/download/tmpl/default.php',
            '/administrator/components/com_banners/views/download/view.html.php',
            '/administrator/components/com_banners/views/tracks/tmpl/default.php',
            '/administrator/components/com_banners/views/tracks/view.html.php',
            '/administrator/components/com_banners/views/tracks/view.raw.php',
            '/administrator/components/com_cache/cache.php',
            '/administrator/components/com_cache/controller.php',
            '/administrator/components/com_cache/helpers/cache.php',
            '/administrator/components/com_cache/models/cache.php',
            '/administrator/components/com_cache/models/forms/filter_cache.xml',
            '/administrator/components/com_cache/views/cache/tmpl/default.php',
            '/administrator/components/com_cache/views/cache/tmpl/default.xml',
            '/administrator/components/com_cache/views/cache/view.html.php',
            '/administrator/components/com_cache/views/purge/tmpl/default.php',
            '/administrator/components/com_cache/views/purge/tmpl/default.xml',
            '/administrator/components/com_cache/views/purge/view.html.php',
            '/administrator/components/com_categories/categories.php',
            '/administrator/components/com_categories/controller.php',
            '/administrator/components/com_categories/controllers/ajax.json.php',
            '/administrator/components/com_categories/controllers/categories.php',
            '/administrator/components/com_categories/controllers/category.php',
            '/administrator/components/com_categories/helpers/association.php',
            '/administrator/components/com_categories/helpers/html/categoriesadministrator.php',
            '/administrator/components/com_categories/models/categories.php',
            '/administrator/components/com_categories/models/category.php',
            '/administrator/components/com_categories/models/fields/categoryedit.php',
            '/administrator/components/com_categories/models/fields/categoryparent.php',
            '/administrator/components/com_categories/models/fields/modal/category.php',
            '/administrator/components/com_categories/models/forms/category.xml',
            '/administrator/components/com_categories/models/forms/filter_categories.xml',
            '/administrator/components/com_categories/tables/category.php',
            '/administrator/components/com_categories/views/categories/tmpl/default.php',
            '/administrator/components/com_categories/views/categories/tmpl/default.xml',
            '/administrator/components/com_categories/views/categories/tmpl/default_batch_body.php',
            '/administrator/components/com_categories/views/categories/tmpl/default_batch_footer.php',
            '/administrator/components/com_categories/views/categories/tmpl/modal.php',
            '/administrator/components/com_categories/views/categories/view.html.php',
            '/administrator/components/com_categories/views/category/tmpl/edit.php',
            '/administrator/components/com_categories/views/category/tmpl/edit.xml',
            '/administrator/components/com_categories/views/category/tmpl/edit_associations.php',
            '/administrator/components/com_categories/views/category/tmpl/edit_metadata.php',
            '/administrator/components/com_categories/views/category/tmpl/modal.php',
            '/administrator/components/com_categories/views/category/tmpl/modal_associations.php',
            '/administrator/components/com_categories/views/category/tmpl/modal_extrafields.php',
            '/administrator/components/com_categories/views/category/tmpl/modal_metadata.php',
            '/administrator/components/com_categories/views/category/tmpl/modal_options.php',
            '/administrator/components/com_categories/views/category/view.html.php',
            '/administrator/components/com_checkin/checkin.php',
            '/administrator/components/com_checkin/controller.php',
            '/administrator/components/com_checkin/models/checkin.php',
            '/administrator/components/com_checkin/models/forms/filter_checkin.xml',
            '/administrator/components/com_checkin/views/checkin/tmpl/default.php',
            '/administrator/components/com_checkin/views/checkin/tmpl/default.xml',
            '/administrator/components/com_checkin/views/checkin/view.html.php',
            '/administrator/components/com_config/config.php',
            '/administrator/components/com_config/controller.php',
            '/administrator/components/com_config/controller/application/cancel.php',
            '/administrator/components/com_config/controller/application/display.php',
            '/administrator/components/com_config/controller/application/removeroot.php',
            '/administrator/components/com_config/controller/application/save.php',
            '/administrator/components/com_config/controller/application/sendtestmail.php',
            '/administrator/components/com_config/controller/application/store.php',
            '/administrator/components/com_config/controller/component/cancel.php',
            '/administrator/components/com_config/controller/component/display.php',
            '/administrator/components/com_config/controller/component/save.php',
            '/administrator/components/com_config/controllers/application.php',
            '/administrator/components/com_config/controllers/component.php',
            '/administrator/components/com_config/helper/config.php',
            '/administrator/components/com_config/model/application.php',
            '/administrator/components/com_config/model/component.php',
            '/administrator/components/com_config/model/field/configcomponents.php',
            '/administrator/components/com_config/model/field/filters.php',
            '/administrator/components/com_config/model/form/application.xml',
            '/administrator/components/com_config/models/application.php',
            '/administrator/components/com_config/models/component.php',
            '/administrator/components/com_config/view/application/html.php',
            '/administrator/components/com_config/view/application/json.php',
            '/administrator/components/com_config/view/application/tmpl/default.php',
            '/administrator/components/com_config/view/application/tmpl/default.xml',
            '/administrator/components/com_config/view/application/tmpl/default_cache.php',
            '/administrator/components/com_config/view/application/tmpl/default_cookie.php',
            '/administrator/components/com_config/view/application/tmpl/default_database.php',
            '/administrator/components/com_config/view/application/tmpl/default_debug.php',
            '/administrator/components/com_config/view/application/tmpl/default_filters.php',
            '/administrator/components/com_config/view/application/tmpl/default_ftp.php',
            '/administrator/components/com_config/view/application/tmpl/default_ftplogin.php',
            '/administrator/components/com_config/view/application/tmpl/default_locale.php',
            '/administrator/components/com_config/view/application/tmpl/default_mail.php',
            '/administrator/components/com_config/view/application/tmpl/default_metadata.php',
            '/administrator/components/com_config/view/application/tmpl/default_navigation.php',
            '/administrator/components/com_config/view/application/tmpl/default_permissions.php',
            '/administrator/components/com_config/view/application/tmpl/default_proxy.php',
            '/administrator/components/com_config/view/application/tmpl/default_seo.php',
            '/administrator/components/com_config/view/application/tmpl/default_server.php',
            '/administrator/components/com_config/view/application/tmpl/default_session.php',
            '/administrator/components/com_config/view/application/tmpl/default_site.php',
            '/administrator/components/com_config/view/application/tmpl/default_system.php',
            '/administrator/components/com_config/view/component/html.php',
            '/administrator/components/com_config/view/component/tmpl/default.php',
            '/administrator/components/com_config/view/component/tmpl/default.xml',
            '/administrator/components/com_config/view/component/tmpl/default_navigation.php',
            '/administrator/components/com_contact/contact.php',
            '/administrator/components/com_contact/controller.php',
            '/administrator/components/com_contact/controllers/ajax.json.php',
            '/administrator/components/com_contact/controllers/contact.php',
            '/administrator/components/com_contact/controllers/contacts.php',
            '/administrator/components/com_contact/helpers/associations.php',
            '/administrator/components/com_contact/helpers/html/contact.php',
            '/administrator/components/com_contact/models/contact.php',
            '/administrator/components/com_contact/models/contacts.php',
            '/administrator/components/com_contact/models/fields/modal/contact.php',
            '/administrator/components/com_contact/models/forms/contact.xml',
            '/administrator/components/com_contact/models/forms/fields/mail.xml',
            '/administrator/components/com_contact/models/forms/filter_contacts.xml',
            '/administrator/components/com_contact/tables/contact.php',
            '/administrator/components/com_contact/views/contact/tmpl/edit.php',
            '/administrator/components/com_contact/views/contact/tmpl/edit_associations.php',
            '/administrator/components/com_contact/views/contact/tmpl/edit_metadata.php',
            '/administrator/components/com_contact/views/contact/tmpl/edit_params.php',
            '/administrator/components/com_contact/views/contact/tmpl/modal.php',
            '/administrator/components/com_contact/views/contact/tmpl/modal_associations.php',
            '/administrator/components/com_contact/views/contact/tmpl/modal_metadata.php',
            '/administrator/components/com_contact/views/contact/tmpl/modal_params.php',
            '/administrator/components/com_contact/views/contact/view.html.php',
            '/administrator/components/com_contact/views/contacts/tmpl/default.php',
            '/administrator/components/com_contact/views/contacts/tmpl/default_batch.php',
            '/administrator/components/com_contact/views/contacts/tmpl/default_batch_body.php',
            '/administrator/components/com_contact/views/contacts/tmpl/default_batch_footer.php',
            '/administrator/components/com_contact/views/contacts/tmpl/modal.php',
            '/administrator/components/com_contact/views/contacts/view.html.php',
            '/administrator/components/com_content/content.php',
            '/administrator/components/com_content/controller.php',
            '/administrator/components/com_content/controllers/ajax.json.php',
            '/administrator/components/com_content/controllers/article.php',
            '/administrator/components/com_content/controllers/articles.php',
            '/administrator/components/com_content/controllers/featured.php',
            '/administrator/components/com_content/helpers/associations.php',
            '/administrator/components/com_content/helpers/html/contentadministrator.php',
            '/administrator/components/com_content/models/article.php',
            '/administrator/components/com_content/models/articles.php',
            '/administrator/components/com_content/models/feature.php',
            '/administrator/components/com_content/models/featured.php',
            '/administrator/components/com_content/models/fields/modal/article.php',
            '/administrator/components/com_content/models/fields/voteradio.php',
            '/administrator/components/com_content/models/forms/article.xml',
            '/administrator/components/com_content/models/forms/filter_articles.xml',
            '/administrator/components/com_content/models/forms/filter_featured.xml',
            '/administrator/components/com_content/tables/featured.php',
            '/administrator/components/com_content/views/article/tmpl/edit.php',
            '/administrator/components/com_content/views/article/tmpl/edit.xml',
            '/administrator/components/com_content/views/article/tmpl/edit_associations.php',
            '/administrator/components/com_content/views/article/tmpl/edit_metadata.php',
            '/administrator/components/com_content/views/article/tmpl/modal.php',
            '/administrator/components/com_content/views/article/tmpl/modal_associations.php',
            '/administrator/components/com_content/views/article/tmpl/modal_metadata.php',
            '/administrator/components/com_content/views/article/tmpl/pagebreak.php',
            '/administrator/components/com_content/views/article/view.html.php',
            '/administrator/components/com_content/views/articles/tmpl/default.php',
            '/administrator/components/com_content/views/articles/tmpl/default.xml',
            '/administrator/components/com_content/views/articles/tmpl/default_batch_body.php',
            '/administrator/components/com_content/views/articles/tmpl/default_batch_footer.php',
            '/administrator/components/com_content/views/articles/tmpl/modal.php',
            '/administrator/components/com_content/views/articles/view.html.php',
            '/administrator/components/com_content/views/featured/tmpl/default.php',
            '/administrator/components/com_content/views/featured/tmpl/default.xml',
            '/administrator/components/com_content/views/featured/view.html.php',
            '/administrator/components/com_contenthistory/contenthistory.php',
            '/administrator/components/com_contenthistory/controller.php',
            '/administrator/components/com_contenthistory/controllers/history.php',
            '/administrator/components/com_contenthistory/controllers/preview.php',
            '/administrator/components/com_contenthistory/helpers/html/textdiff.php',
            '/administrator/components/com_contenthistory/models/compare.php',
            '/administrator/components/com_contenthistory/models/history.php',
            '/administrator/components/com_contenthistory/models/preview.php',
            '/administrator/components/com_contenthistory/views/compare/tmpl/compare.php',
            '/administrator/components/com_contenthistory/views/compare/view.html.php',
            '/administrator/components/com_contenthistory/views/history/tmpl/modal.php',
            '/administrator/components/com_contenthistory/views/history/view.html.php',
            '/administrator/components/com_contenthistory/views/preview/tmpl/preview.php',
            '/administrator/components/com_contenthistory/views/preview/view.html.php',
            '/administrator/components/com_cpanel/controller.php',
            '/administrator/components/com_cpanel/cpanel.php',
            '/administrator/components/com_cpanel/views/cpanel/tmpl/default.php',
            '/administrator/components/com_cpanel/views/cpanel/tmpl/default.xml',
            '/administrator/components/com_cpanel/views/cpanel/view.html.php',
            '/administrator/components/com_fields/controller.php',
            '/administrator/components/com_fields/controllers/field.php',
            '/administrator/components/com_fields/controllers/fields.php',
            '/administrator/components/com_fields/controllers/group.php',
            '/administrator/components/com_fields/controllers/groups.php',
            '/administrator/components/com_fields/fields.php',
            '/administrator/components/com_fields/libraries/fieldslistplugin.php',
            '/administrator/components/com_fields/libraries/fieldsplugin.php',
            '/administrator/components/com_fields/models/field.php',
            '/administrator/components/com_fields/models/fields.php',
            '/administrator/components/com_fields/models/fields/fieldcontexts.php',
            '/administrator/components/com_fields/models/fields/fieldgroups.php',
            '/administrator/components/com_fields/models/fields/fieldlayout.php',
            '/administrator/components/com_fields/models/fields/section.php',
            '/administrator/components/com_fields/models/fields/type.php',
            '/administrator/components/com_fields/models/forms/field.xml',
            '/administrator/components/com_fields/models/forms/filter_fields.xml',
            '/administrator/components/com_fields/models/forms/filter_groups.xml',
            '/administrator/components/com_fields/models/forms/group.xml',
            '/administrator/components/com_fields/models/group.php',
            '/administrator/components/com_fields/models/groups.php',
            '/administrator/components/com_fields/tables/field.php',
            '/administrator/components/com_fields/tables/group.php',
            '/administrator/components/com_fields/views/field/tmpl/edit.php',
            '/administrator/components/com_fields/views/field/view.html.php',
            '/administrator/components/com_fields/views/fields/tmpl/default.php',
            '/administrator/components/com_fields/views/fields/tmpl/default_batch_body.php',
            '/administrator/components/com_fields/views/fields/tmpl/default_batch_footer.php',
            '/administrator/components/com_fields/views/fields/tmpl/modal.php',
            '/administrator/components/com_fields/views/fields/view.html.php',
            '/administrator/components/com_fields/views/group/tmpl/edit.php',
            '/administrator/components/com_fields/views/group/view.html.php',
            '/administrator/components/com_fields/views/groups/tmpl/default.php',
            '/administrator/components/com_fields/views/groups/tmpl/default_batch_body.php',
            '/administrator/components/com_fields/views/groups/tmpl/default_batch_footer.php',
            '/administrator/components/com_fields/views/groups/view.html.php',
            '/administrator/components/com_finder/controller.php',
            '/administrator/components/com_finder/controllers/filter.php',
            '/administrator/components/com_finder/controllers/filters.php',
            '/administrator/components/com_finder/controllers/index.php',
            '/administrator/components/com_finder/controllers/indexer.json.php',
            '/administrator/components/com_finder/controllers/maps.php',
            '/administrator/components/com_finder/finder.php',
            '/administrator/components/com_finder/helpers/finder.php',
            '/administrator/components/com_finder/helpers/html/finder.php',
            '/administrator/components/com_finder/helpers/indexer/driver/mysql.php',
            '/administrator/components/com_finder/helpers/indexer/driver/postgresql.php',
            '/administrator/components/com_finder/helpers/indexer/driver/sqlsrv.php',
            '/administrator/components/com_finder/helpers/indexer/indexer.php',
            '/administrator/components/com_finder/helpers/indexer/parser/html.php',
            '/administrator/components/com_finder/helpers/indexer/parser/rtf.php',
            '/administrator/components/com_finder/helpers/indexer/parser/txt.php',
            '/administrator/components/com_finder/helpers/indexer/stemmer.php',
            '/administrator/components/com_finder/helpers/indexer/stemmer/fr.php',
            '/administrator/components/com_finder/helpers/indexer/stemmer/porter_en.php',
            '/administrator/components/com_finder/helpers/indexer/stemmer/snowball.php',
            '/administrator/components/com_finder/models/fields/branches.php',
            '/administrator/components/com_finder/models/fields/contentmap.php',
            '/administrator/components/com_finder/models/fields/contenttypes.php',
            '/administrator/components/com_finder/models/fields/directories.php',
            '/administrator/components/com_finder/models/fields/searchfilter.php',
            '/administrator/components/com_finder/models/filter.php',
            '/administrator/components/com_finder/models/filters.php',
            '/administrator/components/com_finder/models/forms/filter.xml',
            '/administrator/components/com_finder/models/forms/filter_filters.xml',
            '/administrator/components/com_finder/models/forms/filter_index.xml',
            '/administrator/components/com_finder/models/forms/filter_maps.xml',
            '/administrator/components/com_finder/models/index.php',
            '/administrator/components/com_finder/models/indexer.php',
            '/administrator/components/com_finder/models/maps.php',
            '/administrator/components/com_finder/models/statistics.php',
            '/administrator/components/com_finder/tables/filter.php',
            '/administrator/components/com_finder/tables/link.php',
            '/administrator/components/com_finder/tables/map.php',
            '/administrator/components/com_finder/views/filter/tmpl/edit.php',
            '/administrator/components/com_finder/views/filter/view.html.php',
            '/administrator/components/com_finder/views/filters/tmpl/default.php',
            '/administrator/components/com_finder/views/filters/view.html.php',
            '/administrator/components/com_finder/views/index/tmpl/default.php',
            '/administrator/components/com_finder/views/index/view.html.php',
            '/administrator/components/com_finder/views/indexer/tmpl/default.php',
            '/administrator/components/com_finder/views/indexer/view.html.php',
            '/administrator/components/com_finder/views/maps/tmpl/default.php',
            '/administrator/components/com_finder/views/maps/view.html.php',
            '/administrator/components/com_finder/views/statistics/tmpl/default.php',
            '/administrator/components/com_finder/views/statistics/view.html.php',
            '/administrator/components/com_installer/controller.php',
            '/administrator/components/com_installer/controllers/database.php',
            '/administrator/components/com_installer/controllers/discover.php',
            '/administrator/components/com_installer/controllers/install.php',
            '/administrator/components/com_installer/controllers/manage.php',
            '/administrator/components/com_installer/controllers/update.php',
            '/administrator/components/com_installer/controllers/updatesites.php',
            '/administrator/components/com_installer/helpers/html/manage.php',
            '/administrator/components/com_installer/helpers/html/updatesites.php',
            '/administrator/components/com_installer/installer.php',
            '/administrator/components/com_installer/models/database.php',
            '/administrator/components/com_installer/models/discover.php',
            '/administrator/components/com_installer/models/extension.php',
            '/administrator/components/com_installer/models/fields/extensionstatus.php',
            '/administrator/components/com_installer/models/fields/folder.php',
            '/administrator/components/com_installer/models/fields/location.php',
            '/administrator/components/com_installer/models/fields/type.php',
            '/administrator/components/com_installer/models/forms/filter_discover.xml',
            '/administrator/components/com_installer/models/forms/filter_languages.xml',
            '/administrator/components/com_installer/models/forms/filter_manage.xml',
            '/administrator/components/com_installer/models/forms/filter_update.xml',
            '/administrator/components/com_installer/models/forms/filter_updatesites.xml',
            '/administrator/components/com_installer/models/install.php',
            '/administrator/components/com_installer/models/languages.php',
            '/administrator/components/com_installer/models/manage.php',
            '/administrator/components/com_installer/models/update.php',
            '/administrator/components/com_installer/models/updatesites.php',
            '/administrator/components/com_installer/models/warnings.php',
            '/administrator/components/com_installer/views/database/tmpl/default.php',
            '/administrator/components/com_installer/views/database/tmpl/default.xml',
            '/administrator/components/com_installer/views/database/view.html.php',
            '/administrator/components/com_installer/views/default/tmpl/default_ftp.php',
            '/administrator/components/com_installer/views/default/tmpl/default_message.php',
            '/administrator/components/com_installer/views/default/view.php',
            '/administrator/components/com_installer/views/discover/tmpl/default.php',
            '/administrator/components/com_installer/views/discover/tmpl/default.xml',
            '/administrator/components/com_installer/views/discover/tmpl/default_item.php',
            '/administrator/components/com_installer/views/discover/view.html.php',
            '/administrator/components/com_installer/views/install/tmpl/default.php',
            '/administrator/components/com_installer/views/install/tmpl/default.xml',
            '/administrator/components/com_installer/views/install/view.html.php',
            '/administrator/components/com_installer/views/languages/tmpl/default.php',
            '/administrator/components/com_installer/views/languages/tmpl/default.xml',
            '/administrator/components/com_installer/views/languages/view.html.php',
            '/administrator/components/com_installer/views/manage/tmpl/default.php',
            '/administrator/components/com_installer/views/manage/tmpl/default.xml',
            '/administrator/components/com_installer/views/manage/view.html.php',
            '/administrator/components/com_installer/views/update/tmpl/default.php',
            '/administrator/components/com_installer/views/update/tmpl/default.xml',
            '/administrator/components/com_installer/views/update/view.html.php',
            '/administrator/components/com_installer/views/updatesites/tmpl/default.php',
            '/administrator/components/com_installer/views/updatesites/tmpl/default.xml',
            '/administrator/components/com_installer/views/updatesites/view.html.php',
            '/administrator/components/com_installer/views/warnings/tmpl/default.php',
            '/administrator/components/com_installer/views/warnings/tmpl/default.xml',
            '/administrator/components/com_installer/views/warnings/view.html.php',
            '/administrator/components/com_joomlaupdate/controller.php',
            '/administrator/components/com_joomlaupdate/controllers/update.php',
            '/administrator/components/com_joomlaupdate/helpers/joomlaupdate.php',
            '/administrator/components/com_joomlaupdate/helpers/select.php',
            '/administrator/components/com_joomlaupdate/joomlaupdate.php',
            '/administrator/components/com_joomlaupdate/models/default.php',
            '/administrator/components/com_joomlaupdate/restore.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/complete.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default.xml',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_nodownload.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_noupdate.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_preupdatecheck.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_reinstall.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_update.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_updatemefirst.php',
            '/administrator/components/com_joomlaupdate/views/default/tmpl/default_upload.php',
            '/administrator/components/com_joomlaupdate/views/default/view.html.php',
            '/administrator/components/com_joomlaupdate/views/update/tmpl/default.php',
            '/administrator/components/com_joomlaupdate/views/update/tmpl/finaliseconfirm.php',
            '/administrator/components/com_joomlaupdate/views/update/view.html.php',
            '/administrator/components/com_joomlaupdate/views/upload/tmpl/captive.php',
            '/administrator/components/com_joomlaupdate/views/upload/view.html.php',
            '/administrator/components/com_languages/controller.php',
            '/administrator/components/com_languages/controllers/installed.php',
            '/administrator/components/com_languages/controllers/language.php',
            '/administrator/components/com_languages/controllers/languages.php',
            '/administrator/components/com_languages/controllers/override.php',
            '/administrator/components/com_languages/controllers/overrides.php',
            '/administrator/components/com_languages/controllers/strings.json.php',
            '/administrator/components/com_languages/helpers/html/languages.php',
            '/administrator/components/com_languages/helpers/jsonresponse.php',
            '/administrator/components/com_languages/helpers/languages.php',
            '/administrator/components/com_languages/helpers/multilangstatus.php',
            '/administrator/components/com_languages/languages.php',
            '/administrator/components/com_languages/layouts/joomla/searchtools/default/bar.php',
            '/administrator/components/com_languages/models/fields/languageclient.php',
            '/administrator/components/com_languages/models/forms/filter_installed.xml',
            '/administrator/components/com_languages/models/forms/filter_languages.xml',
            '/administrator/components/com_languages/models/forms/filter_overrides.xml',
            '/administrator/components/com_languages/models/forms/language.xml',
            '/administrator/components/com_languages/models/forms/override.xml',
            '/administrator/components/com_languages/models/installed.php',
            '/administrator/components/com_languages/models/language.php',
            '/administrator/components/com_languages/models/languages.php',
            '/administrator/components/com_languages/models/override.php',
            '/administrator/components/com_languages/models/overrides.php',
            '/administrator/components/com_languages/models/strings.php',
            '/administrator/components/com_languages/views/installed/tmpl/default.php',
            '/administrator/components/com_languages/views/installed/tmpl/default.xml',
            '/administrator/components/com_languages/views/installed/view.html.php',
            '/administrator/components/com_languages/views/language/tmpl/edit.php',
            '/administrator/components/com_languages/views/language/view.html.php',
            '/administrator/components/com_languages/views/languages/tmpl/default.php',
            '/administrator/components/com_languages/views/languages/tmpl/default.xml',
            '/administrator/components/com_languages/views/languages/view.html.php',
            '/administrator/components/com_languages/views/multilangstatus/tmpl/default.php',
            '/administrator/components/com_languages/views/multilangstatus/view.html.php',
            '/administrator/components/com_languages/views/override/tmpl/edit.php',
            '/administrator/components/com_languages/views/override/view.html.php',
            '/administrator/components/com_languages/views/overrides/tmpl/default.php',
            '/administrator/components/com_languages/views/overrides/tmpl/default.xml',
            '/administrator/components/com_languages/views/overrides/view.html.php',
            '/administrator/components/com_login/controller.php',
            '/administrator/components/com_login/login.php',
            '/administrator/components/com_login/models/login.php',
            '/administrator/components/com_login/views/login/tmpl/default.php',
            '/administrator/components/com_login/views/login/view.html.php',
            '/administrator/components/com_media/controller.php',
            '/administrator/components/com_media/controllers/file.json.php',
            '/administrator/components/com_media/controllers/file.php',
            '/administrator/components/com_media/controllers/folder.php',
            '/administrator/components/com_media/layouts/toolbar/deletemedia.php',
            '/administrator/components/com_media/layouts/toolbar/newfolder.php',
            '/administrator/components/com_media/layouts/toolbar/uploadmedia.php',
            '/administrator/components/com_media/media.php',
            '/administrator/components/com_media/models/list.php',
            '/administrator/components/com_media/models/manager.php',
            '/administrator/components/com_media/views/images/tmpl/default.php',
            '/administrator/components/com_media/views/images/view.html.php',
            '/administrator/components/com_media/views/imageslist/tmpl/default.php',
            '/administrator/components/com_media/views/imageslist/tmpl/default_folder.php',
            '/administrator/components/com_media/views/imageslist/tmpl/default_image.php',
            '/administrator/components/com_media/views/imageslist/view.html.php',
            '/administrator/components/com_media/views/media/tmpl/default.php',
            '/administrator/components/com_media/views/media/tmpl/default.xml',
            '/administrator/components/com_media/views/media/tmpl/default_folders.php',
            '/administrator/components/com_media/views/media/tmpl/default_navigation.php',
            '/administrator/components/com_media/views/media/view.html.php',
            '/administrator/components/com_media/views/medialist/tmpl/default.php',
            '/administrator/components/com_media/views/medialist/tmpl/details.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_doc.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_docs.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_folder.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_folders.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_img.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_imgs.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_up.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_video.php',
            '/administrator/components/com_media/views/medialist/tmpl/details_videos.php',
            '/administrator/components/com_media/views/medialist/tmpl/thumbs.php',
            '/administrator/components/com_media/views/medialist/tmpl/thumbs_docs.php',
            '/administrator/components/com_media/views/medialist/tmpl/thumbs_folders.php',
            '/administrator/components/com_media/views/medialist/tmpl/thumbs_imgs.php',
            '/administrator/components/com_media/views/medialist/tmpl/thumbs_up.php',
            '/administrator/components/com_media/views/medialist/tmpl/thumbs_videos.php',
            '/administrator/components/com_media/views/medialist/view.html.php',
            '/administrator/components/com_menus/controller.php',
            '/administrator/components/com_menus/controllers/ajax.json.php',
            '/administrator/components/com_menus/controllers/item.php',
            '/administrator/components/com_menus/controllers/items.php',
            '/administrator/components/com_menus/controllers/menu.php',
            '/administrator/components/com_menus/controllers/menus.php',
            '/administrator/components/com_menus/helpers/associations.php',
            '/administrator/components/com_menus/helpers/html/menus.php',
            '/administrator/components/com_menus/layouts/joomla/searchtools/default/bar.php',
            '/administrator/components/com_menus/menus.php',
            '/administrator/components/com_menus/models/fields/componentscategory.php',
            '/administrator/components/com_menus/models/fields/menuitembytype.php',
            '/administrator/components/com_menus/models/fields/menuordering.php',
            '/administrator/components/com_menus/models/fields/menuparent.php',
            '/administrator/components/com_menus/models/fields/menupreset.php',
            '/administrator/components/com_menus/models/fields/menutype.php',
            '/administrator/components/com_menus/models/fields/modal/menu.php',
            '/administrator/components/com_menus/models/forms/filter_items.xml',
            '/administrator/components/com_menus/models/forms/filter_itemsadmin.xml',
            '/administrator/components/com_menus/models/forms/filter_menus.xml',
            '/administrator/components/com_menus/models/forms/item.xml',
            '/administrator/components/com_menus/models/forms/item_alias.xml',
            '/administrator/components/com_menus/models/forms/item_component.xml',
            '/administrator/components/com_menus/models/forms/item_heading.xml',
            '/administrator/components/com_menus/models/forms/item_separator.xml',
            '/administrator/components/com_menus/models/forms/item_url.xml',
            '/administrator/components/com_menus/models/forms/itemadmin.xml',
            '/administrator/components/com_menus/models/forms/itemadmin_alias.xml',
            '/administrator/components/com_menus/models/forms/itemadmin_component.xml',
            '/administrator/components/com_menus/models/forms/itemadmin_container.xml',
            '/administrator/components/com_menus/models/forms/itemadmin_heading.xml',
            '/administrator/components/com_menus/models/forms/itemadmin_separator.xml',
            '/administrator/components/com_menus/models/forms/itemadmin_url.xml',
            '/administrator/components/com_menus/models/forms/menu.xml',
            '/administrator/components/com_menus/models/item.php',
            '/administrator/components/com_menus/models/items.php',
            '/administrator/components/com_menus/models/menu.php',
            '/administrator/components/com_menus/models/menus.php',
            '/administrator/components/com_menus/models/menutypes.php',
            '/administrator/components/com_menus/presets/joomla.xml',
            '/administrator/components/com_menus/presets/modern.xml',
            '/administrator/components/com_menus/tables/menu.php',
            '/administrator/components/com_menus/views/item/tmpl/edit.php',
            '/administrator/components/com_menus/views/item/tmpl/edit.xml',
            '/administrator/components/com_menus/views/item/tmpl/edit_associations.php',
            '/administrator/components/com_menus/views/item/tmpl/edit_container.php',
            '/administrator/components/com_menus/views/item/tmpl/edit_modules.php',
            '/administrator/components/com_menus/views/item/tmpl/edit_options.php',
            '/administrator/components/com_menus/views/item/tmpl/modal.php',
            '/administrator/components/com_menus/views/item/tmpl/modal_associations.php',
            '/administrator/components/com_menus/views/item/tmpl/modal_options.php',
            '/administrator/components/com_menus/views/item/view.html.php',
            '/administrator/components/com_menus/views/items/tmpl/default.php',
            '/administrator/components/com_menus/views/items/tmpl/default.xml',
            '/administrator/components/com_menus/views/items/tmpl/default_batch_body.php',
            '/administrator/components/com_menus/views/items/tmpl/default_batch_footer.php',
            '/administrator/components/com_menus/views/items/tmpl/modal.php',
            '/administrator/components/com_menus/views/items/view.html.php',
            '/administrator/components/com_menus/views/menu/tmpl/edit.php',
            '/administrator/components/com_menus/views/menu/tmpl/edit.xml',
            '/administrator/components/com_menus/views/menu/view.html.php',
            '/administrator/components/com_menus/views/menu/view.xml.php',
            '/administrator/components/com_menus/views/menus/tmpl/default.php',
            '/administrator/components/com_menus/views/menus/tmpl/default.xml',
            '/administrator/components/com_menus/views/menus/view.html.php',
            '/administrator/components/com_menus/views/menutypes/tmpl/default.php',
            '/administrator/components/com_menus/views/menutypes/view.html.php',
            '/administrator/components/com_messages/controller.php',
            '/administrator/components/com_messages/controllers/config.php',
            '/administrator/components/com_messages/controllers/message.php',
            '/administrator/components/com_messages/controllers/messages.php',
            '/administrator/components/com_messages/helpers/html/messages.php',
            '/administrator/components/com_messages/helpers/messages.php',
            '/administrator/components/com_messages/messages.php',
            '/administrator/components/com_messages/models/config.php',
            '/administrator/components/com_messages/models/fields/messagestates.php',
            '/administrator/components/com_messages/models/fields/usermessages.php',
            '/administrator/components/com_messages/models/forms/config.xml',
            '/administrator/components/com_messages/models/forms/filter_messages.xml',
            '/administrator/components/com_messages/models/forms/message.xml',
            '/administrator/components/com_messages/models/message.php',
            '/administrator/components/com_messages/models/messages.php',
            '/administrator/components/com_messages/tables/message.php',
            '/administrator/components/com_messages/views/config/tmpl/default.php',
            '/administrator/components/com_messages/views/config/view.html.php',
            '/administrator/components/com_messages/views/message/tmpl/default.php',
            '/administrator/components/com_messages/views/message/tmpl/edit.php',
            '/administrator/components/com_messages/views/message/view.html.php',
            '/administrator/components/com_messages/views/messages/tmpl/default.php',
            '/administrator/components/com_messages/views/messages/view.html.php',
            '/administrator/components/com_modules/controller.php',
            '/administrator/components/com_modules/controllers/module.php',
            '/administrator/components/com_modules/controllers/modules.php',
            '/administrator/components/com_modules/helpers/html/modules.php',
            '/administrator/components/com_modules/helpers/xml.php',
            '/administrator/components/com_modules/layouts/toolbar/newmodule.php',
            '/administrator/components/com_modules/models/fields/modulesmodule.php',
            '/administrator/components/com_modules/models/fields/modulesposition.php',
            '/administrator/components/com_modules/models/forms/advanced.xml',
            '/administrator/components/com_modules/models/forms/filter_modules.xml',
            '/administrator/components/com_modules/models/forms/filter_modulesadmin.xml',
            '/administrator/components/com_modules/models/forms/module.xml',
            '/administrator/components/com_modules/models/forms/moduleadmin.xml',
            '/administrator/components/com_modules/models/module.php',
            '/administrator/components/com_modules/models/modules.php',
            '/administrator/components/com_modules/models/positions.php',
            '/administrator/components/com_modules/models/select.php',
            '/administrator/components/com_modules/modules.php',
            '/administrator/components/com_modules/views/module/tmpl/edit.php',
            '/administrator/components/com_modules/views/module/tmpl/edit_assignment.php',
            '/administrator/components/com_modules/views/module/tmpl/edit_options.php',
            '/administrator/components/com_modules/views/module/tmpl/edit_positions.php',
            '/administrator/components/com_modules/views/module/tmpl/modal.php',
            '/administrator/components/com_modules/views/module/view.html.php',
            '/administrator/components/com_modules/views/module/view.json.php',
            '/administrator/components/com_modules/views/modules/tmpl/default.php',
            '/administrator/components/com_modules/views/modules/tmpl/default.xml',
            '/administrator/components/com_modules/views/modules/tmpl/default_batch_body.php',
            '/administrator/components/com_modules/views/modules/tmpl/default_batch_footer.php',
            '/administrator/components/com_modules/views/modules/tmpl/modal.php',
            '/administrator/components/com_modules/views/modules/view.html.php',
            '/administrator/components/com_modules/views/positions/tmpl/modal.php',
            '/administrator/components/com_modules/views/positions/view.html.php',
            '/administrator/components/com_modules/views/preview/tmpl/default.php',
            '/administrator/components/com_modules/views/preview/view.html.php',
            '/administrator/components/com_modules/views/select/tmpl/default.php',
            '/administrator/components/com_modules/views/select/view.html.php',
            '/administrator/components/com_newsfeeds/controller.php',
            '/administrator/components/com_newsfeeds/controllers/ajax.json.php',
            '/administrator/components/com_newsfeeds/controllers/newsfeed.php',
            '/administrator/components/com_newsfeeds/controllers/newsfeeds.php',
            '/administrator/components/com_newsfeeds/helpers/associations.php',
            '/administrator/components/com_newsfeeds/helpers/html/newsfeed.php',
            '/administrator/components/com_newsfeeds/models/fields/modal/newsfeed.php',
            '/administrator/components/com_newsfeeds/models/fields/newsfeeds.php',
            '/administrator/components/com_newsfeeds/models/forms/filter_newsfeeds.xml',
            '/administrator/components/com_newsfeeds/models/forms/newsfeed.xml',
            '/administrator/components/com_newsfeeds/models/newsfeed.php',
            '/administrator/components/com_newsfeeds/models/newsfeeds.php',
            '/administrator/components/com_newsfeeds/newsfeeds.php',
            '/administrator/components/com_newsfeeds/tables/newsfeed.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/edit.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/edit_associations.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/edit_display.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/edit_metadata.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/edit_params.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/modal.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/modal_associations.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/modal_display.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/modal_metadata.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl/modal_params.php',
            '/administrator/components/com_newsfeeds/views/newsfeed/view.html.php',
            '/administrator/components/com_newsfeeds/views/newsfeeds/tmpl/default.php',
            '/administrator/components/com_newsfeeds/views/newsfeeds/tmpl/default_batch_body.php',
            '/administrator/components/com_newsfeeds/views/newsfeeds/tmpl/default_batch_footer.php',
            '/administrator/components/com_newsfeeds/views/newsfeeds/tmpl/modal.php',
            '/administrator/components/com_newsfeeds/views/newsfeeds/view.html.php',
            '/administrator/components/com_plugins/controller.php',
            '/administrator/components/com_plugins/controllers/plugin.php',
            '/administrator/components/com_plugins/controllers/plugins.php',
            '/administrator/components/com_plugins/models/fields/pluginelement.php',
            '/administrator/components/com_plugins/models/fields/pluginordering.php',
            '/administrator/components/com_plugins/models/fields/plugintype.php',
            '/administrator/components/com_plugins/models/forms/filter_plugins.xml',
            '/administrator/components/com_plugins/models/forms/plugin.xml',
            '/administrator/components/com_plugins/models/plugin.php',
            '/administrator/components/com_plugins/models/plugins.php',
            '/administrator/components/com_plugins/plugins.php',
            '/administrator/components/com_plugins/views/plugin/tmpl/edit.php',
            '/administrator/components/com_plugins/views/plugin/tmpl/edit_options.php',
            '/administrator/components/com_plugins/views/plugin/tmpl/modal.php',
            '/administrator/components/com_plugins/views/plugin/view.html.php',
            '/administrator/components/com_plugins/views/plugins/tmpl/default.php',
            '/administrator/components/com_plugins/views/plugins/tmpl/default.xml',
            '/administrator/components/com_plugins/views/plugins/view.html.php',
            '/administrator/components/com_postinstall/controllers/message.php',
            '/administrator/components/com_postinstall/fof.xml',
            '/administrator/components/com_postinstall/models/messages.php',
            '/administrator/components/com_postinstall/postinstall.php',
            '/administrator/components/com_postinstall/toolbar.php',
            '/administrator/components/com_postinstall/views/messages/tmpl/default.php',
            '/administrator/components/com_postinstall/views/messages/tmpl/default.xml',
            '/administrator/components/com_postinstall/views/messages/view.html.php',
            '/administrator/components/com_privacy/controller.php',
            '/administrator/components/com_privacy/controllers/consents.php',
            '/administrator/components/com_privacy/controllers/request.php',
            '/administrator/components/com_privacy/controllers/request.xml.php',
            '/administrator/components/com_privacy/controllers/requests.php',
            '/administrator/components/com_privacy/helpers/export/domain.php',
            '/administrator/components/com_privacy/helpers/export/field.php',
            '/administrator/components/com_privacy/helpers/export/item.php',
            '/administrator/components/com_privacy/helpers/html/helper.php',
            '/administrator/components/com_privacy/helpers/plugin.php',
            '/administrator/components/com_privacy/helpers/privacy.php',
            '/administrator/components/com_privacy/helpers/removal/status.php',
            '/administrator/components/com_privacy/models/capabilities.php',
            '/administrator/components/com_privacy/models/consents.php',
            '/administrator/components/com_privacy/models/dashboard.php',
            '/administrator/components/com_privacy/models/export.php',
            '/administrator/components/com_privacy/models/fields/requeststatus.php',
            '/administrator/components/com_privacy/models/fields/requesttype.php',
            '/administrator/components/com_privacy/models/forms/filter_consents.xml',
            '/administrator/components/com_privacy/models/forms/filter_requests.xml',
            '/administrator/components/com_privacy/models/forms/request.xml',
            '/administrator/components/com_privacy/models/remove.php',
            '/administrator/components/com_privacy/models/request.php',
            '/administrator/components/com_privacy/models/requests.php',
            '/administrator/components/com_privacy/privacy.php',
            '/administrator/components/com_privacy/tables/consent.php',
            '/administrator/components/com_privacy/tables/request.php',
            '/administrator/components/com_privacy/views/capabilities/tmpl/default.php',
            '/administrator/components/com_privacy/views/capabilities/view.html.php',
            '/administrator/components/com_privacy/views/consents/tmpl/default.php',
            '/administrator/components/com_privacy/views/consents/tmpl/default.xml',
            '/administrator/components/com_privacy/views/consents/view.html.php',
            '/administrator/components/com_privacy/views/dashboard/tmpl/default.php',
            '/administrator/components/com_privacy/views/dashboard/tmpl/default.xml',
            '/administrator/components/com_privacy/views/dashboard/view.html.php',
            '/administrator/components/com_privacy/views/export/view.xml.php',
            '/administrator/components/com_privacy/views/request/tmpl/default.php',
            '/administrator/components/com_privacy/views/request/tmpl/edit.php',
            '/administrator/components/com_privacy/views/request/view.html.php',
            '/administrator/components/com_privacy/views/requests/tmpl/default.php',
            '/administrator/components/com_privacy/views/requests/tmpl/default.xml',
            '/administrator/components/com_privacy/views/requests/view.html.php',
            '/administrator/components/com_redirect/controller.php',
            '/administrator/components/com_redirect/controllers/link.php',
            '/administrator/components/com_redirect/controllers/links.php',
            '/administrator/components/com_redirect/helpers/html/redirect.php',
            '/administrator/components/com_redirect/models/fields/redirect.php',
            '/administrator/components/com_redirect/models/forms/filter_links.xml',
            '/administrator/components/com_redirect/models/forms/link.xml',
            '/administrator/components/com_redirect/models/link.php',
            '/administrator/components/com_redirect/models/links.php',
            '/administrator/components/com_redirect/redirect.php',
            '/administrator/components/com_redirect/tables/link.php',
            '/administrator/components/com_redirect/views/link/tmpl/edit.php',
            '/administrator/components/com_redirect/views/link/view.html.php',
            '/administrator/components/com_redirect/views/links/tmpl/default.php',
            '/administrator/components/com_redirect/views/links/tmpl/default.xml',
            '/administrator/components/com_redirect/views/links/tmpl/default_addform.php',
            '/administrator/components/com_redirect/views/links/tmpl/default_batch_body.php',
            '/administrator/components/com_redirect/views/links/tmpl/default_batch_footer.php',
            '/administrator/components/com_redirect/views/links/view.html.php',
            '/administrator/components/com_tags/controller.php',
            '/administrator/components/com_tags/controllers/tag.php',
            '/administrator/components/com_tags/controllers/tags.php',
            '/administrator/components/com_tags/helpers/tags.php',
            '/administrator/components/com_tags/models/forms/filter_tags.xml',
            '/administrator/components/com_tags/models/forms/tag.xml',
            '/administrator/components/com_tags/models/tag.php',
            '/administrator/components/com_tags/models/tags.php',
            '/administrator/components/com_tags/tables/tag.php',
            '/administrator/components/com_tags/tags.php',
            '/administrator/components/com_tags/views/tag/tmpl/edit.php',
            '/administrator/components/com_tags/views/tag/tmpl/edit_metadata.php',
            '/administrator/components/com_tags/views/tag/tmpl/edit_options.php',
            '/administrator/components/com_tags/views/tag/view.html.php',
            '/administrator/components/com_tags/views/tags/tmpl/default.php',
            '/administrator/components/com_tags/views/tags/tmpl/default.xml',
            '/administrator/components/com_tags/views/tags/tmpl/default_batch_body.php',
            '/administrator/components/com_tags/views/tags/tmpl/default_batch_footer.php',
            '/administrator/components/com_tags/views/tags/view.html.php',
            '/administrator/components/com_templates/controller.php',
            '/administrator/components/com_templates/controllers/style.php',
            '/administrator/components/com_templates/controllers/styles.php',
            '/administrator/components/com_templates/controllers/template.php',
            '/administrator/components/com_templates/helpers/html/templates.php',
            '/administrator/components/com_templates/models/fields/templatelocation.php',
            '/administrator/components/com_templates/models/fields/templatename.php',
            '/administrator/components/com_templates/models/forms/filter_styles.xml',
            '/administrator/components/com_templates/models/forms/filter_templates.xml',
            '/administrator/components/com_templates/models/forms/source.xml',
            '/administrator/components/com_templates/models/forms/style.xml',
            '/administrator/components/com_templates/models/forms/style_administrator.xml',
            '/administrator/components/com_templates/models/forms/style_site.xml',
            '/administrator/components/com_templates/models/style.php',
            '/administrator/components/com_templates/models/styles.php',
            '/administrator/components/com_templates/models/template.php',
            '/administrator/components/com_templates/models/templates.php',
            '/administrator/components/com_templates/tables/style.php',
            '/administrator/components/com_templates/templates.php',
            '/administrator/components/com_templates/views/style/tmpl/edit.php',
            '/administrator/components/com_templates/views/style/tmpl/edit_assignment.php',
            '/administrator/components/com_templates/views/style/tmpl/edit_options.php',
            '/administrator/components/com_templates/views/style/view.html.php',
            '/administrator/components/com_templates/views/style/view.json.php',
            '/administrator/components/com_templates/views/styles/tmpl/default.php',
            '/administrator/components/com_templates/views/styles/tmpl/default.xml',
            '/administrator/components/com_templates/views/styles/view.html.php',
            '/administrator/components/com_templates/views/template/tmpl/default.php',
            '/administrator/components/com_templates/views/template/tmpl/default_description.php',
            '/administrator/components/com_templates/views/template/tmpl/default_folders.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_copy_body.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_copy_footer.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_delete_body.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_delete_footer.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_file_body.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_file_footer.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_folder_body.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_folder_footer.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_rename_body.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_rename_footer.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_resize_body.php',
            '/administrator/components/com_templates/views/template/tmpl/default_modal_resize_footer.php',
            '/administrator/components/com_templates/views/template/tmpl/default_tree.php',
            '/administrator/components/com_templates/views/template/tmpl/readonly.php',
            '/administrator/components/com_templates/views/template/view.html.php',
            '/administrator/components/com_templates/views/templates/tmpl/default.php',
            '/administrator/components/com_templates/views/templates/tmpl/default.xml',
            '/administrator/components/com_templates/views/templates/view.html.php',
            '/administrator/components/com_users/controller.php',
            '/administrator/components/com_users/controllers/group.php',
            '/administrator/components/com_users/controllers/groups.php',
            '/administrator/components/com_users/controllers/level.php',
            '/administrator/components/com_users/controllers/levels.php',
            '/administrator/components/com_users/controllers/mail.php',
            '/administrator/components/com_users/controllers/note.php',
            '/administrator/components/com_users/controllers/notes.php',
            '/administrator/components/com_users/controllers/user.php',
            '/administrator/components/com_users/controllers/users.php',
            '/administrator/components/com_users/helpers/html/users.php',
            '/administrator/components/com_users/models/debuggroup.php',
            '/administrator/components/com_users/models/debuguser.php',
            '/administrator/components/com_users/models/fields/groupparent.php',
            '/administrator/components/com_users/models/fields/levels.php',
            '/administrator/components/com_users/models/forms/config_domain.xml',
            '/administrator/components/com_users/models/forms/fields/user.xml',
            '/administrator/components/com_users/models/forms/filter_debuggroup.xml',
            '/administrator/components/com_users/models/forms/filter_debuguser.xml',
            '/administrator/components/com_users/models/forms/filter_groups.xml',
            '/administrator/components/com_users/models/forms/filter_levels.xml',
            '/administrator/components/com_users/models/forms/filter_notes.xml',
            '/administrator/components/com_users/models/forms/filter_users.xml',
            '/administrator/components/com_users/models/forms/group.xml',
            '/administrator/components/com_users/models/forms/level.xml',
            '/administrator/components/com_users/models/forms/mail.xml',
            '/administrator/components/com_users/models/forms/note.xml',
            '/administrator/components/com_users/models/forms/user.xml',
            '/administrator/components/com_users/models/group.php',
            '/administrator/components/com_users/models/groups.php',
            '/administrator/components/com_users/models/level.php',
            '/administrator/components/com_users/models/levels.php',
            '/administrator/components/com_users/models/mail.php',
            '/administrator/components/com_users/models/note.php',
            '/administrator/components/com_users/models/notes.php',
            '/administrator/components/com_users/models/user.php',
            '/administrator/components/com_users/models/users.php',
            '/administrator/components/com_users/tables/note.php',
            '/administrator/components/com_users/users.php',
            '/administrator/components/com_users/views/debuggroup/tmpl/default.php',
            '/administrator/components/com_users/views/debuggroup/view.html.php',
            '/administrator/components/com_users/views/debuguser/tmpl/default.php',
            '/administrator/components/com_users/views/debuguser/view.html.php',
            '/administrator/components/com_users/views/group/tmpl/edit.php',
            '/administrator/components/com_users/views/group/tmpl/edit.xml',
            '/administrator/components/com_users/views/group/view.html.php',
            '/administrator/components/com_users/views/groups/tmpl/default.php',
            '/administrator/components/com_users/views/groups/tmpl/default.xml',
            '/administrator/components/com_users/views/groups/view.html.php',
            '/administrator/components/com_users/views/level/tmpl/edit.php',
            '/administrator/components/com_users/views/level/tmpl/edit.xml',
            '/administrator/components/com_users/views/level/view.html.php',
            '/administrator/components/com_users/views/levels/tmpl/default.php',
            '/administrator/components/com_users/views/levels/tmpl/default.xml',
            '/administrator/components/com_users/views/levels/view.html.php',
            '/administrator/components/com_users/views/mail/tmpl/default.php',
            '/administrator/components/com_users/views/mail/tmpl/default.xml',
            '/administrator/components/com_users/views/mail/view.html.php',
            '/administrator/components/com_users/views/note/tmpl/edit.php',
            '/administrator/components/com_users/views/note/tmpl/edit.xml',
            '/administrator/components/com_users/views/note/view.html.php',
            '/administrator/components/com_users/views/notes/tmpl/default.php',
            '/administrator/components/com_users/views/notes/tmpl/default.xml',
            '/administrator/components/com_users/views/notes/tmpl/modal.php',
            '/administrator/components/com_users/views/notes/view.html.php',
            '/administrator/components/com_users/views/user/tmpl/edit.php',
            '/administrator/components/com_users/views/user/tmpl/edit.xml',
            '/administrator/components/com_users/views/user/tmpl/edit_groups.php',
            '/administrator/components/com_users/views/user/view.html.php',
            '/administrator/components/com_users/views/users/tmpl/default.php',
            '/administrator/components/com_users/views/users/tmpl/default.xml',
            '/administrator/components/com_users/views/users/tmpl/default_batch_body.php',
            '/administrator/components/com_users/views/users/tmpl/default_batch_footer.php',
            '/administrator/components/com_users/views/users/tmpl/modal.php',
            '/administrator/components/com_users/views/users/view.html.php',
            '/administrator/help/helpsites.xml',
            '/administrator/includes/helper.php',
            '/administrator/includes/subtoolbar.php',
            '/administrator/language/en-GB/en-GB.com_actionlogs.ini',
            '/administrator/language/en-GB/en-GB.com_actionlogs.sys.ini',
            '/administrator/language/en-GB/en-GB.com_admin.ini',
            '/administrator/language/en-GB/en-GB.com_admin.sys.ini',
            '/administrator/language/en-GB/en-GB.com_ajax.ini',
            '/administrator/language/en-GB/en-GB.com_ajax.sys.ini',
            '/administrator/language/en-GB/en-GB.com_associations.ini',
            '/administrator/language/en-GB/en-GB.com_associations.sys.ini',
            '/administrator/language/en-GB/en-GB.com_banners.ini',
            '/administrator/language/en-GB/en-GB.com_banners.sys.ini',
            '/administrator/language/en-GB/en-GB.com_cache.ini',
            '/administrator/language/en-GB/en-GB.com_cache.sys.ini',
            '/administrator/language/en-GB/en-GB.com_categories.ini',
            '/administrator/language/en-GB/en-GB.com_categories.sys.ini',
            '/administrator/language/en-GB/en-GB.com_checkin.ini',
            '/administrator/language/en-GB/en-GB.com_checkin.sys.ini',
            '/administrator/language/en-GB/en-GB.com_config.ini',
            '/administrator/language/en-GB/en-GB.com_config.sys.ini',
            '/administrator/language/en-GB/en-GB.com_contact.ini',
            '/administrator/language/en-GB/en-GB.com_contact.sys.ini',
            '/administrator/language/en-GB/en-GB.com_content.ini',
            '/administrator/language/en-GB/en-GB.com_content.sys.ini',
            '/administrator/language/en-GB/en-GB.com_contenthistory.ini',
            '/administrator/language/en-GB/en-GB.com_contenthistory.sys.ini',
            '/administrator/language/en-GB/en-GB.com_cpanel.ini',
            '/administrator/language/en-GB/en-GB.com_cpanel.sys.ini',
            '/administrator/language/en-GB/en-GB.com_fields.ini',
            '/administrator/language/en-GB/en-GB.com_fields.sys.ini',
            '/administrator/language/en-GB/en-GB.com_finder.ini',
            '/administrator/language/en-GB/en-GB.com_finder.sys.ini',
            '/administrator/language/en-GB/en-GB.com_installer.ini',
            '/administrator/language/en-GB/en-GB.com_installer.sys.ini',
            '/administrator/language/en-GB/en-GB.com_joomlaupdate.ini',
            '/administrator/language/en-GB/en-GB.com_joomlaupdate.sys.ini',
            '/administrator/language/en-GB/en-GB.com_languages.ini',
            '/administrator/language/en-GB/en-GB.com_languages.sys.ini',
            '/administrator/language/en-GB/en-GB.com_login.ini',
            '/administrator/language/en-GB/en-GB.com_login.sys.ini',
            '/administrator/language/en-GB/en-GB.com_mailto.sys.ini',
            '/administrator/language/en-GB/en-GB.com_media.ini',
            '/administrator/language/en-GB/en-GB.com_media.sys.ini',
            '/administrator/language/en-GB/en-GB.com_menus.ini',
            '/administrator/language/en-GB/en-GB.com_menus.sys.ini',
            '/administrator/language/en-GB/en-GB.com_messages.ini',
            '/administrator/language/en-GB/en-GB.com_messages.sys.ini',
            '/administrator/language/en-GB/en-GB.com_modules.ini',
            '/administrator/language/en-GB/en-GB.com_modules.sys.ini',
            '/administrator/language/en-GB/en-GB.com_newsfeeds.ini',
            '/administrator/language/en-GB/en-GB.com_newsfeeds.sys.ini',
            '/administrator/language/en-GB/en-GB.com_plugins.ini',
            '/administrator/language/en-GB/en-GB.com_plugins.sys.ini',
            '/administrator/language/en-GB/en-GB.com_postinstall.ini',
            '/administrator/language/en-GB/en-GB.com_postinstall.sys.ini',
            '/administrator/language/en-GB/en-GB.com_privacy.ini',
            '/administrator/language/en-GB/en-GB.com_privacy.sys.ini',
            '/administrator/language/en-GB/en-GB.com_redirect.ini',
            '/administrator/language/en-GB/en-GB.com_redirect.sys.ini',
            '/administrator/language/en-GB/en-GB.com_tags.ini',
            '/administrator/language/en-GB/en-GB.com_tags.sys.ini',
            '/administrator/language/en-GB/en-GB.com_templates.ini',
            '/administrator/language/en-GB/en-GB.com_templates.sys.ini',
            '/administrator/language/en-GB/en-GB.com_users.ini',
            '/administrator/language/en-GB/en-GB.com_users.sys.ini',
            '/administrator/language/en-GB/en-GB.com_weblinks.ini',
            '/administrator/language/en-GB/en-GB.com_weblinks.sys.ini',
            '/administrator/language/en-GB/en-GB.com_wrapper.ini',
            '/administrator/language/en-GB/en-GB.com_wrapper.sys.ini',
            '/administrator/language/en-GB/en-GB.ini',
            '/administrator/language/en-GB/en-GB.lib_joomla.ini',
            '/administrator/language/en-GB/en-GB.localise.php',
            '/administrator/language/en-GB/en-GB.mod_custom.ini',
            '/administrator/language/en-GB/en-GB.mod_custom.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_feed.ini',
            '/administrator/language/en-GB/en-GB.mod_feed.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_latest.ini',
            '/administrator/language/en-GB/en-GB.mod_latest.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_latestactions.ini',
            '/administrator/language/en-GB/en-GB.mod_latestactions.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_logged.ini',
            '/administrator/language/en-GB/en-GB.mod_logged.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_login.ini',
            '/administrator/language/en-GB/en-GB.mod_login.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_menu.ini',
            '/administrator/language/en-GB/en-GB.mod_menu.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_multilangstatus.ini',
            '/administrator/language/en-GB/en-GB.mod_multilangstatus.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_popular.ini',
            '/administrator/language/en-GB/en-GB.mod_popular.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_privacy_dashboard.ini',
            '/administrator/language/en-GB/en-GB.mod_privacy_dashboard.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_quickicon.ini',
            '/administrator/language/en-GB/en-GB.mod_quickicon.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_sampledata.ini',
            '/administrator/language/en-GB/en-GB.mod_sampledata.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_stats_admin.ini',
            '/administrator/language/en-GB/en-GB.mod_stats_admin.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_status.ini',
            '/administrator/language/en-GB/en-GB.mod_status.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_submenu.ini',
            '/administrator/language/en-GB/en-GB.mod_submenu.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_title.ini',
            '/administrator/language/en-GB/en-GB.mod_title.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_toolbar.ini',
            '/administrator/language/en-GB/en-GB.mod_toolbar.sys.ini',
            '/administrator/language/en-GB/en-GB.mod_version.ini',
            '/administrator/language/en-GB/en-GB.mod_version.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_actionlog_joomla.ini',
            '/administrator/language/en-GB/en-GB.plg_actionlog_joomla.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_cookie.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_cookie.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_gmail.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_gmail.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_joomla.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_joomla.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_ldap.ini',
            '/administrator/language/en-GB/en-GB.plg_authentication_ldap.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_captcha_recaptcha.ini',
            '/administrator/language/en-GB/en-GB.plg_captcha_recaptcha.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_captcha_recaptcha_invisible.ini',
            '/administrator/language/en-GB/en-GB.plg_captcha_recaptcha_invisible.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_confirmconsent.ini',
            '/administrator/language/en-GB/en-GB.plg_content_confirmconsent.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_contact.ini',
            '/administrator/language/en-GB/en-GB.plg_content_contact.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_emailcloak.ini',
            '/administrator/language/en-GB/en-GB.plg_content_emailcloak.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_fields.ini',
            '/administrator/language/en-GB/en-GB.plg_content_fields.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_finder.ini',
            '/administrator/language/en-GB/en-GB.plg_content_finder.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_joomla.ini',
            '/administrator/language/en-GB/en-GB.plg_content_joomla.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_loadmodule.ini',
            '/administrator/language/en-GB/en-GB.plg_content_loadmodule.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_pagebreak.ini',
            '/administrator/language/en-GB/en-GB.plg_content_pagebreak.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_pagenavigation.ini',
            '/administrator/language/en-GB/en-GB.plg_content_pagenavigation.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_content_vote.ini',
            '/administrator/language/en-GB/en-GB.plg_content_vote.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_article.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_article.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_contact.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_contact.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_fields.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_fields.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_image.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_image.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_menu.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_menu.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_module.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_module.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_pagebreak.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_pagebreak.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_readmore.ini',
            '/administrator/language/en-GB/en-GB.plg_editors-xtd_readmore.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors_codemirror.ini',
            '/administrator/language/en-GB/en-GB.plg_editors_codemirror.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors_none.ini',
            '/administrator/language/en-GB/en-GB.plg_editors_none.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_editors_tinymce.ini',
            '/administrator/language/en-GB/en-GB.plg_editors_tinymce.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_extension_joomla.ini',
            '/administrator/language/en-GB/en-GB.plg_extension_joomla.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_calendar.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_calendar.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_checkboxes.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_checkboxes.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_color.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_color.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_editor.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_editor.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_image.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_image.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_imagelist.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_imagelist.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_integer.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_integer.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_list.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_list.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_media.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_media.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_radio.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_radio.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_sql.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_sql.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_text.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_text.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_textarea.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_textarea.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_url.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_url.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_user.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_user.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_usergrouplist.ini',
            '/administrator/language/en-GB/en-GB.plg_fields_usergrouplist.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_categories.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_categories.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_contacts.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_contacts.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_content.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_content.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_newsfeeds.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_newsfeeds.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_tags.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_tags.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_weblinks.ini',
            '/administrator/language/en-GB/en-GB.plg_finder_weblinks.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_folderinstaller.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_folderinstaller.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_packageinstaller.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_packageinstaller.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_urlinstaller.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_urlinstaller.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_webinstaller.ini',
            '/administrator/language/en-GB/en-GB.plg_installer_webinstaller.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_actionlogs.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_actionlogs.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_consents.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_consents.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_contact.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_contact.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_content.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_content.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_message.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_message.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_user.ini',
            '/administrator/language/en-GB/en-GB.plg_privacy_user.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_extensionupdate.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_extensionupdate.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_joomlaupdate.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_joomlaupdate.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_phpversioncheck.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_phpversioncheck.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_privacycheck.ini',
            '/administrator/language/en-GB/en-GB.plg_quickicon_privacycheck.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_sampledata_blog.ini',
            '/administrator/language/en-GB/en-GB.plg_sampledata_blog.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_actionlogs.ini',
            '/administrator/language/en-GB/en-GB.plg_system_actionlogs.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_cache.ini',
            '/administrator/language/en-GB/en-GB.plg_system_cache.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_debug.ini',
            '/administrator/language/en-GB/en-GB.plg_system_debug.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_fields.ini',
            '/administrator/language/en-GB/en-GB.plg_system_fields.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_highlight.ini',
            '/administrator/language/en-GB/en-GB.plg_system_highlight.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_languagecode.ini',
            '/administrator/language/en-GB/en-GB.plg_system_languagecode.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_languagefilter.ini',
            '/administrator/language/en-GB/en-GB.plg_system_languagefilter.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_log.ini',
            '/administrator/language/en-GB/en-GB.plg_system_log.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_logout.ini',
            '/administrator/language/en-GB/en-GB.plg_system_logout.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_logrotation.ini',
            '/administrator/language/en-GB/en-GB.plg_system_logrotation.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_p3p.ini',
            '/administrator/language/en-GB/en-GB.plg_system_p3p.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_privacyconsent.ini',
            '/administrator/language/en-GB/en-GB.plg_system_privacyconsent.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_redirect.ini',
            '/administrator/language/en-GB/en-GB.plg_system_redirect.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_remember.ini',
            '/administrator/language/en-GB/en-GB.plg_system_remember.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_sef.ini',
            '/administrator/language/en-GB/en-GB.plg_system_sef.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_sessiongc.ini',
            '/administrator/language/en-GB/en-GB.plg_system_sessiongc.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_stats.ini',
            '/administrator/language/en-GB/en-GB.plg_system_stats.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_system_updatenotification.ini',
            '/administrator/language/en-GB/en-GB.plg_system_updatenotification.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_twofactorauth_totp.ini',
            '/administrator/language/en-GB/en-GB.plg_twofactorauth_totp.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_twofactorauth_yubikey.ini',
            '/administrator/language/en-GB/en-GB.plg_twofactorauth_yubikey.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_user_contactcreator.ini',
            '/administrator/language/en-GB/en-GB.plg_user_contactcreator.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_user_joomla.ini',
            '/administrator/language/en-GB/en-GB.plg_user_joomla.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_user_profile.ini',
            '/administrator/language/en-GB/en-GB.plg_user_profile.sys.ini',
            '/administrator/language/en-GB/en-GB.plg_user_terms.ini',
            '/administrator/language/en-GB/en-GB.plg_user_terms.sys.ini',
            '/administrator/language/en-GB/en-GB.tpl_hathor.ini',
            '/administrator/language/en-GB/en-GB.tpl_hathor.sys.ini',
            '/administrator/language/en-GB/en-GB.tpl_isis.ini',
            '/administrator/language/en-GB/en-GB.tpl_isis.sys.ini',
            '/administrator/language/en-GB/en-GB.xml',
            '/administrator/manifests/libraries/fof.xml',
            '/administrator/manifests/libraries/idna_convert.xml',
            '/administrator/manifests/libraries/phputf8.xml',
            '/administrator/modules/mod_feed/helper.php',
            '/administrator/modules/mod_latest/helper.php',
            '/administrator/modules/mod_latestactions/helper.php',
            '/administrator/modules/mod_logged/helper.php',
            '/administrator/modules/mod_login/helper.php',
            '/administrator/modules/mod_menu/helper.php',
            '/administrator/modules/mod_menu/menu.php',
            '/administrator/modules/mod_multilangstatus/language/en-GB/en-GB.mod_multilangstatus.ini',
            '/administrator/modules/mod_multilangstatus/language/en-GB/en-GB.mod_multilangstatus.sys.ini',
            '/administrator/modules/mod_popular/helper.php',
            '/administrator/modules/mod_privacy_dashboard/helper.php',
            '/administrator/modules/mod_quickicon/helper.php',
            '/administrator/modules/mod_quickicon/mod_quickicon.php',
            '/administrator/modules/mod_sampledata/helper.php',
            '/administrator/modules/mod_stats_admin/helper.php',
            '/administrator/modules/mod_stats_admin/language/en-GB.mod_stats_admin.ini',
            '/administrator/modules/mod_stats_admin/language/en-GB.mod_stats_admin.sys.ini',
            '/administrator/modules/mod_status/mod_status.php',
            '/administrator/modules/mod_status/mod_status.xml',
            '/administrator/modules/mod_status/tmpl/default.php',
            '/administrator/modules/mod_version/helper.php',
            '/administrator/modules/mod_version/language/en-GB/en-GB.mod_version.ini',
            '/administrator/modules/mod_version/language/en-GB/en-GB.mod_version.sys.ini',
            '/administrator/templates/hathor/LICENSE.txt',
            '/administrator/templates/hathor/component.php',
            '/administrator/templates/hathor/cpanel.php',
            '/administrator/templates/hathor/css/boldtext.css',
            '/administrator/templates/hathor/css/colour_blue.css',
            '/administrator/templates/hathor/css/colour_blue_rtl.css',
            '/administrator/templates/hathor/css/colour_brown.css',
            '/administrator/templates/hathor/css/colour_brown_rtl.css',
            '/administrator/templates/hathor/css/colour_highcontrast.css',
            '/administrator/templates/hathor/css/colour_highcontrast_rtl.css',
            '/administrator/templates/hathor/css/colour_standard.css',
            '/administrator/templates/hathor/css/colour_standard_rtl.css',
            '/administrator/templates/hathor/css/error.css',
            '/administrator/templates/hathor/css/ie7.css',
            '/administrator/templates/hathor/css/ie8.css',
            '/administrator/templates/hathor/css/template.css',
            '/administrator/templates/hathor/css/template_rtl.css',
            '/administrator/templates/hathor/css/theme.css',
            '/administrator/templates/hathor/error.php',
            '/administrator/templates/hathor/favicon.ico',
            '/administrator/templates/hathor/html/com_admin/help/default.php',
            '/administrator/templates/hathor/html/com_admin/profile/edit.php',
            '/administrator/templates/hathor/html/com_admin/sysinfo/default.php',
            '/administrator/templates/hathor/html/com_admin/sysinfo/default_config.php',
            '/administrator/templates/hathor/html/com_admin/sysinfo/default_directory.php',
            '/administrator/templates/hathor/html/com_admin/sysinfo/default_navigation.php',
            '/administrator/templates/hathor/html/com_admin/sysinfo/default_phpsettings.php',
            '/administrator/templates/hathor/html/com_admin/sysinfo/default_system.php',
            '/administrator/templates/hathor/html/com_associations/associations/default.php',
            '/administrator/templates/hathor/html/com_banners/banner/edit.php',
            '/administrator/templates/hathor/html/com_banners/banners/default.php',
            '/administrator/templates/hathor/html/com_banners/client/edit.php',
            '/administrator/templates/hathor/html/com_banners/clients/default.php',
            '/administrator/templates/hathor/html/com_banners/download/default.php',
            '/administrator/templates/hathor/html/com_banners/tracks/default.php',
            '/administrator/templates/hathor/html/com_cache/cache/default.php',
            '/administrator/templates/hathor/html/com_cache/purge/default.php',
            '/administrator/templates/hathor/html/com_categories/categories/default.php',
            '/administrator/templates/hathor/html/com_categories/category/edit.php',
            '/administrator/templates/hathor/html/com_categories/category/edit_options.php',
            '/administrator/templates/hathor/html/com_checkin/checkin/default.php',
            '/administrator/templates/hathor/html/com_config/application/default.php',
            '/administrator/templates/hathor/html/com_config/application/default_cache.php',
            '/administrator/templates/hathor/html/com_config/application/default_cookie.php',
            '/administrator/templates/hathor/html/com_config/application/default_database.php',
            '/administrator/templates/hathor/html/com_config/application/default_debug.php',
            '/administrator/templates/hathor/html/com_config/application/default_filters.php',
            '/administrator/templates/hathor/html/com_config/application/default_ftp.php',
            '/administrator/templates/hathor/html/com_config/application/default_ftplogin.php',
            '/administrator/templates/hathor/html/com_config/application/default_locale.php',
            '/administrator/templates/hathor/html/com_config/application/default_mail.php',
            '/administrator/templates/hathor/html/com_config/application/default_metadata.php',
            '/administrator/templates/hathor/html/com_config/application/default_navigation.php',
            '/administrator/templates/hathor/html/com_config/application/default_permissions.php',
            '/administrator/templates/hathor/html/com_config/application/default_seo.php',
            '/administrator/templates/hathor/html/com_config/application/default_server.php',
            '/administrator/templates/hathor/html/com_config/application/default_session.php',
            '/administrator/templates/hathor/html/com_config/application/default_site.php',
            '/administrator/templates/hathor/html/com_config/application/default_system.php',
            '/administrator/templates/hathor/html/com_config/component/default.php',
            '/administrator/templates/hathor/html/com_contact/contact/edit.php',
            '/administrator/templates/hathor/html/com_contact/contact/edit_params.php',
            '/administrator/templates/hathor/html/com_contact/contacts/default.php',
            '/administrator/templates/hathor/html/com_contact/contacts/modal.php',
            '/administrator/templates/hathor/html/com_content/article/edit.php',
            '/administrator/templates/hathor/html/com_content/articles/default.php',
            '/administrator/templates/hathor/html/com_content/articles/modal.php',
            '/administrator/templates/hathor/html/com_content/featured/default.php',
            '/administrator/templates/hathor/html/com_contenthistory/history/modal.php',
            '/administrator/templates/hathor/html/com_cpanel/cpanel/default.php',
            '/administrator/templates/hathor/html/com_fields/field/edit.php',
            '/administrator/templates/hathor/html/com_fields/fields/default.php',
            '/administrator/templates/hathor/html/com_fields/group/edit.php',
            '/administrator/templates/hathor/html/com_fields/groups/default.php',
            '/administrator/templates/hathor/html/com_finder/filters/default.php',
            '/administrator/templates/hathor/html/com_finder/index/default.php',
            '/administrator/templates/hathor/html/com_finder/maps/default.php',
            '/administrator/templates/hathor/html/com_installer/database/default.php',
            '/administrator/templates/hathor/html/com_installer/default/default_ftp.php',
            '/administrator/templates/hathor/html/com_installer/discover/default.php',
            '/administrator/templates/hathor/html/com_installer/install/default.php',
            '/administrator/templates/hathor/html/com_installer/install/default_form.php',
            '/administrator/templates/hathor/html/com_installer/languages/default.php',
            '/administrator/templates/hathor/html/com_installer/languages/default_filter.php',
            '/administrator/templates/hathor/html/com_installer/manage/default.php',
            '/administrator/templates/hathor/html/com_installer/manage/default_filter.php',
            '/administrator/templates/hathor/html/com_installer/update/default.php',
            '/administrator/templates/hathor/html/com_installer/warnings/default.php',
            '/administrator/templates/hathor/html/com_joomlaupdate/default/default.php',
            '/administrator/templates/hathor/html/com_languages/installed/default.php',
            '/administrator/templates/hathor/html/com_languages/installed/default_ftp.php',
            '/administrator/templates/hathor/html/com_languages/languages/default.php',
            '/administrator/templates/hathor/html/com_languages/overrides/default.php',
            '/administrator/templates/hathor/html/com_menus/item/edit.php',
            '/administrator/templates/hathor/html/com_menus/item/edit_options.php',
            '/administrator/templates/hathor/html/com_menus/items/default.php',
            '/administrator/templates/hathor/html/com_menus/menu/edit.php',
            '/administrator/templates/hathor/html/com_menus/menus/default.php',
            '/administrator/templates/hathor/html/com_menus/menutypes/default.php',
            '/administrator/templates/hathor/html/com_messages/message/edit.php',
            '/administrator/templates/hathor/html/com_messages/messages/default.php',
            '/administrator/templates/hathor/html/com_modules/module/edit.php',
            '/administrator/templates/hathor/html/com_modules/module/edit_assignment.php',
            '/administrator/templates/hathor/html/com_modules/module/edit_options.php',
            '/administrator/templates/hathor/html/com_modules/modules/default.php',
            '/administrator/templates/hathor/html/com_modules/positions/modal.php',
            '/administrator/templates/hathor/html/com_newsfeeds/newsfeed/edit.php',
            '/administrator/templates/hathor/html/com_newsfeeds/newsfeed/edit_params.php',
            '/administrator/templates/hathor/html/com_newsfeeds/newsfeeds/default.php',
            '/administrator/templates/hathor/html/com_newsfeeds/newsfeeds/modal.php',
            '/administrator/templates/hathor/html/com_plugins/plugin/edit.php',
            '/administrator/templates/hathor/html/com_plugins/plugin/edit_options.php',
            '/administrator/templates/hathor/html/com_plugins/plugins/default.php',
            '/administrator/templates/hathor/html/com_postinstall/messages/default.php',
            '/administrator/templates/hathor/html/com_redirect/links/default.php',
            '/administrator/templates/hathor/html/com_search/searches/default.php',
            '/administrator/templates/hathor/html/com_tags/tag/edit.php',
            '/administrator/templates/hathor/html/com_tags/tag/edit_metadata.php',
            '/administrator/templates/hathor/html/com_tags/tag/edit_options.php',
            '/administrator/templates/hathor/html/com_tags/tags/default.php',
            '/administrator/templates/hathor/html/com_templates/style/edit.php',
            '/administrator/templates/hathor/html/com_templates/style/edit_assignment.php',
            '/administrator/templates/hathor/html/com_templates/style/edit_options.php',
            '/administrator/templates/hathor/html/com_templates/styles/default.php',
            '/administrator/templates/hathor/html/com_templates/template/default.php',
            '/administrator/templates/hathor/html/com_templates/template/default_description.php',
            '/administrator/templates/hathor/html/com_templates/template/default_folders.php',
            '/administrator/templates/hathor/html/com_templates/template/default_tree.php',
            '/administrator/templates/hathor/html/com_templates/templates/default.php',
            '/administrator/templates/hathor/html/com_users/debuggroup/default.php',
            '/administrator/templates/hathor/html/com_users/debuguser/default.php',
            '/administrator/templates/hathor/html/com_users/groups/default.php',
            '/administrator/templates/hathor/html/com_users/levels/default.php',
            '/administrator/templates/hathor/html/com_users/note/edit.php',
            '/administrator/templates/hathor/html/com_users/notes/default.php',
            '/administrator/templates/hathor/html/com_users/user/edit.php',
            '/administrator/templates/hathor/html/com_users/users/default.php',
            '/administrator/templates/hathor/html/com_users/users/modal.php',
            '/administrator/templates/hathor/html/com_weblinks/weblink/edit.php',
            '/administrator/templates/hathor/html/com_weblinks/weblink/edit_params.php',
            '/administrator/templates/hathor/html/com_weblinks/weblinks/default.php',
            '/administrator/templates/hathor/html/layouts/com_media/toolbar/deletemedia.php',
            '/administrator/templates/hathor/html/layouts/com_media/toolbar/newfolder.php',
            '/administrator/templates/hathor/html/layouts/com_media/toolbar/uploadmedia.php',
            '/administrator/templates/hathor/html/layouts/com_messages/toolbar/mysettings.php',
            '/administrator/templates/hathor/html/layouts/com_modules/toolbar/cancelselect.php',
            '/administrator/templates/hathor/html/layouts/com_modules/toolbar/newmodule.php',
            '/administrator/templates/hathor/html/layouts/joomla/edit/details.php',
            '/administrator/templates/hathor/html/layouts/joomla/edit/fieldset.php',
            '/administrator/templates/hathor/html/layouts/joomla/edit/global.php',
            '/administrator/templates/hathor/html/layouts/joomla/edit/metadata.php',
            '/administrator/templates/hathor/html/layouts/joomla/edit/params.php',
            '/administrator/templates/hathor/html/layouts/joomla/quickicons/icon.php',
            '/administrator/templates/hathor/html/layouts/joomla/sidebars/submenu.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/base.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/batch.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/confirm.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/containerclose.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/containeropen.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/help.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/iconclass.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/link.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/modal.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/popup.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/separator.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/slider.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/standard.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/title.php',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar/versions.php',
            '/administrator/templates/hathor/html/layouts/plugins/user/profile/fields/dob.php',
            '/administrator/templates/hathor/html/mod_login/default.php',
            '/administrator/templates/hathor/html/mod_quickicon/default.php',
            '/administrator/templates/hathor/html/modules.php',
            '/administrator/templates/hathor/html/pagination.php',
            '/administrator/templates/hathor/images/admin/blank.png',
            '/administrator/templates/hathor/images/admin/checked_out.png',
            '/administrator/templates/hathor/images/admin/collapseall.png',
            '/administrator/templates/hathor/images/admin/disabled.png',
            '/administrator/templates/hathor/images/admin/downarrow-1.png',
            '/administrator/templates/hathor/images/admin/downarrow.png',
            '/administrator/templates/hathor/images/admin/downarrow0.png',
            '/administrator/templates/hathor/images/admin/expandall.png',
            '/administrator/templates/hathor/images/admin/featured.png',
            '/administrator/templates/hathor/images/admin/filesave.png',
            '/administrator/templates/hathor/images/admin/filter_16.png',
            '/administrator/templates/hathor/images/admin/icon-16-allow.png',
            '/administrator/templates/hathor/images/admin/icon-16-allowinactive.png',
            '/administrator/templates/hathor/images/admin/icon-16-deny.png',
            '/administrator/templates/hathor/images/admin/icon-16-denyinactive.png',
            '/administrator/templates/hathor/images/admin/icon-16-links.png',
            '/administrator/templates/hathor/images/admin/icon-16-notice-note.png',
            '/administrator/templates/hathor/images/admin/icon-16-protected.png',
            '/administrator/templates/hathor/images/admin/menu_divider.png',
            '/administrator/templates/hathor/images/admin/note_add_16.png',
            '/administrator/templates/hathor/images/admin/publish_g.png',
            '/administrator/templates/hathor/images/admin/publish_r.png',
            '/administrator/templates/hathor/images/admin/publish_x.png',
            '/administrator/templates/hathor/images/admin/publish_y.png',
            '/administrator/templates/hathor/images/admin/sort_asc.png',
            '/administrator/templates/hathor/images/admin/sort_desc.png',
            '/administrator/templates/hathor/images/admin/tick.png',
            '/administrator/templates/hathor/images/admin/trash.png',
            '/administrator/templates/hathor/images/admin/uparrow-1.png',
            '/administrator/templates/hathor/images/admin/uparrow.png',
            '/administrator/templates/hathor/images/admin/uparrow0.png',
            '/administrator/templates/hathor/images/arrow.png',
            '/administrator/templates/hathor/images/bg-menu.gif',
            '/administrator/templates/hathor/images/calendar.png',
            '/administrator/templates/hathor/images/header/icon-48-alert.png',
            '/administrator/templates/hathor/images/header/icon-48-apply.png',
            '/administrator/templates/hathor/images/header/icon-48-archive.png',
            '/administrator/templates/hathor/images/header/icon-48-article-add.png',
            '/administrator/templates/hathor/images/header/icon-48-article-edit.png',
            '/administrator/templates/hathor/images/header/icon-48-article.png',
            '/administrator/templates/hathor/images/header/icon-48-assoc.png',
            '/administrator/templates/hathor/images/header/icon-48-banner-categories.png',
            '/administrator/templates/hathor/images/header/icon-48-banner-client.png',
            '/administrator/templates/hathor/images/header/icon-48-banner-tracks.png',
            '/administrator/templates/hathor/images/header/icon-48-banner.png',
            '/administrator/templates/hathor/images/header/icon-48-calendar.png',
            '/administrator/templates/hathor/images/header/icon-48-category-add.png',
            '/administrator/templates/hathor/images/header/icon-48-category.png',
            '/administrator/templates/hathor/images/header/icon-48-checkin.png',
            '/administrator/templates/hathor/images/header/icon-48-clear.png',
            '/administrator/templates/hathor/images/header/icon-48-component.png',
            '/administrator/templates/hathor/images/header/icon-48-config.png',
            '/administrator/templates/hathor/images/header/icon-48-contacts-categories.png',
            '/administrator/templates/hathor/images/header/icon-48-contacts.png',
            '/administrator/templates/hathor/images/header/icon-48-content.png',
            '/administrator/templates/hathor/images/header/icon-48-cpanel.png',
            '/administrator/templates/hathor/images/header/icon-48-default.png',
            '/administrator/templates/hathor/images/header/icon-48-deny.png',
            '/administrator/templates/hathor/images/header/icon-48-download.png',
            '/administrator/templates/hathor/images/header/icon-48-edit.png',
            '/administrator/templates/hathor/images/header/icon-48-extension.png',
            '/administrator/templates/hathor/images/header/icon-48-featured.png',
            '/administrator/templates/hathor/images/header/icon-48-frontpage.png',
            '/administrator/templates/hathor/images/header/icon-48-generic.png',
            '/administrator/templates/hathor/images/header/icon-48-groups-add.png',
            '/administrator/templates/hathor/images/header/icon-48-groups.png',
            '/administrator/templates/hathor/images/header/icon-48-help-forum.png',
            '/administrator/templates/hathor/images/header/icon-48-help-this.png',
            '/administrator/templates/hathor/images/header/icon-48-help_header.png',
            '/administrator/templates/hathor/images/header/icon-48-inbox.png',
            '/administrator/templates/hathor/images/header/icon-48-info.png',
            '/administrator/templates/hathor/images/header/icon-48-install.png',
            '/administrator/templates/hathor/images/header/icon-48-jupdate-updatefound.png',
            '/administrator/templates/hathor/images/header/icon-48-jupdate-uptodate.png',
            '/administrator/templates/hathor/images/header/icon-48-language.png',
            '/administrator/templates/hathor/images/header/icon-48-levels-add.png',
            '/administrator/templates/hathor/images/header/icon-48-levels.png',
            '/administrator/templates/hathor/images/header/icon-48-links-cat.png',
            '/administrator/templates/hathor/images/header/icon-48-links.png',
            '/administrator/templates/hathor/images/header/icon-48-massmail.png',
            '/administrator/templates/hathor/images/header/icon-48-media.png',
            '/administrator/templates/hathor/images/header/icon-48-menu-add.png',
            '/administrator/templates/hathor/images/header/icon-48-menu.png',
            '/administrator/templates/hathor/images/header/icon-48-menumgr.png',
            '/administrator/templates/hathor/images/header/icon-48-module.png',
            '/administrator/templates/hathor/images/header/icon-48-move.png',
            '/administrator/templates/hathor/images/header/icon-48-new-privatemessage.png',
            '/administrator/templates/hathor/images/header/icon-48-newcategory.png',
            '/administrator/templates/hathor/images/header/icon-48-newsfeeds-cat.png',
            '/administrator/templates/hathor/images/header/icon-48-newsfeeds.png',
            '/administrator/templates/hathor/images/header/icon-48-notice.png',
            '/administrator/templates/hathor/images/header/icon-48-plugin.png',
            '/administrator/templates/hathor/images/header/icon-48-preview.png',
            '/administrator/templates/hathor/images/header/icon-48-print.png',
            '/administrator/templates/hathor/images/header/icon-48-purge.png',
            '/administrator/templates/hathor/images/header/icon-48-puzzle.png',
            '/administrator/templates/hathor/images/header/icon-48-read-privatemessage.png',
            '/administrator/templates/hathor/images/header/icon-48-readmess.png',
            '/administrator/templates/hathor/images/header/icon-48-redirect.png',
            '/administrator/templates/hathor/images/header/icon-48-revert.png',
            '/administrator/templates/hathor/images/header/icon-48-search.png',
            '/administrator/templates/hathor/images/header/icon-48-section.png',
            '/administrator/templates/hathor/images/header/icon-48-send.png',
            '/administrator/templates/hathor/images/header/icon-48-static.png',
            '/administrator/templates/hathor/images/header/icon-48-stats.png',
            '/administrator/templates/hathor/images/header/icon-48-tags.png',
            '/administrator/templates/hathor/images/header/icon-48-themes.png',
            '/administrator/templates/hathor/images/header/icon-48-trash.png',
            '/administrator/templates/hathor/images/header/icon-48-unarchive.png',
            '/administrator/templates/hathor/images/header/icon-48-upload.png',
            '/administrator/templates/hathor/images/header/icon-48-user-add.png',
            '/administrator/templates/hathor/images/header/icon-48-user-edit.png',
            '/administrator/templates/hathor/images/header/icon-48-user-profile.png',
            '/administrator/templates/hathor/images/header/icon-48-user.png',
            '/administrator/templates/hathor/images/header/icon-48-writemess.png',
            '/administrator/templates/hathor/images/header/icon-messaging.png',
            '/administrator/templates/hathor/images/j_arrow.png',
            '/administrator/templates/hathor/images/j_arrow_down.png',
            '/administrator/templates/hathor/images/j_arrow_left.png',
            '/administrator/templates/hathor/images/j_arrow_right.png',
            '/administrator/templates/hathor/images/j_login_lock.png',
            '/administrator/templates/hathor/images/j_logo.png',
            '/administrator/templates/hathor/images/logo.png',
            '/administrator/templates/hathor/images/menu/icon-16-alert.png',
            '/administrator/templates/hathor/images/menu/icon-16-apply.png',
            '/administrator/templates/hathor/images/menu/icon-16-archive.png',
            '/administrator/templates/hathor/images/menu/icon-16-article.png',
            '/administrator/templates/hathor/images/menu/icon-16-assoc.png',
            '/administrator/templates/hathor/images/menu/icon-16-back-user.png',
            '/administrator/templates/hathor/images/menu/icon-16-banner-categories.png',
            '/administrator/templates/hathor/images/menu/icon-16-banner-client.png',
            '/administrator/templates/hathor/images/menu/icon-16-banner-tracks.png',
            '/administrator/templates/hathor/images/menu/icon-16-banner.png',
            '/administrator/templates/hathor/images/menu/icon-16-calendar.png',
            '/administrator/templates/hathor/images/menu/icon-16-category.png',
            '/administrator/templates/hathor/images/menu/icon-16-checkin.png',
            '/administrator/templates/hathor/images/menu/icon-16-clear.png',
            '/administrator/templates/hathor/images/menu/icon-16-component.png',
            '/administrator/templates/hathor/images/menu/icon-16-config.png',
            '/administrator/templates/hathor/images/menu/icon-16-contacts-categories.png',
            '/administrator/templates/hathor/images/menu/icon-16-contacts.png',
            '/administrator/templates/hathor/images/menu/icon-16-content.png',
            '/administrator/templates/hathor/images/menu/icon-16-cpanel.png',
            '/administrator/templates/hathor/images/menu/icon-16-default.png',
            '/administrator/templates/hathor/images/menu/icon-16-delete.png',
            '/administrator/templates/hathor/images/menu/icon-16-deny.png',
            '/administrator/templates/hathor/images/menu/icon-16-download.png',
            '/administrator/templates/hathor/images/menu/icon-16-edit.png',
            '/administrator/templates/hathor/images/menu/icon-16-featured.png',
            '/administrator/templates/hathor/images/menu/icon-16-frontpage.png',
            '/administrator/templates/hathor/images/menu/icon-16-generic.png',
            '/administrator/templates/hathor/images/menu/icon-16-groups.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-community.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-dev.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-docs.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-forum.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-jed.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-jrd.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-security.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-shop.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-this.png',
            '/administrator/templates/hathor/images/menu/icon-16-help-trans.png',
            '/administrator/templates/hathor/images/menu/icon-16-help.png',
            '/administrator/templates/hathor/images/menu/icon-16-inbox.png',
            '/administrator/templates/hathor/images/menu/icon-16-info.png',
            '/administrator/templates/hathor/images/menu/icon-16-install.png',
            '/administrator/templates/hathor/images/menu/icon-16-language.png',
            '/administrator/templates/hathor/images/menu/icon-16-levels.png',
            '/administrator/templates/hathor/images/menu/icon-16-links-cat.png',
            '/administrator/templates/hathor/images/menu/icon-16-links.png',
            '/administrator/templates/hathor/images/menu/icon-16-logout.png',
            '/administrator/templates/hathor/images/menu/icon-16-maintenance.png',
            '/administrator/templates/hathor/images/menu/icon-16-massmail.png',
            '/administrator/templates/hathor/images/menu/icon-16-media.png',
            '/administrator/templates/hathor/images/menu/icon-16-menu.png',
            '/administrator/templates/hathor/images/menu/icon-16-menumgr.png',
            '/administrator/templates/hathor/images/menu/icon-16-messages.png',
            '/administrator/templates/hathor/images/menu/icon-16-messaging.png',
            '/administrator/templates/hathor/images/menu/icon-16-module.png',
            '/administrator/templates/hathor/images/menu/icon-16-move.png',
            '/administrator/templates/hathor/images/menu/icon-16-new-privatemessage.png',
            '/administrator/templates/hathor/images/menu/icon-16-new.png',
            '/administrator/templates/hathor/images/menu/icon-16-newarticle.png',
            '/administrator/templates/hathor/images/menu/icon-16-newcategory.png',
            '/administrator/templates/hathor/images/menu/icon-16-newgroup.png',
            '/administrator/templates/hathor/images/menu/icon-16-newlevel.png',
            '/administrator/templates/hathor/images/menu/icon-16-newsfeeds-cat.png',
            '/administrator/templates/hathor/images/menu/icon-16-newsfeeds.png',
            '/administrator/templates/hathor/images/menu/icon-16-newuser.png',
            '/administrator/templates/hathor/images/menu/icon-16-nopreview.png',
            '/administrator/templates/hathor/images/menu/icon-16-notdefault.png',
            '/administrator/templates/hathor/images/menu/icon-16-notice.png',
            '/administrator/templates/hathor/images/menu/icon-16-plugin.png',
            '/administrator/templates/hathor/images/menu/icon-16-preview.png',
            '/administrator/templates/hathor/images/menu/icon-16-print.png',
            '/administrator/templates/hathor/images/menu/icon-16-purge.png',
            '/administrator/templates/hathor/images/menu/icon-16-puzzle.png',
            '/administrator/templates/hathor/images/menu/icon-16-read-privatemessage.png',
            '/administrator/templates/hathor/images/menu/icon-16-readmess.png',
            '/administrator/templates/hathor/images/menu/icon-16-redirect.png',
            '/administrator/templates/hathor/images/menu/icon-16-revert.png',
            '/administrator/templates/hathor/images/menu/icon-16-search.png',
            '/administrator/templates/hathor/images/menu/icon-16-send.png',
            '/administrator/templates/hathor/images/menu/icon-16-stats.png',
            '/administrator/templates/hathor/images/menu/icon-16-tags.png',
            '/administrator/templates/hathor/images/menu/icon-16-themes.png',
            '/administrator/templates/hathor/images/menu/icon-16-trash.png',
            '/administrator/templates/hathor/images/menu/icon-16-unarticle.png',
            '/administrator/templates/hathor/images/menu/icon-16-upload.png',
            '/administrator/templates/hathor/images/menu/icon-16-user-dd.png',
            '/administrator/templates/hathor/images/menu/icon-16-user-note.png',
            '/administrator/templates/hathor/images/menu/icon-16-user.png',
            '/administrator/templates/hathor/images/menu/icon-16-viewsite.png',
            '/administrator/templates/hathor/images/menu/icon-16-writemess.png',
            '/administrator/templates/hathor/images/mini_icon.png',
            '/administrator/templates/hathor/images/notice-alert.png',
            '/administrator/templates/hathor/images/notice-info.png',
            '/administrator/templates/hathor/images/notice-note.png',
            '/administrator/templates/hathor/images/required.png',
            '/administrator/templates/hathor/images/selector-arrow-hc.png',
            '/administrator/templates/hathor/images/selector-arrow-rtl.png',
            '/administrator/templates/hathor/images/selector-arrow-std.png',
            '/administrator/templates/hathor/images/selector-arrow.png',
            '/administrator/templates/hathor/images/system/calendar.png',
            '/administrator/templates/hathor/images/system/selector-arrow.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-adduser.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-alert.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-apply.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-archive.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-article-add.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-article.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-back.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-banner-categories.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-banner-client.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-banner-tracks.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-banner.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-batch.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-calendar.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-cancel.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-checkin.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-cog.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-component.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-config.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-contacts-categories.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-contacts.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-copy.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-css.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-default.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-delete-style.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-delete.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-deny.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-download.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-edit.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-error.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-export.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-extension.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-featured.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-forward.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-help.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-html.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-inbox.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-info.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-links.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-lock.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-menu.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-messaging.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-messanging.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-module.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-move.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-new-privatemessage.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-new-style.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-new.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-notice.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-preview.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-print.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-publish.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-purge.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-read-privatemessage.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-refresh.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-remove.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-revert.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-save-copy.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-save-new.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-save.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-search.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-send.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-stats.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-trash.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-unarchive.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-unblock.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-unpublish.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-upload.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-user-add.png',
            '/administrator/templates/hathor/images/toolbar/icon-32-xml.png',
            '/administrator/templates/hathor/index.php',
            '/administrator/templates/hathor/js/template.js',
            '/administrator/templates/hathor/language/en-GB/en-GB.tpl_hathor.ini',
            '/administrator/templates/hathor/language/en-GB/en-GB.tpl_hathor.sys.ini',
            '/administrator/templates/hathor/less/buttons.less',
            '/administrator/templates/hathor/less/colour_baseline.less',
            '/administrator/templates/hathor/less/colour_blue.less',
            '/administrator/templates/hathor/less/colour_brown.less',
            '/administrator/templates/hathor/less/colour_standard.less',
            '/administrator/templates/hathor/less/forms.less',
            '/administrator/templates/hathor/less/hathor_variables.less',
            '/administrator/templates/hathor/less/icomoon.less',
            '/administrator/templates/hathor/less/modals.less',
            '/administrator/templates/hathor/less/template.less',
            '/administrator/templates/hathor/less/variables.less',
            '/administrator/templates/hathor/login.php',
            '/administrator/templates/hathor/postinstall/hathormessage.php',
            '/administrator/templates/hathor/templateDetails.xml',
            '/administrator/templates/hathor/template_preview.png',
            '/administrator/templates/hathor/template_thumbnail.png',
            '/administrator/templates/isis/component.php',
            '/administrator/templates/isis/cpanel.php',
            '/administrator/templates/isis/css/template-rtl.css',
            '/administrator/templates/isis/css/template.css',
            '/administrator/templates/isis/error.php',
            '/administrator/templates/isis/favicon.ico',
            '/administrator/templates/isis/html/com_media/imageslist/default_folder.php',
            '/administrator/templates/isis/html/com_media/imageslist/default_image.php',
            '/administrator/templates/isis/html/com_media/medialist/thumbs_folders.php',
            '/administrator/templates/isis/html/com_media/medialist/thumbs_imgs.php',
            '/administrator/templates/isis/html/editor_content.css',
            '/administrator/templates/isis/html/layouts/joomla/form/field/media.php',
            '/administrator/templates/isis/html/layouts/joomla/form/field/user.php',
            '/administrator/templates/isis/html/layouts/joomla/pagination/link.php',
            '/administrator/templates/isis/html/layouts/joomla/pagination/links.php',
            '/administrator/templates/isis/html/layouts/joomla/system/message.php',
            '/administrator/templates/isis/html/layouts/joomla/toolbar/versions.php',
            '/administrator/templates/isis/html/mod_version/default.php',
            '/administrator/templates/isis/html/modules.php',
            '/administrator/templates/isis/html/pagination.php',
            '/administrator/templates/isis/images/admin/blank.png',
            '/administrator/templates/isis/images/admin/checked_out.png',
            '/administrator/templates/isis/images/admin/collapseall.png',
            '/administrator/templates/isis/images/admin/disabled.png',
            '/administrator/templates/isis/images/admin/downarrow-1.png',
            '/administrator/templates/isis/images/admin/downarrow.png',
            '/administrator/templates/isis/images/admin/downarrow0.png',
            '/administrator/templates/isis/images/admin/expandall.png',
            '/administrator/templates/isis/images/admin/featured.png',
            '/administrator/templates/isis/images/admin/filesave.png',
            '/administrator/templates/isis/images/admin/filter_16.png',
            '/administrator/templates/isis/images/admin/icon-16-add.png',
            '/administrator/templates/isis/images/admin/icon-16-allow.png',
            '/administrator/templates/isis/images/admin/icon-16-allowinactive.png',
            '/administrator/templates/isis/images/admin/icon-16-deny.png',
            '/administrator/templates/isis/images/admin/icon-16-denyinactive.png',
            '/administrator/templates/isis/images/admin/icon-16-links.png',
            '/administrator/templates/isis/images/admin/icon-16-notice-note.png',
            '/administrator/templates/isis/images/admin/icon-16-protected.png',
            '/administrator/templates/isis/images/admin/menu_divider.png',
            '/administrator/templates/isis/images/admin/note_add_16.png',
            '/administrator/templates/isis/images/admin/publish_g.png',
            '/administrator/templates/isis/images/admin/publish_r.png',
            '/administrator/templates/isis/images/admin/publish_x.png',
            '/administrator/templates/isis/images/admin/publish_y.png',
            '/administrator/templates/isis/images/admin/sort_asc.png',
            '/administrator/templates/isis/images/admin/sort_desc.png',
            '/administrator/templates/isis/images/admin/tick.png',
            '/administrator/templates/isis/images/admin/trash.png',
            '/administrator/templates/isis/images/admin/uparrow-1.png',
            '/administrator/templates/isis/images/admin/uparrow.png',
            '/administrator/templates/isis/images/admin/uparrow0.png',
            '/administrator/templates/isis/images/emailButton.png',
            '/administrator/templates/isis/images/joomla.png',
            '/administrator/templates/isis/images/login-joomla-inverse.png',
            '/administrator/templates/isis/images/login-joomla.png',
            '/administrator/templates/isis/images/logo-inverse.png',
            '/administrator/templates/isis/images/logo.png',
            '/administrator/templates/isis/images/pdf_button.png',
            '/administrator/templates/isis/images/printButton.png',
            '/administrator/templates/isis/images/system/sort_asc.png',
            '/administrator/templates/isis/images/system/sort_desc.png',
            '/administrator/templates/isis/img/glyphicons-halflings-white.png',
            '/administrator/templates/isis/img/glyphicons-halflings.png',
            '/administrator/templates/isis/index.php',
            '/administrator/templates/isis/js/application.js',
            '/administrator/templates/isis/js/classes.js',
            '/administrator/templates/isis/js/template.js',
            '/administrator/templates/isis/language/en-GB/en-GB.tpl_isis.ini',
            '/administrator/templates/isis/language/en-GB/en-GB.tpl_isis.sys.ini',
            '/administrator/templates/isis/less/blocks/_chzn-override.less',
            '/administrator/templates/isis/less/blocks/_custom.less',
            '/administrator/templates/isis/less/blocks/_editors.less',
            '/administrator/templates/isis/less/blocks/_forms.less',
            '/administrator/templates/isis/less/blocks/_global.less',
            '/administrator/templates/isis/less/blocks/_header.less',
            '/administrator/templates/isis/less/blocks/_login.less',
            '/administrator/templates/isis/less/blocks/_media.less',
            '/administrator/templates/isis/less/blocks/_modals.less',
            '/administrator/templates/isis/less/blocks/_navbar.less',
            '/administrator/templates/isis/less/blocks/_quickicons.less',
            '/administrator/templates/isis/less/blocks/_sidebar.less',
            '/administrator/templates/isis/less/blocks/_status.less',
            '/administrator/templates/isis/less/blocks/_tables.less',
            '/administrator/templates/isis/less/blocks/_toolbar.less',
            '/administrator/templates/isis/less/blocks/_treeselect.less',
            '/administrator/templates/isis/less/blocks/_utility-classes.less',
            '/administrator/templates/isis/less/bootstrap/button-groups.less',
            '/administrator/templates/isis/less/bootstrap/buttons.less',
            '/administrator/templates/isis/less/bootstrap/mixins.less',
            '/administrator/templates/isis/less/bootstrap/responsive-1200px-min.less',
            '/administrator/templates/isis/less/bootstrap/responsive-768px-979px.less',
            '/administrator/templates/isis/less/bootstrap/wells.less',
            '/administrator/templates/isis/less/icomoon.less',
            '/administrator/templates/isis/less/pages/_com_cpanel.less',
            '/administrator/templates/isis/less/pages/_com_postinstall.less',
            '/administrator/templates/isis/less/pages/_com_privacy.less',
            '/administrator/templates/isis/less/pages/_com_templates.less',
            '/administrator/templates/isis/less/template-rtl.less',
            '/administrator/templates/isis/less/template.less',
            '/administrator/templates/isis/less/variables.less',
            '/administrator/templates/isis/login.php',
            '/administrator/templates/isis/templateDetails.xml',
            '/administrator/templates/isis/template_preview.png',
            '/administrator/templates/isis/template_thumbnail.png',
            '/administrator/templates/system/html/modules.php',
            '/bin/index.html',
            '/bin/keychain.php',
            '/cli/deletefiles.php',
            '/cli/finder_indexer.php',
            '/cli/garbagecron.php',
            '/cli/sessionGc.php',
            '/cli/sessionMetadataGc.php',
            '/cli/update_cron.php',
            '/components/com_banners/banners.php',
            '/components/com_banners/controller.php',
            '/components/com_banners/helpers/banner.php',
            '/components/com_banners/helpers/category.php',
            '/components/com_banners/models/banner.php',
            '/components/com_banners/models/banners.php',
            '/components/com_banners/router.php',
            '/components/com_config/config.php',
            '/components/com_config/controller/cancel.php',
            '/components/com_config/controller/canceladmin.php',
            '/components/com_config/controller/cmsbase.php',
            '/components/com_config/controller/config/display.php',
            '/components/com_config/controller/config/save.php',
            '/components/com_config/controller/display.php',
            '/components/com_config/controller/helper.php',
            '/components/com_config/controller/modules/cancel.php',
            '/components/com_config/controller/modules/display.php',
            '/components/com_config/controller/modules/save.php',
            '/components/com_config/controller/templates/display.php',
            '/components/com_config/controller/templates/save.php',
            '/components/com_config/model/cms.php',
            '/components/com_config/model/config.php',
            '/components/com_config/model/form.php',
            '/components/com_config/model/form/config.xml',
            '/components/com_config/model/form/modules.xml',
            '/components/com_config/model/form/modules_advanced.xml',
            '/components/com_config/model/form/templates.xml',
            '/components/com_config/model/modules.php',
            '/components/com_config/model/templates.php',
            '/components/com_config/view/cms/html.php',
            '/components/com_config/view/cms/json.php',
            '/components/com_config/view/config/html.php',
            '/components/com_config/view/config/tmpl/default.php',
            '/components/com_config/view/config/tmpl/default.xml',
            '/components/com_config/view/config/tmpl/default_metadata.php',
            '/components/com_config/view/config/tmpl/default_seo.php',
            '/components/com_config/view/config/tmpl/default_site.php',
            '/components/com_config/view/modules/html.php',
            '/components/com_config/view/modules/tmpl/default.php',
            '/components/com_config/view/modules/tmpl/default_options.php',
            '/components/com_config/view/modules/tmpl/default_positions.php',
            '/components/com_config/view/templates/html.php',
            '/components/com_config/view/templates/tmpl/default.php',
            '/components/com_config/view/templates/tmpl/default.xml',
            '/components/com_config/view/templates/tmpl/default_options.php',
            '/components/com_contact/contact.php',
            '/components/com_contact/controller.php',
            '/components/com_contact/controllers/contact.php',
            '/components/com_contact/helpers/association.php',
            '/components/com_contact/helpers/category.php',
            '/components/com_contact/helpers/legacyrouter.php',
            '/components/com_contact/layouts/joomla/form/renderfield.php',
            '/components/com_contact/models/categories.php',
            '/components/com_contact/models/category.php',
            '/components/com_contact/models/contact.php',
            '/components/com_contact/models/featured.php',
            '/components/com_contact/models/forms/contact.xml',
            '/components/com_contact/models/forms/filter_contacts.xml',
            '/components/com_contact/models/forms/form.xml',
            '/components/com_contact/models/rules/contactemail.php',
            '/components/com_contact/models/rules/contactemailmessage.php',
            '/components/com_contact/models/rules/contactemailsubject.php',
            '/components/com_contact/router.php',
            '/components/com_contact/views/categories/tmpl/default.php',
            '/components/com_contact/views/categories/tmpl/default.xml',
            '/components/com_contact/views/categories/tmpl/default_items.php',
            '/components/com_contact/views/categories/view.html.php',
            '/components/com_contact/views/category/tmpl/default.php',
            '/components/com_contact/views/category/tmpl/default.xml',
            '/components/com_contact/views/category/tmpl/default_children.php',
            '/components/com_contact/views/category/tmpl/default_items.php',
            '/components/com_contact/views/category/view.feed.php',
            '/components/com_contact/views/category/view.html.php',
            '/components/com_contact/views/contact/tmpl/default.php',
            '/components/com_contact/views/contact/tmpl/default.xml',
            '/components/com_contact/views/contact/tmpl/default_address.php',
            '/components/com_contact/views/contact/tmpl/default_articles.php',
            '/components/com_contact/views/contact/tmpl/default_form.php',
            '/components/com_contact/views/contact/tmpl/default_links.php',
            '/components/com_contact/views/contact/tmpl/default_profile.php',
            '/components/com_contact/views/contact/tmpl/default_user_custom_fields.php',
            '/components/com_contact/views/contact/view.html.php',
            '/components/com_contact/views/contact/view.vcf.php',
            '/components/com_contact/views/featured/tmpl/default.php',
            '/components/com_contact/views/featured/tmpl/default.xml',
            '/components/com_contact/views/featured/tmpl/default_items.php',
            '/components/com_contact/views/featured/view.html.php',
            '/components/com_content/content.php',
            '/components/com_content/controller.php',
            '/components/com_content/controllers/article.php',
            '/components/com_content/helpers/association.php',
            '/components/com_content/helpers/category.php',
            '/components/com_content/helpers/legacyrouter.php',
            '/components/com_content/helpers/query.php',
            '/components/com_content/helpers/route.php',
            '/components/com_content/models/archive.php',
            '/components/com_content/models/article.php',
            '/components/com_content/models/articles.php',
            '/components/com_content/models/categories.php',
            '/components/com_content/models/category.php',
            '/components/com_content/models/featured.php',
            '/components/com_content/models/form.php',
            '/components/com_content/models/forms/article.xml',
            '/components/com_content/models/forms/filter_articles.xml',
            '/components/com_content/router.php',
            '/components/com_content/views/archive/tmpl/default.php',
            '/components/com_content/views/archive/tmpl/default.xml',
            '/components/com_content/views/archive/tmpl/default_items.php',
            '/components/com_content/views/archive/view.html.php',
            '/components/com_content/views/article/tmpl/default.php',
            '/components/com_content/views/article/tmpl/default.xml',
            '/components/com_content/views/article/tmpl/default_links.php',
            '/components/com_content/views/article/view.html.php',
            '/components/com_content/views/categories/tmpl/default.php',
            '/components/com_content/views/categories/tmpl/default.xml',
            '/components/com_content/views/categories/tmpl/default_items.php',
            '/components/com_content/views/categories/view.html.php',
            '/components/com_content/views/category/tmpl/blog.php',
            '/components/com_content/views/category/tmpl/blog.xml',
            '/components/com_content/views/category/tmpl/blog_children.php',
            '/components/com_content/views/category/tmpl/blog_item.php',
            '/components/com_content/views/category/tmpl/blog_links.php',
            '/components/com_content/views/category/tmpl/default.php',
            '/components/com_content/views/category/tmpl/default.xml',
            '/components/com_content/views/category/tmpl/default_articles.php',
            '/components/com_content/views/category/tmpl/default_children.php',
            '/components/com_content/views/category/view.feed.php',
            '/components/com_content/views/category/view.html.php',
            '/components/com_content/views/featured/tmpl/default.php',
            '/components/com_content/views/featured/tmpl/default.xml',
            '/components/com_content/views/featured/tmpl/default_item.php',
            '/components/com_content/views/featured/tmpl/default_links.php',
            '/components/com_content/views/featured/view.feed.php',
            '/components/com_content/views/featured/view.html.php',
            '/components/com_content/views/form/tmpl/edit.php',
            '/components/com_content/views/form/tmpl/edit.xml',
            '/components/com_content/views/form/view.html.php',
            '/components/com_contenthistory/contenthistory.php',
            '/components/com_fields/controller.php',
            '/components/com_fields/fields.php',
            '/components/com_fields/models/forms/filter_fields.xml',
            '/components/com_finder/controller.php',
            '/components/com_finder/controllers/suggestions.json.php',
            '/components/com_finder/finder.php',
            '/components/com_finder/helpers/html/filter.php',
            '/components/com_finder/helpers/html/query.php',
            '/components/com_finder/models/search.php',
            '/components/com_finder/models/suggestions.php',
            '/components/com_finder/router.php',
            '/components/com_finder/views/search/tmpl/default.php',
            '/components/com_finder/views/search/tmpl/default.xml',
            '/components/com_finder/views/search/tmpl/default_form.php',
            '/components/com_finder/views/search/tmpl/default_result.php',
            '/components/com_finder/views/search/tmpl/default_results.php',
            '/components/com_finder/views/search/view.feed.php',
            '/components/com_finder/views/search/view.html.php',
            '/components/com_finder/views/search/view.opensearch.php',
            '/components/com_mailto/controller.php',
            '/components/com_mailto/helpers/mailto.php',
            '/components/com_mailto/mailto.php',
            '/components/com_mailto/mailto.xml',
            '/components/com_mailto/models/forms/mailto.xml',
            '/components/com_mailto/models/mailto.php',
            '/components/com_mailto/views/mailto/tmpl/default.php',
            '/components/com_mailto/views/mailto/view.html.php',
            '/components/com_mailto/views/sent/tmpl/default.php',
            '/components/com_mailto/views/sent/view.html.php',
            '/components/com_media/media.php',
            '/components/com_menus/controller.php',
            '/components/com_menus/menus.php',
            '/components/com_menus/models/forms/filter_items.xml',
            '/components/com_modules/controller.php',
            '/components/com_modules/models/forms/filter_modules.xml',
            '/components/com_modules/modules.php',
            '/components/com_newsfeeds/controller.php',
            '/components/com_newsfeeds/helpers/association.php',
            '/components/com_newsfeeds/helpers/category.php',
            '/components/com_newsfeeds/helpers/legacyrouter.php',
            '/components/com_newsfeeds/models/categories.php',
            '/components/com_newsfeeds/models/category.php',
            '/components/com_newsfeeds/models/newsfeed.php',
            '/components/com_newsfeeds/newsfeeds.php',
            '/components/com_newsfeeds/router.php',
            '/components/com_newsfeeds/views/categories/tmpl/default.php',
            '/components/com_newsfeeds/views/categories/tmpl/default.xml',
            '/components/com_newsfeeds/views/categories/tmpl/default_items.php',
            '/components/com_newsfeeds/views/categories/view.html.php',
            '/components/com_newsfeeds/views/category/tmpl/default.php',
            '/components/com_newsfeeds/views/category/tmpl/default.xml',
            '/components/com_newsfeeds/views/category/tmpl/default_children.php',
            '/components/com_newsfeeds/views/category/tmpl/default_items.php',
            '/components/com_newsfeeds/views/category/view.html.php',
            '/components/com_newsfeeds/views/newsfeed/tmpl/default.php',
            '/components/com_newsfeeds/views/newsfeed/tmpl/default.xml',
            '/components/com_newsfeeds/views/newsfeed/view.html.php',
            '/components/com_privacy/controller.php',
            '/components/com_privacy/controllers/request.php',
            '/components/com_privacy/models/confirm.php',
            '/components/com_privacy/models/forms/confirm.xml',
            '/components/com_privacy/models/forms/remind.xml',
            '/components/com_privacy/models/forms/request.xml',
            '/components/com_privacy/models/remind.php',
            '/components/com_privacy/models/request.php',
            '/components/com_privacy/privacy.php',
            '/components/com_privacy/router.php',
            '/components/com_privacy/views/confirm/tmpl/default.php',
            '/components/com_privacy/views/confirm/tmpl/default.xml',
            '/components/com_privacy/views/confirm/view.html.php',
            '/components/com_privacy/views/remind/tmpl/default.php',
            '/components/com_privacy/views/remind/tmpl/default.xml',
            '/components/com_privacy/views/remind/view.html.php',
            '/components/com_privacy/views/request/tmpl/default.php',
            '/components/com_privacy/views/request/tmpl/default.xml',
            '/components/com_privacy/views/request/view.html.php',
            '/components/com_tags/controller.php',
            '/components/com_tags/controllers/tags.php',
            '/components/com_tags/models/tag.php',
            '/components/com_tags/models/tags.php',
            '/components/com_tags/router.php',
            '/components/com_tags/tags.php',
            '/components/com_tags/views/tag/tmpl/default.php',
            '/components/com_tags/views/tag/tmpl/default.xml',
            '/components/com_tags/views/tag/tmpl/default_items.php',
            '/components/com_tags/views/tag/tmpl/list.php',
            '/components/com_tags/views/tag/tmpl/list.xml',
            '/components/com_tags/views/tag/tmpl/list_items.php',
            '/components/com_tags/views/tag/view.feed.php',
            '/components/com_tags/views/tag/view.html.php',
            '/components/com_tags/views/tags/tmpl/default.php',
            '/components/com_tags/views/tags/tmpl/default.xml',
            '/components/com_tags/views/tags/tmpl/default_items.php',
            '/components/com_tags/views/tags/view.feed.php',
            '/components/com_tags/views/tags/view.html.php',
            '/components/com_users/controller.php',
            '/components/com_users/controllers/profile.php',
            '/components/com_users/controllers/registration.php',
            '/components/com_users/controllers/remind.php',
            '/components/com_users/controllers/reset.php',
            '/components/com_users/controllers/user.php',
            '/components/com_users/helpers/html/users.php',
            '/components/com_users/helpers/legacyrouter.php',
            '/components/com_users/helpers/route.php',
            '/components/com_users/layouts/joomla/form/renderfield.php',
            '/components/com_users/models/forms/frontend.xml',
            '/components/com_users/models/forms/frontend_admin.xml',
            '/components/com_users/models/forms/login.xml',
            '/components/com_users/models/forms/profile.xml',
            '/components/com_users/models/forms/registration.xml',
            '/components/com_users/models/forms/remind.xml',
            '/components/com_users/models/forms/reset_complete.xml',
            '/components/com_users/models/forms/reset_confirm.xml',
            '/components/com_users/models/forms/reset_request.xml',
            '/components/com_users/models/forms/sitelang.xml',
            '/components/com_users/models/login.php',
            '/components/com_users/models/profile.php',
            '/components/com_users/models/registration.php',
            '/components/com_users/models/remind.php',
            '/components/com_users/models/reset.php',
            '/components/com_users/models/rules/loginuniquefield.php',
            '/components/com_users/models/rules/logoutuniquefield.php',
            '/components/com_users/router.php',
            '/components/com_users/users.php',
            '/components/com_users/views/login/tmpl/default.php',
            '/components/com_users/views/login/tmpl/default.xml',
            '/components/com_users/views/login/tmpl/default_login.php',
            '/components/com_users/views/login/tmpl/default_logout.php',
            '/components/com_users/views/login/tmpl/logout.xml',
            '/components/com_users/views/login/view.html.php',
            '/components/com_users/views/profile/tmpl/default.php',
            '/components/com_users/views/profile/tmpl/default.xml',
            '/components/com_users/views/profile/tmpl/default_core.php',
            '/components/com_users/views/profile/tmpl/default_custom.php',
            '/components/com_users/views/profile/tmpl/default_params.php',
            '/components/com_users/views/profile/tmpl/edit.php',
            '/components/com_users/views/profile/tmpl/edit.xml',
            '/components/com_users/views/profile/view.html.php',
            '/components/com_users/views/registration/tmpl/complete.php',
            '/components/com_users/views/registration/tmpl/default.php',
            '/components/com_users/views/registration/tmpl/default.xml',
            '/components/com_users/views/registration/view.html.php',
            '/components/com_users/views/remind/tmpl/default.php',
            '/components/com_users/views/remind/tmpl/default.xml',
            '/components/com_users/views/remind/view.html.php',
            '/components/com_users/views/reset/tmpl/complete.php',
            '/components/com_users/views/reset/tmpl/confirm.php',
            '/components/com_users/views/reset/tmpl/default.php',
            '/components/com_users/views/reset/tmpl/default.xml',
            '/components/com_users/views/reset/view.html.php',
            '/components/com_wrapper/controller.php',
            '/components/com_wrapper/router.php',
            '/components/com_wrapper/views/wrapper/tmpl/default.php',
            '/components/com_wrapper/views/wrapper/tmpl/default.xml',
            '/components/com_wrapper/views/wrapper/view.html.php',
            '/components/com_wrapper/wrapper.php',
            '/components/com_wrapper/wrapper.xml',
            '/language/en-GB/en-GB.com_ajax.ini',
            '/language/en-GB/en-GB.com_config.ini',
            '/language/en-GB/en-GB.com_contact.ini',
            '/language/en-GB/en-GB.com_content.ini',
            '/language/en-GB/en-GB.com_finder.ini',
            '/language/en-GB/en-GB.com_mailto.ini',
            '/language/en-GB/en-GB.com_media.ini',
            '/language/en-GB/en-GB.com_messages.ini',
            '/language/en-GB/en-GB.com_newsfeeds.ini',
            '/language/en-GB/en-GB.com_privacy.ini',
            '/language/en-GB/en-GB.com_tags.ini',
            '/language/en-GB/en-GB.com_users.ini',
            '/language/en-GB/en-GB.com_weblinks.ini',
            '/language/en-GB/en-GB.com_wrapper.ini',
            '/language/en-GB/en-GB.files_joomla.sys.ini',
            '/language/en-GB/en-GB.finder_cli.ini',
            '/language/en-GB/en-GB.ini',
            '/language/en-GB/en-GB.lib_fof.ini',
            '/language/en-GB/en-GB.lib_fof.sys.ini',
            '/language/en-GB/en-GB.lib_idna_convert.sys.ini',
            '/language/en-GB/en-GB.lib_joomla.ini',
            '/language/en-GB/en-GB.lib_joomla.sys.ini',
            '/language/en-GB/en-GB.lib_phpass.sys.ini',
            '/language/en-GB/en-GB.lib_phputf8.sys.ini',
            '/language/en-GB/en-GB.lib_simplepie.sys.ini',
            '/language/en-GB/en-GB.localise.php',
            '/language/en-GB/en-GB.mod_articles_archive.ini',
            '/language/en-GB/en-GB.mod_articles_archive.sys.ini',
            '/language/en-GB/en-GB.mod_articles_categories.ini',
            '/language/en-GB/en-GB.mod_articles_categories.sys.ini',
            '/language/en-GB/en-GB.mod_articles_category.ini',
            '/language/en-GB/en-GB.mod_articles_category.sys.ini',
            '/language/en-GB/en-GB.mod_articles_latest.ini',
            '/language/en-GB/en-GB.mod_articles_latest.sys.ini',
            '/language/en-GB/en-GB.mod_articles_news.ini',
            '/language/en-GB/en-GB.mod_articles_news.sys.ini',
            '/language/en-GB/en-GB.mod_articles_popular.ini',
            '/language/en-GB/en-GB.mod_articles_popular.sys.ini',
            '/language/en-GB/en-GB.mod_banners.ini',
            '/language/en-GB/en-GB.mod_banners.sys.ini',
            '/language/en-GB/en-GB.mod_breadcrumbs.ini',
            '/language/en-GB/en-GB.mod_breadcrumbs.sys.ini',
            '/language/en-GB/en-GB.mod_custom.ini',
            '/language/en-GB/en-GB.mod_custom.sys.ini',
            '/language/en-GB/en-GB.mod_feed.ini',
            '/language/en-GB/en-GB.mod_feed.sys.ini',
            '/language/en-GB/en-GB.mod_finder.ini',
            '/language/en-GB/en-GB.mod_finder.sys.ini',
            '/language/en-GB/en-GB.mod_footer.ini',
            '/language/en-GB/en-GB.mod_footer.sys.ini',
            '/language/en-GB/en-GB.mod_languages.ini',
            '/language/en-GB/en-GB.mod_languages.sys.ini',
            '/language/en-GB/en-GB.mod_login.ini',
            '/language/en-GB/en-GB.mod_login.sys.ini',
            '/language/en-GB/en-GB.mod_menu.ini',
            '/language/en-GB/en-GB.mod_menu.sys.ini',
            '/language/en-GB/en-GB.mod_random_image.ini',
            '/language/en-GB/en-GB.mod_random_image.sys.ini',
            '/language/en-GB/en-GB.mod_related_items.ini',
            '/language/en-GB/en-GB.mod_related_items.sys.ini',
            '/language/en-GB/en-GB.mod_stats.ini',
            '/language/en-GB/en-GB.mod_stats.sys.ini',
            '/language/en-GB/en-GB.mod_syndicate.ini',
            '/language/en-GB/en-GB.mod_syndicate.sys.ini',
            '/language/en-GB/en-GB.mod_tags_popular.ini',
            '/language/en-GB/en-GB.mod_tags_popular.sys.ini',
            '/language/en-GB/en-GB.mod_tags_similar.ini',
            '/language/en-GB/en-GB.mod_tags_similar.sys.ini',
            '/language/en-GB/en-GB.mod_users_latest.ini',
            '/language/en-GB/en-GB.mod_users_latest.sys.ini',
            '/language/en-GB/en-GB.mod_weblinks.ini',
            '/language/en-GB/en-GB.mod_weblinks.sys.ini',
            '/language/en-GB/en-GB.mod_whosonline.ini',
            '/language/en-GB/en-GB.mod_whosonline.sys.ini',
            '/language/en-GB/en-GB.mod_wrapper.ini',
            '/language/en-GB/en-GB.mod_wrapper.sys.ini',
            '/language/en-GB/en-GB.tpl_beez3.ini',
            '/language/en-GB/en-GB.tpl_beez3.sys.ini',
            '/language/en-GB/en-GB.tpl_protostar.ini',
            '/language/en-GB/en-GB.tpl_protostar.sys.ini',
            '/language/en-GB/en-GB.xml',
            '/layouts/joomla/content/blog_style_default_links.php',
            '/layouts/joomla/content/icons/email.php',
            '/layouts/joomla/content/icons/print_popup.php',
            '/layouts/joomla/content/icons/print_screen.php',
            '/layouts/joomla/content/info_block/block.php',
            '/layouts/joomla/edit/details.php',
            '/layouts/joomla/edit/item_title.php',
            '/layouts/joomla/form/field/radio.php',
            '/layouts/joomla/html/formbehavior/ajaxchosen.php',
            '/layouts/joomla/html/formbehavior/chosen.php',
            '/layouts/joomla/html/sortablelist.php',
            '/layouts/joomla/html/tag.php',
            '/layouts/joomla/modal/body.php',
            '/layouts/joomla/modal/footer.php',
            '/layouts/joomla/modal/header.php',
            '/layouts/joomla/modal/iframe.php',
            '/layouts/joomla/modal/main.php',
            '/layouts/joomla/sidebars/toggle.php',
            '/layouts/joomla/tinymce/buttons.php',
            '/layouts/joomla/tinymce/buttons/button.php',
            '/layouts/joomla/toolbar/confirm.php',
            '/layouts/joomla/toolbar/help.php',
            '/layouts/joomla/toolbar/modal.php',
            '/layouts/joomla/toolbar/slider.php',
            '/layouts/libraries/cms/html/bootstrap/addtab.php',
            '/layouts/libraries/cms/html/bootstrap/addtabscript.php',
            '/layouts/libraries/cms/html/bootstrap/endtab.php',
            '/layouts/libraries/cms/html/bootstrap/endtabset.php',
            '/layouts/libraries/cms/html/bootstrap/starttabset.php',
            '/layouts/libraries/cms/html/bootstrap/starttabsetscript.php',
            '/libraries/cms/class/loader.php',
            '/libraries/cms/html/access.php',
            '/libraries/cms/html/actionsdropdown.php',
            '/libraries/cms/html/adminlanguage.php',
            '/libraries/cms/html/batch.php',
            '/libraries/cms/html/behavior.php',
            '/libraries/cms/html/bootstrap.php',
            '/libraries/cms/html/category.php',
            '/libraries/cms/html/content.php',
            '/libraries/cms/html/contentlanguage.php',
            '/libraries/cms/html/date.php',
            '/libraries/cms/html/debug.php',
            '/libraries/cms/html/dropdown.php',
            '/libraries/cms/html/email.php',
            '/libraries/cms/html/form.php',
            '/libraries/cms/html/formbehavior.php',
            '/libraries/cms/html/grid.php',
            '/libraries/cms/html/icons.php',
            '/libraries/cms/html/jgrid.php',
            '/libraries/cms/html/jquery.php',
            '/libraries/cms/html/language/en-GB/en-GB.jhtmldate.ini',
            '/libraries/cms/html/links.php',
            '/libraries/cms/html/list.php',
            '/libraries/cms/html/menu.php',
            '/libraries/cms/html/number.php',
            '/libraries/cms/html/rules.php',
            '/libraries/cms/html/searchtools.php',
            '/libraries/cms/html/select.php',
            '/libraries/cms/html/sidebar.php',
            '/libraries/cms/html/sliders.php',
            '/libraries/cms/html/sortablelist.php',
            '/libraries/cms/html/string.php',
            '/libraries/cms/html/tabs.php',
            '/libraries/cms/html/tag.php',
            '/libraries/cms/html/tel.php',
            '/libraries/cms/html/user.php',
            '/libraries/cms/less/formatter/joomla.php',
            '/libraries/cms/less/less.php',
            '/libraries/fof/LICENSE.txt',
            '/libraries/fof/autoloader/component.php',
            '/libraries/fof/autoloader/fof.php',
            '/libraries/fof/config/domain/dispatcher.php',
            '/libraries/fof/config/domain/interface.php',
            '/libraries/fof/config/domain/tables.php',
            '/libraries/fof/config/domain/views.php',
            '/libraries/fof/config/provider.php',
            '/libraries/fof/controller/controller.php',
            '/libraries/fof/database/database.php',
            '/libraries/fof/database/driver.php',
            '/libraries/fof/database/driver/joomla.php',
            '/libraries/fof/database/driver/mysql.php',
            '/libraries/fof/database/driver/mysqli.php',
            '/libraries/fof/database/driver/oracle.php',
            '/libraries/fof/database/driver/pdo.php',
            '/libraries/fof/database/driver/pdomysql.php',
            '/libraries/fof/database/driver/postgresql.php',
            '/libraries/fof/database/driver/sqlazure.php',
            '/libraries/fof/database/driver/sqlite.php',
            '/libraries/fof/database/driver/sqlsrv.php',
            '/libraries/fof/database/factory.php',
            '/libraries/fof/database/installer.php',
            '/libraries/fof/database/interface.php',
            '/libraries/fof/database/iterator.php',
            '/libraries/fof/database/iterator/azure.php',
            '/libraries/fof/database/iterator/mysql.php',
            '/libraries/fof/database/iterator/mysqli.php',
            '/libraries/fof/database/iterator/oracle.php',
            '/libraries/fof/database/iterator/pdo.php',
            '/libraries/fof/database/iterator/pdomysql.php',
            '/libraries/fof/database/iterator/postgresql.php',
            '/libraries/fof/database/iterator/sqlite.php',
            '/libraries/fof/database/iterator/sqlsrv.php',
            '/libraries/fof/database/query.php',
            '/libraries/fof/database/query/element.php',
            '/libraries/fof/database/query/limitable.php',
            '/libraries/fof/database/query/mysql.php',
            '/libraries/fof/database/query/mysqli.php',
            '/libraries/fof/database/query/oracle.php',
            '/libraries/fof/database/query/pdo.php',
            '/libraries/fof/database/query/pdomysql.php',
            '/libraries/fof/database/query/postgresql.php',
            '/libraries/fof/database/query/preparable.php',
            '/libraries/fof/database/query/sqlazure.php',
            '/libraries/fof/database/query/sqlite.php',
            '/libraries/fof/database/query/sqlsrv.php',
            '/libraries/fof/dispatcher/dispatcher.php',
            '/libraries/fof/download/adapter/abstract.php',
            '/libraries/fof/download/adapter/cacert.pem',
            '/libraries/fof/download/adapter/curl.php',
            '/libraries/fof/download/adapter/fopen.php',
            '/libraries/fof/download/download.php',
            '/libraries/fof/download/interface.php',
            '/libraries/fof/encrypt/aes.php',
            '/libraries/fof/encrypt/aes/abstract.php',
            '/libraries/fof/encrypt/aes/interface.php',
            '/libraries/fof/encrypt/aes/mcrypt.php',
            '/libraries/fof/encrypt/aes/openssl.php',
            '/libraries/fof/encrypt/base32.php',
            '/libraries/fof/encrypt/randval.php',
            '/libraries/fof/encrypt/randvalinterface.php',
            '/libraries/fof/encrypt/totp.php',
            '/libraries/fof/form/field.php',
            '/libraries/fof/form/field/accesslevel.php',
            '/libraries/fof/form/field/actions.php',
            '/libraries/fof/form/field/button.php',
            '/libraries/fof/form/field/cachehandler.php',
            '/libraries/fof/form/field/calendar.php',
            '/libraries/fof/form/field/captcha.php',
            '/libraries/fof/form/field/checkbox.php',
            '/libraries/fof/form/field/checkboxes.php',
            '/libraries/fof/form/field/components.php',
            '/libraries/fof/form/field/editor.php',
            '/libraries/fof/form/field/email.php',
            '/libraries/fof/form/field/groupedbutton.php',
            '/libraries/fof/form/field/groupedlist.php',
            '/libraries/fof/form/field/hidden.php',
            '/libraries/fof/form/field/image.php',
            '/libraries/fof/form/field/imagelist.php',
            '/libraries/fof/form/field/integer.php',
            '/libraries/fof/form/field/language.php',
            '/libraries/fof/form/field/list.php',
            '/libraries/fof/form/field/media.php',
            '/libraries/fof/form/field/model.php',
            '/libraries/fof/form/field/ordering.php',
            '/libraries/fof/form/field/password.php',
            '/libraries/fof/form/field/plugins.php',
            '/libraries/fof/form/field/published.php',
            '/libraries/fof/form/field/radio.php',
            '/libraries/fof/form/field/relation.php',
            '/libraries/fof/form/field/rules.php',
            '/libraries/fof/form/field/selectrow.php',
            '/libraries/fof/form/field/sessionhandler.php',
            '/libraries/fof/form/field/spacer.php',
            '/libraries/fof/form/field/sql.php',
            '/libraries/fof/form/field/tag.php',
            '/libraries/fof/form/field/tel.php',
            '/libraries/fof/form/field/text.php',
            '/libraries/fof/form/field/textarea.php',
            '/libraries/fof/form/field/timezone.php',
            '/libraries/fof/form/field/title.php',
            '/libraries/fof/form/field/url.php',
            '/libraries/fof/form/field/user.php',
            '/libraries/fof/form/field/usergroup.php',
            '/libraries/fof/form/form.php',
            '/libraries/fof/form/header.php',
            '/libraries/fof/form/header/accesslevel.php',
            '/libraries/fof/form/header/field.php',
            '/libraries/fof/form/header/fielddate.php',
            '/libraries/fof/form/header/fieldfilterable.php',
            '/libraries/fof/form/header/fieldsearchable.php',
            '/libraries/fof/form/header/fieldselectable.php',
            '/libraries/fof/form/header/fieldsql.php',
            '/libraries/fof/form/header/filterdate.php',
            '/libraries/fof/form/header/filterfilterable.php',
            '/libraries/fof/form/header/filtersearchable.php',
            '/libraries/fof/form/header/filterselectable.php',
            '/libraries/fof/form/header/filtersql.php',
            '/libraries/fof/form/header/language.php',
            '/libraries/fof/form/header/model.php',
            '/libraries/fof/form/header/ordering.php',
            '/libraries/fof/form/header/published.php',
            '/libraries/fof/form/header/rowselect.php',
            '/libraries/fof/form/helper.php',
            '/libraries/fof/hal/document.php',
            '/libraries/fof/hal/link.php',
            '/libraries/fof/hal/links.php',
            '/libraries/fof/hal/render/interface.php',
            '/libraries/fof/hal/render/json.php',
            '/libraries/fof/include.php',
            '/libraries/fof/inflector/inflector.php',
            '/libraries/fof/input/input.php',
            '/libraries/fof/input/jinput/cli.php',
            '/libraries/fof/input/jinput/cookie.php',
            '/libraries/fof/input/jinput/files.php',
            '/libraries/fof/input/jinput/input.php',
            '/libraries/fof/input/jinput/json.php',
            '/libraries/fof/integration/joomla/filesystem/filesystem.php',
            '/libraries/fof/integration/joomla/platform.php',
            '/libraries/fof/layout/file.php',
            '/libraries/fof/layout/helper.php',
            '/libraries/fof/less/formatter/classic.php',
            '/libraries/fof/less/formatter/compressed.php',
            '/libraries/fof/less/formatter/joomla.php',
            '/libraries/fof/less/formatter/lessjs.php',
            '/libraries/fof/less/less.php',
            '/libraries/fof/less/parser/parser.php',
            '/libraries/fof/model/behavior.php',
            '/libraries/fof/model/behavior/access.php',
            '/libraries/fof/model/behavior/emptynonzero.php',
            '/libraries/fof/model/behavior/enabled.php',
            '/libraries/fof/model/behavior/filters.php',
            '/libraries/fof/model/behavior/language.php',
            '/libraries/fof/model/behavior/private.php',
            '/libraries/fof/model/dispatcher/behavior.php',
            '/libraries/fof/model/field.php',
            '/libraries/fof/model/field/boolean.php',
            '/libraries/fof/model/field/date.php',
            '/libraries/fof/model/field/number.php',
            '/libraries/fof/model/field/text.php',
            '/libraries/fof/model/model.php',
            '/libraries/fof/platform/filesystem/filesystem.php',
            '/libraries/fof/platform/filesystem/interface.php',
            '/libraries/fof/platform/interface.php',
            '/libraries/fof/platform/platform.php',
            '/libraries/fof/query/abstract.php',
            '/libraries/fof/render/abstract.php',
            '/libraries/fof/render/joomla.php',
            '/libraries/fof/render/joomla3.php',
            '/libraries/fof/render/strapper.php',
            '/libraries/fof/string/utils.php',
            '/libraries/fof/table/behavior.php',
            '/libraries/fof/table/behavior/assets.php',
            '/libraries/fof/table/behavior/contenthistory.php',
            '/libraries/fof/table/behavior/tags.php',
            '/libraries/fof/table/dispatcher/behavior.php',
            '/libraries/fof/table/nested.php',
            '/libraries/fof/table/relations.php',
            '/libraries/fof/table/table.php',
            '/libraries/fof/template/utils.php',
            '/libraries/fof/toolbar/toolbar.php',
            '/libraries/fof/utils/array/array.php',
            '/libraries/fof/utils/cache/cleaner.php',
            '/libraries/fof/utils/config/helper.php',
            '/libraries/fof/utils/filescheck/filescheck.php',
            '/libraries/fof/utils/ini/parser.php',
            '/libraries/fof/utils/installscript/installscript.php',
            '/libraries/fof/utils/ip/ip.php',
            '/libraries/fof/utils/object/object.php',
            '/libraries/fof/utils/observable/dispatcher.php',
            '/libraries/fof/utils/observable/event.php',
            '/libraries/fof/utils/phpfunc/phpfunc.php',
            '/libraries/fof/utils/timer/timer.php',
            '/libraries/fof/utils/update/collection.php',
            '/libraries/fof/utils/update/extension.php',
            '/libraries/fof/utils/update/joomla.php',
            '/libraries/fof/utils/update/update.php',
            '/libraries/fof/version.txt',
            '/libraries/fof/view/csv.php',
            '/libraries/fof/view/form.php',
            '/libraries/fof/view/html.php',
            '/libraries/fof/view/json.php',
            '/libraries/fof/view/raw.php',
            '/libraries/fof/view/view.php',
            '/libraries/idna_convert/LICENCE',
            '/libraries/idna_convert/ReadMe.txt',
            '/libraries/idna_convert/idna_convert.class.php',
            '/libraries/idna_convert/transcode_wrapper.php',
            '/libraries/idna_convert/uctc.php',
            '/libraries/joomla/application/web/router.php',
            '/libraries/joomla/application/web/router/base.php',
            '/libraries/joomla/application/web/router/rest.php',
            '/libraries/joomla/archive/archive.php',
            '/libraries/joomla/archive/bzip2.php',
            '/libraries/joomla/archive/extractable.php',
            '/libraries/joomla/archive/gzip.php',
            '/libraries/joomla/archive/tar.php',
            '/libraries/joomla/archive/wrapper/archive.php',
            '/libraries/joomla/archive/zip.php',
            '/libraries/joomla/controller/base.php',
            '/libraries/joomla/controller/controller.php',
            '/libraries/joomla/database/database.php',
            '/libraries/joomla/database/driver.php',
            '/libraries/joomla/database/driver/mysql.php',
            '/libraries/joomla/database/driver/mysqli.php',
            '/libraries/joomla/database/driver/oracle.php',
            '/libraries/joomla/database/driver/pdo.php',
            '/libraries/joomla/database/driver/pdomysql.php',
            '/libraries/joomla/database/driver/pgsql.php',
            '/libraries/joomla/database/driver/postgresql.php',
            '/libraries/joomla/database/driver/sqlazure.php',
            '/libraries/joomla/database/driver/sqlite.php',
            '/libraries/joomla/database/driver/sqlsrv.php',
            '/libraries/joomla/database/exception/connecting.php',
            '/libraries/joomla/database/exception/executing.php',
            '/libraries/joomla/database/exception/unsupported.php',
            '/libraries/joomla/database/exporter.php',
            '/libraries/joomla/database/exporter/mysql.php',
            '/libraries/joomla/database/exporter/mysqli.php',
            '/libraries/joomla/database/exporter/pdomysql.php',
            '/libraries/joomla/database/exporter/pgsql.php',
            '/libraries/joomla/database/exporter/postgresql.php',
            '/libraries/joomla/database/factory.php',
            '/libraries/joomla/database/importer.php',
            '/libraries/joomla/database/importer/mysql.php',
            '/libraries/joomla/database/importer/mysqli.php',
            '/libraries/joomla/database/importer/pdomysql.php',
            '/libraries/joomla/database/importer/pgsql.php',
            '/libraries/joomla/database/importer/postgresql.php',
            '/libraries/joomla/database/interface.php',
            '/libraries/joomla/database/iterator.php',
            '/libraries/joomla/database/iterator/mysql.php',
            '/libraries/joomla/database/iterator/mysqli.php',
            '/libraries/joomla/database/iterator/oracle.php',
            '/libraries/joomla/database/iterator/pdo.php',
            '/libraries/joomla/database/iterator/pdomysql.php',
            '/libraries/joomla/database/iterator/pgsql.php',
            '/libraries/joomla/database/iterator/postgresql.php',
            '/libraries/joomla/database/iterator/sqlazure.php',
            '/libraries/joomla/database/iterator/sqlite.php',
            '/libraries/joomla/database/iterator/sqlsrv.php',
            '/libraries/joomla/database/query.php',
            '/libraries/joomla/database/query/element.php',
            '/libraries/joomla/database/query/limitable.php',
            '/libraries/joomla/database/query/mysql.php',
            '/libraries/joomla/database/query/mysqli.php',
            '/libraries/joomla/database/query/oracle.php',
            '/libraries/joomla/database/query/pdo.php',
            '/libraries/joomla/database/query/pdomysql.php',
            '/libraries/joomla/database/query/pgsql.php',
            '/libraries/joomla/database/query/postgresql.php',
            '/libraries/joomla/database/query/preparable.php',
            '/libraries/joomla/database/query/sqlazure.php',
            '/libraries/joomla/database/query/sqlite.php',
            '/libraries/joomla/database/query/sqlsrv.php',
            '/libraries/joomla/event/dispatcher.php',
            '/libraries/joomla/event/event.php',
            '/libraries/joomla/facebook/album.php',
            '/libraries/joomla/facebook/checkin.php',
            '/libraries/joomla/facebook/comment.php',
            '/libraries/joomla/facebook/event.php',
            '/libraries/joomla/facebook/facebook.php',
            '/libraries/joomla/facebook/group.php',
            '/libraries/joomla/facebook/link.php',
            '/libraries/joomla/facebook/note.php',
            '/libraries/joomla/facebook/oauth.php',
            '/libraries/joomla/facebook/object.php',
            '/libraries/joomla/facebook/photo.php',
            '/libraries/joomla/facebook/post.php',
            '/libraries/joomla/facebook/status.php',
            '/libraries/joomla/facebook/user.php',
            '/libraries/joomla/facebook/video.php',
            '/libraries/joomla/form/fields/accesslevel.php',
            '/libraries/joomla/form/fields/aliastag.php',
            '/libraries/joomla/form/fields/cachehandler.php',
            '/libraries/joomla/form/fields/calendar.php',
            '/libraries/joomla/form/fields/checkbox.php',
            '/libraries/joomla/form/fields/checkboxes.php',
            '/libraries/joomla/form/fields/color.php',
            '/libraries/joomla/form/fields/combo.php',
            '/libraries/joomla/form/fields/components.php',
            '/libraries/joomla/form/fields/databaseconnection.php',
            '/libraries/joomla/form/fields/email.php',
            '/libraries/joomla/form/fields/file.php',
            '/libraries/joomla/form/fields/filelist.php',
            '/libraries/joomla/form/fields/folderlist.php',
            '/libraries/joomla/form/fields/groupedlist.php',
            '/libraries/joomla/form/fields/hidden.php',
            '/libraries/joomla/form/fields/imagelist.php',
            '/libraries/joomla/form/fields/integer.php',
            '/libraries/joomla/form/fields/language.php',
            '/libraries/joomla/form/fields/list.php',
            '/libraries/joomla/form/fields/meter.php',
            '/libraries/joomla/form/fields/note.php',
            '/libraries/joomla/form/fields/number.php',
            '/libraries/joomla/form/fields/password.php',
            '/libraries/joomla/form/fields/plugins.php',
            '/libraries/joomla/form/fields/predefinedlist.php',
            '/libraries/joomla/form/fields/radio.php',
            '/libraries/joomla/form/fields/range.php',
            '/libraries/joomla/form/fields/repeatable.php',
            '/libraries/joomla/form/fields/rules.php',
            '/libraries/joomla/form/fields/sessionhandler.php',
            '/libraries/joomla/form/fields/spacer.php',
            '/libraries/joomla/form/fields/sql.php',
            '/libraries/joomla/form/fields/subform.php',
            '/libraries/joomla/form/fields/tel.php',
            '/libraries/joomla/form/fields/text.php',
            '/libraries/joomla/form/fields/textarea.php',
            '/libraries/joomla/form/fields/timezone.php',
            '/libraries/joomla/form/fields/url.php',
            '/libraries/joomla/form/fields/usergroup.php',
            '/libraries/joomla/github/account.php',
            '/libraries/joomla/github/commits.php',
            '/libraries/joomla/github/forks.php',
            '/libraries/joomla/github/github.php',
            '/libraries/joomla/github/hooks.php',
            '/libraries/joomla/github/http.php',
            '/libraries/joomla/github/meta.php',
            '/libraries/joomla/github/milestones.php',
            '/libraries/joomla/github/object.php',
            '/libraries/joomla/github/package.php',
            '/libraries/joomla/github/package/activity.php',
            '/libraries/joomla/github/package/activity/events.php',
            '/libraries/joomla/github/package/activity/notifications.php',
            '/libraries/joomla/github/package/activity/starring.php',
            '/libraries/joomla/github/package/activity/watching.php',
            '/libraries/joomla/github/package/authorization.php',
            '/libraries/joomla/github/package/data.php',
            '/libraries/joomla/github/package/data/blobs.php',
            '/libraries/joomla/github/package/data/commits.php',
            '/libraries/joomla/github/package/data/refs.php',
            '/libraries/joomla/github/package/data/tags.php',
            '/libraries/joomla/github/package/data/trees.php',
            '/libraries/joomla/github/package/gists.php',
            '/libraries/joomla/github/package/gists/comments.php',
            '/libraries/joomla/github/package/gitignore.php',
            '/libraries/joomla/github/package/issues.php',
            '/libraries/joomla/github/package/issues/assignees.php',
            '/libraries/joomla/github/package/issues/comments.php',
            '/libraries/joomla/github/package/issues/events.php',
            '/libraries/joomla/github/package/issues/labels.php',
            '/libraries/joomla/github/package/issues/milestones.php',
            '/libraries/joomla/github/package/markdown.php',
            '/libraries/joomla/github/package/orgs.php',
            '/libraries/joomla/github/package/orgs/members.php',
            '/libraries/joomla/github/package/orgs/teams.php',
            '/libraries/joomla/github/package/pulls.php',
            '/libraries/joomla/github/package/pulls/comments.php',
            '/libraries/joomla/github/package/repositories.php',
            '/libraries/joomla/github/package/repositories/collaborators.php',
            '/libraries/joomla/github/package/repositories/comments.php',
            '/libraries/joomla/github/package/repositories/commits.php',
            '/libraries/joomla/github/package/repositories/contents.php',
            '/libraries/joomla/github/package/repositories/downloads.php',
            '/libraries/joomla/github/package/repositories/forks.php',
            '/libraries/joomla/github/package/repositories/hooks.php',
            '/libraries/joomla/github/package/repositories/keys.php',
            '/libraries/joomla/github/package/repositories/merging.php',
            '/libraries/joomla/github/package/repositories/statistics.php',
            '/libraries/joomla/github/package/repositories/statuses.php',
            '/libraries/joomla/github/package/search.php',
            '/libraries/joomla/github/package/users.php',
            '/libraries/joomla/github/package/users/emails.php',
            '/libraries/joomla/github/package/users/followers.php',
            '/libraries/joomla/github/package/users/keys.php',
            '/libraries/joomla/github/refs.php',
            '/libraries/joomla/github/statuses.php',
            '/libraries/joomla/google/auth.php',
            '/libraries/joomla/google/auth/oauth2.php',
            '/libraries/joomla/google/data.php',
            '/libraries/joomla/google/data/adsense.php',
            '/libraries/joomla/google/data/calendar.php',
            '/libraries/joomla/google/data/picasa.php',
            '/libraries/joomla/google/data/picasa/album.php',
            '/libraries/joomla/google/data/picasa/photo.php',
            '/libraries/joomla/google/data/plus.php',
            '/libraries/joomla/google/data/plus/activities.php',
            '/libraries/joomla/google/data/plus/comments.php',
            '/libraries/joomla/google/data/plus/people.php',
            '/libraries/joomla/google/embed.php',
            '/libraries/joomla/google/embed/analytics.php',
            '/libraries/joomla/google/embed/maps.php',
            '/libraries/joomla/google/google.php',
            '/libraries/joomla/grid/grid.php',
            '/libraries/joomla/keychain/keychain.php',
            '/libraries/joomla/linkedin/communications.php',
            '/libraries/joomla/linkedin/companies.php',
            '/libraries/joomla/linkedin/groups.php',
            '/libraries/joomla/linkedin/jobs.php',
            '/libraries/joomla/linkedin/linkedin.php',
            '/libraries/joomla/linkedin/oauth.php',
            '/libraries/joomla/linkedin/object.php',
            '/libraries/joomla/linkedin/people.php',
            '/libraries/joomla/linkedin/stream.php',
            '/libraries/joomla/mediawiki/categories.php',
            '/libraries/joomla/mediawiki/http.php',
            '/libraries/joomla/mediawiki/images.php',
            '/libraries/joomla/mediawiki/links.php',
            '/libraries/joomla/mediawiki/mediawiki.php',
            '/libraries/joomla/mediawiki/object.php',
            '/libraries/joomla/mediawiki/pages.php',
            '/libraries/joomla/mediawiki/search.php',
            '/libraries/joomla/mediawiki/sites.php',
            '/libraries/joomla/mediawiki/users.php',
            '/libraries/joomla/model/base.php',
            '/libraries/joomla/model/database.php',
            '/libraries/joomla/model/model.php',
            '/libraries/joomla/oauth1/client.php',
            '/libraries/joomla/oauth2/client.php',
            '/libraries/joomla/observable/interface.php',
            '/libraries/joomla/observer/interface.php',
            '/libraries/joomla/observer/mapper.php',
            '/libraries/joomla/observer/updater.php',
            '/libraries/joomla/observer/updater/interface.php',
            '/libraries/joomla/observer/wrapper/mapper.php',
            '/libraries/joomla/openstreetmap/changesets.php',
            '/libraries/joomla/openstreetmap/elements.php',
            '/libraries/joomla/openstreetmap/gps.php',
            '/libraries/joomla/openstreetmap/info.php',
            '/libraries/joomla/openstreetmap/oauth.php',
            '/libraries/joomla/openstreetmap/object.php',
            '/libraries/joomla/openstreetmap/openstreetmap.php',
            '/libraries/joomla/openstreetmap/user.php',
            '/libraries/joomla/platform.php',
            '/libraries/joomla/route/wrapper/route.php',
            '/libraries/joomla/session/handler/interface.php',
            '/libraries/joomla/session/handler/joomla.php',
            '/libraries/joomla/session/handler/native.php',
            '/libraries/joomla/session/storage.php',
            '/libraries/joomla/session/storage/apc.php',
            '/libraries/joomla/session/storage/apcu.php',
            '/libraries/joomla/session/storage/database.php',
            '/libraries/joomla/session/storage/memcache.php',
            '/libraries/joomla/session/storage/memcached.php',
            '/libraries/joomla/session/storage/none.php',
            '/libraries/joomla/session/storage/redis.php',
            '/libraries/joomla/session/storage/wincache.php',
            '/libraries/joomla/session/storage/xcache.php',
            '/libraries/joomla/string/string.php',
            '/libraries/joomla/string/wrapper/normalise.php',
            '/libraries/joomla/string/wrapper/punycode.php',
            '/libraries/joomla/twitter/block.php',
            '/libraries/joomla/twitter/directmessages.php',
            '/libraries/joomla/twitter/favorites.php',
            '/libraries/joomla/twitter/friends.php',
            '/libraries/joomla/twitter/help.php',
            '/libraries/joomla/twitter/lists.php',
            '/libraries/joomla/twitter/oauth.php',
            '/libraries/joomla/twitter/object.php',
            '/libraries/joomla/twitter/places.php',
            '/libraries/joomla/twitter/profile.php',
            '/libraries/joomla/twitter/search.php',
            '/libraries/joomla/twitter/statuses.php',
            '/libraries/joomla/twitter/trends.php',
            '/libraries/joomla/twitter/twitter.php',
            '/libraries/joomla/twitter/users.php',
            '/libraries/joomla/utilities/arrayhelper.php',
            '/libraries/joomla/view/base.php',
            '/libraries/joomla/view/html.php',
            '/libraries/joomla/view/view.php',
            '/libraries/legacy/application/application.php',
            '/libraries/legacy/base/node.php',
            '/libraries/legacy/base/observable.php',
            '/libraries/legacy/base/observer.php',
            '/libraries/legacy/base/tree.php',
            '/libraries/legacy/database/exception.php',
            '/libraries/legacy/database/mysql.php',
            '/libraries/legacy/database/mysqli.php',
            '/libraries/legacy/database/sqlazure.php',
            '/libraries/legacy/database/sqlsrv.php',
            '/libraries/legacy/dispatcher/dispatcher.php',
            '/libraries/legacy/error/error.php',
            '/libraries/legacy/exception/exception.php',
            '/libraries/legacy/form/field/category.php',
            '/libraries/legacy/form/field/componentlayout.php',
            '/libraries/legacy/form/field/modulelayout.php',
            '/libraries/legacy/log/logexception.php',
            '/libraries/legacy/request/request.php',
            '/libraries/legacy/response/response.php',
            '/libraries/legacy/simplecrypt/simplecrypt.php',
            '/libraries/legacy/simplepie/factory.php',
            '/libraries/legacy/table/session.php',
            '/libraries/legacy/utilities/xmlelement.php',
            '/libraries/phputf8/LICENSE',
            '/libraries/phputf8/README',
            '/libraries/phputf8/mbstring/core.php',
            '/libraries/phputf8/native/core.php',
            '/libraries/phputf8/ord.php',
            '/libraries/phputf8/str_ireplace.php',
            '/libraries/phputf8/str_pad.php',
            '/libraries/phputf8/str_split.php',
            '/libraries/phputf8/strcasecmp.php',
            '/libraries/phputf8/strcspn.php',
            '/libraries/phputf8/stristr.php',
            '/libraries/phputf8/strrev.php',
            '/libraries/phputf8/strspn.php',
            '/libraries/phputf8/substr_replace.php',
            '/libraries/phputf8/trim.php',
            '/libraries/phputf8/ucfirst.php',
            '/libraries/phputf8/ucwords.php',
            '/libraries/phputf8/utf8.php',
            '/libraries/phputf8/utils/ascii.php',
            '/libraries/phputf8/utils/bad.php',
            '/libraries/phputf8/utils/patterns.php',
            '/libraries/phputf8/utils/position.php',
            '/libraries/phputf8/utils/specials.php',
            '/libraries/phputf8/utils/unicode.php',
            '/libraries/phputf8/utils/validation.php',
            '/libraries/src/Access/Wrapper/Access.php',
            '/libraries/src/Cache/Storage/ApcStorage.php',
            '/libraries/src/Cache/Storage/CacheliteStorage.php',
            '/libraries/src/Cache/Storage/MemcacheStorage.php',
            '/libraries/src/Cache/Storage/XcacheStorage.php',
            '/libraries/src/Client/ClientWrapper.php',
            '/libraries/src/Crypt/Cipher/BlowfishCipher.php',
            '/libraries/src/Crypt/Cipher/McryptCipher.php',
            '/libraries/src/Crypt/Cipher/Rijndael256Cipher.php',
            '/libraries/src/Crypt/Cipher/SimpleCipher.php',
            '/libraries/src/Crypt/Cipher/TripleDesCipher.php',
            '/libraries/src/Crypt/CipherInterface.php',
            '/libraries/src/Crypt/CryptPassword.php',
            '/libraries/src/Crypt/Key.php',
            '/libraries/src/Crypt/Password/SimpleCryptPassword.php',
            '/libraries/src/Crypt/README.md',
            '/libraries/src/Filesystem/Wrapper/FileWrapper.php',
            '/libraries/src/Filesystem/Wrapper/FolderWrapper.php',
            '/libraries/src/Filesystem/Wrapper/PathWrapper.php',
            '/libraries/src/Filter/Wrapper/OutputFilterWrapper.php',
            '/libraries/src/Form/Field/HelpsiteField.php',
            '/libraries/src/Form/FormWrapper.php',
            '/libraries/src/Helper/ContentHistoryHelper.php',
            '/libraries/src/Helper/SearchHelper.php',
            '/libraries/src/Http/Transport/cacert.pem',
            '/libraries/src/Http/Wrapper/FactoryWrapper.php',
            '/libraries/src/Language/LanguageStemmer.php',
            '/libraries/src/Language/Stemmer/Porteren.php',
            '/libraries/src/Language/Wrapper/JTextWrapper.php',
            '/libraries/src/Language/Wrapper/LanguageHelperWrapper.php',
            '/libraries/src/Language/Wrapper/TransliterateWrapper.php',
            '/libraries/src/Mail/MailWrapper.php',
            '/libraries/src/Menu/MenuHelper.php',
            '/libraries/src/Menu/Node.php',
            '/libraries/src/Menu/Node/Component.php',
            '/libraries/src/Menu/Node/Container.php',
            '/libraries/src/Menu/Node/Heading.php',
            '/libraries/src/Menu/Node/Separator.php',
            '/libraries/src/Menu/Node/Url.php',
            '/libraries/src/Menu/Tree.php',
            '/libraries/src/Table/Observer/AbstractObserver.php',
            '/libraries/src/Table/Observer/ContentHistory.php',
            '/libraries/src/Table/Observer/Tags.php',
            '/libraries/src/Toolbar/Button/SliderButton.php',
            '/libraries/src/User/UserWrapper.php',
            '/libraries/vendor/.htaccess',
            '/libraries/vendor/brumann/polyfill-unserialize/LICENSE',
            '/libraries/vendor/brumann/polyfill-unserialize/composer.json',
            '/libraries/vendor/brumann/polyfill-unserialize/src/DisallowedClassesSubstitutor.php',
            '/libraries/vendor/brumann/polyfill-unserialize/src/Unserialize.php',
            '/libraries/vendor/ircmaxell/password-compat/LICENSE.md',
            '/libraries/vendor/ircmaxell/password-compat/lib/password.php',
            '/libraries/vendor/joomla/application/src/AbstractCliApplication.php',
            '/libraries/vendor/joomla/application/src/AbstractDaemonApplication.php',
            '/libraries/vendor/joomla/application/src/Cli/CliInput.php',
            '/libraries/vendor/joomla/application/src/Cli/CliOutput.php',
            '/libraries/vendor/joomla/application/src/Cli/ColorProcessor.php',
            '/libraries/vendor/joomla/application/src/Cli/ColorStyle.php',
            '/libraries/vendor/joomla/application/src/Cli/Output/Processor/ColorProcessor.php',
            '/libraries/vendor/joomla/application/src/Cli/Output/Processor/ProcessorInterface.php',
            '/libraries/vendor/joomla/application/src/Cli/Output/Stdout.php',
            '/libraries/vendor/joomla/application/src/Cli/Output/Xml.php',
            '/libraries/vendor/joomla/compat/LICENSE',
            '/libraries/vendor/joomla/compat/src/CallbackFilterIterator.php',
            '/libraries/vendor/joomla/compat/src/JsonSerializable.php',
            '/libraries/vendor/joomla/event/src/DelegatingDispatcher.php',
            '/libraries/vendor/joomla/filesystem/src/Stream/String.php',
            '/libraries/vendor/joomla/image/LICENSE',
            '/libraries/vendor/joomla/image/src/Filter/Backgroundfill.php',
            '/libraries/vendor/joomla/image/src/Filter/Brightness.php',
            '/libraries/vendor/joomla/image/src/Filter/Contrast.php',
            '/libraries/vendor/joomla/image/src/Filter/Edgedetect.php',
            '/libraries/vendor/joomla/image/src/Filter/Emboss.php',
            '/libraries/vendor/joomla/image/src/Filter/Grayscale.php',
            '/libraries/vendor/joomla/image/src/Filter/Negate.php',
            '/libraries/vendor/joomla/image/src/Filter/Sketchy.php',
            '/libraries/vendor/joomla/image/src/Filter/Smooth.php',
            '/libraries/vendor/joomla/image/src/Image.php',
            '/libraries/vendor/joomla/image/src/ImageFilter.php',
            '/libraries/vendor/joomla/input/src/Cli.php',
            '/libraries/vendor/joomla/registry/src/AbstractRegistryFormat.php',
            '/libraries/vendor/joomla/session/Joomla/Session/LICENSE',
            '/libraries/vendor/joomla/session/Joomla/Session/Session.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Apc.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Apcu.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Database.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Memcache.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Memcached.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/None.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Wincache.php',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage/Xcache.php',
            '/libraries/vendor/joomla/string/src/String.php',
            '/libraries/vendor/leafo/lessphp/LICENSE',
            '/libraries/vendor/leafo/lessphp/lessc.inc.php',
            '/libraries/vendor/leafo/lessphp/lessify',
            '/libraries/vendor/leafo/lessphp/lessify.inc.php',
            '/libraries/vendor/leafo/lessphp/plessc',
            '/libraries/vendor/paragonie/random_compat/LICENSE',
            '/libraries/vendor/paragonie/random_compat/lib/byte_safe_strings.php',
            '/libraries/vendor/paragonie/random_compat/lib/cast_to_int.php',
            '/libraries/vendor/paragonie/random_compat/lib/error_polyfill.php',
            '/libraries/vendor/paragonie/random_compat/lib/random.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_bytes_com_dotnet.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_bytes_dev_urandom.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_bytes_libsodium.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_bytes_libsodium_legacy.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_bytes_mcrypt.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_bytes_openssl.php',
            '/libraries/vendor/paragonie/random_compat/lib/random_int.php',
            '/libraries/vendor/paragonie/sodium_compat/src/Core32/Curve25519/README.md',
            '/libraries/vendor/phpmailer/phpmailer/PHPMailerAutoload.php',
            '/libraries/vendor/phpmailer/phpmailer/class.phpmailer.php',
            '/libraries/vendor/phpmailer/phpmailer/class.phpmaileroauth.php',
            '/libraries/vendor/phpmailer/phpmailer/class.phpmaileroauthgoogle.php',
            '/libraries/vendor/phpmailer/phpmailer/class.pop3.php',
            '/libraries/vendor/phpmailer/phpmailer/class.smtp.php',
            '/libraries/vendor/phpmailer/phpmailer/extras/EasyPeasyICS.php',
            '/libraries/vendor/phpmailer/phpmailer/extras/htmlfilter.php',
            '/libraries/vendor/phpmailer/phpmailer/extras/ntlm_sasl_client.php',
            '/libraries/vendor/simplepie/simplepie/LICENSE.txt',
            '/libraries/vendor/simplepie/simplepie/autoloader.php',
            '/libraries/vendor/simplepie/simplepie/db.sql',
            '/libraries/vendor/simplepie/simplepie/idn/LICENCE',
            '/libraries/vendor/simplepie/simplepie/idn/idna_convert.class.php',
            '/libraries/vendor/simplepie/simplepie/idn/npdata.ser',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Author.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache/Base.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache/DB.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache/File.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache/Memcache.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache/MySQL.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Caption.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Category.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Content/Type/Sniffer.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Copyright.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Core.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Credit.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Decode/HTML/Entities.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Enclosure.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Exception.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/File.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/HTTP/Parser.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/IRI.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Item.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Locator.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Misc.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Net/IPv6.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Parse/Date.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Parser.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Rating.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Registry.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Restriction.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Sanitize.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Source.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/XML/Declaration/Parser.php',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/gzdecode.php',
            '/libraries/vendor/symfony/polyfill-php55/LICENSE',
            '/libraries/vendor/symfony/polyfill-php55/Php55.php',
            '/libraries/vendor/symfony/polyfill-php55/Php55ArrayColumn.php',
            '/libraries/vendor/symfony/polyfill-php55/bootstrap.php',
            '/libraries/vendor/symfony/polyfill-php56/LICENSE',
            '/libraries/vendor/symfony/polyfill-php56/Php56.php',
            '/libraries/vendor/symfony/polyfill-php56/bootstrap.php',
            '/libraries/vendor/symfony/polyfill-php71/LICENSE',
            '/libraries/vendor/symfony/polyfill-php71/Php71.php',
            '/libraries/vendor/symfony/polyfill-php71/bootstrap.php',
            '/libraries/vendor/symfony/polyfill-util/Binary.php',
            '/libraries/vendor/symfony/polyfill-util/BinaryNoFuncOverload.php',
            '/libraries/vendor/symfony/polyfill-util/BinaryOnFuncOverload.php',
            '/libraries/vendor/symfony/polyfill-util/LICENSE',
            '/libraries/vendor/typo3/phar-stream-wrapper/composer.json',
            '/libraries/vendor/web.config',
            '/media/cms/css/debug.css',
            '/media/com_associations/js/sidebyside-uncompressed.js',
            '/media/com_contenthistory/css/jquery.pretty-text-diff.css',
            '/media/com_contenthistory/js/diff_match_patch.js',
            '/media/com_contenthistory/js/jquery.pretty-text-diff.js',
            '/media/com_contenthistory/js/jquery.pretty-text-diff.min.js',
            '/media/com_finder/js/autocompleter.js',
            '/media/com_joomlaupdate/js/encryption.js',
            '/media/com_joomlaupdate/js/encryption.min.js',
            '/media/com_joomlaupdate/js/json2.js',
            '/media/com_joomlaupdate/js/json2.min.js',
            '/media/com_joomlaupdate/js/update.js',
            '/media/com_joomlaupdate/js/update.min.js',
            '/media/contacts/images/con_address.png',
            '/media/contacts/images/con_fax.png',
            '/media/contacts/images/con_info.png',
            '/media/contacts/images/con_mobile.png',
            '/media/contacts/images/con_tel.png',
            '/media/contacts/images/emailButton.png',
            '/media/editors/codemirror/LICENSE',
            '/media/editors/codemirror/addon/comment/comment.js',
            '/media/editors/codemirror/addon/comment/comment.min.js',
            '/media/editors/codemirror/addon/comment/continuecomment.js',
            '/media/editors/codemirror/addon/comment/continuecomment.min.js',
            '/media/editors/codemirror/addon/dialog/dialog.css',
            '/media/editors/codemirror/addon/dialog/dialog.js',
            '/media/editors/codemirror/addon/dialog/dialog.min.css',
            '/media/editors/codemirror/addon/dialog/dialog.min.js',
            '/media/editors/codemirror/addon/display/autorefresh.js',
            '/media/editors/codemirror/addon/display/autorefresh.min.js',
            '/media/editors/codemirror/addon/display/fullscreen.css',
            '/media/editors/codemirror/addon/display/fullscreen.js',
            '/media/editors/codemirror/addon/display/fullscreen.min.css',
            '/media/editors/codemirror/addon/display/fullscreen.min.js',
            '/media/editors/codemirror/addon/display/panel.js',
            '/media/editors/codemirror/addon/display/panel.min.js',
            '/media/editors/codemirror/addon/display/placeholder.js',
            '/media/editors/codemirror/addon/display/placeholder.min.js',
            '/media/editors/codemirror/addon/display/rulers.js',
            '/media/editors/codemirror/addon/display/rulers.min.js',
            '/media/editors/codemirror/addon/edit/closebrackets.js',
            '/media/editors/codemirror/addon/edit/closebrackets.min.js',
            '/media/editors/codemirror/addon/edit/closetag.js',
            '/media/editors/codemirror/addon/edit/closetag.min.js',
            '/media/editors/codemirror/addon/edit/continuelist.js',
            '/media/editors/codemirror/addon/edit/continuelist.min.js',
            '/media/editors/codemirror/addon/edit/matchbrackets.js',
            '/media/editors/codemirror/addon/edit/matchbrackets.min.js',
            '/media/editors/codemirror/addon/edit/matchtags.js',
            '/media/editors/codemirror/addon/edit/matchtags.min.js',
            '/media/editors/codemirror/addon/edit/trailingspace.js',
            '/media/editors/codemirror/addon/edit/trailingspace.min.js',
            '/media/editors/codemirror/addon/fold/brace-fold.js',
            '/media/editors/codemirror/addon/fold/brace-fold.min.js',
            '/media/editors/codemirror/addon/fold/comment-fold.js',
            '/media/editors/codemirror/addon/fold/comment-fold.min.js',
            '/media/editors/codemirror/addon/fold/foldcode.js',
            '/media/editors/codemirror/addon/fold/foldcode.min.js',
            '/media/editors/codemirror/addon/fold/foldgutter.css',
            '/media/editors/codemirror/addon/fold/foldgutter.js',
            '/media/editors/codemirror/addon/fold/foldgutter.min.css',
            '/media/editors/codemirror/addon/fold/foldgutter.min.js',
            '/media/editors/codemirror/addon/fold/indent-fold.js',
            '/media/editors/codemirror/addon/fold/indent-fold.min.js',
            '/media/editors/codemirror/addon/fold/markdown-fold.js',
            '/media/editors/codemirror/addon/fold/markdown-fold.min.js',
            '/media/editors/codemirror/addon/fold/xml-fold.js',
            '/media/editors/codemirror/addon/fold/xml-fold.min.js',
            '/media/editors/codemirror/addon/hint/anyword-hint.js',
            '/media/editors/codemirror/addon/hint/anyword-hint.min.js',
            '/media/editors/codemirror/addon/hint/css-hint.js',
            '/media/editors/codemirror/addon/hint/css-hint.min.js',
            '/media/editors/codemirror/addon/hint/html-hint.js',
            '/media/editors/codemirror/addon/hint/html-hint.min.js',
            '/media/editors/codemirror/addon/hint/javascript-hint.js',
            '/media/editors/codemirror/addon/hint/javascript-hint.min.js',
            '/media/editors/codemirror/addon/hint/show-hint.css',
            '/media/editors/codemirror/addon/hint/show-hint.js',
            '/media/editors/codemirror/addon/hint/show-hint.min.css',
            '/media/editors/codemirror/addon/hint/show-hint.min.js',
            '/media/editors/codemirror/addon/hint/sql-hint.js',
            '/media/editors/codemirror/addon/hint/sql-hint.min.js',
            '/media/editors/codemirror/addon/hint/xml-hint.js',
            '/media/editors/codemirror/addon/hint/xml-hint.min.js',
            '/media/editors/codemirror/addon/lint/coffeescript-lint.js',
            '/media/editors/codemirror/addon/lint/coffeescript-lint.min.js',
            '/media/editors/codemirror/addon/lint/css-lint.js',
            '/media/editors/codemirror/addon/lint/css-lint.min.js',
            '/media/editors/codemirror/addon/lint/html-lint.js',
            '/media/editors/codemirror/addon/lint/html-lint.min.js',
            '/media/editors/codemirror/addon/lint/javascript-lint.js',
            '/media/editors/codemirror/addon/lint/javascript-lint.min.js',
            '/media/editors/codemirror/addon/lint/json-lint.js',
            '/media/editors/codemirror/addon/lint/json-lint.min.js',
            '/media/editors/codemirror/addon/lint/lint.css',
            '/media/editors/codemirror/addon/lint/lint.js',
            '/media/editors/codemirror/addon/lint/lint.min.css',
            '/media/editors/codemirror/addon/lint/lint.min.js',
            '/media/editors/codemirror/addon/lint/yaml-lint.js',
            '/media/editors/codemirror/addon/lint/yaml-lint.min.js',
            '/media/editors/codemirror/addon/merge/merge.css',
            '/media/editors/codemirror/addon/merge/merge.js',
            '/media/editors/codemirror/addon/merge/merge.min.css',
            '/media/editors/codemirror/addon/merge/merge.min.js',
            '/media/editors/codemirror/addon/mode/loadmode.js',
            '/media/editors/codemirror/addon/mode/loadmode.min.js',
            '/media/editors/codemirror/addon/mode/multiplex.js',
            '/media/editors/codemirror/addon/mode/multiplex.min.js',
            '/media/editors/codemirror/addon/mode/multiplex_test.js',
            '/media/editors/codemirror/addon/mode/multiplex_test.min.js',
            '/media/editors/codemirror/addon/mode/overlay.js',
            '/media/editors/codemirror/addon/mode/overlay.min.js',
            '/media/editors/codemirror/addon/mode/simple.js',
            '/media/editors/codemirror/addon/mode/simple.min.js',
            '/media/editors/codemirror/addon/runmode/colorize.js',
            '/media/editors/codemirror/addon/runmode/colorize.min.js',
            '/media/editors/codemirror/addon/runmode/runmode-standalone.js',
            '/media/editors/codemirror/addon/runmode/runmode-standalone.min.js',
            '/media/editors/codemirror/addon/runmode/runmode.js',
            '/media/editors/codemirror/addon/runmode/runmode.min.js',
            '/media/editors/codemirror/addon/runmode/runmode.node.js',
            '/media/editors/codemirror/addon/scroll/annotatescrollbar.js',
            '/media/editors/codemirror/addon/scroll/annotatescrollbar.min.js',
            '/media/editors/codemirror/addon/scroll/scrollpastend.js',
            '/media/editors/codemirror/addon/scroll/scrollpastend.min.js',
            '/media/editors/codemirror/addon/scroll/simplescrollbars.css',
            '/media/editors/codemirror/addon/scroll/simplescrollbars.js',
            '/media/editors/codemirror/addon/scroll/simplescrollbars.min.css',
            '/media/editors/codemirror/addon/scroll/simplescrollbars.min.js',
            '/media/editors/codemirror/addon/search/jump-to-line.js',
            '/media/editors/codemirror/addon/search/jump-to-line.min.js',
            '/media/editors/codemirror/addon/search/match-highlighter.js',
            '/media/editors/codemirror/addon/search/match-highlighter.min.js',
            '/media/editors/codemirror/addon/search/matchesonscrollbar.css',
            '/media/editors/codemirror/addon/search/matchesonscrollbar.js',
            '/media/editors/codemirror/addon/search/matchesonscrollbar.min.css',
            '/media/editors/codemirror/addon/search/matchesonscrollbar.min.js',
            '/media/editors/codemirror/addon/search/search.js',
            '/media/editors/codemirror/addon/search/search.min.js',
            '/media/editors/codemirror/addon/search/searchcursor.js',
            '/media/editors/codemirror/addon/search/searchcursor.min.js',
            '/media/editors/codemirror/addon/selection/active-line.js',
            '/media/editors/codemirror/addon/selection/active-line.min.js',
            '/media/editors/codemirror/addon/selection/mark-selection.js',
            '/media/editors/codemirror/addon/selection/mark-selection.min.js',
            '/media/editors/codemirror/addon/selection/selection-pointer.js',
            '/media/editors/codemirror/addon/selection/selection-pointer.min.js',
            '/media/editors/codemirror/addon/tern/tern.css',
            '/media/editors/codemirror/addon/tern/tern.js',
            '/media/editors/codemirror/addon/tern/tern.min.css',
            '/media/editors/codemirror/addon/tern/tern.min.js',
            '/media/editors/codemirror/addon/tern/worker.js',
            '/media/editors/codemirror/addon/tern/worker.min.js',
            '/media/editors/codemirror/addon/wrap/hardwrap.js',
            '/media/editors/codemirror/addon/wrap/hardwrap.min.js',
            '/media/editors/codemirror/keymap/emacs.js',
            '/media/editors/codemirror/keymap/emacs.min.js',
            '/media/editors/codemirror/keymap/sublime.js',
            '/media/editors/codemirror/keymap/sublime.min.js',
            '/media/editors/codemirror/keymap/vim.js',
            '/media/editors/codemirror/keymap/vim.min.js',
            '/media/editors/codemirror/lib/addons.css',
            '/media/editors/codemirror/lib/addons.js',
            '/media/editors/codemirror/lib/addons.min.css',
            '/media/editors/codemirror/lib/addons.min.js',
            '/media/editors/codemirror/lib/codemirror.css',
            '/media/editors/codemirror/lib/codemirror.js',
            '/media/editors/codemirror/lib/codemirror.min.css',
            '/media/editors/codemirror/lib/codemirror.min.js',
            '/media/editors/codemirror/mode/apl/apl.js',
            '/media/editors/codemirror/mode/apl/apl.min.js',
            '/media/editors/codemirror/mode/asciiarmor/asciiarmor.js',
            '/media/editors/codemirror/mode/asciiarmor/asciiarmor.min.js',
            '/media/editors/codemirror/mode/asn.1/asn.1.js',
            '/media/editors/codemirror/mode/asn.1/asn.min.js',
            '/media/editors/codemirror/mode/asterisk/asterisk.js',
            '/media/editors/codemirror/mode/asterisk/asterisk.min.js',
            '/media/editors/codemirror/mode/brainfuck/brainfuck.js',
            '/media/editors/codemirror/mode/brainfuck/brainfuck.min.js',
            '/media/editors/codemirror/mode/clike/clike.js',
            '/media/editors/codemirror/mode/clike/clike.min.js',
            '/media/editors/codemirror/mode/clojure/clojure.js',
            '/media/editors/codemirror/mode/clojure/clojure.min.js',
            '/media/editors/codemirror/mode/cmake/cmake.js',
            '/media/editors/codemirror/mode/cmake/cmake.min.js',
            '/media/editors/codemirror/mode/cobol/cobol.js',
            '/media/editors/codemirror/mode/cobol/cobol.min.js',
            '/media/editors/codemirror/mode/coffeescript/coffeescript.js',
            '/media/editors/codemirror/mode/coffeescript/coffeescript.min.js',
            '/media/editors/codemirror/mode/commonlisp/commonlisp.js',
            '/media/editors/codemirror/mode/commonlisp/commonlisp.min.js',
            '/media/editors/codemirror/mode/crystal/crystal.js',
            '/media/editors/codemirror/mode/crystal/crystal.min.js',
            '/media/editors/codemirror/mode/css/css.js',
            '/media/editors/codemirror/mode/css/css.min.js',
            '/media/editors/codemirror/mode/cypher/cypher.js',
            '/media/editors/codemirror/mode/cypher/cypher.min.js',
            '/media/editors/codemirror/mode/d/d.js',
            '/media/editors/codemirror/mode/d/d.min.js',
            '/media/editors/codemirror/mode/dart/dart.js',
            '/media/editors/codemirror/mode/dart/dart.min.js',
            '/media/editors/codemirror/mode/diff/diff.js',
            '/media/editors/codemirror/mode/diff/diff.min.js',
            '/media/editors/codemirror/mode/django/django.js',
            '/media/editors/codemirror/mode/django/django.min.js',
            '/media/editors/codemirror/mode/dockerfile/dockerfile.js',
            '/media/editors/codemirror/mode/dockerfile/dockerfile.min.js',
            '/media/editors/codemirror/mode/dtd/dtd.js',
            '/media/editors/codemirror/mode/dtd/dtd.min.js',
            '/media/editors/codemirror/mode/dylan/dylan.js',
            '/media/editors/codemirror/mode/dylan/dylan.min.js',
            '/media/editors/codemirror/mode/ebnf/ebnf.js',
            '/media/editors/codemirror/mode/ebnf/ebnf.min.js',
            '/media/editors/codemirror/mode/ecl/ecl.js',
            '/media/editors/codemirror/mode/ecl/ecl.min.js',
            '/media/editors/codemirror/mode/eiffel/eiffel.js',
            '/media/editors/codemirror/mode/eiffel/eiffel.min.js',
            '/media/editors/codemirror/mode/elm/elm.js',
            '/media/editors/codemirror/mode/elm/elm.min.js',
            '/media/editors/codemirror/mode/erlang/erlang.js',
            '/media/editors/codemirror/mode/erlang/erlang.min.js',
            '/media/editors/codemirror/mode/factor/factor.js',
            '/media/editors/codemirror/mode/factor/factor.min.js',
            '/media/editors/codemirror/mode/fcl/fcl.js',
            '/media/editors/codemirror/mode/fcl/fcl.min.js',
            '/media/editors/codemirror/mode/forth/forth.js',
            '/media/editors/codemirror/mode/forth/forth.min.js',
            '/media/editors/codemirror/mode/fortran/fortran.js',
            '/media/editors/codemirror/mode/fortran/fortran.min.js',
            '/media/editors/codemirror/mode/gas/gas.js',
            '/media/editors/codemirror/mode/gas/gas.min.js',
            '/media/editors/codemirror/mode/gfm/gfm.js',
            '/media/editors/codemirror/mode/gfm/gfm.min.js',
            '/media/editors/codemirror/mode/gherkin/gherkin.js',
            '/media/editors/codemirror/mode/gherkin/gherkin.min.js',
            '/media/editors/codemirror/mode/go/go.js',
            '/media/editors/codemirror/mode/go/go.min.js',
            '/media/editors/codemirror/mode/groovy/groovy.js',
            '/media/editors/codemirror/mode/groovy/groovy.min.js',
            '/media/editors/codemirror/mode/haml/haml.js',
            '/media/editors/codemirror/mode/haml/haml.min.js',
            '/media/editors/codemirror/mode/handlebars/handlebars.js',
            '/media/editors/codemirror/mode/handlebars/handlebars.min.js',
            '/media/editors/codemirror/mode/haskell-literate/haskell-literate.js',
            '/media/editors/codemirror/mode/haskell-literate/haskell-literate.min.js',
            '/media/editors/codemirror/mode/haskell/haskell.js',
            '/media/editors/codemirror/mode/haskell/haskell.min.js',
            '/media/editors/codemirror/mode/haxe/haxe.js',
            '/media/editors/codemirror/mode/haxe/haxe.min.js',
            '/media/editors/codemirror/mode/htmlembedded/htmlembedded.js',
            '/media/editors/codemirror/mode/htmlembedded/htmlembedded.min.js',
            '/media/editors/codemirror/mode/htmlmixed/htmlmixed.js',
            '/media/editors/codemirror/mode/htmlmixed/htmlmixed.min.js',
            '/media/editors/codemirror/mode/http/http.js',
            '/media/editors/codemirror/mode/http/http.min.js',
            '/media/editors/codemirror/mode/idl/idl.js',
            '/media/editors/codemirror/mode/idl/idl.min.js',
            '/media/editors/codemirror/mode/javascript/javascript.js',
            '/media/editors/codemirror/mode/javascript/javascript.min.js',
            '/media/editors/codemirror/mode/jinja2/jinja2.js',
            '/media/editors/codemirror/mode/jinja2/jinja2.min.js',
            '/media/editors/codemirror/mode/jsx/jsx.js',
            '/media/editors/codemirror/mode/jsx/jsx.min.js',
            '/media/editors/codemirror/mode/julia/julia.js',
            '/media/editors/codemirror/mode/julia/julia.min.js',
            '/media/editors/codemirror/mode/livescript/livescript.js',
            '/media/editors/codemirror/mode/livescript/livescript.min.js',
            '/media/editors/codemirror/mode/lua/lua.js',
            '/media/editors/codemirror/mode/lua/lua.min.js',
            '/media/editors/codemirror/mode/markdown/markdown.js',
            '/media/editors/codemirror/mode/markdown/markdown.min.js',
            '/media/editors/codemirror/mode/mathematica/mathematica.js',
            '/media/editors/codemirror/mode/mathematica/mathematica.min.js',
            '/media/editors/codemirror/mode/mbox/mbox.js',
            '/media/editors/codemirror/mode/mbox/mbox.min.js',
            '/media/editors/codemirror/mode/meta.js',
            '/media/editors/codemirror/mode/meta.min.js',
            '/media/editors/codemirror/mode/mirc/mirc.js',
            '/media/editors/codemirror/mode/mirc/mirc.min.js',
            '/media/editors/codemirror/mode/mllike/mllike.js',
            '/media/editors/codemirror/mode/mllike/mllike.min.js',
            '/media/editors/codemirror/mode/modelica/modelica.js',
            '/media/editors/codemirror/mode/modelica/modelica.min.js',
            '/media/editors/codemirror/mode/mscgen/mscgen.js',
            '/media/editors/codemirror/mode/mscgen/mscgen.min.js',
            '/media/editors/codemirror/mode/mumps/mumps.js',
            '/media/editors/codemirror/mode/mumps/mumps.min.js',
            '/media/editors/codemirror/mode/nginx/nginx.js',
            '/media/editors/codemirror/mode/nginx/nginx.min.js',
            '/media/editors/codemirror/mode/nsis/nsis.js',
            '/media/editors/codemirror/mode/nsis/nsis.min.js',
            '/media/editors/codemirror/mode/ntriples/ntriples.js',
            '/media/editors/codemirror/mode/ntriples/ntriples.min.js',
            '/media/editors/codemirror/mode/octave/octave.js',
            '/media/editors/codemirror/mode/octave/octave.min.js',
            '/media/editors/codemirror/mode/oz/oz.js',
            '/media/editors/codemirror/mode/oz/oz.min.js',
            '/media/editors/codemirror/mode/pascal/pascal.js',
            '/media/editors/codemirror/mode/pascal/pascal.min.js',
            '/media/editors/codemirror/mode/pegjs/pegjs.js',
            '/media/editors/codemirror/mode/pegjs/pegjs.min.js',
            '/media/editors/codemirror/mode/perl/perl.js',
            '/media/editors/codemirror/mode/perl/perl.min.js',
            '/media/editors/codemirror/mode/php/php.js',
            '/media/editors/codemirror/mode/php/php.min.js',
            '/media/editors/codemirror/mode/pig/pig.js',
            '/media/editors/codemirror/mode/pig/pig.min.js',
            '/media/editors/codemirror/mode/powershell/powershell.js',
            '/media/editors/codemirror/mode/powershell/powershell.min.js',
            '/media/editors/codemirror/mode/properties/properties.js',
            '/media/editors/codemirror/mode/properties/properties.min.js',
            '/media/editors/codemirror/mode/protobuf/protobuf.js',
            '/media/editors/codemirror/mode/protobuf/protobuf.min.js',
            '/media/editors/codemirror/mode/pug/pug.js',
            '/media/editors/codemirror/mode/pug/pug.min.js',
            '/media/editors/codemirror/mode/puppet/puppet.js',
            '/media/editors/codemirror/mode/puppet/puppet.min.js',
            '/media/editors/codemirror/mode/python/python.js',
            '/media/editors/codemirror/mode/python/python.min.js',
            '/media/editors/codemirror/mode/q/q.js',
            '/media/editors/codemirror/mode/q/q.min.js',
            '/media/editors/codemirror/mode/r/r.js',
            '/media/editors/codemirror/mode/r/r.min.js',
            '/media/editors/codemirror/mode/rpm/changes/index.html',
            '/media/editors/codemirror/mode/rpm/rpm.js',
            '/media/editors/codemirror/mode/rpm/rpm.min.js',
            '/media/editors/codemirror/mode/rst/rst.js',
            '/media/editors/codemirror/mode/rst/rst.min.js',
            '/media/editors/codemirror/mode/ruby/ruby.js',
            '/media/editors/codemirror/mode/ruby/ruby.min.js',
            '/media/editors/codemirror/mode/rust/rust.js',
            '/media/editors/codemirror/mode/rust/rust.min.js',
            '/media/editors/codemirror/mode/sas/sas.js',
            '/media/editors/codemirror/mode/sas/sas.min.js',
            '/media/editors/codemirror/mode/sass/sass.js',
            '/media/editors/codemirror/mode/sass/sass.min.js',
            '/media/editors/codemirror/mode/scheme/scheme.js',
            '/media/editors/codemirror/mode/scheme/scheme.min.js',
            '/media/editors/codemirror/mode/shell/shell.js',
            '/media/editors/codemirror/mode/shell/shell.min.js',
            '/media/editors/codemirror/mode/sieve/sieve.js',
            '/media/editors/codemirror/mode/sieve/sieve.min.js',
            '/media/editors/codemirror/mode/slim/slim.js',
            '/media/editors/codemirror/mode/slim/slim.min.js',
            '/media/editors/codemirror/mode/smalltalk/smalltalk.js',
            '/media/editors/codemirror/mode/smalltalk/smalltalk.min.js',
            '/media/editors/codemirror/mode/smarty/smarty.js',
            '/media/editors/codemirror/mode/smarty/smarty.min.js',
            '/media/editors/codemirror/mode/solr/solr.js',
            '/media/editors/codemirror/mode/solr/solr.min.js',
            '/media/editors/codemirror/mode/soy/soy.js',
            '/media/editors/codemirror/mode/soy/soy.min.js',
            '/media/editors/codemirror/mode/sparql/sparql.js',
            '/media/editors/codemirror/mode/sparql/sparql.min.js',
            '/media/editors/codemirror/mode/spreadsheet/spreadsheet.js',
            '/media/editors/codemirror/mode/spreadsheet/spreadsheet.min.js',
            '/media/editors/codemirror/mode/sql/sql.js',
            '/media/editors/codemirror/mode/sql/sql.min.js',
            '/media/editors/codemirror/mode/stex/stex.js',
            '/media/editors/codemirror/mode/stex/stex.min.js',
            '/media/editors/codemirror/mode/stylus/stylus.js',
            '/media/editors/codemirror/mode/stylus/stylus.min.js',
            '/media/editors/codemirror/mode/swift/swift.js',
            '/media/editors/codemirror/mode/swift/swift.min.js',
            '/media/editors/codemirror/mode/tcl/tcl.js',
            '/media/editors/codemirror/mode/tcl/tcl.min.js',
            '/media/editors/codemirror/mode/textile/textile.js',
            '/media/editors/codemirror/mode/textile/textile.min.js',
            '/media/editors/codemirror/mode/tiddlywiki/tiddlywiki.css',
            '/media/editors/codemirror/mode/tiddlywiki/tiddlywiki.js',
            '/media/editors/codemirror/mode/tiddlywiki/tiddlywiki.min.css',
            '/media/editors/codemirror/mode/tiddlywiki/tiddlywiki.min.js',
            '/media/editors/codemirror/mode/tiki/tiki.css',
            '/media/editors/codemirror/mode/tiki/tiki.js',
            '/media/editors/codemirror/mode/tiki/tiki.min.css',
            '/media/editors/codemirror/mode/tiki/tiki.min.js',
            '/media/editors/codemirror/mode/toml/toml.js',
            '/media/editors/codemirror/mode/toml/toml.min.js',
            '/media/editors/codemirror/mode/tornado/tornado.js',
            '/media/editors/codemirror/mode/tornado/tornado.min.js',
            '/media/editors/codemirror/mode/troff/troff.js',
            '/media/editors/codemirror/mode/troff/troff.min.js',
            '/media/editors/codemirror/mode/ttcn-cfg/ttcn-cfg.js',
            '/media/editors/codemirror/mode/ttcn-cfg/ttcn-cfg.min.js',
            '/media/editors/codemirror/mode/ttcn/ttcn.js',
            '/media/editors/codemirror/mode/ttcn/ttcn.min.js',
            '/media/editors/codemirror/mode/turtle/turtle.js',
            '/media/editors/codemirror/mode/turtle/turtle.min.js',
            '/media/editors/codemirror/mode/twig/twig.js',
            '/media/editors/codemirror/mode/twig/twig.min.js',
            '/media/editors/codemirror/mode/vb/vb.js',
            '/media/editors/codemirror/mode/vb/vb.min.js',
            '/media/editors/codemirror/mode/vbscript/vbscript.js',
            '/media/editors/codemirror/mode/vbscript/vbscript.min.js',
            '/media/editors/codemirror/mode/velocity/velocity.js',
            '/media/editors/codemirror/mode/velocity/velocity.min.js',
            '/media/editors/codemirror/mode/verilog/verilog.js',
            '/media/editors/codemirror/mode/verilog/verilog.min.js',
            '/media/editors/codemirror/mode/vhdl/vhdl.js',
            '/media/editors/codemirror/mode/vhdl/vhdl.min.js',
            '/media/editors/codemirror/mode/vue/vue.js',
            '/media/editors/codemirror/mode/vue/vue.min.js',
            '/media/editors/codemirror/mode/wast/wast.js',
            '/media/editors/codemirror/mode/wast/wast.min.js',
            '/media/editors/codemirror/mode/webidl/webidl.js',
            '/media/editors/codemirror/mode/webidl/webidl.min.js',
            '/media/editors/codemirror/mode/xml/xml.js',
            '/media/editors/codemirror/mode/xml/xml.min.js',
            '/media/editors/codemirror/mode/xquery/xquery.js',
            '/media/editors/codemirror/mode/xquery/xquery.min.js',
            '/media/editors/codemirror/mode/yacas/yacas.js',
            '/media/editors/codemirror/mode/yacas/yacas.min.js',
            '/media/editors/codemirror/mode/yaml-frontmatter/yaml-frontmatter.js',
            '/media/editors/codemirror/mode/yaml-frontmatter/yaml-frontmatter.min.js',
            '/media/editors/codemirror/mode/yaml/yaml.js',
            '/media/editors/codemirror/mode/yaml/yaml.min.js',
            '/media/editors/codemirror/mode/z80/z80.js',
            '/media/editors/codemirror/mode/z80/z80.min.js',
            '/media/editors/codemirror/theme/3024-day.css',
            '/media/editors/codemirror/theme/3024-night.css',
            '/media/editors/codemirror/theme/abcdef.css',
            '/media/editors/codemirror/theme/ambiance-mobile.css',
            '/media/editors/codemirror/theme/ambiance.css',
            '/media/editors/codemirror/theme/ayu-dark.css',
            '/media/editors/codemirror/theme/ayu-mirage.css',
            '/media/editors/codemirror/theme/base16-dark.css',
            '/media/editors/codemirror/theme/base16-light.css',
            '/media/editors/codemirror/theme/bespin.css',
            '/media/editors/codemirror/theme/blackboard.css',
            '/media/editors/codemirror/theme/cobalt.css',
            '/media/editors/codemirror/theme/colorforth.css',
            '/media/editors/codemirror/theme/darcula.css',
            '/media/editors/codemirror/theme/dracula.css',
            '/media/editors/codemirror/theme/duotone-dark.css',
            '/media/editors/codemirror/theme/duotone-light.css',
            '/media/editors/codemirror/theme/eclipse.css',
            '/media/editors/codemirror/theme/elegant.css',
            '/media/editors/codemirror/theme/erlang-dark.css',
            '/media/editors/codemirror/theme/gruvbox-dark.css',
            '/media/editors/codemirror/theme/hopscotch.css',
            '/media/editors/codemirror/theme/icecoder.css',
            '/media/editors/codemirror/theme/idea.css',
            '/media/editors/codemirror/theme/isotope.css',
            '/media/editors/codemirror/theme/lesser-dark.css',
            '/media/editors/codemirror/theme/liquibyte.css',
            '/media/editors/codemirror/theme/lucario.css',
            '/media/editors/codemirror/theme/material-darker.css',
            '/media/editors/codemirror/theme/material-ocean.css',
            '/media/editors/codemirror/theme/material-palenight.css',
            '/media/editors/codemirror/theme/material.css',
            '/media/editors/codemirror/theme/mbo.css',
            '/media/editors/codemirror/theme/mdn-like.css',
            '/media/editors/codemirror/theme/midnight.css',
            '/media/editors/codemirror/theme/monokai.css',
            '/media/editors/codemirror/theme/moxer.css',
            '/media/editors/codemirror/theme/neat.css',
            '/media/editors/codemirror/theme/neo.css',
            '/media/editors/codemirror/theme/night.css',
            '/media/editors/codemirror/theme/nord.css',
            '/media/editors/codemirror/theme/oceanic-next.css',
            '/media/editors/codemirror/theme/panda-syntax.css',
            '/media/editors/codemirror/theme/paraiso-dark.css',
            '/media/editors/codemirror/theme/paraiso-light.css',
            '/media/editors/codemirror/theme/pastel-on-dark.css',
            '/media/editors/codemirror/theme/railscasts.css',
            '/media/editors/codemirror/theme/rubyblue.css',
            '/media/editors/codemirror/theme/seti.css',
            '/media/editors/codemirror/theme/shadowfox.css',
            '/media/editors/codemirror/theme/solarized.css',
            '/media/editors/codemirror/theme/ssms.css',
            '/media/editors/codemirror/theme/the-matrix.css',
            '/media/editors/codemirror/theme/tomorrow-night-bright.css',
            '/media/editors/codemirror/theme/tomorrow-night-eighties.css',
            '/media/editors/codemirror/theme/ttcn.css',
            '/media/editors/codemirror/theme/twilight.css',
            '/media/editors/codemirror/theme/vibrant-ink.css',
            '/media/editors/codemirror/theme/xq-dark.css',
            '/media/editors/codemirror/theme/xq-light.css',
            '/media/editors/codemirror/theme/yeti.css',
            '/media/editors/codemirror/theme/yonce.css',
            '/media/editors/codemirror/theme/zenburn.css',
            '/media/editors/none/js/none.js',
            '/media/editors/none/js/none.min.js',
            '/media/editors/tinymce/changelog.txt',
            '/media/editors/tinymce/js/plugins/dragdrop/plugin.js',
            '/media/editors/tinymce/js/plugins/dragdrop/plugin.min.js',
            '/media/editors/tinymce/js/tiny-close.js',
            '/media/editors/tinymce/js/tiny-close.min.js',
            '/media/editors/tinymce/js/tinymce-builder.js',
            '/media/editors/tinymce/js/tinymce.js',
            '/media/editors/tinymce/js/tinymce.min.js',
            '/media/editors/tinymce/langs/af.js',
            '/media/editors/tinymce/langs/ar.js',
            '/media/editors/tinymce/langs/be.js',
            '/media/editors/tinymce/langs/bg.js',
            '/media/editors/tinymce/langs/bs.js',
            '/media/editors/tinymce/langs/ca.js',
            '/media/editors/tinymce/langs/cs.js',
            '/media/editors/tinymce/langs/cy.js',
            '/media/editors/tinymce/langs/da.js',
            '/media/editors/tinymce/langs/de.js',
            '/media/editors/tinymce/langs/el.js',
            '/media/editors/tinymce/langs/es.js',
            '/media/editors/tinymce/langs/et.js',
            '/media/editors/tinymce/langs/eu.js',
            '/media/editors/tinymce/langs/fa.js',
            '/media/editors/tinymce/langs/fi.js',
            '/media/editors/tinymce/langs/fo.js',
            '/media/editors/tinymce/langs/fr.js',
            '/media/editors/tinymce/langs/ga.js',
            '/media/editors/tinymce/langs/gl.js',
            '/media/editors/tinymce/langs/he.js',
            '/media/editors/tinymce/langs/hr.js',
            '/media/editors/tinymce/langs/hu.js',
            '/media/editors/tinymce/langs/id.js',
            '/media/editors/tinymce/langs/it.js',
            '/media/editors/tinymce/langs/ja.js',
            '/media/editors/tinymce/langs/ka.js',
            '/media/editors/tinymce/langs/kk.js',
            '/media/editors/tinymce/langs/km.js',
            '/media/editors/tinymce/langs/ko.js',
            '/media/editors/tinymce/langs/lb.js',
            '/media/editors/tinymce/langs/lt.js',
            '/media/editors/tinymce/langs/lv.js',
            '/media/editors/tinymce/langs/mk.js',
            '/media/editors/tinymce/langs/ms.js',
            '/media/editors/tinymce/langs/nb.js',
            '/media/editors/tinymce/langs/nl.js',
            '/media/editors/tinymce/langs/pl.js',
            '/media/editors/tinymce/langs/pt-BR.js',
            '/media/editors/tinymce/langs/pt-PT.js',
            '/media/editors/tinymce/langs/readme.md',
            '/media/editors/tinymce/langs/ro.js',
            '/media/editors/tinymce/langs/ru.js',
            '/media/editors/tinymce/langs/si-LK.js',
            '/media/editors/tinymce/langs/sk.js',
            '/media/editors/tinymce/langs/sl.js',
            '/media/editors/tinymce/langs/sr.js',
            '/media/editors/tinymce/langs/sv.js',
            '/media/editors/tinymce/langs/sw.js',
            '/media/editors/tinymce/langs/sy.js',
            '/media/editors/tinymce/langs/ta.js',
            '/media/editors/tinymce/langs/th.js',
            '/media/editors/tinymce/langs/tr.js',
            '/media/editors/tinymce/langs/ug.js',
            '/media/editors/tinymce/langs/uk.js',
            '/media/editors/tinymce/langs/vi.js',
            '/media/editors/tinymce/langs/zh-CN.js',
            '/media/editors/tinymce/langs/zh-TW.js',
            '/media/editors/tinymce/license.txt',
            '/media/editors/tinymce/plugins/advlist/plugin.min.js',
            '/media/editors/tinymce/plugins/anchor/plugin.min.js',
            '/media/editors/tinymce/plugins/autolink/plugin.min.js',
            '/media/editors/tinymce/plugins/autoresize/plugin.min.js',
            '/media/editors/tinymce/plugins/autosave/plugin.min.js',
            '/media/editors/tinymce/plugins/bbcode/plugin.min.js',
            '/media/editors/tinymce/plugins/charmap/plugin.min.js',
            '/media/editors/tinymce/plugins/code/plugin.min.js',
            '/media/editors/tinymce/plugins/codesample/css/prism.css',
            '/media/editors/tinymce/plugins/codesample/plugin.min.js',
            '/media/editors/tinymce/plugins/colorpicker/plugin.min.js',
            '/media/editors/tinymce/plugins/contextmenu/plugin.min.js',
            '/media/editors/tinymce/plugins/directionality/plugin.min.js',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-cool.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-cry.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-embarassed.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-foot-in-mouth.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-frown.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-innocent.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-kiss.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-laughing.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-money-mouth.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-sealed.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-smile.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-surprised.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-tongue-out.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-undecided.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-wink.gif',
            '/media/editors/tinymce/plugins/emoticons/img/smiley-yell.gif',
            '/media/editors/tinymce/plugins/emoticons/plugin.min.js',
            '/media/editors/tinymce/plugins/example/dialog.html',
            '/media/editors/tinymce/plugins/example/plugin.min.js',
            '/media/editors/tinymce/plugins/example_dependency/plugin.min.js',
            '/media/editors/tinymce/plugins/fullpage/plugin.min.js',
            '/media/editors/tinymce/plugins/fullscreen/plugin.min.js',
            '/media/editors/tinymce/plugins/hr/plugin.min.js',
            '/media/editors/tinymce/plugins/image/plugin.min.js',
            '/media/editors/tinymce/plugins/imagetools/plugin.min.js',
            '/media/editors/tinymce/plugins/importcss/plugin.min.js',
            '/media/editors/tinymce/plugins/insertdatetime/plugin.min.js',
            '/media/editors/tinymce/plugins/layer/plugin.min.js',
            '/media/editors/tinymce/plugins/legacyoutput/plugin.min.js',
            '/media/editors/tinymce/plugins/link/plugin.min.js',
            '/media/editors/tinymce/plugins/lists/plugin.min.js',
            '/media/editors/tinymce/plugins/media/plugin.min.js',
            '/media/editors/tinymce/plugins/nonbreaking/plugin.min.js',
            '/media/editors/tinymce/plugins/noneditable/plugin.min.js',
            '/media/editors/tinymce/plugins/pagebreak/plugin.min.js',
            '/media/editors/tinymce/plugins/paste/plugin.min.js',
            '/media/editors/tinymce/plugins/preview/plugin.min.js',
            '/media/editors/tinymce/plugins/print/plugin.min.js',
            '/media/editors/tinymce/plugins/save/plugin.min.js',
            '/media/editors/tinymce/plugins/searchreplace/plugin.min.js',
            '/media/editors/tinymce/plugins/spellchecker/plugin.min.js',
            '/media/editors/tinymce/plugins/tabfocus/plugin.min.js',
            '/media/editors/tinymce/plugins/table/plugin.min.js',
            '/media/editors/tinymce/plugins/template/plugin.min.js',
            '/media/editors/tinymce/plugins/textcolor/plugin.min.js',
            '/media/editors/tinymce/plugins/textpattern/plugin.min.js',
            '/media/editors/tinymce/plugins/toc/plugin.min.js',
            '/media/editors/tinymce/plugins/visualblocks/css/visualblocks.css',
            '/media/editors/tinymce/plugins/visualblocks/plugin.min.js',
            '/media/editors/tinymce/plugins/visualchars/plugin.min.js',
            '/media/editors/tinymce/plugins/wordcount/plugin.min.js',
            '/media/editors/tinymce/skins/lightgray/content.inline.min.css',
            '/media/editors/tinymce/skins/lightgray/content.min.css',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce-small.eot',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce-small.svg',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce-small.ttf',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce-small.woff',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce.eot',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce.svg',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce.ttf',
            '/media/editors/tinymce/skins/lightgray/fonts/tinymce.woff',
            '/media/editors/tinymce/skins/lightgray/img/anchor.gif',
            '/media/editors/tinymce/skins/lightgray/img/loader.gif',
            '/media/editors/tinymce/skins/lightgray/img/object.gif',
            '/media/editors/tinymce/skins/lightgray/img/trans.gif',
            '/media/editors/tinymce/skins/lightgray/skin.ie7.min.css',
            '/media/editors/tinymce/skins/lightgray/skin.min.css',
            '/media/editors/tinymce/templates/layout1.html',
            '/media/editors/tinymce/templates/snippet1.html',
            '/media/editors/tinymce/themes/modern/theme.min.js',
            '/media/editors/tinymce/tinymce.min.js',
            '/media/jui/css/bootstrap-extended.css',
            '/media/jui/css/bootstrap-responsive.css',
            '/media/jui/css/bootstrap-responsive.min.css',
            '/media/jui/css/bootstrap-rtl.css',
            '/media/jui/css/bootstrap-tooltip-extended.css',
            '/media/jui/css/bootstrap.css',
            '/media/jui/css/bootstrap.min.css',
            '/media/jui/css/chosen-sprite.png',
            '/media/jui/css/chosen-sprite@2x.png',
            '/media/jui/css/chosen.css',
            '/media/jui/css/icomoon.css',
            '/media/jui/css/jquery.minicolors.css',
            '/media/jui/css/jquery.searchtools.css',
            '/media/jui/css/jquery.simplecolors.css',
            '/media/jui/css/sortablelist.css',
            '/media/jui/fonts/IcoMoon.dev.commented.svg',
            '/media/jui/fonts/IcoMoon.dev.svg',
            '/media/jui/fonts/IcoMoon.eot',
            '/media/jui/fonts/IcoMoon.svg',
            '/media/jui/fonts/IcoMoon.ttf',
            '/media/jui/fonts/IcoMoon.woff',
            '/media/jui/fonts/icomoon-license.txt',
            '/media/jui/images/ajax-loader.gif',
            '/media/jui/img/ajax-loader.gif',
            '/media/jui/img/alpha.png',
            '/media/jui/img/bg-overlay.png',
            '/media/jui/img/glyphicons-halflings-white.png',
            '/media/jui/img/glyphicons-halflings.png',
            '/media/jui/img/hue.png',
            '/media/jui/img/joomla.png',
            '/media/jui/img/jquery.minicolors.png',
            '/media/jui/img/saturation.png',
            '/media/jui/js/ajax-chosen.js',
            '/media/jui/js/ajax-chosen.min.js',
            '/media/jui/js/bootstrap-tooltip-extended.js',
            '/media/jui/js/bootstrap-tooltip-extended.min.js',
            '/media/jui/js/bootstrap.js',
            '/media/jui/js/bootstrap.min.js',
            '/media/jui/js/chosen.jquery.js',
            '/media/jui/js/chosen.jquery.min.js',
            '/media/jui/js/cms-uncompressed.js',
            '/media/jui/js/cms.js',
            '/media/jui/js/fielduser.js',
            '/media/jui/js/fielduser.min.js',
            '/media/jui/js/html5-uncompressed.js',
            '/media/jui/js/html5.js',
            '/media/jui/js/icomoon-lte-ie7.js',
            '/media/jui/js/jquery-migrate.js',
            '/media/jui/js/jquery-migrate.min.js',
            '/media/jui/js/jquery-noconflict.js',
            '/media/jui/js/jquery.autocomplete.js',
            '/media/jui/js/jquery.autocomplete.min.js',
            '/media/jui/js/jquery.js',
            '/media/jui/js/jquery.min.js',
            '/media/jui/js/jquery.minicolors.js',
            '/media/jui/js/jquery.minicolors.min.js',
            '/media/jui/js/jquery.searchtools.js',
            '/media/jui/js/jquery.searchtools.min.js',
            '/media/jui/js/jquery.simplecolors.js',
            '/media/jui/js/jquery.simplecolors.min.js',
            '/media/jui/js/jquery.ui.core.js',
            '/media/jui/js/jquery.ui.core.min.js',
            '/media/jui/js/jquery.ui.sortable.js',
            '/media/jui/js/jquery.ui.sortable.min.js',
            '/media/jui/js/sortablelist.js',
            '/media/jui/js/treeselectmenu.jquery.js',
            '/media/jui/js/treeselectmenu.jquery.min.js',
            '/media/jui/less/accordion.less',
            '/media/jui/less/alerts.less',
            '/media/jui/less/bootstrap-extended.less',
            '/media/jui/less/bootstrap-rtl.less',
            '/media/jui/less/bootstrap.less',
            '/media/jui/less/breadcrumbs.less',
            '/media/jui/less/button-groups.less',
            '/media/jui/less/buttons.less',
            '/media/jui/less/carousel.less',
            '/media/jui/less/close.less',
            '/media/jui/less/code.less',
            '/media/jui/less/component-animations.less',
            '/media/jui/less/dropdowns.less',
            '/media/jui/less/forms.less',
            '/media/jui/less/grid.less',
            '/media/jui/less/hero-unit.less',
            '/media/jui/less/icomoon.less',
            '/media/jui/less/labels-badges.less',
            '/media/jui/less/layouts.less',
            '/media/jui/less/media.less',
            '/media/jui/less/mixins.less',
            '/media/jui/less/modals.joomla.less',
            '/media/jui/less/modals.less',
            '/media/jui/less/navbar.less',
            '/media/jui/less/navs.less',
            '/media/jui/less/pager.less',
            '/media/jui/less/pagination.less',
            '/media/jui/less/popovers.less',
            '/media/jui/less/progress-bars.less',
            '/media/jui/less/reset.less',
            '/media/jui/less/responsive-1200px-min.less',
            '/media/jui/less/responsive-767px-max.joomla.less',
            '/media/jui/less/responsive-767px-max.less',
            '/media/jui/less/responsive-768px-979px.less',
            '/media/jui/less/responsive-navbar.less',
            '/media/jui/less/responsive-utilities.less',
            '/media/jui/less/responsive.less',
            '/media/jui/less/scaffolding.less',
            '/media/jui/less/sprites.less',
            '/media/jui/less/tables.less',
            '/media/jui/less/thumbnails.less',
            '/media/jui/less/tooltip.less',
            '/media/jui/less/type.less',
            '/media/jui/less/utilities.less',
            '/media/jui/less/variables.less',
            '/media/jui/less/wells.less',
            '/media/media/css/background.png',
            '/media/media/css/bigplay.fw.png',
            '/media/media/css/bigplay.png',
            '/media/media/css/bigplay.svg',
            '/media/media/css/controls-ted.png',
            '/media/media/css/controls-wmp-bg.png',
            '/media/media/css/controls-wmp.png',
            '/media/media/css/controls.fw.png',
            '/media/media/css/controls.png',
            '/media/media/css/controls.svg',
            '/media/media/css/jumpforward.png',
            '/media/media/css/loading.gif',
            '/media/media/css/mediaelementplayer.css',
            '/media/media/css/mediaelementplayer.min.css',
            '/media/media/css/medialist-details.css',
            '/media/media/css/medialist-details_rtl.css',
            '/media/media/css/medialist-thumbs.css',
            '/media/media/css/medialist-thumbs_rtl.css',
            '/media/media/css/mediamanager.css',
            '/media/media/css/mediamanager_rtl.css',
            '/media/media/css/mejs-skins.css',
            '/media/media/css/popup-imagelist.css',
            '/media/media/css/popup-imagelist_rtl.css',
            '/media/media/css/popup-imagemanager.css',
            '/media/media/css/popup-imagemanager_rtl.css',
            '/media/media/css/skipback.png',
            '/media/media/images/bar.gif',
            '/media/media/images/con_info.png',
            '/media/media/images/delete.png',
            '/media/media/images/dots.gif',
            '/media/media/images/failed.png',
            '/media/media/images/folder.gif',
            '/media/media/images/folder.png',
            '/media/media/images/folder_sm.png',
            '/media/media/images/folderup_16.png',
            '/media/media/images/folderup_32.png',
            '/media/media/images/mime-icon-16/avi.png',
            '/media/media/images/mime-icon-16/doc.png',
            '/media/media/images/mime-icon-16/mov.png',
            '/media/media/images/mime-icon-16/mp3.png',
            '/media/media/images/mime-icon-16/mp4.png',
            '/media/media/images/mime-icon-16/odc.png',
            '/media/media/images/mime-icon-16/odd.png',
            '/media/media/images/mime-icon-16/odt.png',
            '/media/media/images/mime-icon-16/ogg.png',
            '/media/media/images/mime-icon-16/pdf.png',
            '/media/media/images/mime-icon-16/ppt.png',
            '/media/media/images/mime-icon-16/rar.png',
            '/media/media/images/mime-icon-16/rtf.png',
            '/media/media/images/mime-icon-16/svg.png',
            '/media/media/images/mime-icon-16/sxd.png',
            '/media/media/images/mime-icon-16/tar.png',
            '/media/media/images/mime-icon-16/tgz.png',
            '/media/media/images/mime-icon-16/wma.png',
            '/media/media/images/mime-icon-16/wmv.png',
            '/media/media/images/mime-icon-16/xls.png',
            '/media/media/images/mime-icon-16/zip.png',
            '/media/media/images/mime-icon-32/avi.png',
            '/media/media/images/mime-icon-32/doc.png',
            '/media/media/images/mime-icon-32/mov.png',
            '/media/media/images/mime-icon-32/mp3.png',
            '/media/media/images/mime-icon-32/mp4.png',
            '/media/media/images/mime-icon-32/odc.png',
            '/media/media/images/mime-icon-32/odd.png',
            '/media/media/images/mime-icon-32/odt.png',
            '/media/media/images/mime-icon-32/ogg.png',
            '/media/media/images/mime-icon-32/pdf.png',
            '/media/media/images/mime-icon-32/ppt.png',
            '/media/media/images/mime-icon-32/rar.png',
            '/media/media/images/mime-icon-32/rtf.png',
            '/media/media/images/mime-icon-32/svg.png',
            '/media/media/images/mime-icon-32/sxd.png',
            '/media/media/images/mime-icon-32/tar.png',
            '/media/media/images/mime-icon-32/tgz.png',
            '/media/media/images/mime-icon-32/wma.png',
            '/media/media/images/mime-icon-32/wmv.png',
            '/media/media/images/mime-icon-32/xls.png',
            '/media/media/images/mime-icon-32/zip.png',
            '/media/media/images/progress.gif',
            '/media/media/images/remove.png',
            '/media/media/images/success.png',
            '/media/media/images/upload.png',
            '/media/media/images/uploading.png',
            '/media/media/js/flashmediaelement-cdn.swf',
            '/media/media/js/flashmediaelement.swf',
            '/media/media/js/mediaelement-and-player.js',
            '/media/media/js/mediaelement-and-player.min.js',
            '/media/media/js/mediafield-mootools.js',
            '/media/media/js/mediafield-mootools.min.js',
            '/media/media/js/mediafield.js',
            '/media/media/js/mediafield.min.js',
            '/media/media/js/mediamanager.js',
            '/media/media/js/mediamanager.min.js',
            '/media/media/js/popup-imagemanager.js',
            '/media/media/js/popup-imagemanager.min.js',
            '/media/media/js/silverlightmediaelement.xap',
            '/media/overrider/css/overrider.css',
            '/media/overrider/js/overrider.js',
            '/media/overrider/js/overrider.min.js',
            '/media/plg_system_highlight/highlight.css',
            '/media/plg_twofactorauth_totp/js/qrcode.js',
            '/media/plg_twofactorauth_totp/js/qrcode.min.js',
            '/media/plg_twofactorauth_totp/js/qrcode_SJIS.js',
            '/media/plg_twofactorauth_totp/js/qrcode_UTF8.js',
            '/media/system/css/adminlist.css',
            '/media/system/css/jquery.Jcrop.min.css',
            '/media/system/css/modal.css',
            '/media/system/css/system.css',
            '/media/system/js/associations-edit-uncompressed.js',
            '/media/system/js/associations-edit.js',
            '/media/system/js/calendar-setup-uncompressed.js',
            '/media/system/js/calendar-setup.js',
            '/media/system/js/calendar-uncompressed.js',
            '/media/system/js/calendar.js',
            '/media/system/js/caption-uncompressed.js',
            '/media/system/js/caption.js',
            '/media/system/js/color-field-adv-init.js',
            '/media/system/js/color-field-adv-init.min.js',
            '/media/system/js/color-field-init.js',
            '/media/system/js/color-field-init.min.js',
            '/media/system/js/combobox-uncompressed.js',
            '/media/system/js/combobox.js',
            '/media/system/js/core-uncompressed.js',
            '/media/system/js/fields/calendar-locales/af.js',
            '/media/system/js/fields/calendar-locales/ar.js',
            '/media/system/js/fields/calendar-locales/bg.js',
            '/media/system/js/fields/calendar-locales/bn.js',
            '/media/system/js/fields/calendar-locales/bs.js',
            '/media/system/js/fields/calendar-locales/ca.js',
            '/media/system/js/fields/calendar-locales/cs.js',
            '/media/system/js/fields/calendar-locales/cy.js',
            '/media/system/js/fields/calendar-locales/da.js',
            '/media/system/js/fields/calendar-locales/de.js',
            '/media/system/js/fields/calendar-locales/el.js',
            '/media/system/js/fields/calendar-locales/en.js',
            '/media/system/js/fields/calendar-locales/es.js',
            '/media/system/js/fields/calendar-locales/eu.js',
            '/media/system/js/fields/calendar-locales/fa-ir.js',
            '/media/system/js/fields/calendar-locales/fi.js',
            '/media/system/js/fields/calendar-locales/fr.js',
            '/media/system/js/fields/calendar-locales/ga.js',
            '/media/system/js/fields/calendar-locales/hr.js',
            '/media/system/js/fields/calendar-locales/hu.js',
            '/media/system/js/fields/calendar-locales/it.js',
            '/media/system/js/fields/calendar-locales/ja.js',
            '/media/system/js/fields/calendar-locales/ka.js',
            '/media/system/js/fields/calendar-locales/kk.js',
            '/media/system/js/fields/calendar-locales/ko.js',
            '/media/system/js/fields/calendar-locales/lt.js',
            '/media/system/js/fields/calendar-locales/mk.js',
            '/media/system/js/fields/calendar-locales/nb.js',
            '/media/system/js/fields/calendar-locales/nl.js',
            '/media/system/js/fields/calendar-locales/pl.js',
            '/media/system/js/fields/calendar-locales/prs-af.js',
            '/media/system/js/fields/calendar-locales/pt.js',
            '/media/system/js/fields/calendar-locales/ru.js',
            '/media/system/js/fields/calendar-locales/sk.js',
            '/media/system/js/fields/calendar-locales/sl.js',
            '/media/system/js/fields/calendar-locales/sr-rs.js',
            '/media/system/js/fields/calendar-locales/sr-yu.js',
            '/media/system/js/fields/calendar-locales/sv.js',
            '/media/system/js/fields/calendar-locales/sw.js',
            '/media/system/js/fields/calendar-locales/ta.js',
            '/media/system/js/fields/calendar-locales/th.js',
            '/media/system/js/fields/calendar-locales/uk.js',
            '/media/system/js/fields/calendar-locales/zh-CN.js',
            '/media/system/js/fields/calendar-locales/zh-TW.js',
            '/media/system/js/frontediting-uncompressed.js',
            '/media/system/js/frontediting.js',
            '/media/system/js/helpsite.js',
            '/media/system/js/highlighter-uncompressed.js',
            '/media/system/js/highlighter.js',
            '/media/system/js/html5fallback-uncompressed.js',
            '/media/system/js/html5fallback.js',
            '/media/system/js/jquery.Jcrop.js',
            '/media/system/js/jquery.Jcrop.min.js',
            '/media/system/js/keepalive-uncompressed.js',
            '/media/system/js/modal-fields-uncompressed.js',
            '/media/system/js/modal-fields.js',
            '/media/system/js/modal-uncompressed.js',
            '/media/system/js/modal.js',
            '/media/system/js/moduleorder.js',
            '/media/system/js/mootools-core-uncompressed.js',
            '/media/system/js/mootools-core.js',
            '/media/system/js/mootools-more-uncompressed.js',
            '/media/system/js/mootools-more.js',
            '/media/system/js/mootree-uncompressed.js',
            '/media/system/js/mootree.js',
            '/media/system/js/multiselect-uncompressed.js',
            '/media/system/js/passwordstrength.js',
            '/media/system/js/permissions-uncompressed.js',
            '/media/system/js/permissions.js',
            '/media/system/js/polyfill.classlist-uncompressed.js',
            '/media/system/js/polyfill.classlist.js',
            '/media/system/js/polyfill.event-uncompressed.js',
            '/media/system/js/polyfill.event.js',
            '/media/system/js/polyfill.filter-uncompressed.js',
            '/media/system/js/polyfill.filter.js',
            '/media/system/js/polyfill.map-uncompressed.js',
            '/media/system/js/polyfill.map.js',
            '/media/system/js/polyfill.xpath-uncompressed.js',
            '/media/system/js/polyfill.xpath.js',
            '/media/system/js/progressbar-uncompressed.js',
            '/media/system/js/progressbar.js',
            '/media/system/js/punycode-uncompressed.js',
            '/media/system/js/punycode.js',
            '/media/system/js/repeatable-uncompressed.js',
            '/media/system/js/repeatable.js',
            '/media/system/js/sendtestmail-uncompressed.js',
            '/media/system/js/sendtestmail.js',
            '/media/system/js/subform-repeatable-uncompressed.js',
            '/media/system/js/subform-repeatable.js',
            '/media/system/js/switcher-uncompressed.js',
            '/media/system/js/switcher.js',
            '/media/system/js/tabs-state-uncompressed.js',
            '/media/system/js/tabs-state.js',
            '/media/system/js/tabs.js',
            '/media/system/js/validate-uncompressed.js',
            '/media/system/js/validate.js',
            '/modules/mod_articles_archive/helper.php',
            '/modules/mod_articles_categories/helper.php',
            '/modules/mod_articles_category/helper.php',
            '/modules/mod_articles_latest/helper.php',
            '/modules/mod_articles_news/helper.php',
            '/modules/mod_articles_popular/helper.php',
            '/modules/mod_banners/helper.php',
            '/modules/mod_breadcrumbs/helper.php',
            '/modules/mod_feed/helper.php',
            '/modules/mod_finder/helper.php',
            '/modules/mod_languages/helper.php',
            '/modules/mod_login/helper.php',
            '/modules/mod_menu/helper.php',
            '/modules/mod_random_image/helper.php',
            '/modules/mod_related_items/helper.php',
            '/modules/mod_stats/helper.php',
            '/modules/mod_syndicate/helper.php',
            '/modules/mod_tags_popular/helper.php',
            '/modules/mod_tags_similar/helper.php',
            '/modules/mod_users_latest/helper.php',
            '/modules/mod_whosonline/helper.php',
            '/modules/mod_wrapper/helper.php',
            '/plugins/authentication/gmail/gmail.php',
            '/plugins/authentication/gmail/gmail.xml',
            '/plugins/captcha/recaptcha/postinstall/actions.php',
            '/plugins/content/confirmconsent/fields/consentbox.php',
            '/plugins/editors/codemirror/fonts.php',
            '/plugins/editors/codemirror/layouts/editors/codemirror/init.php',
            '/plugins/editors/tinymce/field/skins.php',
            '/plugins/editors/tinymce/field/tinymcebuilder.php',
            '/plugins/editors/tinymce/field/uploaddirs.php',
            '/plugins/editors/tinymce/form/setoptions.xml',
            '/plugins/quickicon/joomlaupdate/joomlaupdate.php',
            '/plugins/system/languagecode/language/en-GB/en-GB.plg_system_languagecode.ini',
            '/plugins/system/languagecode/language/en-GB/en-GB.plg_system_languagecode.sys.ini',
            '/plugins/system/p3p/p3p.php',
            '/plugins/system/p3p/p3p.xml',
            '/plugins/system/privacyconsent/field/privacy.php',
            '/plugins/system/privacyconsent/privacyconsent/privacyconsent.xml',
            '/plugins/system/stats/field/base.php',
            '/plugins/system/stats/field/data.php',
            '/plugins/system/stats/field/uniqueid.php',
            '/plugins/user/profile/field/dob.php',
            '/plugins/user/profile/field/tos.php',
            '/plugins/user/profile/profiles/profile.xml',
            '/plugins/user/terms/field/terms.php',
            '/plugins/user/terms/terms/terms.xml',
            '/templates/beez3/component.php',
            '/templates/beez3/css/general.css',
            '/templates/beez3/css/ie7only.css',
            '/templates/beez3/css/ieonly.css',
            '/templates/beez3/css/layout.css',
            '/templates/beez3/css/nature.css',
            '/templates/beez3/css/nature_rtl.css',
            '/templates/beez3/css/personal.css',
            '/templates/beez3/css/personal_rtl.css',
            '/templates/beez3/css/position.css',
            '/templates/beez3/css/print.css',
            '/templates/beez3/css/red.css',
            '/templates/beez3/css/template.css',
            '/templates/beez3/css/template_rtl.css',
            '/templates/beez3/css/turq.css',
            '/templates/beez3/css/turq.less',
            '/templates/beez3/error.php',
            '/templates/beez3/favicon.ico',
            '/templates/beez3/html/com_contact/categories/default.php',
            '/templates/beez3/html/com_contact/categories/default_items.php',
            '/templates/beez3/html/com_contact/category/default.php',
            '/templates/beez3/html/com_contact/category/default_children.php',
            '/templates/beez3/html/com_contact/category/default_items.php',
            '/templates/beez3/html/com_contact/contact/default.php',
            '/templates/beez3/html/com_contact/contact/default_address.php',
            '/templates/beez3/html/com_contact/contact/default_articles.php',
            '/templates/beez3/html/com_contact/contact/default_form.php',
            '/templates/beez3/html/com_contact/contact/default_links.php',
            '/templates/beez3/html/com_contact/contact/default_profile.php',
            '/templates/beez3/html/com_contact/contact/default_user_custom_fields.php',
            '/templates/beez3/html/com_contact/contact/encyclopedia.php',
            '/templates/beez3/html/com_content/archive/default.php',
            '/templates/beez3/html/com_content/archive/default_items.php',
            '/templates/beez3/html/com_content/article/default.php',
            '/templates/beez3/html/com_content/article/default_links.php',
            '/templates/beez3/html/com_content/categories/default.php',
            '/templates/beez3/html/com_content/categories/default_items.php',
            '/templates/beez3/html/com_content/category/blog.php',
            '/templates/beez3/html/com_content/category/blog_children.php',
            '/templates/beez3/html/com_content/category/blog_item.php',
            '/templates/beez3/html/com_content/category/blog_links.php',
            '/templates/beez3/html/com_content/category/default.php',
            '/templates/beez3/html/com_content/category/default_articles.php',
            '/templates/beez3/html/com_content/category/default_children.php',
            '/templates/beez3/html/com_content/featured/default.php',
            '/templates/beez3/html/com_content/featured/default_item.php',
            '/templates/beez3/html/com_content/featured/default_links.php',
            '/templates/beez3/html/com_content/form/edit.php',
            '/templates/beez3/html/com_newsfeeds/categories/default.php',
            '/templates/beez3/html/com_newsfeeds/categories/default_items.php',
            '/templates/beez3/html/com_newsfeeds/category/default.php',
            '/templates/beez3/html/com_newsfeeds/category/default_children.php',
            '/templates/beez3/html/com_newsfeeds/category/default_items.php',
            '/templates/beez3/html/com_weblinks/categories/default.php',
            '/templates/beez3/html/com_weblinks/categories/default_items.php',
            '/templates/beez3/html/com_weblinks/category/default.php',
            '/templates/beez3/html/com_weblinks/category/default_children.php',
            '/templates/beez3/html/com_weblinks/category/default_items.php',
            '/templates/beez3/html/com_weblinks/form/edit.php',
            '/templates/beez3/html/layouts/joomla/system/message.php',
            '/templates/beez3/html/mod_breadcrumbs/default.php',
            '/templates/beez3/html/mod_languages/default.php',
            '/templates/beez3/html/mod_login/default.php',
            '/templates/beez3/html/mod_login/default_logout.php',
            '/templates/beez3/html/modules.php',
            '/templates/beez3/images/all_bg.gif',
            '/templates/beez3/images/arrow.png',
            '/templates/beez3/images/arrow2_grey.png',
            '/templates/beez3/images/arrow_white_grey.png',
            '/templates/beez3/images/blog_more.gif',
            '/templates/beez3/images/blog_more_hover.gif',
            '/templates/beez3/images/close.png',
            '/templates/beez3/images/content_bg.gif',
            '/templates/beez3/images/footer_bg.gif',
            '/templates/beez3/images/footer_bg.png',
            '/templates/beez3/images/header-bg.gif',
            '/templates/beez3/images/minus.png',
            '/templates/beez3/images/nature/arrow1.gif',
            '/templates/beez3/images/nature/arrow1_rtl.gif',
            '/templates/beez3/images/nature/arrow2.gif',
            '/templates/beez3/images/nature/arrow2_grey.png',
            '/templates/beez3/images/nature/arrow2_rtl.gif',
            '/templates/beez3/images/nature/arrow_nav.gif',
            '/templates/beez3/images/nature/arrow_small.png',
            '/templates/beez3/images/nature/arrow_small_rtl.png',
            '/templates/beez3/images/nature/blog_more.gif',
            '/templates/beez3/images/nature/box.png',
            '/templates/beez3/images/nature/box1.png',
            '/templates/beez3/images/nature/grey_bg.png',
            '/templates/beez3/images/nature/headingback.png',
            '/templates/beez3/images/nature/karo.gif',
            '/templates/beez3/images/nature/level4.png',
            '/templates/beez3/images/nature/nav_level1_a.gif',
            '/templates/beez3/images/nature/nav_level_1.gif',
            '/templates/beez3/images/nature/pfeil.gif',
            '/templates/beez3/images/nature/readmore_arrow.png',
            '/templates/beez3/images/nature/searchbutton.png',
            '/templates/beez3/images/nature/tabs.gif',
            '/templates/beez3/images/nav_level_1.gif',
            '/templates/beez3/images/news.gif',
            '/templates/beez3/images/personal/arrow2_grey.jpg',
            '/templates/beez3/images/personal/arrow2_grey.png',
            '/templates/beez3/images/personal/bg2.png',
            '/templates/beez3/images/personal/button.png',
            '/templates/beez3/images/personal/dot.png',
            '/templates/beez3/images/personal/ecke.gif',
            '/templates/beez3/images/personal/footer.jpg',
            '/templates/beez3/images/personal/grey_bg.png',
            '/templates/beez3/images/personal/navi_active.png',
            '/templates/beez3/images/personal/personal2.png',
            '/templates/beez3/images/personal/readmore_arrow.png',
            '/templates/beez3/images/personal/readmore_arrow_hover.png',
            '/templates/beez3/images/personal/tabs_back.png',
            '/templates/beez3/images/plus.png',
            '/templates/beez3/images/req.png',
            '/templates/beez3/images/slider_minus.png',
            '/templates/beez3/images/slider_minus_rtl.png',
            '/templates/beez3/images/slider_plus.png',
            '/templates/beez3/images/slider_plus_rtl.png',
            '/templates/beez3/images/system/arrow.png',
            '/templates/beez3/images/system/arrow_rtl.png',
            '/templates/beez3/images/system/calendar.png',
            '/templates/beez3/images/system/j_button2_blank.png',
            '/templates/beez3/images/system/j_button2_image.png',
            '/templates/beez3/images/system/j_button2_left.png',
            '/templates/beez3/images/system/j_button2_pagebreak.png',
            '/templates/beez3/images/system/j_button2_readmore.png',
            '/templates/beez3/images/system/notice-alert.png',
            '/templates/beez3/images/system/notice-alert_rtl.png',
            '/templates/beez3/images/system/notice-info.png',
            '/templates/beez3/images/system/notice-info_rtl.png',
            '/templates/beez3/images/system/notice-note.png',
            '/templates/beez3/images/system/notice-note_rtl.png',
            '/templates/beez3/images/system/selector-arrow.png',
            '/templates/beez3/images/table_footer.gif',
            '/templates/beez3/images/trans.gif',
            '/templates/beez3/index.php',
            '/templates/beez3/javascript/hide.js',
            '/templates/beez3/javascript/md_stylechanger.js',
            '/templates/beez3/javascript/respond.js',
            '/templates/beez3/javascript/respond.src.js',
            '/templates/beez3/javascript/template.js',
            '/templates/beez3/jsstrings.php',
            '/templates/beez3/language/en-GB/en-GB.tpl_beez3.ini',
            '/templates/beez3/language/en-GB/en-GB.tpl_beez3.sys.ini',
            '/templates/beez3/templateDetails.xml',
            '/templates/beez3/template_preview.png',
            '/templates/beez3/template_thumbnail.png',
            '/templates/protostar/component.php',
            '/templates/protostar/css/offline.css',
            '/templates/protostar/css/template.css',
            '/templates/protostar/error.php',
            '/templates/protostar/favicon.ico',
            '/templates/protostar/html/com_media/imageslist/default_folder.php',
            '/templates/protostar/html/com_media/imageslist/default_image.php',
            '/templates/protostar/html/layouts/joomla/form/field/contenthistory.php',
            '/templates/protostar/html/layouts/joomla/form/field/media.php',
            '/templates/protostar/html/layouts/joomla/form/field/user.php',
            '/templates/protostar/html/layouts/joomla/system/message.php',
            '/templates/protostar/html/modules.php',
            '/templates/protostar/html/pagination.php',
            '/templates/protostar/images/logo.png',
            '/templates/protostar/images/system/rating_star.png',
            '/templates/protostar/images/system/rating_star_blank.png',
            '/templates/protostar/images/system/sort_asc.png',
            '/templates/protostar/images/system/sort_desc.png',
            '/templates/protostar/img/glyphicons-halflings-white.png',
            '/templates/protostar/img/glyphicons-halflings.png',
            '/templates/protostar/index.php',
            '/templates/protostar/js/application.js',
            '/templates/protostar/js/classes.js',
            '/templates/protostar/js/template.js',
            '/templates/protostar/language/en-GB/en-GB.tpl_protostar.ini',
            '/templates/protostar/language/en-GB/en-GB.tpl_protostar.sys.ini',
            '/templates/protostar/less/icomoon.less',
            '/templates/protostar/less/template.less',
            '/templates/protostar/less/template_rtl.less',
            '/templates/protostar/less/variables.less',
            '/templates/protostar/offline.php',
            '/templates/protostar/templateDetails.xml',
            '/templates/protostar/template_preview.png',
            '/templates/protostar/template_thumbnail.png',
            '/templates/system/css/system.css',
            '/templates/system/css/toolbar.css',
            '/templates/system/html/modules.php',
            '/templates/system/images/calendar.png',
            '/templates/system/images/j_button2_blank.png',
            '/templates/system/images/j_button2_image.png',
            '/templates/system/images/j_button2_left.png',
            '/templates/system/images/j_button2_pagebreak.png',
            '/templates/system/images/j_button2_readmore.png',
            '/templates/system/images/j_button2_right.png',
            '/templates/system/images/selector-arrow.png',
            // 4.0 from Beta 1 to Beta 2
            '/administrator/components/com_finder/src/Indexer/Driver/Mysql.php',
            '/administrator/components/com_finder/src/Indexer/Driver/Postgresql.php',
            '/administrator/components/com_workflow/access.xml',
            '/api/components/com_installer/src/Controller/LanguagesController.php',
            '/api/components/com_installer/src/View/Languages/JsonapiView.php',
            '/libraries/vendor/joomla/controller/LICENSE',
            '/libraries/vendor/joomla/controller/src/AbstractController.php',
            '/libraries/vendor/joomla/controller/src/ControllerInterface.php',
            '/media/com_users/js/admin-users-user.es6.js',
            '/media/com_users/js/admin-users-user.es6.min.js',
            '/media/com_users/js/admin-users-user.es6.min.js.gz',
            '/media/com_users/js/admin-users-user.js',
            '/media/com_users/js/admin-users-user.min.js',
            '/media/com_users/js/admin-users-user.min.js.gz',
            // 4.0 from Beta 2 to Beta 3
            '/administrator/templates/atum/images/logo-blue.svg',
            '/administrator/templates/atum/images/logo-joomla-blue.svg',
            '/administrator/templates/atum/images/logo-joomla-white.svg',
            '/administrator/templates/atum/images/logo.svg',
            // 4.0 from Beta 3 to Beta 4
            '/components/com_config/src/Model/CmsModel.php',
            // 4.0 from Beta 4 to Beta 5
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-06-11.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-04-18.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-06-11.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-04-18.sql',
            '/administrator/components/com_config/tmpl/application/default_system.php',
            '/administrator/language/en-GB/plg_content_imagelazyload.sys.ini',
            '/administrator/language/en-GB/plg_fields_image.ini',
            '/administrator/language/en-GB/plg_fields_image.sys.ini',
            '/administrator/templates/atum/scss/vendor/bootstrap/_nav.scss',
            '/libraries/vendor/spomky-labs/base64url/phpstan.neon',
            '/media/plg_system_webauthn/images/webauthn-black.png',
            '/media/plg_system_webauthn/images/webauthn-color.png',
            '/media/plg_system_webauthn/images/webauthn-white.png',
            '/media/system/css/system.min.css',
            '/media/system/css/system.min.css.gz',
            '/plugins/content/imagelazyload/imagelazyload.php',
            '/plugins/content/imagelazyload/imagelazyload.xml',
            '/templates/cassiopeia/html/layouts/chromes/cardGrey.php',
            '/templates/cassiopeia/html/layouts/chromes/default.php',
            '/templates/cassiopeia/scss/vendor/bootstrap/_card.scss',
            // 4.0 from Beta 5 to Beta 6
            '/administrator/modules/mod_multilangstatus/src/Helper/MultilangstatusAdminHelper.php',
            '/administrator/templates/atum/favicon.ico',
            '/libraries/vendor/nyholm/psr7/phpstan.baseline.dist',
            '/libraries/vendor/spomky-labs/base64url/.php_cs.dist',
            '/libraries/vendor/spomky-labs/base64url/infection.json.dist',
            '/media/layouts/js/joomla/html/batch/batch-language.es6.js',
            '/media/layouts/js/joomla/html/batch/batch-language.es6.min.js',
            '/media/layouts/js/joomla/html/batch/batch-language.es6.min.js.gz',
            '/media/layouts/js/joomla/html/batch/batch-language.js',
            '/media/layouts/js/joomla/html/batch/batch-language.min.js',
            '/media/layouts/js/joomla/html/batch/batch-language.min.js.gz',
            '/media/plg_system_webauthn/images/webauthn-black.svg',
            '/media/plg_system_webauthn/images/webauthn-white.svg',
            '/media/system/js/core.es6/ajax.es6',
            '/media/system/js/core.es6/customevent.es6',
            '/media/system/js/core.es6/event.es6',
            '/media/system/js/core.es6/form.es6',
            '/media/system/js/core.es6/message.es6',
            '/media/system/js/core.es6/options.es6',
            '/media/system/js/core.es6/text.es6',
            '/media/system/js/core.es6/token.es6',
            '/media/system/js/core.es6/webcomponent.es6',
            '/templates/cassiopeia/favicon.ico',
            '/templates/cassiopeia/scss/_mixin.scss',
            '/templates/cassiopeia/scss/_variables.scss',
            '/templates/cassiopeia/scss/blocks/_demo-styling.scss',
            // 4.0 from Beta 6 to Beta 7
            '/media/legacy/js/bootstrap-init.js',
            '/media/legacy/js/bootstrap-init.min.js',
            '/media/legacy/js/bootstrap-init.min.js.gz',
            '/media/legacy/js/frontediting.js',
            '/media/legacy/js/frontediting.min.js',
            '/media/legacy/js/frontediting.min.js.gz',
            '/media/vendor/bootstrap/js/bootstrap.bundle.js',
            '/media/vendor/bootstrap/js/bootstrap.bundle.min.js',
            '/media/vendor/bootstrap/js/bootstrap.bundle.min.js.gz',
            '/media/vendor/bootstrap/js/bootstrap.bundle.min.js.map',
            '/media/vendor/bootstrap/js/bootstrap.js',
            '/media/vendor/bootstrap/js/bootstrap.min.js',
            '/media/vendor/bootstrap/js/bootstrap.min.js.gz',
            '/media/vendor/bootstrap/scss/_code.scss',
            '/media/vendor/bootstrap/scss/_custom-forms.scss',
            '/media/vendor/bootstrap/scss/_input-group.scss',
            '/media/vendor/bootstrap/scss/_jumbotron.scss',
            '/media/vendor/bootstrap/scss/_media.scss',
            '/media/vendor/bootstrap/scss/_print.scss',
            '/media/vendor/bootstrap/scss/mixins/_background-variant.scss',
            '/media/vendor/bootstrap/scss/mixins/_badge.scss',
            '/media/vendor/bootstrap/scss/mixins/_float.scss',
            '/media/vendor/bootstrap/scss/mixins/_grid-framework.scss',
            '/media/vendor/bootstrap/scss/mixins/_hover.scss',
            '/media/vendor/bootstrap/scss/mixins/_nav-divider.scss',
            '/media/vendor/bootstrap/scss/mixins/_screen-reader.scss',
            '/media/vendor/bootstrap/scss/mixins/_size.scss',
            '/media/vendor/bootstrap/scss/mixins/_table-row.scss',
            '/media/vendor/bootstrap/scss/mixins/_text-emphasis.scss',
            '/media/vendor/bootstrap/scss/mixins/_text-hide.scss',
            '/media/vendor/bootstrap/scss/mixins/_visibility.scss',
            '/media/vendor/bootstrap/scss/utilities/_align.scss',
            '/media/vendor/bootstrap/scss/utilities/_background.scss',
            '/media/vendor/bootstrap/scss/utilities/_borders.scss',
            '/media/vendor/bootstrap/scss/utilities/_clearfix.scss',
            '/media/vendor/bootstrap/scss/utilities/_display.scss',
            '/media/vendor/bootstrap/scss/utilities/_embed.scss',
            '/media/vendor/bootstrap/scss/utilities/_flex.scss',
            '/media/vendor/bootstrap/scss/utilities/_float.scss',
            '/media/vendor/bootstrap/scss/utilities/_interactions.scss',
            '/media/vendor/bootstrap/scss/utilities/_overflow.scss',
            '/media/vendor/bootstrap/scss/utilities/_position.scss',
            '/media/vendor/bootstrap/scss/utilities/_screenreaders.scss',
            '/media/vendor/bootstrap/scss/utilities/_shadows.scss',
            '/media/vendor/bootstrap/scss/utilities/_sizing.scss',
            '/media/vendor/bootstrap/scss/utilities/_spacing.scss',
            '/media/vendor/bootstrap/scss/utilities/_stretched-link.scss',
            '/media/vendor/bootstrap/scss/utilities/_text.scss',
            '/media/vendor/bootstrap/scss/utilities/_visibility.scss',
            '/media/vendor/skipto/css/SkipTo.css',
            '/media/vendor/skipto/js/dropMenu.js',
            // 4.0 from Beta 7 to RC 1
            '/administrator/components/com_admin/forms/profile.xml',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2016-07-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2016-09-22.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2016-09-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2016-10-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2016-10-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2017-03-18.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2017-04-25.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2017-05-31.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2017-06-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2017-10-10.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-02-24.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-06-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-06-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-07-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-08-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-09-12.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2018-10-18.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-01-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-01-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-02-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-03-31.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-05-05.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-06-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-07-02.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-07-14.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-07-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-08-03.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-08-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-08-21.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-14.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-23.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-24.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-25.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-26.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-27.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-09-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-10-13.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-10-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-11-07.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2019-11-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-02-08.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-02-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-02-22.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-02-29.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-04-11.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-04-16.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-05-21.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-09-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-09-22.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-12-08.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2020-12-19.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-02-28.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-04-11.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-04-20.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-05-01.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-05-04.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-05-07.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-05-10.sql',
            '/administrator/components/com_admin/sql/updates/mysql/4.0.0-2021-05-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2016-07-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2016-09-22.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2016-09-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2016-10-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2016-10-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2017-03-18.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2017-04-25.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2017-05-31.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2017-06-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2017-10-10.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-02-24.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-06-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-06-26.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-07-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-08-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-09-12.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2018-10-18.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-01-05.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-01-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-02-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-03-31.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-05-05.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-06-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-07-02.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-07-14.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-07-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-08-03.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-08-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-08-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-14.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-23.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-24.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-25.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-26.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-27.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-09-29.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-10-13.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-10-29.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-11-07.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2019-11-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-02-08.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-02-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-02-22.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-02-29.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-04-11.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-04-16.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-05-21.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-09-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-09-22.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-12-08.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2020-12-19.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-02-28.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-04-11.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-04-20.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-05-01.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-05-04.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-05-07.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-05-10.sql',
            '/administrator/components/com_admin/sql/updates/postgresql/4.0.0-2021-05-21.sql',
            '/administrator/components/com_admin/src/Controller/ProfileController.php',
            '/administrator/components/com_admin/src/Model/ProfileModel.php',
            '/administrator/components/com_admin/src/View/Profile/HtmlView.php',
            '/administrator/components/com_admin/tmpl/profile/edit.php',
            '/administrator/components/com_config/tmpl/application/default_ftp.php',
            '/administrator/components/com_config/tmpl/application/default_ftplogin.php',
            '/administrator/components/com_csp/access.xml',
            '/administrator/components/com_csp/config.xml',
            '/administrator/components/com_csp/csp.xml',
            '/administrator/components/com_csp/forms/filter_reports.xml',
            '/administrator/components/com_csp/services/provider.php',
            '/administrator/components/com_csp/src/Controller/DisplayController.php',
            '/administrator/components/com_csp/src/Controller/ReportsController.php',
            '/administrator/components/com_csp/src/Helper/ReporterHelper.php',
            '/administrator/components/com_csp/src/Model/ReportModel.php',
            '/administrator/components/com_csp/src/Model/ReportsModel.php',
            '/administrator/components/com_csp/src/Table/ReportTable.php',
            '/administrator/components/com_csp/src/View/Reports/HtmlView.php',
            '/administrator/components/com_csp/tmpl/reports/default.php',
            '/administrator/components/com_csp/tmpl/reports/default.xml',
            '/administrator/components/com_fields/src/Field/SubfieldstypeField.php',
            '/administrator/components/com_installer/tmpl/installer/default_ftp.php',
            '/administrator/components/com_joomlaupdate/src/Helper/Select.php',
            '/administrator/language/en-GB/com_csp.ini',
            '/administrator/language/en-GB/com_csp.sys.ini',
            '/administrator/language/en-GB/plg_fields_subfields.ini',
            '/administrator/language/en-GB/plg_fields_subfields.sys.ini',
            '/administrator/templates/atum/Service/HTML/Atum.php',
            '/components/com_csp/src/Controller/ReportController.php',
            '/components/com_menus/src/Controller/DisplayController.php',
            '/libraries/vendor/algo26-matthias/idna-convert/CODE_OF_CONDUCT.md',
            '/libraries/vendor/algo26-matthias/idna-convert/UPGRADING.md',
            '/libraries/vendor/algo26-matthias/idna-convert/docker-compose.yml',
            '/libraries/vendor/beberlei/assert/phpstan-code.neon',
            '/libraries/vendor/beberlei/assert/phpstan-tests.neon',
            '/libraries/vendor/bin/generate-defuse-key',
            '/libraries/vendor/bin/var-dump-server',
            '/libraries/vendor/bin/yaml-lint',
            '/libraries/vendor/brick/math/psalm-baseline.xml',
            '/libraries/vendor/doctrine/inflector/phpstan.neon.dist',
            '/libraries/vendor/jakeasmith/http_build_url/readme.md',
            '/libraries/vendor/nyholm/psr7/src/LowercaseTrait.php',
            '/libraries/vendor/ozdemirburak/iris/LICENSE.md',
            '/libraries/vendor/ozdemirburak/iris/src/BaseColor.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Factory.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Hex.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Hsl.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Hsla.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Hsv.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Rgb.php',
            '/libraries/vendor/ozdemirburak/iris/src/Color/Rgba.php',
            '/libraries/vendor/ozdemirburak/iris/src/Exceptions/AmbiguousColorString.php',
            '/libraries/vendor/ozdemirburak/iris/src/Exceptions/InvalidColorException.php',
            '/libraries/vendor/ozdemirburak/iris/src/Helpers/DefinedColor.php',
            '/libraries/vendor/ozdemirburak/iris/src/Traits/AlphaTrait.php',
            '/libraries/vendor/ozdemirburak/iris/src/Traits/HsTrait.php',
            '/libraries/vendor/ozdemirburak/iris/src/Traits/HslTrait.php',
            '/libraries/vendor/ozdemirburak/iris/src/Traits/RgbTrait.php',
            '/libraries/vendor/paragonie/random_compat/dist/random_compat.phar.pubkey',
            '/libraries/vendor/paragonie/random_compat/dist/random_compat.phar.pubkey.asc',
            '/libraries/vendor/psr/http-factory/.pullapprove.yml',
            '/libraries/vendor/spomky-labs/cbor-php/.php_cs.dist',
            '/libraries/vendor/spomky-labs/cbor-php/CODE_OF_CONDUCT.md',
            '/libraries/vendor/spomky-labs/cbor-php/infection.json.dist',
            '/libraries/vendor/spomky-labs/cbor-php/phpstan.neon',
            '/libraries/vendor/typo3/phar-stream-wrapper/_config.yml',
            '/libraries/vendor/voku/portable-utf8/SUMMARY.md',
            '/libraries/vendor/willdurand/negotiation/src/Negotiation/Match.php',
            '/media/com_actionlogs/js/admin-actionlogs-default.es6.js',
            '/media/com_actionlogs/js/admin-actionlogs-default.es6.min.js',
            '/media/com_actionlogs/js/admin-actionlogs-default.es6.min.js.gz',
            '/media/com_associations/js/admin-associations-default.es6.js',
            '/media/com_associations/js/admin-associations-default.es6.min.js',
            '/media/com_associations/js/admin-associations-default.es6.min.js.gz',
            '/media/com_associations/js/admin-associations-modal.es6.js',
            '/media/com_associations/js/admin-associations-modal.es6.min.js',
            '/media/com_associations/js/admin-associations-modal.es6.min.js.gz',
            '/media/com_associations/js/associations-edit.es6.js',
            '/media/com_associations/js/associations-edit.es6.min.js',
            '/media/com_associations/js/associations-edit.es6.min.js.gz',
            '/media/com_banners/js/admin-banner-edit.es6.js',
            '/media/com_banners/js/admin-banner-edit.es6.min.js',
            '/media/com_banners/js/admin-banner-edit.es6.min.js.gz',
            '/media/com_cache/js/admin-cache-default.es6.js',
            '/media/com_cache/js/admin-cache-default.es6.min.js',
            '/media/com_cache/js/admin-cache-default.es6.min.js.gz',
            '/media/com_categories/js/shared-categories-accordion.es6.js',
            '/media/com_categories/js/shared-categories-accordion.es6.min.js',
            '/media/com_categories/js/shared-categories-accordion.es6.min.js.gz',
            '/media/com_config/js/config-default.es6.js',
            '/media/com_config/js/config-default.es6.min.js',
            '/media/com_config/js/config-default.es6.min.js.gz',
            '/media/com_config/js/modules-default.es6.js',
            '/media/com_config/js/modules-default.es6.min.js',
            '/media/com_config/js/modules-default.es6.min.js.gz',
            '/media/com_config/js/templates-default.es6.js',
            '/media/com_config/js/templates-default.es6.min.js',
            '/media/com_config/js/templates-default.es6.min.js.gz',
            '/media/com_contact/js/admin-contacts-modal.es6.js',
            '/media/com_contact/js/admin-contacts-modal.es6.min.js',
            '/media/com_contact/js/admin-contacts-modal.es6.min.js.gz',
            '/media/com_contact/js/contacts-list.es6.js',
            '/media/com_contact/js/contacts-list.es6.min.js',
            '/media/com_contact/js/contacts-list.es6.min.js.gz',
            '/media/com_content/js/admin-article-pagebreak.es6.js',
            '/media/com_content/js/admin-article-pagebreak.es6.min.js',
            '/media/com_content/js/admin-article-pagebreak.es6.min.js.gz',
            '/media/com_content/js/admin-article-readmore.es6.js',
            '/media/com_content/js/admin-article-readmore.es6.min.js',
            '/media/com_content/js/admin-article-readmore.es6.min.js.gz',
            '/media/com_content/js/admin-articles-default-batch-footer.es6.js',
            '/media/com_content/js/admin-articles-default-batch-footer.es6.min.js',
            '/media/com_content/js/admin-articles-default-batch-footer.es6.min.js.gz',
            '/media/com_content/js/admin-articles-default-stage-footer.es6.js',
            '/media/com_content/js/admin-articles-default-stage-footer.es6.min.js',
            '/media/com_content/js/admin-articles-default-stage-footer.es6.min.js.gz',
            '/media/com_content/js/admin-articles-modal.es6.js',
            '/media/com_content/js/admin-articles-modal.es6.min.js',
            '/media/com_content/js/admin-articles-modal.es6.min.js.gz',
            '/media/com_content/js/articles-list.es6.js',
            '/media/com_content/js/articles-list.es6.min.js',
            '/media/com_content/js/articles-list.es6.min.js.gz',
            '/media/com_content/js/form-edit.es6.js',
            '/media/com_content/js/form-edit.es6.min.js',
            '/media/com_content/js/form-edit.es6.min.js.gz',
            '/media/com_contenthistory/js/admin-compare-compare.es6.js',
            '/media/com_contenthistory/js/admin-compare-compare.es6.min.js',
            '/media/com_contenthistory/js/admin-compare-compare.es6.min.js.gz',
            '/media/com_contenthistory/js/admin-history-modal.es6.js',
            '/media/com_contenthistory/js/admin-history-modal.es6.min.js',
            '/media/com_contenthistory/js/admin-history-modal.es6.min.js.gz',
            '/media/com_contenthistory/js/admin-history-versions.es6.js',
            '/media/com_contenthistory/js/admin-history-versions.es6.min.js',
            '/media/com_contenthistory/js/admin-history-versions.es6.min.js.gz',
            '/media/com_cpanel/js/admin-add_module.es6.js',
            '/media/com_cpanel/js/admin-add_module.es6.min.js',
            '/media/com_cpanel/js/admin-add_module.es6.min.js.gz',
            '/media/com_cpanel/js/admin-cpanel-default.es6.js',
            '/media/com_cpanel/js/admin-cpanel-default.es6.min.js',
            '/media/com_cpanel/js/admin-cpanel-default.es6.min.js.gz',
            '/media/com_cpanel/js/admin-system-loader.es6.js',
            '/media/com_cpanel/js/admin-system-loader.es6.min.js',
            '/media/com_cpanel/js/admin-system-loader.es6.min.js.gz',
            '/media/com_fields/js/admin-field-changecontext.es6.js',
            '/media/com_fields/js/admin-field-changecontext.es6.min.js',
            '/media/com_fields/js/admin-field-changecontext.es6.min.js.gz',
            '/media/com_fields/js/admin-field-edit-modal.es6.js',
            '/media/com_fields/js/admin-field-edit-modal.es6.min.js',
            '/media/com_fields/js/admin-field-edit-modal.es6.min.js.gz',
            '/media/com_fields/js/admin-field-edit.es6.js',
            '/media/com_fields/js/admin-field-edit.es6.min.js',
            '/media/com_fields/js/admin-field-edit.es6.min.js.gz',
            '/media/com_fields/js/admin-field-typehaschanged.es6.js',
            '/media/com_fields/js/admin-field-typehaschanged.es6.min.js',
            '/media/com_fields/js/admin-field-typehaschanged.es6.min.js.gz',
            '/media/com_fields/js/admin-fields-default-batch.es6.js',
            '/media/com_fields/js/admin-fields-default-batch.es6.min.js',
            '/media/com_fields/js/admin-fields-default-batch.es6.min.js.gz',
            '/media/com_fields/js/admin-fields-modal.es6.js',
            '/media/com_fields/js/admin-fields-modal.es6.min.js',
            '/media/com_fields/js/admin-fields-modal.es6.min.js.gz',
            '/media/com_finder/js/filters.es6.js',
            '/media/com_finder/js/filters.es6.min.js',
            '/media/com_finder/js/filters.es6.min.js.gz',
            '/media/com_finder/js/finder-edit.es6.js',
            '/media/com_finder/js/finder-edit.es6.min.js',
            '/media/com_finder/js/finder-edit.es6.min.js.gz',
            '/media/com_finder/js/finder.es6.js',
            '/media/com_finder/js/finder.es6.min.js',
            '/media/com_finder/js/finder.es6.min.js.gz',
            '/media/com_finder/js/index.es6.js',
            '/media/com_finder/js/index.es6.min.js',
            '/media/com_finder/js/index.es6.min.js.gz',
            '/media/com_finder/js/indexer.es6.js',
            '/media/com_finder/js/indexer.es6.min.js',
            '/media/com_finder/js/indexer.es6.min.js.gz',
            '/media/com_finder/js/maps.es6.js',
            '/media/com_finder/js/maps.es6.min.js',
            '/media/com_finder/js/maps.es6.min.js.gz',
            '/media/com_installer/js/changelog.es6.js',
            '/media/com_installer/js/changelog.es6.min.js',
            '/media/com_installer/js/changelog.es6.min.js.gz',
            '/media/com_installer/js/installer.es6.js',
            '/media/com_installer/js/installer.es6.min.js',
            '/media/com_installer/js/installer.es6.min.js.gz',
            '/media/com_joomlaupdate/js/admin-update-default.es6.js',
            '/media/com_joomlaupdate/js/admin-update-default.es6.min.js',
            '/media/com_joomlaupdate/js/admin-update-default.es6.min.js.gz',
            '/media/com_languages/js/admin-language-edit-change-flag.es6.js',
            '/media/com_languages/js/admin-language-edit-change-flag.es6.min.js',
            '/media/com_languages/js/admin-language-edit-change-flag.es6.min.js.gz',
            '/media/com_languages/js/admin-override-edit-refresh-searchstring.es6.js',
            '/media/com_languages/js/admin-override-edit-refresh-searchstring.es6.min.js',
            '/media/com_languages/js/admin-override-edit-refresh-searchstring.es6.min.js.gz',
            '/media/com_languages/js/overrider.es6.js',
            '/media/com_languages/js/overrider.es6.min.js',
            '/media/com_languages/js/overrider.es6.min.js.gz',
            '/media/com_mails/js/admin-email-template-edit.es6.js',
            '/media/com_mails/js/admin-email-template-edit.es6.min.js',
            '/media/com_mails/js/admin-email-template-edit.es6.min.js.gz',
            '/media/com_media/css/mediamanager.min.css',
            '/media/com_media/css/mediamanager.min.css.gz',
            '/media/com_media/css/mediamanager.min.css.map',
            '/media/com_media/js/edit-images.es6.js',
            '/media/com_media/js/edit-images.es6.min.js',
            '/media/com_media/js/mediamanager.min.js',
            '/media/com_media/js/mediamanager.min.js.gz',
            '/media/com_media/js/mediamanager.min.js.map',
            '/media/com_menus/js/admin-item-edit.es6.js',
            '/media/com_menus/js/admin-item-edit.es6.min.js',
            '/media/com_menus/js/admin-item-edit.es6.min.js.gz',
            '/media/com_menus/js/admin-item-edit_container.es6.js',
            '/media/com_menus/js/admin-item-edit_container.es6.min.js',
            '/media/com_menus/js/admin-item-edit_container.es6.min.js.gz',
            '/media/com_menus/js/admin-item-edit_modules.es6.js',
            '/media/com_menus/js/admin-item-edit_modules.es6.min.js',
            '/media/com_menus/js/admin-item-edit_modules.es6.min.js.gz',
            '/media/com_menus/js/admin-item-modal.es6.js',
            '/media/com_menus/js/admin-item-modal.es6.min.js',
            '/media/com_menus/js/admin-item-modal.es6.min.js.gz',
            '/media/com_menus/js/admin-items-modal.es6.js',
            '/media/com_menus/js/admin-items-modal.es6.min.js',
            '/media/com_menus/js/admin-items-modal.es6.min.js.gz',
            '/media/com_menus/js/admin-menus-default.es6.js',
            '/media/com_menus/js/admin-menus-default.es6.min.js',
            '/media/com_menus/js/admin-menus-default.es6.min.js.gz',
            '/media/com_menus/js/default-batch-body.es6.js',
            '/media/com_menus/js/default-batch-body.es6.min.js',
            '/media/com_menus/js/default-batch-body.es6.min.js.gz',
            '/media/com_modules/js/admin-module-edit.es6.js',
            '/media/com_modules/js/admin-module-edit.es6.min.js',
            '/media/com_modules/js/admin-module-edit.es6.min.js.gz',
            '/media/com_modules/js/admin-module-edit_assignment.es6.js',
            '/media/com_modules/js/admin-module-edit_assignment.es6.min.js',
            '/media/com_modules/js/admin-module-edit_assignment.es6.min.js.gz',
            '/media/com_modules/js/admin-module-search.es6.js',
            '/media/com_modules/js/admin-module-search.es6.min.js',
            '/media/com_modules/js/admin-module-search.es6.min.js.gz',
            '/media/com_modules/js/admin-modules-modal.es6.js',
            '/media/com_modules/js/admin-modules-modal.es6.min.js',
            '/media/com_modules/js/admin-modules-modal.es6.min.js.gz',
            '/media/com_modules/js/admin-select-modal.es6.js',
            '/media/com_modules/js/admin-select-modal.es6.min.js',
            '/media/com_modules/js/admin-select-modal.es6.min.js.gz',
            '/media/com_tags/js/tag-default.es6.js',
            '/media/com_tags/js/tag-default.es6.min.js',
            '/media/com_tags/js/tag-default.es6.min.js.gz',
            '/media/com_tags/js/tag-list.es6.js',
            '/media/com_tags/js/tag-list.es6.min.js',
            '/media/com_tags/js/tag-list.es6.min.js.gz',
            '/media/com_tags/js/tags-default.es6.js',
            '/media/com_tags/js/tags-default.es6.min.js',
            '/media/com_tags/js/tags-default.es6.min.js.gz',
            '/media/com_templates/js/admin-template-compare.es6.js',
            '/media/com_templates/js/admin-template-compare.es6.min.js',
            '/media/com_templates/js/admin-template-compare.es6.min.js.gz',
            '/media/com_templates/js/admin-template-toggle-assignment.es6.js',
            '/media/com_templates/js/admin-template-toggle-assignment.es6.min.js',
            '/media/com_templates/js/admin-template-toggle-assignment.es6.min.js.gz',
            '/media/com_templates/js/admin-template-toggle-switch.es6.js',
            '/media/com_templates/js/admin-template-toggle-switch.es6.min.js',
            '/media/com_templates/js/admin-template-toggle-switch.es6.min.js.gz',
            '/media/com_templates/js/admin-templates-default.es6.js',
            '/media/com_templates/js/admin-templates-default.es6.min.js',
            '/media/com_templates/js/admin-templates-default.es6.min.js.gz',
            '/media/com_users/js/admin-users-groups.es6.js',
            '/media/com_users/js/admin-users-groups.es6.min.js',
            '/media/com_users/js/admin-users-groups.es6.min.js.gz',
            '/media/com_users/js/admin-users-mail.es6.js',
            '/media/com_users/js/admin-users-mail.es6.min.js',
            '/media/com_users/js/admin-users-mail.es6.min.js.gz',
            '/media/com_users/js/two-factor-switcher.es6.js',
            '/media/com_users/js/two-factor-switcher.es6.min.js',
            '/media/com_users/js/two-factor-switcher.es6.min.js.gz',
            '/media/com_workflow/js/admin-items-workflow-buttons.es6.js',
            '/media/com_workflow/js/admin-items-workflow-buttons.es6.min.js',
            '/media/com_workflow/js/admin-items-workflow-buttons.es6.min.js.gz',
            '/media/com_wrapper/js/iframe-height.es6.js',
            '/media/com_wrapper/js/iframe-height.es6.min.js',
            '/media/com_wrapper/js/iframe-height.es6.min.js.gz',
            '/media/layouts/js/joomla/form/field/category-change.es6.js',
            '/media/layouts/js/joomla/form/field/category-change.es6.min.js',
            '/media/layouts/js/joomla/form/field/category-change.es6.min.js.gz',
            '/media/layouts/js/joomla/html/batch/batch-copymove.es6.js',
            '/media/layouts/js/joomla/html/batch/batch-copymove.es6.min.js',
            '/media/layouts/js/joomla/html/batch/batch-copymove.es6.min.js.gz',
            '/media/legacy/js/highlighter.js',
            '/media/legacy/js/highlighter.min.js',
            '/media/legacy/js/highlighter.min.js.gz',
            '/media/mod_login/js/admin-login.es6.js',
            '/media/mod_login/js/admin-login.es6.min.js',
            '/media/mod_login/js/admin-login.es6.min.js.gz',
            '/media/mod_menu/js/admin-menu.es6.js',
            '/media/mod_menu/js/admin-menu.es6.min.js',
            '/media/mod_menu/js/admin-menu.es6.min.js.gz',
            '/media/mod_menu/js/menu.es6.js',
            '/media/mod_menu/js/menu.es6.min.js',
            '/media/mod_menu/js/menu.es6.min.js.gz',
            '/media/mod_multilangstatus/js/admin-multilangstatus.es6.js',
            '/media/mod_multilangstatus/js/admin-multilangstatus.es6.min.js',
            '/media/mod_multilangstatus/js/admin-multilangstatus.es6.min.js.gz',
            '/media/mod_quickicon/js/quickicon.es6.js',
            '/media/mod_quickicon/js/quickicon.es6.min.js',
            '/media/mod_quickicon/js/quickicon.es6.min.js.gz',
            '/media/mod_sampledata/js/sampledata-process.es6.js',
            '/media/mod_sampledata/js/sampledata-process.es6.min.js',
            '/media/mod_sampledata/js/sampledata-process.es6.min.js.gz',
            '/media/plg_captcha_recaptcha/js/recaptcha.es6.js',
            '/media/plg_captcha_recaptcha/js/recaptcha.es6.min.js',
            '/media/plg_captcha_recaptcha/js/recaptcha.es6.min.js.gz',
            '/media/plg_captcha_recaptcha_invisible/js/recaptcha.es6.js',
            '/media/plg_captcha_recaptcha_invisible/js/recaptcha.es6.min.js',
            '/media/plg_captcha_recaptcha_invisible/js/recaptcha.es6.min.js.gz',
            '/media/plg_editors_tinymce/js/plugins/dragdrop/plugin.es6.js',
            '/media/plg_editors_tinymce/js/plugins/dragdrop/plugin.es6.min.js',
            '/media/plg_editors_tinymce/js/plugins/dragdrop/plugin.es6.min.js.gz',
            '/media/plg_editors_tinymce/js/tinymce-builder.es6.js',
            '/media/plg_editors_tinymce/js/tinymce-builder.es6.min.js',
            '/media/plg_editors_tinymce/js/tinymce-builder.es6.min.js.gz',
            '/media/plg_editors_tinymce/js/tinymce.es6.js',
            '/media/plg_editors_tinymce/js/tinymce.es6.min.js',
            '/media/plg_editors_tinymce/js/tinymce.es6.min.js.gz',
            '/media/plg_installer_folderinstaller/js/folderinstaller.es6.js',
            '/media/plg_installer_folderinstaller/js/folderinstaller.es6.min.js',
            '/media/plg_installer_folderinstaller/js/folderinstaller.es6.min.js.gz',
            '/media/plg_installer_packageinstaller/js/packageinstaller.es6.js',
            '/media/plg_installer_packageinstaller/js/packageinstaller.es6.min.js',
            '/media/plg_installer_packageinstaller/js/packageinstaller.es6.min.js.gz',
            '/media/plg_installer_urlinstaller/js/urlinstaller.es6.js',
            '/media/plg_installer_urlinstaller/js/urlinstaller.es6.min.js',
            '/media/plg_installer_urlinstaller/js/urlinstaller.es6.min.js.gz',
            '/media/plg_installer_webinstaller/js/client.es6.js',
            '/media/plg_installer_webinstaller/js/client.es6.min.js',
            '/media/plg_installer_webinstaller/js/client.es6.min.js.gz',
            '/media/plg_media-action_crop/js/crop.es6.js',
            '/media/plg_media-action_crop/js/crop.es6.min.js',
            '/media/plg_media-action_crop/js/crop.es6.min.js.gz',
            '/media/plg_media-action_resize/js/resize.es6.js',
            '/media/plg_media-action_resize/js/resize.es6.min.js',
            '/media/plg_media-action_resize/js/resize.es6.min.js.gz',
            '/media/plg_media-action_rotate/js/rotate.es6.js',
            '/media/plg_media-action_rotate/js/rotate.es6.min.js',
            '/media/plg_media-action_rotate/js/rotate.es6.min.js.gz',
            '/media/plg_quickicon_extensionupdate/js/extensionupdatecheck.es6.js',
            '/media/plg_quickicon_extensionupdate/js/extensionupdatecheck.es6.min.js',
            '/media/plg_quickicon_extensionupdate/js/extensionupdatecheck.es6.min.js.gz',
            '/media/plg_quickicon_joomlaupdate/js/jupdatecheck.es6.js',
            '/media/plg_quickicon_joomlaupdate/js/jupdatecheck.es6.min.js',
            '/media/plg_quickicon_joomlaupdate/js/jupdatecheck.es6.min.js.gz',
            '/media/plg_quickicon_overridecheck/js/overridecheck.es6.js',
            '/media/plg_quickicon_overridecheck/js/overridecheck.es6.min.js',
            '/media/plg_quickicon_overridecheck/js/overridecheck.es6.min.js.gz',
            '/media/plg_quickicon_privacycheck/js/privacycheck.es6.js',
            '/media/plg_quickicon_privacycheck/js/privacycheck.es6.min.js',
            '/media/plg_quickicon_privacycheck/js/privacycheck.es6.min.js.gz',
            '/media/plg_system_debug/js/debug.es6.js',
            '/media/plg_system_debug/js/debug.es6.min.js',
            '/media/plg_system_debug/js/debug.es6.min.js.gz',
            '/media/plg_system_highlight/highlight.min.css',
            '/media/plg_system_highlight/highlight.min.css.gz',
            '/media/plg_system_stats/js/stats-message.es6.js',
            '/media/plg_system_stats/js/stats-message.es6.min.js',
            '/media/plg_system_stats/js/stats-message.es6.min.js.gz',
            '/media/plg_system_stats/js/stats.es6.js',
            '/media/plg_system_stats/js/stats.es6.min.js',
            '/media/plg_system_stats/js/stats.es6.min.js.gz',
            '/media/plg_system_webauthn/js/login.es6.js',
            '/media/plg_system_webauthn/js/login.es6.min.js',
            '/media/plg_system_webauthn/js/login.es6.min.js.gz',
            '/media/plg_system_webauthn/js/management.es6.js',
            '/media/plg_system_webauthn/js/management.es6.min.js',
            '/media/plg_system_webauthn/js/management.es6.min.js.gz',
            '/media/plg_user_token/js/token.es6.js',
            '/media/plg_user_token/js/token.es6.min.js',
            '/media/plg_user_token/js/token.es6.min.js.gz',
            '/media/system/js/core.es6.js',
            '/media/system/js/core.es6.min.js',
            '/media/system/js/core.es6.min.js.gz',
            '/media/system/js/draggable.es6.js',
            '/media/system/js/draggable.es6.min.js',
            '/media/system/js/draggable.es6.min.js.gz',
            '/media/system/js/fields/joomla-field-color-slider.es6.js',
            '/media/system/js/fields/joomla-field-color-slider.es6.min.js',
            '/media/system/js/fields/joomla-field-color-slider.es6.min.js.gz',
            '/media/system/js/fields/passwordstrength.es6.js',
            '/media/system/js/fields/passwordstrength.es6.min.js',
            '/media/system/js/fields/passwordstrength.es6.min.js.gz',
            '/media/system/js/fields/passwordview.es6.js',
            '/media/system/js/fields/passwordview.es6.min.js',
            '/media/system/js/fields/passwordview.es6.min.js.gz',
            '/media/system/js/fields/select-colour.es6.js',
            '/media/system/js/fields/select-colour.es6.min.js',
            '/media/system/js/fields/select-colour.es6.min.js.gz',
            '/media/system/js/fields/validate.es6.js',
            '/media/system/js/fields/validate.es6.min.js',
            '/media/system/js/fields/validate.es6.min.js.gz',
            '/media/system/js/keepalive.es6.js',
            '/media/system/js/keepalive.es6.min.js',
            '/media/system/js/keepalive.es6.min.js.gz',
            '/media/system/js/multiselect.es6.js',
            '/media/system/js/multiselect.es6.min.js',
            '/media/system/js/multiselect.es6.min.js.gz',
            '/media/system/js/searchtools.es6.js',
            '/media/system/js/searchtools.es6.min.js',
            '/media/system/js/searchtools.es6.min.js.gz',
            '/media/system/js/showon.es6.js',
            '/media/system/js/showon.es6.min.js',
            '/media/system/js/showon.es6.min.js.gz',
            '/media/templates/atum/js/template.es6.js',
            '/media/templates/atum/js/template.es6.min.js',
            '/media/templates/atum/js/template.es6.min.js.gz',
            '/media/templates/atum/js/template.js',
            '/media/templates/atum/js/template.min.js',
            '/media/templates/atum/js/template.min.js.gz',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu.es6.js',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu.es6.min.js',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu.es6.min.js.gz',
            '/media/vendor/bootstrap/js/alert.es6.js',
            '/media/vendor/bootstrap/js/alert.es6.min.js',
            '/media/vendor/bootstrap/js/alert.es6.min.js.gz',
            '/media/vendor/bootstrap/js/bootstrap.es5.js',
            '/media/vendor/bootstrap/js/bootstrap.es5.min.js',
            '/media/vendor/bootstrap/js/bootstrap.es5.min.js.gz',
            '/media/vendor/bootstrap/js/button.es6.js',
            '/media/vendor/bootstrap/js/button.es6.min.js',
            '/media/vendor/bootstrap/js/button.es6.min.js.gz',
            '/media/vendor/bootstrap/js/carousel.es6.js',
            '/media/vendor/bootstrap/js/carousel.es6.min.js',
            '/media/vendor/bootstrap/js/carousel.es6.min.js.gz',
            '/media/vendor/bootstrap/js/collapse.es6.js',
            '/media/vendor/bootstrap/js/collapse.es6.min.js',
            '/media/vendor/bootstrap/js/collapse.es6.min.js.gz',
            '/media/vendor/bootstrap/js/dom-8eef6b5f.js',
            '/media/vendor/bootstrap/js/dropdown.es6.js',
            '/media/vendor/bootstrap/js/dropdown.es6.min.js',
            '/media/vendor/bootstrap/js/dropdown.es6.min.js.gz',
            '/media/vendor/bootstrap/js/modal.es6.js',
            '/media/vendor/bootstrap/js/modal.es6.min.js',
            '/media/vendor/bootstrap/js/modal.es6.min.js.gz',
            '/media/vendor/bootstrap/js/popover.es6.js',
            '/media/vendor/bootstrap/js/popover.es6.min.js',
            '/media/vendor/bootstrap/js/popover.es6.min.js.gz',
            '/media/vendor/bootstrap/js/popper-5304749a.js',
            '/media/vendor/bootstrap/js/scrollspy.es6.js',
            '/media/vendor/bootstrap/js/scrollspy.es6.min.js',
            '/media/vendor/bootstrap/js/scrollspy.es6.min.js.gz',
            '/media/vendor/bootstrap/js/tab.es6.js',
            '/media/vendor/bootstrap/js/tab.es6.min.js',
            '/media/vendor/bootstrap/js/tab.es6.min.js.gz',
            '/media/vendor/bootstrap/js/toast.es6.js',
            '/media/vendor/bootstrap/js/toast.es6.min.js',
            '/media/vendor/bootstrap/js/toast.es6.min.js.gz',
            '/media/vendor/codemirror/lib/codemirror-ce.js',
            '/media/vendor/codemirror/lib/codemirror-ce.min.js',
            '/media/vendor/codemirror/lib/codemirror-ce.min.js.gz',
            '/media/vendor/punycode/js/punycode.js',
            '/media/vendor/punycode/js/punycode.min.js',
            '/media/vendor/punycode/js/punycode.min.js.gz',
            '/media/vendor/tinymce/changelog.txt',
            '/media/vendor/webcomponentsjs/js/webcomponents-ce.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-ce.min.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-ce.min.js.gz',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd-ce-pf.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd-ce-pf.min.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd-ce-pf.min.js.gz',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd-ce.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd-ce.min.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd-ce.min.js.gz',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd.min.js',
            '/media/vendor/webcomponentsjs/js/webcomponents-sd.min.js.gz',
            '/plugins/fields/subfields/params/subfields.xml',
            '/plugins/fields/subfields/subfields.php',
            '/plugins/fields/subfields/subfields.xml',
            '/plugins/fields/subfields/tmpl/subfields.php',
            '/templates/cassiopeia/images/system/rating_star.png',
            '/templates/cassiopeia/images/system/rating_star_blank.png',
            '/templates/cassiopeia/scss/tools/mixins/_margin.scss',
            '/templates/cassiopeia/scss/tools/mixins/_visually-hidden.scss',
            '/templates/system/js/error-locales.js',
            // 4.0 from RC 1 to RC 2
            '/administrator/components/com_fields/tmpl/field/modal.php',
            '/administrator/templates/atum/scss/pages/_com_admin.scss',
            '/administrator/templates/atum/scss/pages/_com_finder.scss',
            '/libraries/src/Error/JsonApi/InstallLanguageExceptionHandler.php',
            '/libraries/src/MVC/Controller/Exception/InstallLanguage.php',
            '/media/com_fields/js/admin-field-edit-modal-es5.js',
            '/media/com_fields/js/admin-field-edit-modal-es5.min.js',
            '/media/com_fields/js/admin-field-edit-modal-es5.min.js.gz',
            '/media/com_fields/js/admin-field-edit-modal.js',
            '/media/com_fields/js/admin-field-edit-modal.min.js',
            '/media/com_fields/js/admin-field-edit-modal.min.js.gz',
            // 4.0 from RC 3 to RC 4
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_nodownload.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_noupdate.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_preupdatecheck.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_reinstall.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_update.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_updatemefirst.php',
            '/administrator/components/com_joomlaupdate/tmpl/joomlaupdate/default_upload.php',
            '/language/en-GB/com_messages.ini',
            '/media/system/css/fields/joomla-image-select.css',
            '/media/system/css/fields/joomla-image-select.min.css',
            '/media/system/css/fields/joomla-image-select.min.css.gz',
            '/media/system/js/fields/joomla-image-select-es5.js',
            '/media/system/js/fields/joomla-image-select-es5.min.js',
            '/media/system/js/fields/joomla-image-select-es5.min.js.gz',
            '/media/system/js/fields/joomla-image-select.js',
            '/media/system/js/fields/joomla-image-select.min.js',
            '/media/system/js/fields/joomla-image-select.min.js.gz',
            // 4.0 from RC 4 to RC 5
            '/media/system/js/fields/calendar-locales/af.min.js',
            '/media/system/js/fields/calendar-locales/af.min.js.gz',
            '/media/system/js/fields/calendar-locales/ar.min.js',
            '/media/system/js/fields/calendar-locales/ar.min.js.gz',
            '/media/system/js/fields/calendar-locales/bg.min.js',
            '/media/system/js/fields/calendar-locales/bg.min.js.gz',
            '/media/system/js/fields/calendar-locales/bn.min.js',
            '/media/system/js/fields/calendar-locales/bn.min.js.gz',
            '/media/system/js/fields/calendar-locales/bs.min.js',
            '/media/system/js/fields/calendar-locales/bs.min.js.gz',
            '/media/system/js/fields/calendar-locales/ca.min.js',
            '/media/system/js/fields/calendar-locales/ca.min.js.gz',
            '/media/system/js/fields/calendar-locales/cs.min.js',
            '/media/system/js/fields/calendar-locales/cs.min.js.gz',
            '/media/system/js/fields/calendar-locales/cy.min.js',
            '/media/system/js/fields/calendar-locales/cy.min.js.gz',
            '/media/system/js/fields/calendar-locales/da.min.js',
            '/media/system/js/fields/calendar-locales/da.min.js.gz',
            '/media/system/js/fields/calendar-locales/de.min.js',
            '/media/system/js/fields/calendar-locales/de.min.js.gz',
            '/media/system/js/fields/calendar-locales/el.min.js',
            '/media/system/js/fields/calendar-locales/el.min.js.gz',
            '/media/system/js/fields/calendar-locales/en.min.js',
            '/media/system/js/fields/calendar-locales/en.min.js.gz',
            '/media/system/js/fields/calendar-locales/es.min.js',
            '/media/system/js/fields/calendar-locales/es.min.js.gz',
            '/media/system/js/fields/calendar-locales/eu.min.js',
            '/media/system/js/fields/calendar-locales/eu.min.js.gz',
            '/media/system/js/fields/calendar-locales/fa-ir.min.js',
            '/media/system/js/fields/calendar-locales/fa-ir.min.js.gz',
            '/media/system/js/fields/calendar-locales/fi.min.js',
            '/media/system/js/fields/calendar-locales/fi.min.js.gz',
            '/media/system/js/fields/calendar-locales/fr.min.js',
            '/media/system/js/fields/calendar-locales/fr.min.js.gz',
            '/media/system/js/fields/calendar-locales/ga.min.js',
            '/media/system/js/fields/calendar-locales/ga.min.js.gz',
            '/media/system/js/fields/calendar-locales/hr.min.js',
            '/media/system/js/fields/calendar-locales/hr.min.js.gz',
            '/media/system/js/fields/calendar-locales/hu.min.js',
            '/media/system/js/fields/calendar-locales/hu.min.js.gz',
            '/media/system/js/fields/calendar-locales/it.min.js',
            '/media/system/js/fields/calendar-locales/it.min.js.gz',
            '/media/system/js/fields/calendar-locales/ja.min.js',
            '/media/system/js/fields/calendar-locales/ja.min.js.gz',
            '/media/system/js/fields/calendar-locales/ka.min.js',
            '/media/system/js/fields/calendar-locales/ka.min.js.gz',
            '/media/system/js/fields/calendar-locales/kk.min.js',
            '/media/system/js/fields/calendar-locales/kk.min.js.gz',
            '/media/system/js/fields/calendar-locales/ko.min.js',
            '/media/system/js/fields/calendar-locales/ko.min.js.gz',
            '/media/system/js/fields/calendar-locales/lt.min.js',
            '/media/system/js/fields/calendar-locales/lt.min.js.gz',
            '/media/system/js/fields/calendar-locales/mk.min.js',
            '/media/system/js/fields/calendar-locales/mk.min.js.gz',
            '/media/system/js/fields/calendar-locales/nb.min.js',
            '/media/system/js/fields/calendar-locales/nb.min.js.gz',
            '/media/system/js/fields/calendar-locales/nl.min.js',
            '/media/system/js/fields/calendar-locales/nl.min.js.gz',
            '/media/system/js/fields/calendar-locales/pl.min.js',
            '/media/system/js/fields/calendar-locales/pl.min.js.gz',
            '/media/system/js/fields/calendar-locales/prs-af.min.js',
            '/media/system/js/fields/calendar-locales/prs-af.min.js.gz',
            '/media/system/js/fields/calendar-locales/pt.min.js',
            '/media/system/js/fields/calendar-locales/pt.min.js.gz',
            '/media/system/js/fields/calendar-locales/ru.min.js',
            '/media/system/js/fields/calendar-locales/ru.min.js.gz',
            '/media/system/js/fields/calendar-locales/sk.min.js',
            '/media/system/js/fields/calendar-locales/sk.min.js.gz',
            '/media/system/js/fields/calendar-locales/sl.min.js',
            '/media/system/js/fields/calendar-locales/sl.min.js.gz',
            '/media/system/js/fields/calendar-locales/sr-rs.min.js',
            '/media/system/js/fields/calendar-locales/sr-rs.min.js.gz',
            '/media/system/js/fields/calendar-locales/sr-yu.min.js',
            '/media/system/js/fields/calendar-locales/sr-yu.min.js.gz',
            '/media/system/js/fields/calendar-locales/sv.min.js',
            '/media/system/js/fields/calendar-locales/sv.min.js.gz',
            '/media/system/js/fields/calendar-locales/sw.min.js',
            '/media/system/js/fields/calendar-locales/sw.min.js.gz',
            '/media/system/js/fields/calendar-locales/ta.min.js',
            '/media/system/js/fields/calendar-locales/ta.min.js.gz',
            '/media/system/js/fields/calendar-locales/th.min.js',
            '/media/system/js/fields/calendar-locales/th.min.js.gz',
            '/media/system/js/fields/calendar-locales/uk.min.js',
            '/media/system/js/fields/calendar-locales/uk.min.js.gz',
            '/media/system/js/fields/calendar-locales/zh-CN.min.js',
            '/media/system/js/fields/calendar-locales/zh-CN.min.js.gz',
            '/media/system/js/fields/calendar-locales/zh-TW.min.js',
            '/media/system/js/fields/calendar-locales/zh-TW.min.js.gz',
            // 4.0 from RC 5 to RC 6
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu-es5.js',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu-es5.min.js',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu-es5.min.js.gz',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu.js',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu.min.js',
            '/media/templates/cassiopeia/js/mod_menu/menu-metismenu.min.js.gz',
            '/templates/cassiopeia/css/vendor/fontawesome-free/fontawesome.css',
            '/templates/cassiopeia/css/vendor/fontawesome-free/fontawesome.min.css',
            '/templates/cassiopeia/css/vendor/fontawesome-free/fontawesome.min.css.gz',
            '/templates/cassiopeia/scss/vendor/fontawesome-free/fontawesome.scss',
            // 4.0 from RC 6 to 4.0.0 (stable)
            '/libraries/vendor/algo26-matthias/idna-convert/tests/integration/ToIdnTest.php',
            '/libraries/vendor/algo26-matthias/idna-convert/tests/integration/ToUnicodeTest.php',
            '/libraries/vendor/algo26-matthias/idna-convert/tests/unit/.gitkeep',
            '/libraries/vendor/algo26-matthias/idna-convert/tests/unit/namePrepTest.php',
            '/libraries/vendor/doctrine/inflector/docs/en/index.rst',
            '/libraries/vendor/jakeasmith/http_build_url/tests/HttpBuildUrlTest.php',
            '/libraries/vendor/jakeasmith/http_build_url/tests/bootstrap.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/AcceptLanguageTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/AcceptTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/BaseAcceptTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/CharsetNegotiatorTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/EncodingNegotiatorTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/LanguageNegotiatorTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/MatchTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/NegotiatorTest.php',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests/TestCase.php',
            '/libraries/vendor/willdurand/negotiation/tests/bootstrap.php',
            // From 4.0.2 to 4.0.3
            '/templates/cassiopeia/css/global/fonts-web_fira-sans.css',
            '/templates/cassiopeia/css/global/fonts-web_fira-sans.min.css',
            '/templates/cassiopeia/css/global/fonts-web_fira-sans.min.css.gz',
            '/templates/cassiopeia/css/global/fonts-web_roboto+noto-sans.css',
            '/templates/cassiopeia/css/global/fonts-web_roboto+noto-sans.min.css',
            '/templates/cassiopeia/css/global/fonts-web_roboto+noto-sans.min.css.gz',
            '/templates/cassiopeia/scss/global/fonts-web_fira-sans.scss',
            '/templates/cassiopeia/scss/global/fonts-web_roboto+noto-sans.scss',
            // From 4.0.3 to 4.0.4
            '/administrator/templates/atum/scss/_mixin.scss',
            '/media/com_joomlaupdate/js/encryption.min.js.gz',
            '/media/com_joomlaupdate/js/update.min.js.gz',
            '/templates/cassiopeia/images/system/sort_asc.png',
            '/templates/cassiopeia/images/system/sort_desc.png',
            // From 4.0.4 to 4.0.5
            '/media/vendor/codemirror/lib/#codemirror.js#',
            // From 4.0.5 to 4.0.6
            '/media/vendor/mediaelement/css/mejs-controls.png',
            // From 4.0.x to 4.1.0-beta1
            '/administrator/templates/atum/css/system/searchtools/searchtools.css',
            '/administrator/templates/atum/css/system/searchtools/searchtools.min.css',
            '/administrator/templates/atum/css/system/searchtools/searchtools.min.css.gz',
            '/administrator/templates/atum/css/template-rtl.css',
            '/administrator/templates/atum/css/template-rtl.min.css',
            '/administrator/templates/atum/css/template-rtl.min.css.gz',
            '/administrator/templates/atum/css/template.css',
            '/administrator/templates/atum/css/template.min.css',
            '/administrator/templates/atum/css/template.min.css.gz',
            '/administrator/templates/atum/css/vendor/awesomplete/awesomplete.css',
            '/administrator/templates/atum/css/vendor/awesomplete/awesomplete.min.css',
            '/administrator/templates/atum/css/vendor/awesomplete/awesomplete.min.css.gz',
            '/administrator/templates/atum/css/vendor/choicesjs/choices.css',
            '/administrator/templates/atum/css/vendor/choicesjs/choices.min.css',
            '/administrator/templates/atum/css/vendor/choicesjs/choices.min.css.gz',
            '/administrator/templates/atum/css/vendor/fontawesome-free/fontawesome.css',
            '/administrator/templates/atum/css/vendor/fontawesome-free/fontawesome.min.css',
            '/administrator/templates/atum/css/vendor/fontawesome-free/fontawesome.min.css.gz',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements/joomla-alert.css',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements/joomla-alert.min.css',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements/joomla-alert.min.css.gz',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements/joomla-tab.css',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements/joomla-tab.min.css',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements/joomla-tab.min.css.gz',
            '/administrator/templates/atum/css/vendor/minicolors/minicolors.css',
            '/administrator/templates/atum/css/vendor/minicolors/minicolors.min.css',
            '/administrator/templates/atum/css/vendor/minicolors/minicolors.min.css.gz',
            '/administrator/templates/atum/images/joomla-pattern.svg',
            '/administrator/templates/atum/images/logos/brand-large.svg',
            '/administrator/templates/atum/images/logos/brand-small.svg',
            '/administrator/templates/atum/images/logos/login.svg',
            '/administrator/templates/atum/images/select-bg-active-rtl.svg',
            '/administrator/templates/atum/images/select-bg-active.svg',
            '/administrator/templates/atum/images/select-bg-rtl.svg',
            '/administrator/templates/atum/images/select-bg.svg',
            '/administrator/templates/atum/scss/_root.scss',
            '/administrator/templates/atum/scss/_variables.scss',
            '/administrator/templates/atum/scss/blocks/_alerts.scss',
            '/administrator/templates/atum/scss/blocks/_edit.scss',
            '/administrator/templates/atum/scss/blocks/_form.scss',
            '/administrator/templates/atum/scss/blocks/_global.scss',
            '/administrator/templates/atum/scss/blocks/_header.scss',
            '/administrator/templates/atum/scss/blocks/_icons.scss',
            '/administrator/templates/atum/scss/blocks/_iframe.scss',
            '/administrator/templates/atum/scss/blocks/_layout.scss',
            '/administrator/templates/atum/scss/blocks/_lists.scss',
            '/administrator/templates/atum/scss/blocks/_login.scss',
            '/administrator/templates/atum/scss/blocks/_modals.scss',
            '/administrator/templates/atum/scss/blocks/_quickicons.scss',
            '/administrator/templates/atum/scss/blocks/_sidebar-nav.scss',
            '/administrator/templates/atum/scss/blocks/_sidebar.scss',
            '/administrator/templates/atum/scss/blocks/_switcher.scss',
            '/administrator/templates/atum/scss/blocks/_toolbar.scss',
            '/administrator/templates/atum/scss/blocks/_treeselect.scss',
            '/administrator/templates/atum/scss/blocks/_utilities.scss',
            '/administrator/templates/atum/scss/pages/_com_config.scss',
            '/administrator/templates/atum/scss/pages/_com_content.scss',
            '/administrator/templates/atum/scss/pages/_com_cpanel.scss',
            '/administrator/templates/atum/scss/pages/_com_joomlaupdate.scss',
            '/administrator/templates/atum/scss/pages/_com_modules.scss',
            '/administrator/templates/atum/scss/pages/_com_privacy.scss',
            '/administrator/templates/atum/scss/pages/_com_tags.scss',
            '/administrator/templates/atum/scss/pages/_com_templates.scss',
            '/administrator/templates/atum/scss/pages/_com_users.scss',
            '/administrator/templates/atum/scss/system/searchtools/searchtools.scss',
            '/administrator/templates/atum/scss/template-rtl.scss',
            '/administrator/templates/atum/scss/template.scss',
            '/administrator/templates/atum/scss/vendor/_bootstrap.scss',
            '/administrator/templates/atum/scss/vendor/_codemirror.scss',
            '/administrator/templates/atum/scss/vendor/_dragula.scss',
            '/administrator/templates/atum/scss/vendor/_tinymce.scss',
            '/administrator/templates/atum/scss/vendor/awesomplete/awesomplete.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_badge.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_bootstrap-rtl.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_buttons.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_card.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_collapse.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_custom-forms.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_dropdown.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_form.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_lists.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_modal.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_pagination.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_reboot.scss',
            '/administrator/templates/atum/scss/vendor/bootstrap/_table.scss',
            '/administrator/templates/atum/scss/vendor/choicesjs/choices.scss',
            '/administrator/templates/atum/scss/vendor/fontawesome-free/fontawesome.scss',
            '/administrator/templates/atum/scss/vendor/joomla-custom-elements/joomla-alert.scss',
            '/administrator/templates/atum/scss/vendor/joomla-custom-elements/joomla-tab.scss',
            '/administrator/templates/atum/scss/vendor/minicolors/minicolors.scss',
            '/administrator/templates/atum/template_preview.png',
            '/administrator/templates/atum/template_thumbnail.png',
            '/administrator/templates/system/css/error.css',
            '/administrator/templates/system/css/error.min.css',
            '/administrator/templates/system/css/error.min.css.gz',
            '/administrator/templates/system/css/system.css',
            '/administrator/templates/system/css/system.min.css',
            '/administrator/templates/system/css/system.min.css.gz',
            '/administrator/templates/system/images/calendar.png',
            '/administrator/templates/system/scss/error.scss',
            '/administrator/templates/system/scss/system.scss',
            '/templates/cassiopeia/css/editor.css',
            '/templates/cassiopeia/css/editor.min.css',
            '/templates/cassiopeia/css/editor.min.css.gz',
            '/templates/cassiopeia/css/global/colors_alternative.css',
            '/templates/cassiopeia/css/global/colors_alternative.min.css',
            '/templates/cassiopeia/css/global/colors_alternative.min.css.gz',
            '/templates/cassiopeia/css/global/colors_standard.css',
            '/templates/cassiopeia/css/global/colors_standard.min.css',
            '/templates/cassiopeia/css/global/colors_standard.min.css.gz',
            '/templates/cassiopeia/css/global/fonts-local_roboto.css',
            '/templates/cassiopeia/css/global/fonts-local_roboto.min.css',
            '/templates/cassiopeia/css/global/fonts-local_roboto.min.css.gz',
            '/templates/cassiopeia/css/offline.css',
            '/templates/cassiopeia/css/offline.min.css',
            '/templates/cassiopeia/css/offline.min.css.gz',
            '/templates/cassiopeia/css/system/searchtools/searchtools.css',
            '/templates/cassiopeia/css/system/searchtools/searchtools.min.css',
            '/templates/cassiopeia/css/system/searchtools/searchtools.min.css.gz',
            '/templates/cassiopeia/css/template-rtl.css',
            '/templates/cassiopeia/css/template-rtl.min.css',
            '/templates/cassiopeia/css/template-rtl.min.css.gz',
            '/templates/cassiopeia/css/template.css',
            '/templates/cassiopeia/css/template.min.css',
            '/templates/cassiopeia/css/template.min.css.gz',
            '/templates/cassiopeia/css/vendor/choicesjs/choices.css',
            '/templates/cassiopeia/css/vendor/choicesjs/choices.min.css',
            '/templates/cassiopeia/css/vendor/choicesjs/choices.min.css.gz',
            '/templates/cassiopeia/css/vendor/joomla-custom-elements/joomla-alert.css',
            '/templates/cassiopeia/css/vendor/joomla-custom-elements/joomla-alert.min.css',
            '/templates/cassiopeia/css/vendor/joomla-custom-elements/joomla-alert.min.css.gz',
            '/templates/cassiopeia/images/logo.svg',
            '/templates/cassiopeia/images/select-bg-active-rtl.svg',
            '/templates/cassiopeia/images/select-bg-active.svg',
            '/templates/cassiopeia/images/select-bg-rtl.svg',
            '/templates/cassiopeia/images/select-bg.svg',
            '/templates/cassiopeia/js/template.es5.js',
            '/templates/cassiopeia/js/template.js',
            '/templates/cassiopeia/js/template.min.js',
            '/templates/cassiopeia/js/template.min.js.gz',
            '/templates/cassiopeia/scss/blocks/_alerts.scss',
            '/templates/cassiopeia/scss/blocks/_back-to-top.scss',
            '/templates/cassiopeia/scss/blocks/_banner.scss',
            '/templates/cassiopeia/scss/blocks/_css-grid.scss',
            '/templates/cassiopeia/scss/blocks/_footer.scss',
            '/templates/cassiopeia/scss/blocks/_form.scss',
            '/templates/cassiopeia/scss/blocks/_frontend-edit.scss',
            '/templates/cassiopeia/scss/blocks/_global.scss',
            '/templates/cassiopeia/scss/blocks/_header.scss',
            '/templates/cassiopeia/scss/blocks/_icons.scss',
            '/templates/cassiopeia/scss/blocks/_iframe.scss',
            '/templates/cassiopeia/scss/blocks/_layout.scss',
            '/templates/cassiopeia/scss/blocks/_legacy.scss',
            '/templates/cassiopeia/scss/blocks/_modals.scss',
            '/templates/cassiopeia/scss/blocks/_modifiers.scss',
            '/templates/cassiopeia/scss/blocks/_tags.scss',
            '/templates/cassiopeia/scss/blocks/_toolbar.scss',
            '/templates/cassiopeia/scss/blocks/_utilities.scss',
            '/templates/cassiopeia/scss/editor.scss',
            '/templates/cassiopeia/scss/global/colors_alternative.scss',
            '/templates/cassiopeia/scss/global/colors_standard.scss',
            '/templates/cassiopeia/scss/global/fonts-local_roboto.scss',
            '/templates/cassiopeia/scss/offline.scss',
            '/templates/cassiopeia/scss/system/searchtools/searchtools.scss',
            '/templates/cassiopeia/scss/template-rtl.scss',
            '/templates/cassiopeia/scss/template.scss',
            '/templates/cassiopeia/scss/tools/_tools.scss',
            '/templates/cassiopeia/scss/tools/functions/_max-width.scss',
            '/templates/cassiopeia/scss/tools/variables/_variables.scss',
            '/templates/cassiopeia/scss/vendor/_awesomplete.scss',
            '/templates/cassiopeia/scss/vendor/_chosen.scss',
            '/templates/cassiopeia/scss/vendor/_dragula.scss',
            '/templates/cassiopeia/scss/vendor/_minicolors.scss',
            '/templates/cassiopeia/scss/vendor/_tinymce.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_bootstrap-rtl.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_buttons.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_collapse.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_custom-forms.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_dropdown.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_forms.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_lists.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_modal.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_nav.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_pagination.scss',
            '/templates/cassiopeia/scss/vendor/bootstrap/_table.scss',
            '/templates/cassiopeia/scss/vendor/choicesjs/choices.scss',
            '/templates/cassiopeia/scss/vendor/joomla-custom-elements/joomla-alert.scss',
            '/templates/cassiopeia/scss/vendor/metismenu/_metismenu.scss',
            '/templates/cassiopeia/template_preview.png',
            '/templates/cassiopeia/template_thumbnail.png',
            '/templates/system/css/editor.css',
            '/templates/system/css/editor.min.css',
            '/templates/system/css/editor.min.css.gz',
            '/templates/system/css/error.css',
            '/templates/system/css/error.min.css',
            '/templates/system/css/error.min.css.gz',
            '/templates/system/css/error_rtl.css',
            '/templates/system/css/error_rtl.min.css',
            '/templates/system/css/error_rtl.min.css.gz',
            '/templates/system/css/general.css',
            '/templates/system/css/general.min.css',
            '/templates/system/css/general.min.css.gz',
            '/templates/system/css/offline.css',
            '/templates/system/css/offline.min.css',
            '/templates/system/css/offline.min.css.gz',
            '/templates/system/css/offline_rtl.css',
            '/templates/system/css/offline_rtl.min.css',
            '/templates/system/css/offline_rtl.min.css.gz',
            '/templates/system/scss/editor.scss',
            '/templates/system/scss/error.scss',
            '/templates/system/scss/error_rtl.scss',
            '/templates/system/scss/general.scss',
            '/templates/system/scss/offline.scss',
            '/templates/system/scss/offline_rtl.scss',
            // From 4.1.0-beta3 to 4.1.0-rc1
            '/api/components/com_media/src/Helper/AdapterTrait.php',
            // From 4.1.0 to 4.1.1
            '/libraries/vendor/tobscure/json-api/.git/HEAD',
            '/libraries/vendor/tobscure/json-api/.git/ORIG_HEAD',
            '/libraries/vendor/tobscure/json-api/.git/config',
            '/libraries/vendor/tobscure/json-api/.git/description',
            '/libraries/vendor/tobscure/json-api/.git/hooks/applypatch-msg.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/commit-msg.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/fsmonitor-watchman.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/post-update.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/pre-applypatch.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/pre-commit.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/pre-merge-commit.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/pre-push.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/pre-rebase.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/pre-receive.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/prepare-commit-msg.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/push-to-checkout.sample',
            '/libraries/vendor/tobscure/json-api/.git/hooks/update.sample',
            '/libraries/vendor/tobscure/json-api/.git/index',
            '/libraries/vendor/tobscure/json-api/.git/info/exclude',
            '/libraries/vendor/tobscure/json-api/.git/info/refs',
            '/libraries/vendor/tobscure/json-api/.git/logs/HEAD',
            '/libraries/vendor/tobscure/json-api/.git/logs/refs/heads/joomla-backports',
            '/libraries/vendor/tobscure/json-api/.git/logs/refs/remotes/origin/HEAD',
            '/libraries/vendor/tobscure/json-api/.git/objects/info/packs',
            '/libraries/vendor/tobscure/json-api/.git/objects/pack/pack-51530cba04703b17f3c11b9e8458a171092cf5e3.idx',
            '/libraries/vendor/tobscure/json-api/.git/objects/pack/pack-51530cba04703b17f3c11b9e8458a171092cf5e3.pack',
            '/libraries/vendor/tobscure/json-api/.git/packed-refs',
            '/libraries/vendor/tobscure/json-api/.git/refs/heads/joomla-backports',
            '/libraries/vendor/tobscure/json-api/.git/refs/remotes/origin/HEAD',
            '/libraries/vendor/tobscure/json-api/.php_cs',
            '/libraries/vendor/tobscure/json-api/tests/AbstractSerializerTest.php',
            '/libraries/vendor/tobscure/json-api/tests/AbstractTestCase.php',
            '/libraries/vendor/tobscure/json-api/tests/CollectionTest.php',
            '/libraries/vendor/tobscure/json-api/tests/DocumentTest.php',
            '/libraries/vendor/tobscure/json-api/tests/ErrorHandlerTest.php',
            '/libraries/vendor/tobscure/json-api/tests/Exception/Handler/FallbackExceptionHandlerTest.php',
            '/libraries/vendor/tobscure/json-api/tests/Exception/Handler/InvalidParameterExceptionHandlerTest.php',
            '/libraries/vendor/tobscure/json-api/tests/LinksTraitTest.php',
            '/libraries/vendor/tobscure/json-api/tests/ParametersTest.php',
            '/libraries/vendor/tobscure/json-api/tests/ResourceTest.php',
            '/libraries/vendor/tobscure/json-api/tests/UtilTest.php',
            // From 4.1.1 to 4.1.2
            '/administrator/components/com_users/src/Field/PrimaryauthprovidersField.php',
            // From 4.1.2 to 4.1.3
            '/libraries/vendor/webmozart/assert/.php_cs',
            // From 4.1.3 to 4.1.4
            '/libraries/vendor/maximebf/debugbar/.bowerrc',
            '/libraries/vendor/maximebf/debugbar/bower.json',
            '/libraries/vendor/maximebf/debugbar/build/namespaceFontAwesome.php',
            '/libraries/vendor/maximebf/debugbar/demo/ajax.php',
            '/libraries/vendor/maximebf/debugbar/demo/ajax_exception.php',
            '/libraries/vendor/maximebf/debugbar/demo/bootstrap.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/cachecache/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/bootstrap.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/build.sh',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/cli-config.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/src/Demo/Product.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/monolog/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/propel/build.properties',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/propel/build.sh',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/propel/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/propel/runtime-conf.xml',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/propel/schema.xml',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/slim/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/swiftmailer/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/twig/foobar.html',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/twig/hello.html',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/twig/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/dump_assets.php',
            '/libraries/vendor/maximebf/debugbar/demo/index.php',
            '/libraries/vendor/maximebf/debugbar/demo/open.php',
            '/libraries/vendor/maximebf/debugbar/demo/pdo.php',
            '/libraries/vendor/maximebf/debugbar/demo/stack.php',
            '/libraries/vendor/maximebf/debugbar/docs/ajax_and_stack.md',
            '/libraries/vendor/maximebf/debugbar/docs/base_collectors.md',
            '/libraries/vendor/maximebf/debugbar/docs/bridge_collectors.md',
            '/libraries/vendor/maximebf/debugbar/docs/data_collectors.md',
            '/libraries/vendor/maximebf/debugbar/docs/data_formatter.md',
            '/libraries/vendor/maximebf/debugbar/docs/http_drivers.md',
            '/libraries/vendor/maximebf/debugbar/docs/javascript_bar.md',
            '/libraries/vendor/maximebf/debugbar/docs/manifest.json',
            '/libraries/vendor/maximebf/debugbar/docs/openhandler.md',
            '/libraries/vendor/maximebf/debugbar/docs/rendering.md',
            '/libraries/vendor/maximebf/debugbar/docs/screenshot.png',
            '/libraries/vendor/maximebf/debugbar/docs/storage.md',
            '/libraries/vendor/maximebf/debugbar/docs/style.css',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector/AggregatedCollectorTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector/ConfigCollectorTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector/MessagesCollectorTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector/MockCollector.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector/Propel2CollectorTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector/TimeDataCollectorTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataFormatter/DataFormatterTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataFormatter/DebugBarVarDumperTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DebugBarTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DebugBarTestCase.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/JavascriptRendererTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/MockHttpDriver.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/OpenHandlerTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/Storage/FileStorageTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/Storage/MockStorage.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/TracedStatementTest.php',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/full_init.html',
            '/libraries/vendor/maximebf/debugbar/tests/bootstrap.php',
            // From 4.1 to 4.2.0-beta1
            '/libraries/src/Service/Provider/ApiRouter.php',
            '/libraries/vendor/nyholm/psr7/doc/final.md',
            '/media/com_finder/js/index-es5.js',
            '/media/com_finder/js/index-es5.min.js',
            '/media/com_finder/js/index-es5.min.js.gz',
            '/media/com_finder/js/index.js',
            '/media/com_finder/js/index.min.js',
            '/media/com_finder/js/index.min.js.gz',
            '/media/com_users/js/two-factor-switcher-es5.js',
            '/media/com_users/js/two-factor-switcher-es5.min.js',
            '/media/com_users/js/two-factor-switcher-es5.min.js.gz',
            '/media/com_users/js/two-factor-switcher.js',
            '/media/com_users/js/two-factor-switcher.min.js',
            '/media/com_users/js/two-factor-switcher.min.js.gz',
            '/modules/mod_articles_news/mod_articles_news.php',
            '/plugins/actionlog/joomla/joomla.php',
            '/plugins/api-authentication/basic/basic.php',
            '/plugins/api-authentication/token/token.php',
            '/plugins/system/cache/cache.php',
            '/plugins/twofactorauth/totp/postinstall/actions.php',
            '/plugins/twofactorauth/totp/tmpl/form.php',
            '/plugins/twofactorauth/totp/totp.php',
            '/plugins/twofactorauth/totp/totp.xml',
            '/plugins/twofactorauth/yubikey/tmpl/form.php',
            '/plugins/twofactorauth/yubikey/yubikey.php',
            '/plugins/twofactorauth/yubikey/yubikey.xml',
            // From 4.2.0-beta1 to 4.2.0-beta2
            '/layouts/plugins/user/profile/fields/dob.php',
            '/modules/mod_articles_latest/mod_articles_latest.php',
            '/plugins/behaviour/taggable/taggable.php',
            '/plugins/behaviour/versionable/versionable.php',
            '/plugins/task/requests/requests.php',
            '/plugins/task/sitestatus/sitestatus.php',
            '/plugins/user/profile/src/Field/DobField.php',
            // From 4.2.0-beta2 to 4.2.0-beta3
            '/plugins/system/webauthn/src/Exception/AjaxNonCmsAppException.php',
            '/plugins/system/webauthn/src/Helper/CredentialsCreation.php',
            '/plugins/system/webauthn/src/Helper/Joomla.php',
            '/plugins/system/webauthn/webauthn.php',
            '/plugins/task/checkfiles/checkfiles.php',
            '/plugins/task/demotasks/demotasks.php',
            // From 4.2.0-rc1 to 4.2.0
            '/administrator/language/en-GB/plg_fields_menuitem.ini',
            '/administrator/language/en-GB/plg_fields_menuitem.sys.ini',
            '/plugins/fields/menuitem/menuitem.php',
            '/plugins/fields/menuitem/menuitem.xml',
            '/plugins/fields/menuitem/tmpl/menuitem.php',
            // From 4.2.0 to 4.2.1
            '/media/vendor/hotkeys.js/js/hotkeys.js',
            '/media/vendor/hotkeys.js/js/hotkeys.min.js',
            '/media/vendor/hotkeys.js/js/hotkeys.min.js.gz',
            '/media/vendor/hotkeys.js/LICENSE',
            // From 4.2.1 to 4.2.2
            '/administrator/cache/fido.jwt',
            // From 4.2.6 to 4.2.7
            '/libraries/vendor/maximebf/debugbar/src/DebugBar/DataFormatter/VarDumper/SeekingData.php',
            // From 4.2.8 to 4.2.9
            '/administrator/components/com_scheduler/tmpl/select/modal.php',
            // From 4.2.x to 4.3.0-alpha1
            '/libraries/vendor/paragonie/sodium_compat/autoload-fast.php',
            '/libraries/vendor/paragonie/sodium_compat/autoload-pedantic.php',
            '/libraries/vendor/paragonie/sodium_compat/autoload-phpunit.php',
            '/libraries/vendor/paragonie/sodium_compat/dist/Makefile',
            '/libraries/vendor/paragonie/sodium_compat/dist/box.json',
            '/libraries/vendor/paragonie/sodium_compat/psalm-above-3.xml',
            '/libraries/vendor/paragonie/sodium_compat/psalm-below-3.xml',
            '/libraries/vendor/paragonie/sodium_compat/src/Core/Base64/Common.php',
            '/media/com_menus/css/admin-item-edit_modules.css',
            '/media/com_menus/css/admin-item-edit_modules.min.css',
            '/media/com_menus/css/admin-item-edit_modules.min.css.gz',
            '/media/templates/administrator/atum/scss/vendor/bootstrap/_bootstrap-rtl.scss',
            '/media/templates/site/cassiopeia/scss/vendor/bootstrap/_bootstrap-rtl.scss',
            '/plugins/content/confirmconsent/confirmconsent.php',
            '/plugins/content/contact/contact.php',
            '/plugins/extension/finder/finder.php',
            '/plugins/extension/joomla/joomla.php',
            '/plugins/extension/namespacemap/namespacemap.php',
            '/plugins/quickicon/downloadkey/downloadkey.php',
            '/plugins/quickicon/extensionupdate/extensionupdate.php',
            '/plugins/quickicon/overridecheck/overridecheck.php',
            '/plugins/quickicon/phpversioncheck/phpversioncheck.php',
            '/plugins/quickicon/privacycheck/privacycheck.php',
            // From 4.3.0-alpha1 to 4.3.0-alpha2
            '/plugins/content/emailcloak/emailcloak.php',
            '/plugins/content/fields/fields.php',
            // From 4.3.0-alpha2 to 4.3.0-alpha3
            '/cypress.config.js',
            '/media/templates/administrator/atum/scss/_root.scss',
            '/media/templates/administrator/atum/scss/vendor/_bootstrap.scss',
            '/modules/mod_articles_popular/mod_articles_popular.php',
            '/plugins/authentication/cookie/cookie.php',
            '/plugins/authentication/joomla/joomla.php',
            '/plugins/authentication/ldap/ldap.php',
            '/plugins/editors-xtd/article/article.php',
            '/plugins/editors-xtd/contact/contact.php',
            '/plugins/editors-xtd/fields/fields.php',
            '/plugins/editors-xtd/image/image.php',
            '/plugins/editors-xtd/menu/menu.php',
            '/plugins/editors-xtd/module/module.php',
            '/plugins/editors-xtd/readmore/readmore.php',
            '/plugins/editors/tinymce/tinymce.php',
            // From 4.3.0-alpha3 to 4.3.0-beta1
            '/plugins/editors/codemirror/codemirror.php',
            '/plugins/editors/none/none.php',
            '/plugins/fields/calendar/calendar.php',
            '/plugins/fields/checkboxes/checkboxes.php',
            '/plugins/fields/color/color.php',
            '/plugins/fields/editor/editor.php',
            '/plugins/fields/imagelist/imagelist.php',
            '/plugins/fields/integer/integer.php',
            '/plugins/fields/list/list.php',
            '/plugins/fields/media/media.php',
            '/plugins/fields/radio/radio.php',
            '/plugins/fields/sql/sql.php',
            '/plugins/fields/subform/subform.php',
            '/plugins/fields/text/text.php',
            '/plugins/fields/textarea/textarea.php',
            '/plugins/fields/url/url.php',
            '/plugins/fields/user/user.php',
            '/plugins/fields/usergrouplist/usergrouplist.php',
            // From 4.3.0-beta2 to 4.3.0-beta3
            '/cypress.config.dist.js',
            '/plugins/captcha/recaptcha/recaptcha.php',
            '/plugins/captcha/recaptcha_invisible/recaptcha_invisible.php',
            '/plugins/filesystem/local/local.php',
            '/plugins/finder/categories/categories.php',
            '/plugins/finder/contacts/contacts.php',
            '/plugins/finder/content/content.php',
            '/plugins/finder/newsfeeds/newsfeeds.php',
            '/plugins/finder/tags/tags.php',
            // From 4.3.0-beta3 to 4.3.0-beta4
            '/layouts/joomla/content/categories_default_items.php',
            // From 4.3.0-beta4 to 4.3.0-rc1
            '/administrator/components/com_guidedtours/src/Helper/GuidedtoursHelper.php',
            '/libraries/vendor/voku/portable-ascii/build/docs/base.md',
            '/libraries/vendor/voku/portable-ascii/build/generate_docs.php',
            '/libraries/vendor/voku/portable-ascii/build/generate_max_key_length.php',
            // From 4.3.x to 4.4.0-alpha1
            '/modules/mod_articles_archive/mod_articles_archive.php',
            '/modules/mod_articles_categories/mod_articles_categories.php',
            '/modules/mod_articles_category/mod_articles_category.php',
            '/modules/mod_breadcrumbs/mod_breadcrumbs.php',
            '/modules/mod_custom/mod_custom.php',
            '/modules/mod_footer/mod_footer.php',
            '/modules/mod_related_items/mod_related_items.php',
            '/modules/mod_users_latest/mod_users_latest.php',
            '/plugins/content/finder/finder.php',
            '/plugins/content/joomla/joomla.php',
            '/plugins/content/loadmodule/loadmodule.php',
            '/plugins/content/pagebreak/pagebreak.php',
            '/plugins/content/pagenavigation/pagenavigation.php',
            '/plugins/content/vote/vote.php',
            '/plugins/installer/folderinstaller/folderinstaller.php',
            '/plugins/installer/override/override.php',
            '/plugins/installer/packageinstaller/packageinstaller.php',
            '/plugins/installer/urlinstaller/urlinstaller.php',
            '/plugins/installer/webinstaller/webinstaller.php',
            '/plugins/media-action/crop/crop.php',
            '/plugins/media-action/resize/resize.php',
            '/plugins/media-action/rotate/rotate.php',
            '/plugins/privacy/actionlogs/actionlogs.php',
            '/plugins/privacy/consents/consents.php',
            '/plugins/privacy/contact/contact.php',
            '/plugins/privacy/content/content.php',
            '/plugins/privacy/message/message.php',
            '/plugins/privacy/user/user.php',
            '/plugins/sampledata/blog/blog.php',
            '/plugins/sampledata/multilang/multilang.php',
            '/plugins/system/accessibility/accessibility.php',
            '/plugins/system/actionlogs/actionlogs.php',
            '/plugins/system/debug/debug.php',
            '/plugins/system/fields/fields.php',
            '/plugins/system/highlight/highlight.php',
            '/plugins/system/httpheaders/httpheaders.php',
            '/plugins/system/jooa11y/jooa11y.php',
            '/plugins/system/languagecode/languagecode.php',
            '/plugins/system/languagefilter/languagefilter.php',
            '/plugins/system/log/log.php',
            '/plugins/system/logout/logout.php',
            '/plugins/system/logrotation/logrotation.php',
            '/plugins/system/privacyconsent/privacyconsent.php',
            '/plugins/system/redirect/redirect.php',
            '/plugins/system/remember/remember.php',
            '/plugins/system/schedulerunner/schedulerunner.php',
            '/plugins/system/sef/sef.php',
            '/plugins/system/sessiongc/sessiongc.php',
            '/plugins/system/skipto/skipto.php',
            '/plugins/system/stats/stats.php',
            '/plugins/system/tasknotification/tasknotification.php',
            '/plugins/system/updatenotification/updatenotification.php',
            '/plugins/user/contactcreator/contactcreator.php',
            '/plugins/user/joomla/joomla.php',
            '/plugins/user/profile/profile.php',
            '/plugins/user/terms/terms.php',
            '/plugins/user/token/token.php',
            '/plugins/webservices/banners/banners.php',
            '/plugins/webservices/config/config.php',
            '/plugins/webservices/contact/contact.php',
            '/plugins/webservices/content/content.php',
            '/plugins/webservices/installer/installer.php',
            '/plugins/webservices/languages/languages.php',
            '/plugins/webservices/media/media.php',
            '/plugins/webservices/menus/menus.php',
            '/plugins/webservices/messages/messages.php',
            '/plugins/webservices/modules/modules.php',
            '/plugins/webservices/newsfeeds/newsfeeds.php',
            '/plugins/webservices/plugins/plugins.php',
            '/plugins/webservices/privacy/privacy.php',
            '/plugins/webservices/redirect/redirect.php',
            '/plugins/webservices/tags/tags.php',
            '/plugins/webservices/templates/templates.php',
            '/plugins/webservices/users/users.php',
            '/plugins/workflow/featuring/featuring.php',
            '/plugins/workflow/notification/notification.php',
            '/plugins/workflow/publishing/publishing.php',
            // From 4.4.0-alpha1 to 4.4.0-alpha2
            '/libraries/vendor/jfcherng/php-diff/src/languages/readme.txt',
            '/plugins/editors-xtd/pagebreak/pagebreak.php',
        ];

        $folders = [
            // From 3.10 to 4.3
            '/templates/system/images',
            '/templates/system/html',
            '/templates/protostar/less',
            '/templates/protostar/language/en-GB',
            '/templates/protostar/language',
            '/templates/protostar/js',
            '/templates/protostar/img',
            '/templates/protostar/images/system',
            '/templates/protostar/images',
            '/templates/protostar/html/layouts/joomla/system',
            '/templates/protostar/html/layouts/joomla/form/field',
            '/templates/protostar/html/layouts/joomla/form',
            '/templates/protostar/html/layouts/joomla',
            '/templates/protostar/html/layouts',
            '/templates/protostar/html/com_media/imageslist',
            '/templates/protostar/html/com_media',
            '/templates/protostar/html',
            '/templates/protostar/css',
            '/templates/protostar',
            '/templates/beez3/language/en-GB',
            '/templates/beez3/language',
            '/templates/beez3/javascript',
            '/templates/beez3/images/system',
            '/templates/beez3/images/personal',
            '/templates/beez3/images/nature',
            '/templates/beez3/images',
            '/templates/beez3/html/mod_login',
            '/templates/beez3/html/mod_languages',
            '/templates/beez3/html/mod_breadcrumbs',
            '/templates/beez3/html/layouts/joomla/system',
            '/templates/beez3/html/layouts/joomla',
            '/templates/beez3/html/layouts',
            '/templates/beez3/html/com_weblinks/form',
            '/templates/beez3/html/com_weblinks/category',
            '/templates/beez3/html/com_weblinks/categories',
            '/templates/beez3/html/com_weblinks',
            '/templates/beez3/html/com_newsfeeds/category',
            '/templates/beez3/html/com_newsfeeds/categories',
            '/templates/beez3/html/com_newsfeeds',
            '/templates/beez3/html/com_content/form',
            '/templates/beez3/html/com_content/featured',
            '/templates/beez3/html/com_content/category',
            '/templates/beez3/html/com_content/categories',
            '/templates/beez3/html/com_content/article',
            '/templates/beez3/html/com_content/archive',
            '/templates/beez3/html/com_content',
            '/templates/beez3/html/com_contact/contact',
            '/templates/beez3/html/com_contact/category',
            '/templates/beez3/html/com_contact/categories',
            '/templates/beez3/html/com_contact',
            '/templates/beez3/html',
            '/templates/beez3/css',
            '/templates/beez3',
            '/plugins/user/terms/terms',
            '/plugins/user/terms/field',
            '/plugins/user/profile/profiles',
            '/plugins/user/profile/field',
            '/plugins/system/stats/field',
            '/plugins/system/privacyconsent/privacyconsent',
            '/plugins/system/privacyconsent/field',
            '/plugins/system/p3p',
            '/plugins/system/languagecode/language/en-GB',
            '/plugins/system/languagecode/language',
            '/plugins/editors/tinymce/form',
            '/plugins/editors/tinymce/field',
            '/plugins/content/confirmconsent/fields',
            '/plugins/captcha/recaptcha/postinstall',
            '/plugins/authentication/gmail',
            '/media/plg_twofactorauth_totp/js',
            '/media/plg_twofactorauth_totp',
            '/media/plg_system_highlight',
            '/media/overrider/js',
            '/media/overrider/css',
            '/media/overrider',
            '/media/media/js',
            '/media/media/images/mime-icon-32',
            '/media/media/images/mime-icon-16',
            '/media/media/images',
            '/media/media/css',
            '/media/media',
            '/media/jui/less',
            '/media/jui/js',
            '/media/jui/img',
            '/media/jui/images',
            '/media/jui/fonts',
            '/media/jui/css',
            '/media/jui',
            '/media/editors/tinymce/themes/modern',
            '/media/editors/tinymce/themes',
            '/media/editors/tinymce/templates',
            '/media/editors/tinymce/skins/lightgray/img',
            '/media/editors/tinymce/skins/lightgray/fonts',
            '/media/editors/tinymce/skins/lightgray',
            '/media/editors/tinymce/skins',
            '/media/editors/tinymce/plugins/wordcount',
            '/media/editors/tinymce/plugins/visualchars',
            '/media/editors/tinymce/plugins/visualblocks/css',
            '/media/editors/tinymce/plugins/visualblocks',
            '/media/editors/tinymce/plugins/toc',
            '/media/editors/tinymce/plugins/textpattern',
            '/media/editors/tinymce/plugins/textcolor',
            '/media/editors/tinymce/plugins/template',
            '/media/editors/tinymce/plugins/table',
            '/media/editors/tinymce/plugins/tabfocus',
            '/media/editors/tinymce/plugins/spellchecker',
            '/media/editors/tinymce/plugins/searchreplace',
            '/media/editors/tinymce/plugins/save',
            '/media/editors/tinymce/plugins/print',
            '/media/editors/tinymce/plugins/preview',
            '/media/editors/tinymce/plugins/paste',
            '/media/editors/tinymce/plugins/pagebreak',
            '/media/editors/tinymce/plugins/noneditable',
            '/media/editors/tinymce/plugins/nonbreaking',
            '/media/editors/tinymce/plugins/media',
            '/media/editors/tinymce/plugins/lists',
            '/media/editors/tinymce/plugins/link',
            '/media/editors/tinymce/plugins/legacyoutput',
            '/media/editors/tinymce/plugins/layer',
            '/media/editors/tinymce/plugins/insertdatetime',
            '/media/editors/tinymce/plugins/importcss',
            '/media/editors/tinymce/plugins/imagetools',
            '/media/editors/tinymce/plugins/image',
            '/media/editors/tinymce/plugins/hr',
            '/media/editors/tinymce/plugins/fullscreen',
            '/media/editors/tinymce/plugins/fullpage',
            '/media/editors/tinymce/plugins/example_dependency',
            '/media/editors/tinymce/plugins/example',
            '/media/editors/tinymce/plugins/emoticons/img',
            '/media/editors/tinymce/plugins/emoticons',
            '/media/editors/tinymce/plugins/directionality',
            '/media/editors/tinymce/plugins/contextmenu',
            '/media/editors/tinymce/plugins/colorpicker',
            '/media/editors/tinymce/plugins/codesample/css',
            '/media/editors/tinymce/plugins/codesample',
            '/media/editors/tinymce/plugins/code',
            '/media/editors/tinymce/plugins/charmap',
            '/media/editors/tinymce/plugins/bbcode',
            '/media/editors/tinymce/plugins/autosave',
            '/media/editors/tinymce/plugins/autoresize',
            '/media/editors/tinymce/plugins/autolink',
            '/media/editors/tinymce/plugins/anchor',
            '/media/editors/tinymce/plugins/advlist',
            '/media/editors/tinymce/plugins',
            '/media/editors/tinymce/langs',
            '/media/editors/tinymce/js/plugins/dragdrop',
            '/media/editors/tinymce/js/plugins',
            '/media/editors/tinymce/js',
            '/media/editors/tinymce',
            '/media/editors/none/js',
            '/media/editors/none',
            '/media/editors/codemirror/theme',
            '/media/editors/codemirror/mode/z80',
            '/media/editors/codemirror/mode/yaml-frontmatter',
            '/media/editors/codemirror/mode/yaml',
            '/media/editors/codemirror/mode/yacas',
            '/media/editors/codemirror/mode/xquery',
            '/media/editors/codemirror/mode/xml',
            '/media/editors/codemirror/mode/webidl',
            '/media/editors/codemirror/mode/wast',
            '/media/editors/codemirror/mode/vue',
            '/media/editors/codemirror/mode/vhdl',
            '/media/editors/codemirror/mode/verilog',
            '/media/editors/codemirror/mode/velocity',
            '/media/editors/codemirror/mode/vbscript',
            '/media/editors/codemirror/mode/vb',
            '/media/editors/codemirror/mode/twig',
            '/media/editors/codemirror/mode/turtle',
            '/media/editors/codemirror/mode/ttcn-cfg',
            '/media/editors/codemirror/mode/ttcn',
            '/media/editors/codemirror/mode/troff',
            '/media/editors/codemirror/mode/tornado',
            '/media/editors/codemirror/mode/toml',
            '/media/editors/codemirror/mode/tiki',
            '/media/editors/codemirror/mode/tiddlywiki',
            '/media/editors/codemirror/mode/textile',
            '/media/editors/codemirror/mode/tcl',
            '/media/editors/codemirror/mode/swift',
            '/media/editors/codemirror/mode/stylus',
            '/media/editors/codemirror/mode/stex',
            '/media/editors/codemirror/mode/sql',
            '/media/editors/codemirror/mode/spreadsheet',
            '/media/editors/codemirror/mode/sparql',
            '/media/editors/codemirror/mode/soy',
            '/media/editors/codemirror/mode/solr',
            '/media/editors/codemirror/mode/smarty',
            '/media/editors/codemirror/mode/smalltalk',
            '/media/editors/codemirror/mode/slim',
            '/media/editors/codemirror/mode/sieve',
            '/media/editors/codemirror/mode/shell',
            '/media/editors/codemirror/mode/scheme',
            '/media/editors/codemirror/mode/sass',
            '/media/editors/codemirror/mode/sas',
            '/media/editors/codemirror/mode/rust',
            '/media/editors/codemirror/mode/ruby',
            '/media/editors/codemirror/mode/rst',
            '/media/editors/codemirror/mode/rpm/changes',
            '/media/editors/codemirror/mode/rpm',
            '/media/editors/codemirror/mode/r',
            '/media/editors/codemirror/mode/q',
            '/media/editors/codemirror/mode/python',
            '/media/editors/codemirror/mode/puppet',
            '/media/editors/codemirror/mode/pug',
            '/media/editors/codemirror/mode/protobuf',
            '/media/editors/codemirror/mode/properties',
            '/media/editors/codemirror/mode/powershell',
            '/media/editors/codemirror/mode/pig',
            '/media/editors/codemirror/mode/php',
            '/media/editors/codemirror/mode/perl',
            '/media/editors/codemirror/mode/pegjs',
            '/media/editors/codemirror/mode/pascal',
            '/media/editors/codemirror/mode/oz',
            '/media/editors/codemirror/mode/octave',
            '/media/editors/codemirror/mode/ntriples',
            '/media/editors/codemirror/mode/nsis',
            '/media/editors/codemirror/mode/nginx',
            '/media/editors/codemirror/mode/mumps',
            '/media/editors/codemirror/mode/mscgen',
            '/media/editors/codemirror/mode/modelica',
            '/media/editors/codemirror/mode/mllike',
            '/media/editors/codemirror/mode/mirc',
            '/media/editors/codemirror/mode/mbox',
            '/media/editors/codemirror/mode/mathematica',
            '/media/editors/codemirror/mode/markdown',
            '/media/editors/codemirror/mode/lua',
            '/media/editors/codemirror/mode/livescript',
            '/media/editors/codemirror/mode/julia',
            '/media/editors/codemirror/mode/jsx',
            '/media/editors/codemirror/mode/jinja2',
            '/media/editors/codemirror/mode/javascript',
            '/media/editors/codemirror/mode/idl',
            '/media/editors/codemirror/mode/http',
            '/media/editors/codemirror/mode/htmlmixed',
            '/media/editors/codemirror/mode/htmlembedded',
            '/media/editors/codemirror/mode/haxe',
            '/media/editors/codemirror/mode/haskell-literate',
            '/media/editors/codemirror/mode/haskell',
            '/media/editors/codemirror/mode/handlebars',
            '/media/editors/codemirror/mode/haml',
            '/media/editors/codemirror/mode/groovy',
            '/media/editors/codemirror/mode/go',
            '/media/editors/codemirror/mode/gherkin',
            '/media/editors/codemirror/mode/gfm',
            '/media/editors/codemirror/mode/gas',
            '/media/editors/codemirror/mode/fortran',
            '/media/editors/codemirror/mode/forth',
            '/media/editors/codemirror/mode/fcl',
            '/media/editors/codemirror/mode/factor',
            '/media/editors/codemirror/mode/erlang',
            '/media/editors/codemirror/mode/elm',
            '/media/editors/codemirror/mode/eiffel',
            '/media/editors/codemirror/mode/ecl',
            '/media/editors/codemirror/mode/ebnf',
            '/media/editors/codemirror/mode/dylan',
            '/media/editors/codemirror/mode/dtd',
            '/media/editors/codemirror/mode/dockerfile',
            '/media/editors/codemirror/mode/django',
            '/media/editors/codemirror/mode/diff',
            '/media/editors/codemirror/mode/dart',
            '/media/editors/codemirror/mode/d',
            '/media/editors/codemirror/mode/cypher',
            '/media/editors/codemirror/mode/css',
            '/media/editors/codemirror/mode/crystal',
            '/media/editors/codemirror/mode/commonlisp',
            '/media/editors/codemirror/mode/coffeescript',
            '/media/editors/codemirror/mode/cobol',
            '/media/editors/codemirror/mode/cmake',
            '/media/editors/codemirror/mode/clojure',
            '/media/editors/codemirror/mode/clike',
            '/media/editors/codemirror/mode/brainfuck',
            '/media/editors/codemirror/mode/asterisk',
            '/media/editors/codemirror/mode/asn.1',
            '/media/editors/codemirror/mode/asciiarmor',
            '/media/editors/codemirror/mode/apl',
            '/media/editors/codemirror/mode',
            '/media/editors/codemirror/lib',
            '/media/editors/codemirror/keymap',
            '/media/editors/codemirror/addon/wrap',
            '/media/editors/codemirror/addon/tern',
            '/media/editors/codemirror/addon/selection',
            '/media/editors/codemirror/addon/search',
            '/media/editors/codemirror/addon/scroll',
            '/media/editors/codemirror/addon/runmode',
            '/media/editors/codemirror/addon/mode',
            '/media/editors/codemirror/addon/merge',
            '/media/editors/codemirror/addon/lint',
            '/media/editors/codemirror/addon/hint',
            '/media/editors/codemirror/addon/fold',
            '/media/editors/codemirror/addon/edit',
            '/media/editors/codemirror/addon/display',
            '/media/editors/codemirror/addon/dialog',
            '/media/editors/codemirror/addon/comment',
            '/media/editors/codemirror/addon',
            '/media/editors/codemirror',
            '/media/editors',
            '/media/contacts/images',
            '/media/contacts',
            '/media/com_contenthistory/css',
            '/media/cms/css',
            '/media/cms',
            '/libraries/vendor/symfony/polyfill-util',
            '/libraries/vendor/symfony/polyfill-php71',
            '/libraries/vendor/symfony/polyfill-php56',
            '/libraries/vendor/symfony/polyfill-php55',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/XML/Declaration',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/XML',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Parse',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Net',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/HTTP',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Decode/HTML',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Decode',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Content/Type',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Content',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie/Cache',
            '/libraries/vendor/simplepie/simplepie/library/SimplePie',
            '/libraries/vendor/simplepie/simplepie/library',
            '/libraries/vendor/simplepie/simplepie/idn',
            '/libraries/vendor/simplepie/simplepie',
            '/libraries/vendor/simplepie',
            '/libraries/vendor/phpmailer/phpmailer/extras',
            '/libraries/vendor/paragonie/random_compat/lib',
            '/libraries/vendor/leafo/lessphp',
            '/libraries/vendor/leafo',
            '/libraries/vendor/joomla/session/Joomla/Session/Storage',
            '/libraries/vendor/joomla/session/Joomla/Session',
            '/libraries/vendor/joomla/session/Joomla',
            '/libraries/vendor/joomla/image/src/Filter',
            '/libraries/vendor/joomla/image/src',
            '/libraries/vendor/joomla/image',
            '/libraries/vendor/joomla/compat/src',
            '/libraries/vendor/joomla/compat',
            '/libraries/vendor/joomla/application/src/Cli/Output/Processor',
            '/libraries/vendor/joomla/application/src/Cli/Output',
            '/libraries/vendor/joomla/application/src/Cli',
            '/libraries/vendor/ircmaxell/password-compat/lib',
            '/libraries/vendor/ircmaxell/password-compat',
            '/libraries/vendor/ircmaxell',
            '/libraries/vendor/brumann/polyfill-unserialize/src',
            '/libraries/vendor/brumann/polyfill-unserialize',
            '/libraries/vendor/brumann',
            '/libraries/src/Table/Observer',
            '/libraries/src/Menu/Node',
            '/libraries/src/Language/Wrapper',
            '/libraries/src/Language/Stemmer',
            '/libraries/src/Http/Wrapper',
            '/libraries/src/Filter/Wrapper',
            '/libraries/src/Filesystem/Wrapper',
            '/libraries/src/Crypt/Password',
            '/libraries/src/Access/Wrapper',
            '/libraries/phputf8/utils',
            '/libraries/phputf8/native',
            '/libraries/phputf8/mbstring',
            '/libraries/phputf8',
            '/libraries/legacy/utilities',
            '/libraries/legacy/table',
            '/libraries/legacy/simplepie',
            '/libraries/legacy/simplecrypt',
            '/libraries/legacy/response',
            '/libraries/legacy/request',
            '/libraries/legacy/log',
            '/libraries/legacy/form/field',
            '/libraries/legacy/form',
            '/libraries/legacy/exception',
            '/libraries/legacy/error',
            '/libraries/legacy/dispatcher',
            '/libraries/legacy/database',
            '/libraries/legacy/base',
            '/libraries/legacy/application',
            '/libraries/legacy',
            '/libraries/joomla/view',
            '/libraries/joomla/utilities',
            '/libraries/joomla/twitter',
            '/libraries/joomla/string/wrapper',
            '/libraries/joomla/string',
            '/libraries/joomla/session/storage',
            '/libraries/joomla/session/handler',
            '/libraries/joomla/session',
            '/libraries/joomla/route/wrapper',
            '/libraries/joomla/route',
            '/libraries/joomla/openstreetmap',
            '/libraries/joomla/observer/wrapper',
            '/libraries/joomla/observer/updater',
            '/libraries/joomla/observer',
            '/libraries/joomla/observable',
            '/libraries/joomla/oauth2',
            '/libraries/joomla/oauth1',
            '/libraries/joomla/model',
            '/libraries/joomla/mediawiki',
            '/libraries/joomla/linkedin',
            '/libraries/joomla/keychain',
            '/libraries/joomla/grid',
            '/libraries/joomla/google/embed',
            '/libraries/joomla/google/data/plus',
            '/libraries/joomla/google/data/picasa',
            '/libraries/joomla/google/data',
            '/libraries/joomla/google/auth',
            '/libraries/joomla/google',
            '/libraries/joomla/github/package/users',
            '/libraries/joomla/github/package/repositories',
            '/libraries/joomla/github/package/pulls',
            '/libraries/joomla/github/package/orgs',
            '/libraries/joomla/github/package/issues',
            '/libraries/joomla/github/package/gists',
            '/libraries/joomla/github/package/data',
            '/libraries/joomla/github/package/activity',
            '/libraries/joomla/github/package',
            '/libraries/joomla/github',
            '/libraries/joomla/form/fields',
            '/libraries/joomla/form',
            '/libraries/joomla/facebook',
            '/libraries/joomla/event',
            '/libraries/joomla/database/query',
            '/libraries/joomla/database/iterator',
            '/libraries/joomla/database/importer',
            '/libraries/joomla/database/exporter',
            '/libraries/joomla/database/exception',
            '/libraries/joomla/database/driver',
            '/libraries/joomla/database',
            '/libraries/joomla/controller',
            '/libraries/joomla/archive/wrapper',
            '/libraries/joomla/archive',
            '/libraries/joomla/application/web/router',
            '/libraries/joomla/application/web',
            '/libraries/joomla/application',
            '/libraries/joomla',
            '/libraries/idna_convert',
            '/libraries/fof/view',
            '/libraries/fof/utils/update',
            '/libraries/fof/utils/timer',
            '/libraries/fof/utils/phpfunc',
            '/libraries/fof/utils/observable',
            '/libraries/fof/utils/object',
            '/libraries/fof/utils/ip',
            '/libraries/fof/utils/installscript',
            '/libraries/fof/utils/ini',
            '/libraries/fof/utils/filescheck',
            '/libraries/fof/utils/config',
            '/libraries/fof/utils/cache',
            '/libraries/fof/utils/array',
            '/libraries/fof/utils',
            '/libraries/fof/toolbar',
            '/libraries/fof/template',
            '/libraries/fof/table/dispatcher',
            '/libraries/fof/table/behavior',
            '/libraries/fof/table',
            '/libraries/fof/string',
            '/libraries/fof/render',
            '/libraries/fof/query',
            '/libraries/fof/platform/filesystem',
            '/libraries/fof/platform',
            '/libraries/fof/model/field',
            '/libraries/fof/model/dispatcher',
            '/libraries/fof/model/behavior',
            '/libraries/fof/model',
            '/libraries/fof/less/parser',
            '/libraries/fof/less/formatter',
            '/libraries/fof/less',
            '/libraries/fof/layout',
            '/libraries/fof/integration/joomla/filesystem',
            '/libraries/fof/integration/joomla',
            '/libraries/fof/integration',
            '/libraries/fof/input/jinput',
            '/libraries/fof/input',
            '/libraries/fof/inflector',
            '/libraries/fof/hal/render',
            '/libraries/fof/hal',
            '/libraries/fof/form/header',
            '/libraries/fof/form/field',
            '/libraries/fof/form',
            '/libraries/fof/encrypt/aes',
            '/libraries/fof/encrypt',
            '/libraries/fof/download/adapter',
            '/libraries/fof/download',
            '/libraries/fof/dispatcher',
            '/libraries/fof/database/query',
            '/libraries/fof/database/iterator',
            '/libraries/fof/database/driver',
            '/libraries/fof/database',
            '/libraries/fof/controller',
            '/libraries/fof/config/domain',
            '/libraries/fof/config',
            '/libraries/fof/autoloader',
            '/libraries/fof',
            '/libraries/cms/less/formatter',
            '/libraries/cms/less',
            '/libraries/cms/html/language/en-GB',
            '/libraries/cms/html/language',
            '/libraries/cms/html',
            '/libraries/cms/class',
            '/libraries/cms',
            '/layouts/libraries/cms/html/bootstrap',
            '/layouts/libraries/cms/html',
            '/layouts/libraries/cms',
            '/layouts/joomla/tinymce/buttons',
            '/layouts/joomla/modal',
            '/layouts/joomla/html/formbehavior',
            '/components/com_wrapper/views/wrapper/tmpl',
            '/components/com_wrapper/views/wrapper',
            '/components/com_wrapper/views',
            '/components/com_users/views/reset/tmpl',
            '/components/com_users/views/reset',
            '/components/com_users/views/remind/tmpl',
            '/components/com_users/views/remind',
            '/components/com_users/views/registration/tmpl',
            '/components/com_users/views/registration',
            '/components/com_users/views/profile/tmpl',
            '/components/com_users/views/profile',
            '/components/com_users/views/login/tmpl',
            '/components/com_users/views/login',
            '/components/com_users/views',
            '/components/com_users/models/rules',
            '/components/com_users/models/forms',
            '/components/com_users/models',
            '/components/com_users/layouts/joomla/form',
            '/components/com_users/layouts/joomla',
            '/components/com_users/layouts',
            '/components/com_users/helpers/html',
            '/components/com_users/helpers',
            '/components/com_users/controllers',
            '/components/com_tags/views/tags/tmpl',
            '/components/com_tags/views/tags',
            '/components/com_tags/views/tag/tmpl',
            '/components/com_tags/views/tag',
            '/components/com_tags/views',
            '/components/com_tags/models',
            '/components/com_tags/controllers',
            '/components/com_privacy/views/request/tmpl',
            '/components/com_privacy/views/request',
            '/components/com_privacy/views/remind/tmpl',
            '/components/com_privacy/views/remind',
            '/components/com_privacy/views/confirm/tmpl',
            '/components/com_privacy/views/confirm',
            '/components/com_privacy/views',
            '/components/com_privacy/models/forms',
            '/components/com_privacy/models',
            '/components/com_privacy/controllers',
            '/components/com_newsfeeds/views/newsfeed/tmpl',
            '/components/com_newsfeeds/views/newsfeed',
            '/components/com_newsfeeds/views/category/tmpl',
            '/components/com_newsfeeds/views/category',
            '/components/com_newsfeeds/views/categories/tmpl',
            '/components/com_newsfeeds/views/categories',
            '/components/com_newsfeeds/views',
            '/components/com_newsfeeds/models',
            '/components/com_modules/models/forms',
            '/components/com_modules/models',
            '/components/com_menus/models/forms',
            '/components/com_menus/models',
            '/components/com_mailto/views/sent/tmpl',
            '/components/com_mailto/views/sent',
            '/components/com_mailto/views/mailto/tmpl',
            '/components/com_mailto/views/mailto',
            '/components/com_mailto/views',
            '/components/com_mailto/models/forms',
            '/components/com_mailto/models',
            '/components/com_mailto/helpers',
            '/components/com_mailto',
            '/components/com_finder/views/search/tmpl',
            '/components/com_finder/views/search',
            '/components/com_finder/views',
            '/components/com_finder/models',
            '/components/com_finder/helpers/html',
            '/components/com_finder/controllers',
            '/components/com_fields/models/forms',
            '/components/com_fields/models',
            '/components/com_content/views/form/tmpl',
            '/components/com_content/views/form',
            '/components/com_content/views/featured/tmpl',
            '/components/com_content/views/featured',
            '/components/com_content/views/category/tmpl',
            '/components/com_content/views/category',
            '/components/com_content/views/categories/tmpl',
            '/components/com_content/views/categories',
            '/components/com_content/views/article/tmpl',
            '/components/com_content/views/article',
            '/components/com_content/views/archive/tmpl',
            '/components/com_content/views/archive',
            '/components/com_content/views',
            '/components/com_content/models/forms',
            '/components/com_content/models',
            '/components/com_content/controllers',
            '/components/com_contact/views/featured/tmpl',
            '/components/com_contact/views/featured',
            '/components/com_contact/views/contact/tmpl',
            '/components/com_contact/views/contact',
            '/components/com_contact/views/category/tmpl',
            '/components/com_contact/views/category',
            '/components/com_contact/views/categories/tmpl',
            '/components/com_contact/views/categories',
            '/components/com_contact/views',
            '/components/com_contact/models/rules',
            '/components/com_contact/models/forms',
            '/components/com_contact/models',
            '/components/com_contact/layouts/joomla/form',
            '/components/com_contact/layouts/joomla',
            '/components/com_contact/controllers',
            '/components/com_config/view/templates/tmpl',
            '/components/com_config/view/templates',
            '/components/com_config/view/modules/tmpl',
            '/components/com_config/view/modules',
            '/components/com_config/view/config/tmpl',
            '/components/com_config/view/config',
            '/components/com_config/view/cms',
            '/components/com_config/view',
            '/components/com_config/model/form',
            '/components/com_config/model',
            '/components/com_config/controller/templates',
            '/components/com_config/controller/modules',
            '/components/com_config/controller/config',
            '/components/com_config/controller',
            '/components/com_banners/models',
            '/components/com_banners/helpers',
            '/administrator/templates/system/html',
            '/administrator/templates/isis/less/pages',
            '/administrator/templates/isis/less/bootstrap',
            '/administrator/templates/isis/less/blocks',
            '/administrator/templates/isis/less',
            '/administrator/templates/isis/language/en-GB',
            '/administrator/templates/isis/language',
            '/administrator/templates/isis/js',
            '/administrator/templates/isis/img',
            '/administrator/templates/isis/images/system',
            '/administrator/templates/isis/images/admin',
            '/administrator/templates/isis/images',
            '/administrator/templates/isis/html/mod_version',
            '/administrator/templates/isis/html/layouts/joomla/toolbar',
            '/administrator/templates/isis/html/layouts/joomla/system',
            '/administrator/templates/isis/html/layouts/joomla/pagination',
            '/administrator/templates/isis/html/layouts/joomla/form/field',
            '/administrator/templates/isis/html/layouts/joomla/form',
            '/administrator/templates/isis/html/layouts/joomla',
            '/administrator/templates/isis/html/layouts',
            '/administrator/templates/isis/html/com_media/medialist',
            '/administrator/templates/isis/html/com_media/imageslist',
            '/administrator/templates/isis/html/com_media',
            '/administrator/templates/isis/html',
            '/administrator/templates/isis/css',
            '/administrator/templates/isis',
            '/administrator/templates/hathor/postinstall',
            '/administrator/templates/hathor/less',
            '/administrator/templates/hathor/language/en-GB',
            '/administrator/templates/hathor/language',
            '/administrator/templates/hathor/js',
            '/administrator/templates/hathor/images/toolbar',
            '/administrator/templates/hathor/images/system',
            '/administrator/templates/hathor/images/menu',
            '/administrator/templates/hathor/images/header',
            '/administrator/templates/hathor/images/admin',
            '/administrator/templates/hathor/images',
            '/administrator/templates/hathor/html/mod_quickicon',
            '/administrator/templates/hathor/html/mod_login',
            '/administrator/templates/hathor/html/layouts/plugins/user/profile/fields',
            '/administrator/templates/hathor/html/layouts/plugins/user/profile',
            '/administrator/templates/hathor/html/layouts/plugins/user',
            '/administrator/templates/hathor/html/layouts/plugins',
            '/administrator/templates/hathor/html/layouts/joomla/toolbar',
            '/administrator/templates/hathor/html/layouts/joomla/sidebars',
            '/administrator/templates/hathor/html/layouts/joomla/quickicons',
            '/administrator/templates/hathor/html/layouts/joomla/edit',
            '/administrator/templates/hathor/html/layouts/joomla',
            '/administrator/templates/hathor/html/layouts/com_modules/toolbar',
            '/administrator/templates/hathor/html/layouts/com_modules',
            '/administrator/templates/hathor/html/layouts/com_messages/toolbar',
            '/administrator/templates/hathor/html/layouts/com_messages',
            '/administrator/templates/hathor/html/layouts/com_media/toolbar',
            '/administrator/templates/hathor/html/layouts/com_media',
            '/administrator/templates/hathor/html/layouts',
            '/administrator/templates/hathor/html/com_weblinks/weblinks',
            '/administrator/templates/hathor/html/com_weblinks/weblink',
            '/administrator/templates/hathor/html/com_weblinks',
            '/administrator/templates/hathor/html/com_users/users',
            '/administrator/templates/hathor/html/com_users/user',
            '/administrator/templates/hathor/html/com_users/notes',
            '/administrator/templates/hathor/html/com_users/note',
            '/administrator/templates/hathor/html/com_users/levels',
            '/administrator/templates/hathor/html/com_users/groups',
            '/administrator/templates/hathor/html/com_users/debuguser',
            '/administrator/templates/hathor/html/com_users/debuggroup',
            '/administrator/templates/hathor/html/com_users',
            '/administrator/templates/hathor/html/com_templates/templates',
            '/administrator/templates/hathor/html/com_templates/template',
            '/administrator/templates/hathor/html/com_templates/styles',
            '/administrator/templates/hathor/html/com_templates/style',
            '/administrator/templates/hathor/html/com_templates',
            '/administrator/templates/hathor/html/com_tags/tags',
            '/administrator/templates/hathor/html/com_tags/tag',
            '/administrator/templates/hathor/html/com_tags',
            '/administrator/templates/hathor/html/com_search/searches',
            '/administrator/templates/hathor/html/com_search',
            '/administrator/templates/hathor/html/com_redirect/links',
            '/administrator/templates/hathor/html/com_redirect',
            '/administrator/templates/hathor/html/com_postinstall/messages',
            '/administrator/templates/hathor/html/com_postinstall',
            '/administrator/templates/hathor/html/com_plugins/plugins',
            '/administrator/templates/hathor/html/com_plugins/plugin',
            '/administrator/templates/hathor/html/com_plugins',
            '/administrator/templates/hathor/html/com_newsfeeds/newsfeeds',
            '/administrator/templates/hathor/html/com_newsfeeds/newsfeed',
            '/administrator/templates/hathor/html/com_newsfeeds',
            '/administrator/templates/hathor/html/com_modules/positions',
            '/administrator/templates/hathor/html/com_modules/modules',
            '/administrator/templates/hathor/html/com_modules/module',
            '/administrator/templates/hathor/html/com_modules',
            '/administrator/templates/hathor/html/com_messages/messages',
            '/administrator/templates/hathor/html/com_messages/message',
            '/administrator/templates/hathor/html/com_messages',
            '/administrator/templates/hathor/html/com_menus/menutypes',
            '/administrator/templates/hathor/html/com_menus/menus',
            '/administrator/templates/hathor/html/com_menus/menu',
            '/administrator/templates/hathor/html/com_menus/items',
            '/administrator/templates/hathor/html/com_menus/item',
            '/administrator/templates/hathor/html/com_menus',
            '/administrator/templates/hathor/html/com_languages/overrides',
            '/administrator/templates/hathor/html/com_languages/languages',
            '/administrator/templates/hathor/html/com_languages/installed',
            '/administrator/templates/hathor/html/com_languages',
            '/administrator/templates/hathor/html/com_joomlaupdate/default',
            '/administrator/templates/hathor/html/com_joomlaupdate',
            '/administrator/templates/hathor/html/com_installer/warnings',
            '/administrator/templates/hathor/html/com_installer/update',
            '/administrator/templates/hathor/html/com_installer/manage',
            '/administrator/templates/hathor/html/com_installer/languages',
            '/administrator/templates/hathor/html/com_installer/install',
            '/administrator/templates/hathor/html/com_installer/discover',
            '/administrator/templates/hathor/html/com_installer/default',
            '/administrator/templates/hathor/html/com_installer/database',
            '/administrator/templates/hathor/html/com_installer',
            '/administrator/templates/hathor/html/com_finder/maps',
            '/administrator/templates/hathor/html/com_finder/index',
            '/administrator/templates/hathor/html/com_finder/filters',
            '/administrator/templates/hathor/html/com_finder',
            '/administrator/templates/hathor/html/com_fields/groups',
            '/administrator/templates/hathor/html/com_fields/group',
            '/administrator/templates/hathor/html/com_fields/fields',
            '/administrator/templates/hathor/html/com_fields/field',
            '/administrator/templates/hathor/html/com_fields',
            '/administrator/templates/hathor/html/com_cpanel/cpanel',
            '/administrator/templates/hathor/html/com_cpanel',
            '/administrator/templates/hathor/html/com_contenthistory/history',
            '/administrator/templates/hathor/html/com_contenthistory',
            '/administrator/templates/hathor/html/com_content/featured',
            '/administrator/templates/hathor/html/com_content/articles',
            '/administrator/templates/hathor/html/com_content/article',
            '/administrator/templates/hathor/html/com_content',
            '/administrator/templates/hathor/html/com_contact/contacts',
            '/administrator/templates/hathor/html/com_contact/contact',
            '/administrator/templates/hathor/html/com_contact',
            '/administrator/templates/hathor/html/com_config/component',
            '/administrator/templates/hathor/html/com_config/application',
            '/administrator/templates/hathor/html/com_config',
            '/administrator/templates/hathor/html/com_checkin/checkin',
            '/administrator/templates/hathor/html/com_checkin',
            '/administrator/templates/hathor/html/com_categories/category',
            '/administrator/templates/hathor/html/com_categories/categories',
            '/administrator/templates/hathor/html/com_categories',
            '/administrator/templates/hathor/html/com_cache/purge',
            '/administrator/templates/hathor/html/com_cache/cache',
            '/administrator/templates/hathor/html/com_cache',
            '/administrator/templates/hathor/html/com_banners/tracks',
            '/administrator/templates/hathor/html/com_banners/download',
            '/administrator/templates/hathor/html/com_banners/clients',
            '/administrator/templates/hathor/html/com_banners/client',
            '/administrator/templates/hathor/html/com_banners/banners',
            '/administrator/templates/hathor/html/com_banners/banner',
            '/administrator/templates/hathor/html/com_banners',
            '/administrator/templates/hathor/html/com_associations/associations',
            '/administrator/templates/hathor/html/com_associations',
            '/administrator/templates/hathor/html/com_admin/sysinfo',
            '/administrator/templates/hathor/html/com_admin/profile',
            '/administrator/templates/hathor/html/com_admin/help',
            '/administrator/templates/hathor/html/com_admin',
            '/administrator/templates/hathor/html',
            '/administrator/templates/hathor/css',
            '/administrator/templates/hathor',
            '/administrator/modules/mod_version/language/en-GB',
            '/administrator/modules/mod_version/language',
            '/administrator/modules/mod_status/tmpl',
            '/administrator/modules/mod_status',
            '/administrator/modules/mod_stats_admin/language',
            '/administrator/modules/mod_multilangstatus/language/en-GB',
            '/administrator/modules/mod_multilangstatus/language',
            '/administrator/components/com_users/views/users/tmpl',
            '/administrator/components/com_users/views/users',
            '/administrator/components/com_users/views/user/tmpl',
            '/administrator/components/com_users/views/user',
            '/administrator/components/com_users/views/notes/tmpl',
            '/administrator/components/com_users/views/notes',
            '/administrator/components/com_users/views/note/tmpl',
            '/administrator/components/com_users/views/note',
            '/administrator/components/com_users/views/mail/tmpl',
            '/administrator/components/com_users/views/mail',
            '/administrator/components/com_users/views/levels/tmpl',
            '/administrator/components/com_users/views/levels',
            '/administrator/components/com_users/views/level/tmpl',
            '/administrator/components/com_users/views/level',
            '/administrator/components/com_users/views/groups/tmpl',
            '/administrator/components/com_users/views/groups',
            '/administrator/components/com_users/views/group/tmpl',
            '/administrator/components/com_users/views/group',
            '/administrator/components/com_users/views/debuguser/tmpl',
            '/administrator/components/com_users/views/debuguser',
            '/administrator/components/com_users/views/debuggroup/tmpl',
            '/administrator/components/com_users/views/debuggroup',
            '/administrator/components/com_users/views',
            '/administrator/components/com_users/tables',
            '/administrator/components/com_users/models/forms/fields',
            '/administrator/components/com_users/models/forms',
            '/administrator/components/com_users/models/fields',
            '/administrator/components/com_users/models',
            '/administrator/components/com_users/helpers/html',
            '/administrator/components/com_users/controllers',
            '/administrator/components/com_templates/views/templates/tmpl',
            '/administrator/components/com_templates/views/templates',
            '/administrator/components/com_templates/views/template/tmpl',
            '/administrator/components/com_templates/views/template',
            '/administrator/components/com_templates/views/styles/tmpl',
            '/administrator/components/com_templates/views/styles',
            '/administrator/components/com_templates/views/style/tmpl',
            '/administrator/components/com_templates/views/style',
            '/administrator/components/com_templates/views',
            '/administrator/components/com_templates/tables',
            '/administrator/components/com_templates/models/forms',
            '/administrator/components/com_templates/models/fields',
            '/administrator/components/com_templates/models',
            '/administrator/components/com_templates/helpers/html',
            '/administrator/components/com_templates/controllers',
            '/administrator/components/com_tags/views/tags/tmpl',
            '/administrator/components/com_tags/views/tags',
            '/administrator/components/com_tags/views/tag/tmpl',
            '/administrator/components/com_tags/views/tag',
            '/administrator/components/com_tags/views',
            '/administrator/components/com_tags/tables',
            '/administrator/components/com_tags/models/forms',
            '/administrator/components/com_tags/models',
            '/administrator/components/com_tags/helpers',
            '/administrator/components/com_tags/controllers',
            '/administrator/components/com_redirect/views/links/tmpl',
            '/administrator/components/com_redirect/views/links',
            '/administrator/components/com_redirect/views/link/tmpl',
            '/administrator/components/com_redirect/views/link',
            '/administrator/components/com_redirect/views',
            '/administrator/components/com_redirect/tables',
            '/administrator/components/com_redirect/models/forms',
            '/administrator/components/com_redirect/models/fields',
            '/administrator/components/com_redirect/models',
            '/administrator/components/com_redirect/helpers/html',
            '/administrator/components/com_redirect/controllers',
            '/administrator/components/com_privacy/views/requests/tmpl',
            '/administrator/components/com_privacy/views/requests',
            '/administrator/components/com_privacy/views/request/tmpl',
            '/administrator/components/com_privacy/views/request',
            '/administrator/components/com_privacy/views/export',
            '/administrator/components/com_privacy/views/dashboard/tmpl',
            '/administrator/components/com_privacy/views/dashboard',
            '/administrator/components/com_privacy/views/consents/tmpl',
            '/administrator/components/com_privacy/views/consents',
            '/administrator/components/com_privacy/views/capabilities/tmpl',
            '/administrator/components/com_privacy/views/capabilities',
            '/administrator/components/com_privacy/views',
            '/administrator/components/com_privacy/tables',
            '/administrator/components/com_privacy/models/forms',
            '/administrator/components/com_privacy/models/fields',
            '/administrator/components/com_privacy/models',
            '/administrator/components/com_privacy/helpers/removal',
            '/administrator/components/com_privacy/helpers/html',
            '/administrator/components/com_privacy/helpers/export',
            '/administrator/components/com_privacy/helpers',
            '/administrator/components/com_privacy/controllers',
            '/administrator/components/com_postinstall/views/messages/tmpl',
            '/administrator/components/com_postinstall/views/messages',
            '/administrator/components/com_postinstall/views',
            '/administrator/components/com_postinstall/models',
            '/administrator/components/com_postinstall/controllers',
            '/administrator/components/com_plugins/views/plugins/tmpl',
            '/administrator/components/com_plugins/views/plugins',
            '/administrator/components/com_plugins/views/plugin/tmpl',
            '/administrator/components/com_plugins/views/plugin',
            '/administrator/components/com_plugins/views',
            '/administrator/components/com_plugins/models/forms',
            '/administrator/components/com_plugins/models/fields',
            '/administrator/components/com_plugins/models',
            '/administrator/components/com_plugins/controllers',
            '/administrator/components/com_newsfeeds/views/newsfeeds/tmpl',
            '/administrator/components/com_newsfeeds/views/newsfeeds',
            '/administrator/components/com_newsfeeds/views/newsfeed/tmpl',
            '/administrator/components/com_newsfeeds/views/newsfeed',
            '/administrator/components/com_newsfeeds/views',
            '/administrator/components/com_newsfeeds/tables',
            '/administrator/components/com_newsfeeds/models/forms',
            '/administrator/components/com_newsfeeds/models/fields/modal',
            '/administrator/components/com_newsfeeds/models/fields',
            '/administrator/components/com_newsfeeds/models',
            '/administrator/components/com_newsfeeds/helpers/html',
            '/administrator/components/com_newsfeeds/controllers',
            '/administrator/components/com_modules/views/select/tmpl',
            '/administrator/components/com_modules/views/select',
            '/administrator/components/com_modules/views/preview/tmpl',
            '/administrator/components/com_modules/views/preview',
            '/administrator/components/com_modules/views/positions/tmpl',
            '/administrator/components/com_modules/views/positions',
            '/administrator/components/com_modules/views/modules/tmpl',
            '/administrator/components/com_modules/views/modules',
            '/administrator/components/com_modules/views/module/tmpl',
            '/administrator/components/com_modules/views/module',
            '/administrator/components/com_modules/views',
            '/administrator/components/com_modules/models/forms',
            '/administrator/components/com_modules/models/fields',
            '/administrator/components/com_modules/models',
            '/administrator/components/com_modules/helpers/html',
            '/administrator/components/com_modules/controllers',
            '/administrator/components/com_messages/views/messages/tmpl',
            '/administrator/components/com_messages/views/messages',
            '/administrator/components/com_messages/views/message/tmpl',
            '/administrator/components/com_messages/views/message',
            '/administrator/components/com_messages/views/config/tmpl',
            '/administrator/components/com_messages/views/config',
            '/administrator/components/com_messages/views',
            '/administrator/components/com_messages/tables',
            '/administrator/components/com_messages/models/forms',
            '/administrator/components/com_messages/models/fields',
            '/administrator/components/com_messages/models',
            '/administrator/components/com_messages/helpers/html',
            '/administrator/components/com_messages/helpers',
            '/administrator/components/com_messages/controllers',
            '/administrator/components/com_menus/views/menutypes/tmpl',
            '/administrator/components/com_menus/views/menutypes',
            '/administrator/components/com_menus/views/menus/tmpl',
            '/administrator/components/com_menus/views/menus',
            '/administrator/components/com_menus/views/menu/tmpl',
            '/administrator/components/com_menus/views/menu',
            '/administrator/components/com_menus/views/items/tmpl',
            '/administrator/components/com_menus/views/items',
            '/administrator/components/com_menus/views/item/tmpl',
            '/administrator/components/com_menus/views/item',
            '/administrator/components/com_menus/views',
            '/administrator/components/com_menus/tables',
            '/administrator/components/com_menus/models/forms',
            '/administrator/components/com_menus/models/fields/modal',
            '/administrator/components/com_menus/models/fields',
            '/administrator/components/com_menus/models',
            '/administrator/components/com_menus/layouts/joomla/searchtools/default',
            '/administrator/components/com_menus/helpers/html',
            '/administrator/components/com_menus/controllers',
            '/administrator/components/com_media/views/medialist/tmpl',
            '/administrator/components/com_media/views/medialist',
            '/administrator/components/com_media/views/media/tmpl',
            '/administrator/components/com_media/views/media',
            '/administrator/components/com_media/views/imageslist/tmpl',
            '/administrator/components/com_media/views/imageslist',
            '/administrator/components/com_media/views/images/tmpl',
            '/administrator/components/com_media/views/images',
            '/administrator/components/com_media/views',
            '/administrator/components/com_media/models',
            '/administrator/components/com_media/controllers',
            '/administrator/components/com_login/views/login/tmpl',
            '/administrator/components/com_login/views/login',
            '/administrator/components/com_login/views',
            '/administrator/components/com_login/models',
            '/administrator/components/com_languages/views/overrides/tmpl',
            '/administrator/components/com_languages/views/overrides',
            '/administrator/components/com_languages/views/override/tmpl',
            '/administrator/components/com_languages/views/override',
            '/administrator/components/com_languages/views/multilangstatus/tmpl',
            '/administrator/components/com_languages/views/multilangstatus',
            '/administrator/components/com_languages/views/languages/tmpl',
            '/administrator/components/com_languages/views/languages',
            '/administrator/components/com_languages/views/language/tmpl',
            '/administrator/components/com_languages/views/language',
            '/administrator/components/com_languages/views/installed/tmpl',
            '/administrator/components/com_languages/views/installed',
            '/administrator/components/com_languages/views',
            '/administrator/components/com_languages/models/forms',
            '/administrator/components/com_languages/models/fields',
            '/administrator/components/com_languages/models',
            '/administrator/components/com_languages/layouts/joomla/searchtools/default',
            '/administrator/components/com_languages/layouts/joomla/searchtools',
            '/administrator/components/com_languages/layouts/joomla',
            '/administrator/components/com_languages/layouts',
            '/administrator/components/com_languages/helpers/html',
            '/administrator/components/com_languages/helpers',
            '/administrator/components/com_languages/controllers',
            '/administrator/components/com_joomlaupdate/views/upload/tmpl',
            '/administrator/components/com_joomlaupdate/views/upload',
            '/administrator/components/com_joomlaupdate/views/update/tmpl',
            '/administrator/components/com_joomlaupdate/views/update',
            '/administrator/components/com_joomlaupdate/views/default/tmpl',
            '/administrator/components/com_joomlaupdate/views/default',
            '/administrator/components/com_joomlaupdate/views',
            '/administrator/components/com_joomlaupdate/models',
            '/administrator/components/com_joomlaupdate/helpers',
            '/administrator/components/com_joomlaupdate/controllers',
            '/administrator/components/com_installer/views/warnings/tmpl',
            '/administrator/components/com_installer/views/warnings',
            '/administrator/components/com_installer/views/updatesites/tmpl',
            '/administrator/components/com_installer/views/updatesites',
            '/administrator/components/com_installer/views/update/tmpl',
            '/administrator/components/com_installer/views/update',
            '/administrator/components/com_installer/views/manage/tmpl',
            '/administrator/components/com_installer/views/manage',
            '/administrator/components/com_installer/views/languages/tmpl',
            '/administrator/components/com_installer/views/languages',
            '/administrator/components/com_installer/views/install/tmpl',
            '/administrator/components/com_installer/views/install',
            '/administrator/components/com_installer/views/discover/tmpl',
            '/administrator/components/com_installer/views/discover',
            '/administrator/components/com_installer/views/default/tmpl',
            '/administrator/components/com_installer/views/default',
            '/administrator/components/com_installer/views/database/tmpl',
            '/administrator/components/com_installer/views/database',
            '/administrator/components/com_installer/views',
            '/administrator/components/com_installer/models/forms',
            '/administrator/components/com_installer/models/fields',
            '/administrator/components/com_installer/models',
            '/administrator/components/com_installer/helpers/html',
            '/administrator/components/com_installer/controllers',
            '/administrator/components/com_finder/views/statistics/tmpl',
            '/administrator/components/com_finder/views/statistics',
            '/administrator/components/com_finder/views/maps/tmpl',
            '/administrator/components/com_finder/views/maps',
            '/administrator/components/com_finder/views/indexer/tmpl',
            '/administrator/components/com_finder/views/indexer',
            '/administrator/components/com_finder/views/index/tmpl',
            '/administrator/components/com_finder/views/index',
            '/administrator/components/com_finder/views/filters/tmpl',
            '/administrator/components/com_finder/views/filters',
            '/administrator/components/com_finder/views/filter/tmpl',
            '/administrator/components/com_finder/views/filter',
            '/administrator/components/com_finder/views',
            '/administrator/components/com_finder/tables',
            '/administrator/components/com_finder/models/forms',
            '/administrator/components/com_finder/models/fields',
            '/administrator/components/com_finder/models',
            '/administrator/components/com_finder/helpers/indexer/stemmer',
            '/administrator/components/com_finder/helpers/indexer/parser',
            '/administrator/components/com_finder/helpers/indexer/driver',
            '/administrator/components/com_finder/helpers/html',
            '/administrator/components/com_finder/controllers',
            '/administrator/components/com_fields/views/groups/tmpl',
            '/administrator/components/com_fields/views/groups',
            '/administrator/components/com_fields/views/group/tmpl',
            '/administrator/components/com_fields/views/group',
            '/administrator/components/com_fields/views/fields/tmpl',
            '/administrator/components/com_fields/views/fields',
            '/administrator/components/com_fields/views/field/tmpl',
            '/administrator/components/com_fields/views/field',
            '/administrator/components/com_fields/views',
            '/administrator/components/com_fields/tables',
            '/administrator/components/com_fields/models/forms',
            '/administrator/components/com_fields/models/fields',
            '/administrator/components/com_fields/models',
            '/administrator/components/com_fields/libraries',
            '/administrator/components/com_fields/controllers',
            '/administrator/components/com_cpanel/views/cpanel/tmpl',
            '/administrator/components/com_cpanel/views/cpanel',
            '/administrator/components/com_cpanel/views',
            '/administrator/components/com_contenthistory/views/preview/tmpl',
            '/administrator/components/com_contenthistory/views/preview',
            '/administrator/components/com_contenthistory/views/history/tmpl',
            '/administrator/components/com_contenthistory/views/history',
            '/administrator/components/com_contenthistory/views/compare/tmpl',
            '/administrator/components/com_contenthistory/views/compare',
            '/administrator/components/com_contenthistory/views',
            '/administrator/components/com_contenthistory/models',
            '/administrator/components/com_contenthistory/helpers/html',
            '/administrator/components/com_contenthistory/controllers',
            '/administrator/components/com_content/views/featured/tmpl',
            '/administrator/components/com_content/views/featured',
            '/administrator/components/com_content/views/articles/tmpl',
            '/administrator/components/com_content/views/articles',
            '/administrator/components/com_content/views/article/tmpl',
            '/administrator/components/com_content/views/article',
            '/administrator/components/com_content/views',
            '/administrator/components/com_content/tables',
            '/administrator/components/com_content/models/forms',
            '/administrator/components/com_content/models/fields/modal',
            '/administrator/components/com_content/models/fields',
            '/administrator/components/com_content/models',
            '/administrator/components/com_content/helpers/html',
            '/administrator/components/com_content/controllers',
            '/administrator/components/com_contact/views/contacts/tmpl',
            '/administrator/components/com_contact/views/contacts',
            '/administrator/components/com_contact/views/contact/tmpl',
            '/administrator/components/com_contact/views/contact',
            '/administrator/components/com_contact/views',
            '/administrator/components/com_contact/tables',
            '/administrator/components/com_contact/models/forms/fields',
            '/administrator/components/com_contact/models/forms',
            '/administrator/components/com_contact/models/fields/modal',
            '/administrator/components/com_contact/models/fields',
            '/administrator/components/com_contact/models',
            '/administrator/components/com_contact/helpers/html',
            '/administrator/components/com_contact/controllers',
            '/administrator/components/com_config/view/component/tmpl',
            '/administrator/components/com_config/view/component',
            '/administrator/components/com_config/view/application/tmpl',
            '/administrator/components/com_config/view/application',
            '/administrator/components/com_config/view',
            '/administrator/components/com_config/models',
            '/administrator/components/com_config/model/form',
            '/administrator/components/com_config/model/field',
            '/administrator/components/com_config/model',
            '/administrator/components/com_config/helper',
            '/administrator/components/com_config/controllers',
            '/administrator/components/com_config/controller/component',
            '/administrator/components/com_config/controller/application',
            '/administrator/components/com_config/controller',
            '/administrator/components/com_checkin/views/checkin/tmpl',
            '/administrator/components/com_checkin/views/checkin',
            '/administrator/components/com_checkin/views',
            '/administrator/components/com_checkin/models/forms',
            '/administrator/components/com_checkin/models',
            '/administrator/components/com_categories/views/category/tmpl',
            '/administrator/components/com_categories/views/category',
            '/administrator/components/com_categories/views/categories/tmpl',
            '/administrator/components/com_categories/views/categories',
            '/administrator/components/com_categories/views',
            '/administrator/components/com_categories/tables',
            '/administrator/components/com_categories/models/forms',
            '/administrator/components/com_categories/models/fields/modal',
            '/administrator/components/com_categories/models/fields',
            '/administrator/components/com_categories/models',
            '/administrator/components/com_categories/helpers/html',
            '/administrator/components/com_categories/controllers',
            '/administrator/components/com_cache/views/purge/tmpl',
            '/administrator/components/com_cache/views/purge',
            '/administrator/components/com_cache/views/cache/tmpl',
            '/administrator/components/com_cache/views/cache',
            '/administrator/components/com_cache/views',
            '/administrator/components/com_cache/models/forms',
            '/administrator/components/com_cache/models',
            '/administrator/components/com_cache/helpers',
            '/administrator/components/com_banners/views/tracks/tmpl',
            '/administrator/components/com_banners/views/tracks',
            '/administrator/components/com_banners/views/download/tmpl',
            '/administrator/components/com_banners/views/download',
            '/administrator/components/com_banners/views/clients/tmpl',
            '/administrator/components/com_banners/views/clients',
            '/administrator/components/com_banners/views/client/tmpl',
            '/administrator/components/com_banners/views/client',
            '/administrator/components/com_banners/views/banners/tmpl',
            '/administrator/components/com_banners/views/banners',
            '/administrator/components/com_banners/views/banner/tmpl',
            '/administrator/components/com_banners/views/banner',
            '/administrator/components/com_banners/views',
            '/administrator/components/com_banners/tables',
            '/administrator/components/com_banners/models/forms',
            '/administrator/components/com_banners/models/fields',
            '/administrator/components/com_banners/models',
            '/administrator/components/com_banners/helpers/html',
            '/administrator/components/com_banners/controllers',
            '/administrator/components/com_associations/views/associations/tmpl',
            '/administrator/components/com_associations/views/associations',
            '/administrator/components/com_associations/views/association/tmpl',
            '/administrator/components/com_associations/views/association',
            '/administrator/components/com_associations/views',
            '/administrator/components/com_associations/models/forms',
            '/administrator/components/com_associations/models/fields',
            '/administrator/components/com_associations/models',
            '/administrator/components/com_associations/layouts/joomla/searchtools/default',
            '/administrator/components/com_associations/helpers',
            '/administrator/components/com_associations/controllers',
            '/administrator/components/com_admin/views/sysinfo/tmpl',
            '/administrator/components/com_admin/views/sysinfo',
            '/administrator/components/com_admin/views/profile/tmpl',
            '/administrator/components/com_admin/views/profile',
            '/administrator/components/com_admin/views/help/tmpl',
            '/administrator/components/com_admin/views/help',
            '/administrator/components/com_admin/views',
            '/administrator/components/com_admin/sql/updates/sqlazure',
            '/administrator/components/com_admin/models/forms',
            '/administrator/components/com_admin/models',
            '/administrator/components/com_admin/helpers/html',
            '/administrator/components/com_admin/helpers',
            '/administrator/components/com_admin/controllers',
            '/administrator/components/com_actionlogs/views/actionlogs/tmpl',
            '/administrator/components/com_actionlogs/views/actionlogs',
            '/administrator/components/com_actionlogs/views',
            '/administrator/components/com_actionlogs/models/forms',
            '/administrator/components/com_actionlogs/models/fields',
            '/administrator/components/com_actionlogs/models',
            '/administrator/components/com_actionlogs/libraries',
            '/administrator/components/com_actionlogs/layouts',
            '/administrator/components/com_actionlogs/helpers',
            '/administrator/components/com_actionlogs/controllers',
            // 4.0 from Beta 1 to Beta 2
            '/libraries/vendor/joomla/controller/src',
            '/libraries/vendor/joomla/controller',
            '/api/components/com_installer/src/View/Languages',
            '/administrator/components/com_finder/src/Indexer/Driver',
            // 4.0 from Beta 4 to Beta 5
            '/plugins/content/imagelazyload',
            // 4.0 from Beta 5 to Beta 6
            '/media/system/js/core.es6',
            '/administrator/modules/mod_multilangstatus/src/Helper',
            '/administrator/modules/mod_multilangstatus/src',
            // 4.0 from Beta 6 to Beta 7
            '/media/vendor/skipto/css',
            // 4.0 from Beta 7 to RC 1
            '/templates/system/js',
            '/templates/cassiopeia/scss/tools/mixins',
            '/plugins/fields/subfields/tmpl',
            '/plugins/fields/subfields/params',
            '/plugins/fields/subfields',
            '/media/vendor/punycode/js',
            '/media/templates/atum/js',
            '/media/templates/atum',
            '/libraries/vendor/paragonie/random_compat/dist',
            '/libraries/vendor/paragonie/random_compat',
            '/libraries/vendor/ozdemirburak/iris/src/Traits',
            '/libraries/vendor/ozdemirburak/iris/src/Helpers',
            '/libraries/vendor/ozdemirburak/iris/src/Exceptions',
            '/libraries/vendor/ozdemirburak/iris/src/Color',
            '/libraries/vendor/ozdemirburak/iris/src',
            '/libraries/vendor/ozdemirburak/iris',
            '/libraries/vendor/ozdemirburak',
            '/libraries/vendor/bin',
            '/components/com_menus/src/Controller',
            '/components/com_csp/src/Controller',
            '/components/com_csp/src',
            '/components/com_csp',
            '/administrator/templates/atum/Service/HTML',
            '/administrator/templates/atum/Service',
            '/administrator/components/com_joomlaupdate/src/Helper',
            '/administrator/components/com_csp/tmpl/reports',
            '/administrator/components/com_csp/tmpl',
            '/administrator/components/com_csp/src/View/Reports',
            '/administrator/components/com_csp/src/View',
            '/administrator/components/com_csp/src/Table',
            '/administrator/components/com_csp/src/Model',
            '/administrator/components/com_csp/src/Helper',
            '/administrator/components/com_csp/src/Controller',
            '/administrator/components/com_csp/src',
            '/administrator/components/com_csp/services',
            '/administrator/components/com_csp/forms',
            '/administrator/components/com_csp',
            '/administrator/components/com_admin/tmpl/profile',
            '/administrator/components/com_admin/src/View/Profile',
            '/administrator/components/com_admin/forms',
            // 4.0 from RC 5 to RC 6
            '/templates/cassiopeia/scss/vendor/fontawesome-free',
            '/templates/cassiopeia/css/vendor/fontawesome-free',
            '/media/templates/cassiopeia/js/mod_menu',
            '/media/templates/cassiopeia/js',
            '/media/templates/cassiopeia',
            // 4.0 from RC 6 to 4.0.0 (stable)
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation/Tests',
            '/libraries/vendor/willdurand/negotiation/tests/Negotiation',
            '/libraries/vendor/willdurand/negotiation/tests',
            '/libraries/vendor/jakeasmith/http_build_url/tests',
            '/libraries/vendor/doctrine/inflector/docs/en',
            '/libraries/vendor/doctrine/inflector/docs',
            '/libraries/vendor/algo26-matthias/idna-convert/tests/unit',
            '/libraries/vendor/algo26-matthias/idna-convert/tests/integration',
            '/libraries/vendor/algo26-matthias/idna-convert/tests',
            // From 4.0.3 to 4.0.4
            '/templates/cassiopeia/images/system',
            // From 4.0.x to 4.1.0-beta1
            '/templates/system/scss',
            '/templates/system/css',
            '/templates/cassiopeia/scss/vendor/metismenu',
            '/templates/cassiopeia/scss/vendor/joomla-custom-elements',
            '/templates/cassiopeia/scss/vendor/choicesjs',
            '/templates/cassiopeia/scss/vendor/bootstrap',
            '/templates/cassiopeia/scss/vendor',
            '/templates/cassiopeia/scss/tools/variables',
            '/templates/cassiopeia/scss/tools/functions',
            '/templates/cassiopeia/scss/tools',
            '/templates/cassiopeia/scss/system/searchtools',
            '/templates/cassiopeia/scss/system',
            '/templates/cassiopeia/scss/global',
            '/templates/cassiopeia/scss/blocks',
            '/templates/cassiopeia/scss',
            '/templates/cassiopeia/js',
            '/templates/cassiopeia/images',
            '/templates/cassiopeia/css/vendor/joomla-custom-elements',
            '/templates/cassiopeia/css/vendor/choicesjs',
            '/templates/cassiopeia/css/vendor',
            '/templates/cassiopeia/css/system/searchtools',
            '/templates/cassiopeia/css/system',
            '/templates/cassiopeia/css/global',
            '/templates/cassiopeia/css',
            '/administrator/templates/system/scss',
            '/administrator/templates/system/images',
            '/administrator/templates/system/css',
            '/administrator/templates/atum/scss/vendor/minicolors',
            '/administrator/templates/atum/scss/vendor/joomla-custom-elements',
            '/administrator/templates/atum/scss/vendor/fontawesome-free',
            '/administrator/templates/atum/scss/vendor/choicesjs',
            '/administrator/templates/atum/scss/vendor/bootstrap',
            '/administrator/templates/atum/scss/vendor/awesomplete',
            '/administrator/templates/atum/scss/vendor',
            '/administrator/templates/atum/scss/system/searchtools',
            '/administrator/templates/atum/scss/system',
            '/administrator/templates/atum/scss/pages',
            '/administrator/templates/atum/scss/blocks',
            '/administrator/templates/atum/scss',
            '/administrator/templates/atum/images/logos',
            '/administrator/templates/atum/images',
            '/administrator/templates/atum/css/vendor/minicolors',
            '/administrator/templates/atum/css/vendor/joomla-custom-elements',
            '/administrator/templates/atum/css/vendor/fontawesome-free',
            '/administrator/templates/atum/css/vendor/choicesjs',
            '/administrator/templates/atum/css/vendor/awesomplete',
            '/administrator/templates/atum/css/vendor',
            '/administrator/templates/atum/css/system/searchtools',
            '/administrator/templates/atum/css/system',
            '/administrator/templates/atum/css',
            // From 4.1.0-beta3 to 4.1.0-rc1
            '/api/components/com_media/src/Helper',
            // From 4.1.0 to 4.1.1
            '/libraries/vendor/tobscure/json-api/tests/Exception/Handler',
            '/libraries/vendor/tobscure/json-api/tests/Exception',
            '/libraries/vendor/tobscure/json-api/tests',
            '/libraries/vendor/tobscure/json-api/.git/refs/tags',
            '/libraries/vendor/tobscure/json-api/.git/refs/remotes/origin',
            '/libraries/vendor/tobscure/json-api/.git/refs/remotes',
            '/libraries/vendor/tobscure/json-api/.git/refs/heads',
            '/libraries/vendor/tobscure/json-api/.git/refs',
            '/libraries/vendor/tobscure/json-api/.git/objects/pack',
            '/libraries/vendor/tobscure/json-api/.git/objects/info',
            '/libraries/vendor/tobscure/json-api/.git/objects',
            '/libraries/vendor/tobscure/json-api/.git/logs/refs/remotes/origin',
            '/libraries/vendor/tobscure/json-api/.git/logs/refs/remotes',
            '/libraries/vendor/tobscure/json-api/.git/logs/refs/heads',
            '/libraries/vendor/tobscure/json-api/.git/logs/refs',
            '/libraries/vendor/tobscure/json-api/.git/logs',
            '/libraries/vendor/tobscure/json-api/.git/info',
            '/libraries/vendor/tobscure/json-api/.git/hooks',
            '/libraries/vendor/tobscure/json-api/.git/branches',
            '/libraries/vendor/tobscure/json-api/.git',
            // From 4.1.3 to 4.1.4
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/Storage',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataFormatter',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests/DataCollector',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar/Tests',
            '/libraries/vendor/maximebf/debugbar/tests/DebugBar',
            '/libraries/vendor/maximebf/debugbar/tests',
            '/libraries/vendor/maximebf/debugbar/docs',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/twig',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/swiftmailer',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/slim',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/propel',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/monolog',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/src/Demo',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine/src',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/doctrine',
            '/libraries/vendor/maximebf/debugbar/demo/bridge/cachecache',
            '/libraries/vendor/maximebf/debugbar/demo/bridge',
            '/libraries/vendor/maximebf/debugbar/demo',
            '/libraries/vendor/maximebf/debugbar/build',
            // From 4.1 to 4.2.0-beta1
            '/plugins/twofactorauth/yubikey/tmpl',
            '/plugins/twofactorauth/yubikey',
            '/plugins/twofactorauth/totp/tmpl',
            '/plugins/twofactorauth/totp/postinstall',
            '/plugins/twofactorauth/totp',
            '/plugins/twofactorauth',
            '/libraries/vendor/nyholm/psr7/doc',
            // From 4.2.0-beta1 to 4.2.0-beta2
            '/layouts/plugins/user/profile/fields',
            '/layouts/plugins/user/profile',
            // From 4.2.0-beta2 to 4.2.0-beta3
            '/plugins/system/webauthn/src/Helper',
            '/plugins/system/webauthn/src/Exception',
            // From 4.2.0-rc1 to 4.2.0
            '/plugins/fields/menuitem/tmpl',
            '/plugins/fields/menuitem',
            // From 4.2.0 to 4.2.1
            '/media/vendor/hotkeys.js/js',
            '/media/vendor/hotkeys.js',
            '/libraries/vendor/symfony/string/Resources/bin',
            // From 4.2.x to 4.3.0-alpha1
            '/libraries/vendor/paragonie/sodium_compat/dist',
            // From 4.3.0-beta4 to 4.3.0-rc1
            '/libraries/vendor/voku/portable-ascii/build/docs',
            '/libraries/vendor/voku/portable-ascii/build',
            '/administrator/components/com_guidedtours/src/Helper',
        ];

        $status['files_checked']   = $files;
        $status['folders_checked'] = $folders;

        foreach ($files as $file) {
            if ($fileExists = is_file(JPATH_ROOT . $file)) {
                $status['files_exist'][] = $file;

                if ($dryRun === false) {
                    if (File::delete(JPATH_ROOT . $file)) {
                        $status['files_deleted'][] = $file;
                    } else {
                        $status['files_errors'][] = sprintf('Error on deleting file or folder %s', $file);
                    }
                }
            }
        }

        $this->moveRemainingTemplateFiles();

        foreach ($folders as $folder) {
            if ($folderExists = Folder::exists(JPATH_ROOT . $folder)) {
                $status['folders_exist'][] = $folder;

                if ($dryRun === false) {
                    if (Folder::delete(JPATH_ROOT . $folder)) {
                        $status['folders_deleted'][] = $folder;
                    } else {
                        $status['folders_errors'][] = sprintf('Error on deleting file or folder %s', $folder);
                    }
                }
            }
        }

        $this->fixFilenameCasing();

        /*
         * Needed for updates from 3.10
         * If com_search doesn't exist then assume we can delete the search package manifest (included in the update packages)
         * We deliberately check for the presence of the files in case people have previously uninstalled their search extension
         * but an update has put the files back. In that case it exists even if they don't believe in it!
         */
        if (
            !is_file(JPATH_ROOT . '/administrator/components/com_search/search.php')
            && is_file(JPATH_ROOT . '/administrator/manifests/packages/pkg_search.xml')
        ) {
            File::delete(JPATH_ROOT . '/administrator/manifests/packages/pkg_search.xml');
        }

        if ($suppressOutput === false && count($status['folders_errors'])) {
            echo implode('<br>', $status['folders_errors']);
        }

        if ($suppressOutput === false && count($status['files_errors'])) {
            echo implode('<br>', $status['files_errors']);
        }

        return $status;
    }

    /**
     * Method to create assets for newly installed components
     *
     * @param   Installer  $installer  The class calling this method
     *
     * @return  boolean
     *
     * @since   3.2
     */
    public function updateAssets($installer)
    {
        // List all components added since 4.0
        $newComponents = [
            // Components to be added here
            'com_guidedtours',
            'com_mails',
            'com_scheduler',
            'com_workflow',
        ];

        foreach ($newComponents as $component) {
            /** @var \Joomla\CMS\Table\Asset $asset */
            $asset = Table::getInstance('Asset');

            if ($asset->loadByName($component)) {
                continue;
            }

            $asset->name      = $component;
            $asset->parent_id = 1;
            $asset->rules     = '{}';
            $asset->title     = $component;
            $asset->setLocation(1, 'last-child');

            if (!$asset->store()) {
                $this->collectError(__METHOD__, new \Exception($asset->getError(true)));

                // Install failed, roll back changes
                $installer->abort(Text::sprintf('JLIB_INSTALLER_ABORT_COMP_INSTALL_ROLLBACK', $asset->getError(true)));

                return false;
            }
        }

        return true;
    }

    /**
     * Converts the site's database tables to support UTF-8 Multibyte.
     *
     * @param   boolean  $doDbFixMsg  Flag if message to be shown to check db fix
     *
     * @return  void
     *
     * @since   3.5
     */
    public function convertTablesToUtf8mb4($doDbFixMsg = false)
    {
        $db = Factory::getDbo();

        if ($db->getServerType() !== 'mysql') {
            return;
        }

        // Check if the #__utf8_conversion table exists
        $db->setQuery('SHOW TABLES LIKE ' . $db->quote($db->getPrefix() . 'utf8_conversion'));

        try {
            $rows = $db->loadRowList(0);
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            if ($doDbFixMsg) {
                // Show an error message telling to check database problems
                Factory::getApplication()->enqueueMessage(Text::_('JLIB_DATABASE_ERROR_DATABASE_UPGRADE_FAILED'), 'error');
            }

            return;
        }

        // Nothing to do if the table doesn't exist because the CMS has never been updated from a pre-4.0 version
        if (count($rows) === 0) {
            return;
        }

        // Set required conversion status
        $converted = 5;

        // Check conversion status in database
        $db->setQuery(
            'SELECT ' . $db->quoteName('converted')
            . ' FROM ' . $db->quoteName('#__utf8_conversion')
        );

        try {
            $convertedDB = $db->loadResult();
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            if ($doDbFixMsg) {
                // Show an error message telling to check database problems
                Factory::getApplication()->enqueueMessage(Text::_('JLIB_DATABASE_ERROR_DATABASE_UPGRADE_FAILED'), 'error');
            }

            return;
        }

        // If conversion status from DB is equal to required final status, try to drop the #__utf8_conversion table
        if ($convertedDB === $converted) {
            $this->dropUtf8ConversionTable();

            return;
        }

        // Perform the required conversions of core tables if not done already in a previous step
        if ($convertedDB !== 99) {
            $fileName1 = JPATH_ROOT . '/administrator/components/com_admin/sql/others/mysql/utf8mb4-conversion.sql';

            if (is_file($fileName1)) {
                $fileContents1 = @file_get_contents($fileName1);
                $queries1      = $db->splitSql($fileContents1);

                if (!empty($queries1)) {
                    foreach ($queries1 as $query1) {
                        try {
                            $db->setQuery($query1)->execute();
                        } catch (Exception $e) {
                            $converted = $convertedDB;

                            $this->collectError(__METHOD__, $e);
                        }
                    }
                }
            }
        }

        // If no error before, perform the optional conversions of tables which might or might not exist
        if ($converted === 5) {
            $fileName2 = JPATH_ROOT . '/administrator/components/com_admin/sql/others/mysql/utf8mb4-conversion_optional.sql';

            if (is_file($fileName2)) {
                $fileContents2 = @file_get_contents($fileName2);
                $queries2      = $db->splitSql($fileContents2);

                if (!empty($queries2)) {
                    foreach ($queries2 as $query2) {
                        // Get table name from query
                        if (preg_match('/^ALTER\s+TABLE\s+([^\s]+)\s+/i', $query2, $matches) === 1) {
                            $tableName = str_replace('`', '', $matches[1]);
                            $tableName = str_replace('#__', $db->getPrefix(), $tableName);

                            // Check if the table exists and if yes, run the query
                            try {
                                $db->setQuery('SHOW TABLES LIKE ' . $db->quote($tableName));

                                $rows = $db->loadRowList(0);

                                if (count($rows) > 0) {
                                    $db->setQuery($query2)->execute();
                                }
                            } catch (Exception $e) {
                                $converted = 99;

                                $this->collectError(__METHOD__, $e);
                            }
                        }
                    }
                }
            }
        }

        if ($doDbFixMsg && $converted !== 5) {
            // Show an error message telling to check database problems
            Factory::getApplication()->enqueueMessage(Text::_('JLIB_DATABASE_ERROR_DATABASE_UPGRADE_FAILED'), 'error');
        }

        // If the conversion was successful try to drop the #__utf8_conversion table
        if ($converted === 5 && $this->dropUtf8ConversionTable()) {
            // Table successfully dropped
            return;
        }

        // Set flag in database if the conversion status has changed.
        if ($converted !== $convertedDB) {
            $db->setQuery('UPDATE ' . $db->quoteName('#__utf8_conversion')
                . ' SET ' . $db->quoteName('converted') . ' = ' . $converted . ';')->execute();
        }
    }

    /**
     * This method clean the Joomla Cache using the method `clean` from the com_cache model
     *
     * @return  void
     *
     * @since   3.5.1
     */
    private function cleanJoomlaCache()
    {
        /** @var \Joomla\Component\Cache\Administrator\Model\CacheModel $model */
        $model = Factory::getApplication()->bootComponent('com_cache')->getMVCFactory()
            ->createModel('Cache', 'Administrator', ['ignore_request' => true]);

        // Clean frontend cache
        $model->clean();

        // Clean admin cache
        $model->setState('client_id', 1);
        $model->clean();
    }

    /**
     * This method drops the #__utf8_conversion table
     *
     * @return  boolean  True on success
     *
     * @since   4.0.0
     */
    private function dropUtf8ConversionTable()
    {
        $db = Factory::getDbo();

        try {
            $db->setQuery('DROP TABLE ' . $db->quoteName('#__utf8_conversion') . ';')->execute();
        } catch (Exception $e) {
            return false;
        }

        return true;
    }

    /**
     * Called after any type of action
     *
     * @param   string     $action     Which action is happening (install|uninstall|discover_install|update)
     * @param   Installer  $installer  The class calling this method
     *
     * @return  boolean  True on success
     *
     * @since   4.0.0
     */
    public function postflight($action, $installer)
    {
        if ($action !== 'update') {
            return true;
        }

        if (empty($this->fromVersion) || version_compare($this->fromVersion, '4.0.0', 'ge')) {
            return true;
        }

        // Update UCM content types.
        $this->updateContentTypes();

        $db = Factory::getDbo();
        Table::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_menus/Table/');

        $tableItem   = new \Joomla\Component\Menus\Administrator\Table\MenuTable($db);

        $contactItems = $this->contactItems($tableItem);
        $finderItems  = $this->finderItems($tableItem);

        $menuItems = array_merge($contactItems, $finderItems);

        foreach ($menuItems as $menuItem) {
            // Check an existing record
            $keys = [
                'menutype'  => $menuItem['menutype'],
                'type'      => $menuItem['type'],
                'title'     => $menuItem['title'],
                'parent_id' => $menuItem['parent_id'],
                'client_id' => $menuItem['client_id'],
            ];

            if ($tableItem->load($keys)) {
                continue;
            }

            $newTableItem = new \Joomla\Component\Menus\Administrator\Table\MenuTable($db);

            // Bind the data.
            if (!$newTableItem->bind($menuItem)) {
                return false;
            }

            $newTableItem->setLocation($menuItem['parent_id'], 'last-child');

            // Check the data.
            if (!$newTableItem->check()) {
                return false;
            }

            // Store the data.
            if (!$newTableItem->store()) {
                return false;
            }

            // Rebuild the tree path.
            if (!$newTableItem->rebuildPath($newTableItem->id)) {
                return false;
            }
        }

        // Refresh versionable assets cache.
        Factory::getApplication()->flushAssets();

        return true;
    }

    /**
     * Prepare the contact menu items
     *
     * @return  array  Menu items
     *
     * @since   4.0.0
     */
    private function contactItems(Table $tableItem): array
    {
        // Check for the Contact parent Id Menu Item
        $keys = [
            'menutype'  => 'main',
            'type'      => 'component',
            'title'     => 'com_contact',
            'parent_id' => 1,
            'client_id' => 1,
        ];

        $contactMenuitem = $tableItem->load($keys);

        if (!$contactMenuitem) {
            return [];
        }

        $parentId    = $tableItem->id;
        $componentId = ExtensionHelper::getExtensionRecord('com_fields', 'component')->extension_id;

        // Add Contact Fields Menu Items.
        $menuItems = [
            [
                'menutype'          => 'main',
                'title'             => '-',
                'alias'             => microtime(true),
                'note'              => '',
                'path'              => '',
                'link'              => '#',
                'type'              => 'separator',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
            [
                'menutype'          => 'main',
                'title'             => 'mod_menu_fields',
                'alias'             => 'Contact Custom Fields',
                'note'              => '',
                'path'              => 'contact/Custom Fields',
                'link'              => 'index.php?option=com_fields&context=com_contact.contact',
                'type'              => 'component',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
            [
                'menutype'          => 'main',
                'title'             => 'mod_menu_fields_group',
                'alias'             => 'Contact Custom Fields Group',
                'note'              => '',
                'path'              => 'contact/Custom Fields Group',
                'link'              => 'index.php?option=com_fields&view=groups&context=com_contact.contact',
                'type'              => 'component',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
        ];

        return $menuItems;
    }

    /**
     * Prepare the finder menu items
     *
     * @return  array  Menu items
     *
     * @since   4.0.0
     */
    private function finderItems(Table $tableItem): array
    {
        // Check for the Finder parent Id Menu Item
        $keys = [
            'menutype'  => 'main',
            'type'      => 'component',
            'title'     => 'com_finder',
            'parent_id' => 1,
            'client_id' => 1,
        ];

        $finderMenuitem = $tableItem->load($keys);

        if (!$finderMenuitem) {
            return [];
        }

        $parentId    = $tableItem->id;
        $componentId = ExtensionHelper::getExtensionRecord('com_finder', 'component')->extension_id;

        // Add Finder Fields Menu Items.
        $menuItems = [
            [
                'menutype'          => 'main',
                'title'             => '-',
                'alias'             => microtime(true),
                'note'              => '',
                'path'              => '',
                'link'              => '#',
                'type'              => 'separator',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
            [
                'menutype'          => 'main',
                'title'             => 'com_finder_index',
                'alias'             => 'Smart-Search-Index',
                'note'              => '',
                'path'              => 'Smart Search/Index',
                'link'              => 'index.php?option=com_finder&view=index',
                'type'              => 'component',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
            [
                'menutype'          => 'main',
                'title'             => 'com_finder_maps',
                'alias'             => 'Smart-Search-Maps',
                'note'              => '',
                'path'              => 'Smart Search/Maps',
                'link'              => 'index.php?option=com_finder&view=maps',
                'type'              => 'component',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
            [
                'menutype'          => 'main',
                'title'             => 'com_finder_filters',
                'alias'             => 'Smart-Search-Filters',
                'note'              => '',
                'path'              => 'Smart Search/Filters',
                'link'              => 'index.php?option=com_finder&view=filters',
                'type'              => 'component',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
            [
                'menutype'          => 'main',
                'title'             => 'com_finder_searches',
                'alias'             => 'Smart-Search-Searches',
                'note'              => '',
                'path'              => 'Smart Search/Searches',
                'link'              => 'index.php?option=com_finder&view=searches',
                'type'              => 'component',
                'published'         => 1,
                'parent_id'         => $parentId,
                'level'             => 2,
                'component_id'      => $componentId,
                'checked_out'       => null,
                'checked_out_time'  => null,
                'browserNav'        => 0,
                'access'            => 0,
                'img'               => '',
                'template_style_id' => 0,
                'params'            => '{}',
                'home'              => 0,
                'language'          => '*',
                'client_id'         => 1,
                'publish_up'        => null,
                'publish_down'      => null,
            ],
        ];

        return $menuItems;
    }

    /**
     * Updates content type table classes.
     *
     * @return  void
     *
     * @since   4.0.0
     */
    private function updateContentTypes(): void
    {
        // Content types to update.
        $contentTypes = [
            'com_content.article',
            'com_contact.contact',
            'com_newsfeeds.newsfeed',
            'com_tags.tag',
            'com_banners.banner',
            'com_banners.client',
            'com_users.note',
            'com_content.category',
            'com_contact.category',
            'com_newsfeeds.category',
            'com_banners.category',
            'com_users.category',
            'com_users.user',
        ];

        // Get table definitions.
        $db    = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select(
                [
                    $db->quoteName('type_alias'),
                    $db->quoteName('table'),
                ]
            )
            ->from($db->quoteName('#__content_types'))
            ->whereIn($db->quoteName('type_alias'), $contentTypes, ParameterType::STRING);

        $db->setQuery($query);
        $contentTypes = $db->loadObjectList();

        // Prepare the update query.
        $query = $db->getQuery(true)
            ->update($db->quoteName('#__content_types'))
            ->set($db->quoteName('table') . ' = :table')
            ->where($db->quoteName('type_alias') . ' = :typeAlias')
            ->bind(':table', $table)
            ->bind(':typeAlias', $typeAlias);

        $db->setQuery($query);

        foreach ($contentTypes as $contentType) {
            list($component, $tableType) = explode('.', $contentType->type_alias);

            // Special case for core table classes.
            if ($contentType->type_alias === 'com_users.users' || $tableType === 'category') {
                $tablePrefix = 'Joomla\\CMS\Table\\';
                $tableType   = ucfirst($tableType);
            } else {
                $tablePrefix = 'Joomla\\Component\\' . ucfirst(substr($component, 4)) . '\\Administrator\\Table\\';
                $tableType   = ucfirst($tableType) . 'Table';
            }

            // Bind type alias.
            $typeAlias = $contentType->type_alias;

            $table = json_decode($contentType->table);

            // Update table definitions.
            $table->special->type   = $tableType;
            $table->special->prefix = $tablePrefix;

            // Some content types don't have this property.
            if (!empty($table->common->prefix)) {
                $table->common->prefix  = 'Joomla\\CMS\\Table\\';
            }

            $table = json_encode($table);

            // Execute the query.
            $db->execute();
        }
    }

    /**
     * Renames or removes incorrectly cased files.
     *
     * @return  void
     *
     * @since   3.9.25
     */
    protected function fixFilenameCasing()
    {
        $files = [
            // 3.10 changes
            '/libraries/src/Filesystem/Support/Stringcontroller.php' => '/libraries/src/Filesystem/Support/StringController.php',
            '/libraries/src/Form/Rule/SubFormRule.php'               => '/libraries/src/Form/Rule/SubformRule.php',
            // 4.0.0
            '/media/vendor/skipto/js/skipTo.js' => '/media/vendor/skipto/js/skipto.js',
        ];

        foreach ($files as $old => $expected) {
            $oldRealpath = realpath(JPATH_ROOT . $old);

            // On Unix without incorrectly cased file.
            if ($oldRealpath === false) {
                continue;
            }

            $oldBasename      = basename($oldRealpath);
            $newRealpath      = realpath(JPATH_ROOT . $expected);
            $newBasename      = basename($newRealpath);
            $expectedBasename = basename($expected);

            // On Windows or Unix with only the incorrectly cased file.
            if ($newBasename !== $expectedBasename) {
                // Rename the file.
                File::move(JPATH_ROOT . $old, JPATH_ROOT . $old . '.tmp');
                File::move(JPATH_ROOT . $old . '.tmp', JPATH_ROOT . $expected);

                continue;
            }

            // There might still be an incorrectly cased file on other OS than Windows.
            if ($oldBasename === basename($old)) {
                // Check if case-insensitive file system, eg on OSX.
                if (fileinode($oldRealpath) === fileinode($newRealpath)) {
                    // Check deeper because even realpath or glob might not return the actual case.
                    if (!in_array($expectedBasename, scandir(dirname($newRealpath)))) {
                        // Rename the file.
                        File::move(JPATH_ROOT . $old, JPATH_ROOT . $old . '.tmp');
                        File::move(JPATH_ROOT . $old . '.tmp', JPATH_ROOT . $expected);
                    }
                } else {
                    // On Unix with both files: Delete the incorrectly cased file.
                    File::delete(JPATH_ROOT . $old);
                }
            }
        }
    }

    /**
     * Move core template (s)css or js or image files which are left after deleting
     * obsolete core files to the right place in media folder.
     *
     * @return  void
     *
     * @since   4.1.0
     */
    protected function moveRemainingTemplateFiles()
    {
        $folders = [
            '/administrator/templates/atum/css'    => '/media/templates/administrator/atum/css',
            '/administrator/templates/atum/images' => '/media/templates/administrator/atum/images',
            '/administrator/templates/atum/js'     => '/media/templates/administrator/atum/js',
            '/administrator/templates/atum/scss'   => '/media/templates/administrator/atum/scss',
            '/templates/cassiopeia/css'            => '/media/templates/site/cassiopeia/css',
            '/templates/cassiopeia/images'         => '/media/templates/site/cassiopeia/images',
            '/templates/cassiopeia/js'             => '/media/templates/site/cassiopeia/js',
            '/templates/cassiopeia/scss'           => '/media/templates/site/cassiopeia/scss',
        ];

        foreach ($folders as $oldFolder => $newFolder) {
            if (Folder::exists(JPATH_ROOT . $oldFolder)) {
                $oldPath   = realpath(JPATH_ROOT . $oldFolder);
                $newPath   = realpath(JPATH_ROOT . $newFolder);
                $directory = new \RecursiveDirectoryIterator($oldPath);
                $directory->setFlags(RecursiveDirectoryIterator::SKIP_DOTS);
                $iterator  = new \RecursiveIteratorIterator($directory);

                // Handle all files in this folder and all sub-folders
                foreach ($iterator as $oldFile) {
                    if ($oldFile->isDir()) {
                        continue;
                    }

                    $newFile = $newPath . substr($oldFile, strlen($oldPath));

                    // Create target folder and parent folders if they don't exist yet
                    if (is_dir(dirname($newFile)) || @mkdir(dirname($newFile), 0755, true)) {
                        File::move($oldFile, $newFile);
                    }
                }
            }
        }
    }

    /**
     * Ensure the core templates are correctly moved to the new mode.
     *
     * @return  void
     *
     * @since   4.1.0
     */
    protected function fixTemplateMode(): void
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);

        foreach (['atum', 'cassiopeia'] as $template) {
            $clientId = $template === 'atum' ? 1 : 0;
            $query    = $db->getQuery(true)
                ->update($db->quoteName('#__template_styles'))
                ->set($db->quoteName('inheritable') . ' = 1')
                ->where($db->quoteName('template') . ' = ' . $db->quote($template))
                ->where($db->quoteName('client_id') . ' = ' . $clientId);

            try {
                $db->setQuery($query)->execute();
            } catch (Exception $e) {
                $this->collectError(__METHOD__, $e);
            }
        }
    }

    /**
     * Add the user Auth Provider Column as it could be present from 3.10 already
     *
     * @return  void
     *
     * @since   4.1.1
     */
    protected function addUserAuthProviderColumn(): void
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);

        // Check if the column already exists
        $fields = $db->getTableColumns('#__users');

        // Column exists, skip
        if (isset($fields['authProvider'])) {
            return;
        }

        $query = 'ALTER TABLE ' . $db->quoteName('#__users')
            . ' ADD COLUMN ' . $db->quoteName('authProvider') . ' varchar(100) DEFAULT ' . $db->quote('') . ' NOT NULL';

        // Add column
        try {
            $db->setQuery($query)->execute();
        } catch (Exception $e) {
            $this->collectError(__METHOD__, $e);

            return;
        }
    }
}
