<?php
 /**
 * Jamroom Sitemap Generator module
 *
 * copyright 2022 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrSitemap_meta()
{
    return array(
        'name'        => 'Sitemap Generator',
        'url'         => 'sitemap',
        'version'     => '1.2.0',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Create and maintain an XML Sitemap used by search engines',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/741/sitemap-generator',
        'license'     => 'mpl',
        'requires'    => 'jrCore:6.5.12',
        'category'    => 'tools'
    );
}

/**
 * init
 */
function jrSitemap_init()
{
    // After the core has parsed the URL, we can check for a sitemap call
    jrCore_register_event_listener('jrCore', 'parse_url', 'jrSitemap_parse_url_listener');

    // Tool to manually create sitemap
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrSitemap', 'create', array('Create Site Map', 'Create or Update the Sitemap'));

    // Maintain our Sitemap on a daily basis
    jrCore_register_event_listener('jrCore', 'daily_maintenance', 'jrSitemap_daily_maintenance_listener');

    // Our "map" event trigger
    jrCore_register_event_trigger('jrSitemap', 'sitemap_site_pages', 'Fired when gathering relative URLs for sitemap');

    // We don't need a session when getting a site map
    jrCore_register_module_feature('jrUser', 'skip_session', 'jrSitemap', 'default');

    return true;
}

//-----------------------------
// EVENT LISTENERS
//-----------------------------

/**
 * Generates an XML Sitemap
 * @param $_data array incoming data array from jrCore_save_media_file()
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrSitemap_parse_url_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_data['module_url']) && $_data['module_url'] === 'sitemap.xml') {

        $out = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $out .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        if (jrCore_checktype($_conf['jrSitemap_file_count'], 'number_nz')) {

            $tbl = jrCore_db_table_name('jrCore', 'setting');
            $req = "SELECT `updated` FROM {$tbl} WHERE `module` = 'jrSitemap' AND `name` = 'file_count'";
            $_rt = jrCore_db_query($req, 'SINGLE');

            $mod = (isset($_rt['updated'])) ? $_rt['updated'] : time();
            $mod = date('c', $mod);
            $mrl = jrCore_get_module_url('jrSitemap');
            $num = 1;

            $ttl = jrCore_get_config_value('jrSitemap', 'file_count', 1);
            if ($ttl > 0) {
                while ($num <= $ttl) {
                    $url = "{$_conf['jrCore_base_url']}/{$mrl}/sitemap{$num}.xml";
                    $out .= "<sitemap><loc>{$url}</loc><lastmod>{$mod}</lastmod></sitemap>\n";
                    $num++;
                }
            }

        }
        $out .= '</sitemapindex>';
        header("Content-Type: text/xml; charset=utf-8");
        echo $out;
        exit;
    }
    return $_data;
}

/**
 * Keep Sitemap up to date
 * @param array $_data incoming data array from jrCore_save_media_file()
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrSitemap_daily_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    jrSitemap_create_sitemap();
    return $_data;
}

//-----------------------------
// FUNCTIONS
//-----------------------------

/**
 * Create the site map
 * @param $modal bool set to TRUE to show modal updates
 * @return bool
 */
function jrSitemap_create_sitemap($modal = false)
{
    global $_mods, $_conf;
    jrCore_logger('INF', 'create XML site map - starting');
    if ($modal) {
        jrCore_form_modal_notice('update', 'Creating XML Site Map...');
    }

    // Cleanup old Site map XML files
    $old = jrCore_get_config_value('jrSitemap', 'file_count', 0);

    // SITE PAGES
    $updv   = jrCore_get_config_value('jrSitemap', 'site_freq', 'daily');
    $_map   = array();
    $_map[] = '/';

    // Include all active module indexes...
    foreach ($_mods as $mod => $_inf) {
        if (isset($_inf['module_prefix']) && strlen($_inf['module_prefix']) > 0 && is_file(APP_DIR . "/modules/{$mod}/templates/index.tpl")) {
            switch ($mod) {
                case 'jrUser':
                    break;
                default:
                    if (jrCore_module_is_active($mod)) {
                        $_map[] = "/{$_inf['module_url']}";
                    }
                    break;
            }
        }
    }

    // Let modules know we are looking for pages
    if ($modal) {
        jrCore_form_modal_notice('update', 'triggering modules for site map URLs');
    }
    $_map = jrCore_trigger_event('jrSitemap', 'sitemap_site_pages', $_map);
    jrCore_create_media_directory(0);

    // Create our first output
    $now = date('Y-m-d');
    $out = '<?xml version="1.0" encoding="UTF-8"?>' . "\n" . '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
    foreach ($_map as $url) {
        if (strpos($url, $_conf['jrCore_base_url']) !== 0) {
            $url = "{$_conf['jrCore_base_url']}{$url}";
        }
        $out .= "\n<url>\n<loc>{$url}</loc>\n<priority>1.0</priority>\n<changefreq>{$updv}</changefreq>\n<lastmod>{$now}</lastmod>\n</url>";
    }
    $out .= "\n</urlset>";
    jrCore_write_media_file(0, 'sitemap1.xml', $out, 'public-read');

    // PROFILES
    $updv = 'daily';
    if (isset($_conf['jrSitemap_profile_freq']) && strlen($_conf['jrSitemap_profile_freq']) > 0) {
        $updv = $_conf['jrSitemap_profile_freq'];
    }

    // Get privacy settings for our quotas
    $_qt = array();
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT `quota_id`, `name`, `value` FROM {$tbl} WHERE `module` = 'jrProfile' AND `name` IN('default_privacy', 'privacy_changes')";
    $_tm = jrCore_db_query($req, 'NUMERIC');
    if ($_tm && is_array($_tm)) {
        foreach ($_tm as $_q) {
            $qid = (int) $_q['quota_id'];
            if (!isset($_qt[$qid])) {
                $_qt[$qid] = array();
            }
            $_qt[$qid]["{$_q['name']}"] = $_q['value'];
        }
    }

    // Go through our profiles (1000 at a time)
    $total = 0;
    $mapid = 2;
    $start = 0;
    while (true) {
        $_src = array(
            'search'          => array(
                "_item_id > {$start}"
            ),
            'return_keys'     => array('_profile_id', 'profile_url', 'profile_active', 'profile_private', 'profile_quota_id'),
            'order_by'        => array(
                '_item_id' => 'numerical_asc'
            ),
            'skip_all_checks' => true,
            'limit'           => 1000
        );
        $_rt  = jrCore_db_search_items('jrProfile', $_src);
        if (!empty($_rt['_items'])) {

            $out = '<?xml version="1.0" encoding="UTF-8"?>' . "\n" . '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
            foreach ($_rt['_items'] as $_profile) {
                if ($_profile['_profile_id'] > 0) {
                    $start = $_profile['_profile_id'];
                    if ($_profile['profile_active'] == 1) {
                        // Check for PRIVATE profiles
                        // 0 = Private
                        // 1 = Global
                        // 2 = Shared
                        // 3 = Shared but Visible in Search
                        $prv = $_profile['profile_private'];
                        $qid = (int) $_profile['profile_quota_id'];
                        if (isset($_qt[$qid]) && isset($_qt[$qid]['privacy_changes']) && $_qt[$qid]['privacy_changes'] == 'off') {
                            $prv = (isset($_qt[$qid]['default_privacy'])) ? (int) $_qt[$qid]['default_privacy'] : '1';
                        }
                        if ($prv == 0 || $prv == 2) {
                            // profile is NOT global or searchable - do not add to Sitemap
                            continue;
                        }
                        $out .= "\n<url>\n<loc>{$_conf['jrCore_base_url']}/{$_profile['profile_url']}</loc>\n<priority>1.0</priority>\n<changefreq>{$updv}</changefreq>\n<lastmod>{$now}</lastmod>\n</url>";
                        $total++;
                    }
                }
            }
            $out .= "\n</urlset>";
            jrCore_write_media_file(0, "sitemap{$mapid}.xml", $out, 'public-read');

            $mapid++;
            if ($modal) {
                jrCore_form_modal_notice('update', "created XML Sitemap for " . jrCore_number_format($total) . " profiles");
            }
            if (count($_rt['_items']) < 1000) {
                // We're done
                break;
            }
        }
        else {
            break;
        }
    }

    // Cleanup files we no longer need
    $new = ($mapid - 1);
    if ($new < $old) {
        $i = $mapid;
        while ($i <= $old) {
            jrCore_delete_media_file(0, "sitemap{$i}.xml");
            $i++;
        }
    }

    jrCore_set_setting_value('jrSitemap', 'file_count', $new);
    jrCore_delete_config_cache();
    jrCore_logger('INF', 'create XML site map - completed');

    if ($modal) {
        jrCore_form_modal_notice('update', 'Successfully creating XML Site Map for ' . jrCore_number_format($total) . ' profiles');
    }

    return true;
}
