<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: settings
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrProfile_settings($_post, $_user, $_conf)
{
    jrUser_session_require_login();

    // make sure we get a good profile_id
    if (!empty($_post['id'])) {
        $_post['profile_id'] = (int) $_post['id'];
    }
    if (!isset($_post['profile_id']) || !jrCore_checktype($_post['profile_id'], 'number_nz') || !jrProfile_is_profile_owner($_post['profile_id'])) {
        $_post['profile_id'] = jrUser_get_profile_home_key('_profile_id');
    }

    // Get info
    $_profile = jrUser_get_requested_user_account_info();
    if (!$_profile || !is_array($_profile)) {
        jrCore_notice_page('error', 41);
    }
    jrUser_account_tabs('settings', $_profile);

    $_ln = jrUser_load_lang_strings();

    if ($_profile['_profile_id'] != jrUser_get_profile_home_key('_profile_id')) {
        $notice = "{$_ln['jrProfile'][35]} <b>{$_profile['profile_name']}</b>";
        jrCore_set_form_notice('notice', $notice, false);
    }
    if (!isset($_profile['profile_active']) || $_profile['profile_active'] != '1') {
        if (!isset($_post['hl'])) {
            global $_post;
            $_post['hl'] = 'profile_active';
        }
        jrCore_set_form_notice('error', $_ln['jrProfile'][36], false);
    }

    // If we have a Power User, we can create additional profiles
    $create = null;
    if (jrUser_is_admin() || jrUser_is_power_user()) {
        $max = jrUser_get_profile_home_key('quota_jrUser_power_user_max');
        if (jrUser_is_admin() || (intval($max) > count(explode(',', $_user['user_linked_profile_ids'])))) {
            $create .= jrCore_page_button('profile_create', $_ln['jrProfile'][7], "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/create')");
        }
    }
    $create .= jrCore_page_button('account-tabs-profile-button', "@{$_profile['profile_url']}", "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_profile['profile_url']}')");
    jrCore_page_banner(2, $create, false);

    // Form init
    $_tmp = array(
        'submit_value'     => $_ln['jrCore'][72],
        'cancel'           => 'referrer',
        'form_ajax_submit' => false,
        'values'           => $_profile
    );
    jrCore_form_create($_tmp);

    // If we modifying FROM the Profile Browser, we redirect there on save...
    $ref = jrCore_get_local_referrer();
    if (jrUser_is_admin() && strpos($ref, '/browser')) {
        $_tmp = array(
            'name'  => 'from_browser',
            'type'  => 'hidden',
            'value' => $ref
        );
        jrCore_form_field_create($_tmp);
    }

    if ((jrUser_is_admin() || jrUser_is_power_user() || jrUser_is_multi_user()) && isset($_post['profile_id']) && jrCore_checktype($_post['profile_id'], 'number_nz')) {
        $_tmp = array(
            'name'  => 'profile_id',
            'type'  => 'hidden',
            'value' => $_post['profile_id']
        );
        jrCore_form_field_create($_tmp);
    }

    // Profile Image
    $_img             = $_profile;
    $_img['_item_id'] = $_profile['_profile_id'];
    $_tmp             = array(
        'name'         => 'profile_image',
        'label'        => 6,
        'help'         => 23,
        'type'         => 'image',
        'size'         => 'medium',
        'max'          => (isset($_profile['quota_jrImage_max_image_size'])) ? intval($_profile['quota_jrImage_max_image_size']) : 2097152,
        'required'     => false,
        'image_delete' => true,
        'value'        => $_img
    );
    jrCore_form_field_create($_tmp);

    // Profile Name
    $_tmp = array(
        'name'     => 'profile_name',
        'label'    => 9,
        'help'     => 10,
        'type'     => 'text',
        'required' => true,
        'min'      => 1,
        'validate' => 'printable'
    );
    jrCore_form_field_create($_tmp);

    if (jrUser_is_admin() || (isset($_user['quota_jrProfile_url_changes']) && $_user['quota_jrProfile_url_changes'] == 'on')) {
        $_tmp = array(
            'name'          => 'profile_url',
            'label'         => 42,
            'sublabel'      => 43,
            'help'          => 44,
            'type'          => 'text',
            'required'      => false,
            'validate'      => 'url_name',
            'form_designer' => false
        );
        jrCore_form_field_create($_tmp);
    }

    // See if we can change our Profile Privacy
    if (jrUser_is_admin() || (isset($_user['quota_jrProfile_privacy_changes']) && $_user['quota_jrProfile_privacy_changes'] == 'on')) {
        $_opt = jrProfile_get_privacy_options();
        $priv = 1;
        if (isset($_profile['profile_private']) && jrCore_checktype($_profile['profile_private'], 'number_nn')) {
            $priv = (int) $_profile['profile_private'];
        }
        elseif (isset($_profile['quota_jrProfile_default_privacy']) && jrCore_checktype($_profile['quota_jrProfile_default_privacy'], 'number_nn')) {
            $priv = (int) $_profile['quota_jrProfile_default_privacy'];
        }
        // Profile Privacy
        $_tmp = array(
            'name'          => 'profile_private',
            'label'         => 11,
            'help'          => 12,
            'type'          => 'select',
            'options'       => $_opt,
            'value'         => $priv,
            'required'      => true,
            'min'           => 0,
            'max'           => 2,
            'validate'      => 'number_nn',
            'form_designer' => false // We do not allow the form designer to override this field
        );
        jrCore_form_field_create($_tmp);
    }

    // Bio
    $_tmp = array(
        'name'     => 'profile_bio',
        'label'    => 21,
        'help'     => 22,
        'type'     => 'editor',
        'validate' => 'allowed_html',
        'required' => false
    );
    jrCore_form_field_create($_tmp);

    // Power Users are limited to specific Quotas
    if (jrUser_is_power_user()) {

        // Power Users can change profile quota_id's of profiles that are NOT their home profile
        if (jrUser_get_profile_home_key('_profile_id') != $_profile['_profile_id']) {

            // Profile Quota (power users only)
            $_qot = jrProfile_get_settings_quotas();
            if ($_qot && is_array($_qot)) {
                if (count($_qot) > 1) {
                    $_tmp = array(
                        'name'          => 'profile_quota_id',
                        'label'         => 29,
                        'help'          => 30,
                        'type'          => 'select',
                        'options'       => $_qot,
                        'value'         => $_profile['profile_quota_id'],
                        'required'      => true,
                        'group'         => 'power',
                        'validate'      => 'number_nz',
                        'form_designer' => false // We do not allow the form designer to override this field
                    );
                }
                else {
                    $_tmp = array(
                        'name'          => 'profile_quota_id',
                        'type'          => 'hidden',
                        'value'         => $_profile['profile_quota_id'],
                        'form_designer' => false // We do not allow the form designer to override this field
                    );
                }
                jrCore_form_field_create($_tmp);
            }
        }
    }

    // If we allow multiple free signup quotas, let the user change quotas
    // But only if they are on a FREE quota - otherwise no change
    elseif (isset($_conf['jrProfile_change']) && $_conf['jrProfile_change'] == 'on' && isset($_user['quota_jrUser_allow_signups']) && $_user['quota_jrUser_allow_signups'] == 'on') {
        $_qot = jrProfile_get_signup_quotas();
        if ($_qot && count($_qot) > 1) {
            $_tmp = array(
                'name'          => 'profile_quota_id',
                'label'         => 29,
                'help'          => 30,
                'type'          => 'select',
                'options'       => $_qot,
                'value'         => $_profile['profile_quota_id'],
                'required'      => true,
                'validate'      => 'number_nz',
                'form_designer' => false // We do not allow the form designer to override this field
            );
            jrCore_form_field_create($_tmp);
        }
    }

    // Admin options
    if (jrUser_is_admin()) {

        $_tmp = array(
            'name'          => 'profile_active',
            'label'         => 'profile active',
            'help'          => 'If checked, this profile is active and will be viewable in the system to all users',
            'type'          => 'checkbox',
            'required'      => true,
            'validate'      => 'onoff',
            'value'         => (isset($_profile['profile_active']) && $_profile['profile_active'] == '1') ? 'on' : 'off',
            'form_designer' => false,
            'order'         => 200,
            'section'       => 'admin options'
        );
        jrCore_form_field_create($_tmp);

        $_tmp = array(
            'name'          => 'profile_quota_id',
            'label'         => 29,
            'help'          => 30,
            'type'          => 'select',
            'options'       => 'jrProfile_get_quotas',
            'value'         => $_profile['profile_quota_id'],
            'required'      => true,
            'group'         => 'power',
            'validate'      => 'number_nz',
            'form_designer' => false,
            'order'         => 201
        );
        jrCore_form_field_create($_tmp);
    }

    jrCore_page_display();
}
