<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * Profile Functions
 * @copyright 2003-2022 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Return an array of admin/master profile_id's
 */
function jrProfile_get_admin_profile_ids()
{
    $key = 'jrprofile_get_admin_profile_ids';
    if (!$_pr = jrCore_get_flag($key)) {
        $_pr = array();
        $_ui = jrUser_get_admin_user_ids();
        if ($_ui && is_array($_ui)) {
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            $req = "SELECT profile_id FROM {$tbl} WHERE user_id IN(" . implode(',', $_ui) . ')';
            $_pr = jrCore_db_query($req, 'profile_id', false, 'profile_id');
        }
        jrCore_set_flag($key, $_pr);
    }
    return (count($_pr) > 0) ? $_pr : false;
}

/**
 * Return an array of master profile_id's
 */
function jrProfile_get_master_profile_ids()
{
    $key = 'jrprofile_get_master_profile_ids';
    if (!$_pr = jrCore_get_flag($key)) {
        $_pr = array();
        $_ui = jrUser_get_master_user_ids();
        if ($_ui && is_array($_ui)) {
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            $req = "SELECT profile_id FROM {$tbl} WHERE user_id IN(" . implode(',', $_ui) . ')';
            $_pr = jrCore_db_query($req, 'profile_id', false, 'profile_id');
        }
        jrCore_set_flag($key, $_pr);
    }
    return (count($_pr) > 0) ? $_pr : false;
}

/**
 * Delete a profile from the system
 * @param $profile_id int Profile ID to delete
 * @param $user_check bool Set to FALSE to skip checking for attached users
 * @param $log_message bool Set to FALSE to skip logging delete message
 * @return bool
 */
function jrProfile_delete_profile($profile_id, $user_check = true, $log_message = true)
{
    global $_user, $_conf;
    if (!jrCore_checktype($profile_id, 'number_nz')) {
        return false;
    }
    $pid = (int) $profile_id;

    // Make sure we're good
    $_pr = jrCore_db_get_item('jrProfile', $pid, true, true);
    if (!$_pr || !is_array($_pr)) {
        // Profile does not exist
        return false;
    }

    // Send pre-delete event
    jrCore_trigger_event('jrProfile', 'delete_profile_init', $_pr);

    // Delete Profile
    jrCore_set_flag('jrProfile_delete_user_check', $user_check);
    jrCore_set_flag('jrProfile_delete_log_message', $log_message);

    jrCore_db_delete_item('jrProfile', $pid);
    // @note: after deleting here, the core will have removed the user_id => profile_id linking
    jrProfile_decrement_quota_profile_count($_pr['profile_quota_id']);

    // Remove user -> profile link and all files for this profile if our recycle bin is OFF
    if (isset($_conf['jrCore_recycle_bin']) && $_conf['jrCore_recycle_bin'] == 'off') {
        $dir = jrCore_get_media_directory($pid);
        jrCore_delete_dir_contents($dir, false);
        rmdir($dir);
    }

    // Send out our profile delete trigger so other modules and do extra cleanup if needed
    jrCore_trigger_event('jrProfile', 'delete_profile', $_pr);

    // Delete caches
    jrCore_delete_profile_cache_entries($pid);
    if (jrUser_is_logged_in()) {
        // so deleting user sees the change
        jrUser_reset_cache($_user['_user_id']);
    }

    if ($log_message) {
        jrCore_logger('INF', "profile {$_pr['profile_name']} (profile_id: {$profile_id}) has been deleted");
    }
    jrCore_delete_flag('jrProfile_delete_user_check');
    jrCore_delete_flag('jrProfile_delete_log_message');
    return true;
}

/**
 * Check profile media disk space and show error if over limit
 * @param $profile_id int Profile ID
 * @return bool
 */
function jrProfile_check_disk_usage($profile_id = null)
{
    global $_user;
    if (jrUser_is_admin()) {
        // Admins can always add new items
        return true;
    }
    if (is_null($profile_id)) {
        $profile_id = (int) $_user['user_active_profile_id'];
    }
    if (isset($_user['quota_jrCore_disk']) && $_user['quota_jrCore_disk'] > 0) {
        $tmp = jrCore_get_flag('jrprofile_check_disk_usage');
        if (!$tmp) {
            $tmp = jrProfile_get_disk_usage($profile_id);
            if ($tmp) {
                jrCore_set_flag('jrprofile_check_disk_usage', $tmp);
            }
        }
        if ($tmp && $tmp > ($_user['quota_jrCore_disk'] * 1048576)) {
            $_ln = jrUser_load_lang_strings();
            jrCore_notice_page('error', $_ln['jrProfile'][27], 'referrer');
        }
    }
    return true;
}

/**
 * Get profile media directory disk usage in bytes
 * @param $profile_id int Profile ID
 * @return bool|int
 */
function jrProfile_get_disk_usage($profile_id)
{
    $sum = 0;
    $pid = (int) $profile_id;
    $_fl = jrCore_get_media_files($pid);
    if (is_array($_fl)) {
        foreach ($_fl as $_file) {
            if (!strpos($_file['name'], '/rb_')) {
                $sum += $_file['size'];
            }
        }
    }
    return $sum;
}

/**
 * Custom Data Store browser tool
 * @param $_post array Global $_post
 * @param $_user array Viewing user array
 * @param $_conf array Global config
 * @return bool
 */
function jrProfile_data_browser($_post, $_user, $_conf)
{
    $order_dir = 'desc';
    $order_opp = 'asc';
    if (isset($_post['order_dir']) && ($_post['order_dir'] == 'asc' || $_post['order_dir'] == 'numerical_asc')) {
        $order_dir = 'asc';
        $order_opp = 'desc';
    }

    $order_by = '_item_id';
    if (isset($_post['order_by'])) {
        switch ($_post['order_by']) {
            case '_item_id';
            case '_created';
            case 'profile_disk_usage';
                $order_dir = 'numerical_' . $order_dir;
                $order_opp = 'numerical_' . $order_opp;
                $order_by  = $_post['order_by'];
                break;
            case 'profile_name':
                $order_by = $_post['order_by'];
                break;
        }
    }

    // get our items
    $_pr = array(
        'search'         => array(
            'profile_quota_id > 0'
        ),
        'pagebreak'      => jrCore_get_pager_rows(),
        'page'           => 1,
        'order_by'       => array(
            $order_by => $order_dir
        ),
        'ignore_pending' => true,
        'no_cache'       => true,
        'privacy_check'  => false
    );
    if (isset($_post['p']) && jrCore_checktype($_post['p'], 'number_nz')) {
        $_pr['page'] = (int) $_post['p'];
    }
    // See we have a search condition
    $_ex = false;
    if (isset($_post['search_string']) && strlen($_post['search_string']) > 0) {
        $_ex = array('search_string' => $_post['search_string']);
        if (strpos($_post['search_string'], ':')) {
            // Searching for a value on a specific key
            $_pr['search'] = array(jrCore_get_ds_search_condition($_post['search_string']));
        }
        elseif (strpos($_post['search_string'], '@')) {
            // Searching for a specific EMAIL Address ties to a profile
            $_pr['search'] = array("user_email = {$_post['search_string']}");
            $_pr['limit']  = 1;
            unset($_pr['order_by'], $_pr['page'], $_pr['pagebreak']);
        }
        elseif (jrCore_module_is_active('jrSearch') && !jrSearch_is_disabled_module('jrProfile')) {
            $_pr['search'] = array("profile_name full_text {$_post['search_string']}");
            unset($_pr['order_by']);
        }
        else {
            $_pr['search'] = array(jrCore_get_ds_search_condition($_post['search_string'], array('profile_name', 'profile_url')));
        }
    }
    $_us = jrCore_db_search_items('jrProfile', $_pr);

    // Start our output
    $url             = jrCore_get_base_url() . jrCore_strip_url_params($_post['_uri'], array('order_by', 'order_dir'));
    $dat             = array();
    $dat[1]['title'] = 'img';
    $dat[1]['width'] = '2%';
    $dat[2]['title'] = 'ID';
    $dat[2]['width'] = '2%';
    $dat[3]['title'] = '<a href="' . $url . '/order_by=profile_name/order_dir=' . $order_opp . '">profile name</a>';
    $dat[3]['width'] = '28%';
    $dat[4]['title'] = 'user names';
    $dat[4]['width'] = '20%';
    $dat[5]['title'] = 'user emails';
    $dat[5]['width'] = '20%';
    $dat[6]['title'] = '<a href="' . $url . '/order_by=profile_disk_usage/order_dir=' . $order_opp . '">disk space</a>';
    $dat[6]['width'] = '9%';
    $dat[7]['title'] = '<a href="' . $url . '/order_by=_created/order_dir=' . $order_opp . '">created</a>';
    $dat[7]['width'] = '9%';
    $dat[8]['title'] = 'modify';
    $dat[8]['width'] = '5%';
    $dat[9]['title'] = 'delete';
    $dat[9]['width'] = '5%';
    jrCore_page_table_header($dat);

    if (isset($_us['_items']) && is_array($_us['_items'])) {

        // Get user info for these profiles
        $_pi = array();
        foreach ($_us['_items'] as $_usr) {
            $_pi[] = (int) $_usr['_profile_id'];
        }

        $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
        $req = "SELECT * FROM {$tbl} WHERE profile_id IN(" . implode(',', $_pi) . ")";
        $_ui = jrCore_db_query($req, 'NUMERIC');
        if ($_ui && is_array($_ui)) {

            $_id = array();
            foreach ($_ui as $v) {
                $_id["{$v['user_id']}"] = $v['user_id'];
            }

            // get users linked to profiles
            $_pr = array(
                'search'         => array(
                    '_user_id in ' . implode(',', $_id)
                ),
                'return_keys'    => array('_profile_id', '_user_id', 'user_name', 'user_email', 'user_group', 'user_image_time', 'user_active', 'user_last_login', 'user_blocked'),
                'skip_triggers'  => true,
                'ignore_pending' => true,
                'privacy_check'  => false,
                'no_cache'       => true,
                'limit'          => 250
            );
            $_pi = jrCore_db_search_items('jrUser', $_pr);
            if (isset($_pi['_items']) && is_array($_pi['_items'])) {
                $_ud = array();
                $_pm = array();
                foreach ($_pi['_items'] as $_usr) {
                    $_ud["{$_usr['_user_id']}"] = $_usr;
                    if (!isset($_pm["{$_usr['_profile_id']}"])) {
                        $_pm["{$_usr['_profile_id']}"] = array();
                    }
                    $_pm["{$_usr['_profile_id']}"][] = $_usr;
                }
                unset($_pi);
                $_pr = array();
                $_em = array();
                $url = jrCore_get_module_url('jrUser');
                foreach ($_ui as $v) {
                    $uid = (int) $v['user_id'];
                    if (!isset($_pr["{$v['profile_id']}"])) {
                        $_pr["{$v['profile_id']}"] = array();
                        $_em["{$v['profile_id']}"] = array();
                    }
                    if (isset($_ud[$uid])) {
                        $_pr["{$v['profile_id']}"][] = "<a href=\"{$_conf['jrCore_base_url']}/{$url}/account/user_id={$uid}\">{$_ud[$uid]['user_name']}</a>";
                        $_em["{$v['profile_id']}"][] = $_ud[$uid]['user_email'];
                    }
                }
            }
            unset($_pi);
        }

        $purl            = jrCore_get_module_url('jrProfile');
        $home_profile_id = jrUser_get_profile_home_key('_profile_id');
        foreach ($_us['_items'] as $_prf) {
            $dat             = array();
            $_im             = array(
                'crop'   => 'square',
                'width'  => 36,
                'height' => 36,
                'alt'    => 'profile image',
                'title'  => $_prf['profile_name'],
                'ext'    => (!empty($_prf['profile_image_extension'])) ? $_prf['profile_image_extension'] : '',
                '_v'     => (!empty($_prf['profile_image_time'])) ? $_prf['profile_image_time'] : false
            );
            $dat[1]['title'] = jrImage_get_image_src('jrProfile', 'profile_image', $_prf['_profile_id'], 'small', $_im);
            $dat[2]['title'] = $_prf['_profile_id'];
            $dat[2]['class'] = 'center';
            $profile_url     = $_prf['profile_url'];
            if (strpos(' ' . $profile_url, '%')) {
                $profile_url = rawurldecode($_prf['profile_url']);
            }
            $dat[3]['title'] = "<a href=\"{$_conf['jrCore_base_url']}/{$_prf['profile_url']}\"><h3>{$_prf['profile_name']}</h3></a><br><small><a href=\"{$_conf['jrCore_base_url']}/{$_prf['profile_url']}\">@{$profile_url}</small></a>";
            $dat[3]['class'] = 'word-break';
            $dat[4]['title'] = (isset($_pr["{$_prf['_profile_id']}"])) ? implode('<br>', $_pr["{$_prf['_profile_id']}"]) : '-';
            $dat[4]['class'] = 'center word-break';
            $dat[5]['title'] = (isset($_em["{$_prf['_profile_id']}"])) ? implode('<br>', $_em["{$_prf['_profile_id']}"]) : '-';
            $dat[5]['class'] = 'center word-break';
            $cls             = '';
            if (!isset($_prf['profile_active']) || $_prf['profile_active'] != '1') {
                // Is this profile pending?
                $cls = ' error';
                $txt = 'INACTIVE';
                if (isset($_pm["{$_prf['_profile_id']}"]) && is_array($_pm["{$_prf['_profile_id']}"])) {
                    foreach ($_pm["{$_prf['_profile_id']}"] as $_uinf) {
                        if (isset($_uinf['user_blocked']) && $_uinf['user_blocked'] == '1') {
                            $txt = 'BLOCKED';
                            break;
                        }
                    }
                    if ($txt != 'BLOCKED') {
                        foreach ($_pm["{$_prf['_profile_id']}"] as $_uinf) {
                            if (isset($_uinf['user_last_login']) && $_uinf['user_last_login'] > 0) {
                                break;
                            }
                            $cls = ' notice';
                            $txt = 'PENDING';
                        }
                    }
                }
                $dat[7]['title'] = $txt;
            }
            else {
                $dat[7]['title'] = jrCore_format_time($_prf['_created']);
            }
            $ttl = '?';
            if (isset($_prf['profile_disk_usage'])) {
                $ttl = jrCore_format_size($_prf['profile_disk_usage']);
            }
            $dat[6]['title'] = "<a onclick=\"popwin('{$_conf['jrCore_base_url']}/{$purl}/disk_usage_report/{$_prf['_profile_id']}','disk_usage_{$_prf['_profile_id']}',800,600,'yes');\"><u>" . $ttl . '</u></a>';
            $dat[6]['class'] = 'center';
            $dat[7]['class'] = 'center' . $cls;
            $dat[8]['title'] = jrCore_page_button("profile-modify-{$_prf['_profile_id']}", 'modify', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/settings/profile_id={$_prf['_profile_id']}')");
            // If this profile belongs to a Master User, it can only be deleted by a Master Admin
            $master = false;
            if (isset($_pm["{$_prf['_profile_id']}"]) && is_array($_pm["{$_prf['_profile_id']}"])) {
                foreach ($_pm["{$_prf['_profile_id']}"] as $_uinf) {
                    if (isset($_uinf['user_group']) && $_uinf['user_group'] == 'master') {
                        $master = true;
                        break;
                    }
                }
            }
            $dat[8]['class'] = 'center' . $cls;
            if (($master && !jrUser_is_master()) || $_prf['_profile_id'] == $home_profile_id) {
                $dat[9]['title'] = jrCore_page_button("profile-delete-{$_prf['_profile_id']}", 'delete', 'disabled');
            }
            else {
                $dat[9]['title'] = jrCore_page_button("profile-delete-{$_prf['_profile_id']}", 'delete', "jrCore_confirm('', 'Are you sure you want to delete this profile? Any User Accounts associated with ONLY this profile will also be removed.', function() { jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/delete_save/id={$_prf['_profile_id']}/modal=1') } )");
            }
            $dat[9]['class'] = 'center' . $cls;
            $dat             = jrCore_trigger_event('jrProfile', 'profile_browser_row', $dat, $_prf);
            jrCore_page_table_row($dat);
        }
        jrCore_page_table_pager($_us, $_ex);
    }
    else {
        $dat = array();
        if (isset($_post['search_string'])) {
            $dat[1]['title'] = '<p>No Results found for your Search Criteria.</p>';
        }
        else {
            $dat[1]['title'] = '<p>No Profiles found!</p>';
        }
        $dat[1]['class'] = 'center';
        jrCore_page_table_row($dat);
    }
    jrCore_page_table_footer();
    return true;
}

/**
 * Get available profile privacy choices
 * @return array
 */
function jrProfile_get_privacy_options()
{
    $_lng = jrUser_load_lang_strings();
    $_opt = array(
        0 => $_lng['jrProfile'][14], // Private
        1 => $_lng['jrProfile'][13]  // Global
    );
    if (jrCore_module_is_active('jrFollower')) {
        $_opt[2] = $_lng['jrProfile'][15]; // Followers Only
        $_opt[3] = $_lng['jrProfile'][38]; // Followers Only + Search
    }
    return $_opt;
}

/**
 * Check if a viewing user has access to a profile based on the profile_privacy setting
 * @param $module string Module check is being run for
 * @param $profile_id integer Profile_ID
 * @param $privacy_setting integer Profile_Privacy (one of 0|1|2)
 * @return bool
 */
function jrProfile_privacy_check($module, $profile_id, $privacy_setting)
{
    global $_user;
    // Admins see everything
    if (jrUser_is_admin()) {
        return true;
    }
    $priv = (int) $privacy_setting;
    switch ($priv) {

        case 0:
            // Private Access
            if (jrUser_is_logged_in() && jrProfile_is_profile_owner($profile_id)) {
                return true;
            }
            break;

        case 1:
            // Global Access
            return true;

        case 2:
        case 3:
            // Shared Access
            if (jrUser_is_logged_in()) {
                // If this is a PROFILE IMAGE, and we are Shared + Search - allowed
                if ($priv === 3 && $module == 'jrProfile') {
                    return true;
                }
                // We can always view our own profile
                if ($profile_id == $_user['user_active_profile_id'] || $profile_id == jrUser_get_profile_home_key('_profile')) {
                    return true;
                }
                // If we are a Power User or Multi Profile user, we always
                // can view the profiles we have access to.
                if (jrUser_is_power_user() || jrUser_is_multi_user()) {
                    if (jrProfile_is_profile_owner($profile_id)) {
                        return true;
                    }
                }
                // We're shared - viewer must be a follower of the profile
                if (jrCore_module_is_active('jrFollower')) {
                    if ($_fw = jrFollower_is_follower($_user['_user_id'], $profile_id)) {
                        if (isset($_fw['follow_active']) && $_fw['follow_active'] == '1') {
                            return true;
                        }
                    }
                }
            }
            break;
    }
    return false;
}

/**
 * Return TRUE if user is linked to more than 1 profile
 */
function jrProfile_more_than_one_linked_profile()
{
    global $_user;
    // [user_linked_profile_ids] => 4,18
    if (isset($_user['user_linked_profile_ids']) && strpos($_user['user_linked_profile_ids'], ',')) {
        return true;
    }
    return false;
}

/**
 * Return an array of profile owner emails
 * @param $pid integer Profile ID
 * @return array|false
 */
function jrProfile_get_owner_email($pid)
{
    $pid = (int) $pid;
    $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
    $req = "SELECT user_id FROM {$tbl} WHERE profile_id = '{$pid}'";
    $_rt = jrCore_db_query($req, 'user_id');
    if (!$_rt || !is_array($_rt)) {
        return false;
    }
    $_sc = array(
        'search'         => array(
            '_item_id in ' . implode(',', array_keys($_rt))
        ),
        'return_keys'    => array('_user_id', 'user_email'),
        'skip_triggers'  => true,
        'privacy_check'  => false,
        'ignore_pending' => true,
        'limit'          => count($_rt)
    );
    $_rt = jrCore_db_search_items('jrUser', $_sc);
    if ($_rt && is_array($_rt) && is_array($_rt['_items'])) {
        $_ot = array();
        foreach ($_rt['_items'] as $v) {
            $_ot["{$v['_user_id']}"] = $v['user_email'];
        }
        return $_ot;
    }
    return false;
}

/**
 * Return an array of profile owner info
 * @param $pid integer Profile ID
 * @return array|false
 */
function jrProfile_get_owner_info($pid)
{
    $pid = (int) $pid;
    $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
    $req = "SELECT user_id FROM {$tbl} WHERE profile_id = '{$pid}'";
    $_rt = jrCore_db_query($req, 'user_id');
    if (!$_rt || !is_array($_rt)) {
        return false;
    }
    $_us = jrCore_db_get_multiple_items('jrUser', array_keys($_rt));
    if ($_us && is_array($_us)) {
        foreach ($_us as $k => $v) {
            unset($_us[$k]['user_password']);
            unset($_us[$k]['user_old_password']);
        }
        return $_us;
    }
    return false;
}

/**
 * Register a setting to be shown in Profile Settings
 * @param $module string Module registering setting for
 * @param $_field array Array of setting information
 * @return bool
 * @deprecated Do not use
 */
function jrProfile_register_setting($module, $_field)
{
    if (!isset($_field['name'])) {
        jrCore_set_form_notice('error', "You must provide a valid field name");
        return false;
    }
    if ($_field['name'] == "item_count") {
        jrCore_set_form_notice('error', "Invalid profile_setting name - item_count is reserved for internal use");
        return false;
    }
    $_tmp = jrCore_get_flag('jrprofile_register_setting');
    if (!$_tmp) {
        $_tmp = array();
    }
    if (!isset($_tmp[$module])) {
        $_tmp[$module] = array();
    }
    $_field['name']  = "profile_{$module}_{$_field['name']}";
    $_tmp[$module][] = $_field;
    jrCore_set_flag('jrprofile_register_setting', $_tmp);
    return true;
}

/**
 * Returns true if viewing user can modify the given profile id
 * @param $id integer Profile ID to check
 * @return bool
 */
function jrProfile_is_profile_owner($id)
{
    // validate id
    if (!$id || !jrCore_checktype($id, 'number_nz')) {
        return false;
    }
    if (jrUser_is_admin()) {
        // Admins can always manage a profile
        return true;
    }
    if (isset($_SESSION['user_linked_profile_ids']) && in_array($id, explode(',', $_SESSION['user_linked_profile_ids']))) {
        // The viewing user can manage this profile
        return true;
    }
    return false;
}

/**
 * Create a new link between a User and a Profile
 * @param $profile_id integer Profile ID to link to
 * @param $user_id integer User ID to link to
 * @return bool
 */
function jrProfile_create_user_link($user_id, $profile_id)
{
    $uid = (int) $user_id;
    $pid = (int) $profile_id;
    $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
    $req = "INSERT INTO {$tbl} (user_id,profile_id) VALUES ('{$uid}','{$pid}') ON DUPLICATE KEY UPDATE profile_id = '{$pid}'";
    jrCore_db_query($req);
    return true;
}

/**
 * Temporarily add a profile_id to a user's linked profile_ids
 * @param int $user_id User ID
 * @param int $profile_id Profile ID
 * @return bool
 */
function jrProfile_add_session_linked_profile_id($user_id, $profile_id)
{
    $uid = (int) $user_id;
    $pid = (int) $profile_id;
    $_tm = false;
    if (isset($_SESSION['user_temp_linked_profile_ids'])) {
        $_tm = explode(',', $_SESSION['user_temp_linked_profile_ids']);
    }
    if (!$_tm || !is_array($_tm)) {
        $_SESSION['user_temp_linked_profile_ids'] = array();
    }
    $_SESSION['user_temp_linked_profile_ids'][$pid] = $uid;
    if ($_rt = jrCore_get_flag("jrprofile_linked_profiles_{$uid}")) {
        $_rt[$pid] = $uid;
        jrCore_set_flag("jrprofile_linked_profiles_{$uid}", $_rt);
    }
    return true;
}

/**
 * Delete a temporarily linked profile_id from a user_id
 * @param int $user_id
 * @param int $profile_id
 * @return bool
 */
function jrProfile_delete_session_linked_profile_id($user_id, $profile_id)
{
    $pid = (int) $profile_id;
    if (isset($_SESSION['user_temp_linked_profile_ids'][$pid])) {
        $uid = (int) $user_id;
        unset($_SESSION['user_temp_linked_profile_ids'][$pid]);
        if ($_rt = jrCore_get_flag("jrprofile_linked_profiles_{$uid}")) {
            unset($_rt[$pid]);
            jrCore_set_flag("jrprofile_linked_profiles_{$uid}", $_rt);
        }
    }
    return true;
}

/**
 * Get array of profile_id's that user_id is linked to
 * @param $user_id integer User_ID to get profile id's for
 * @return bool|mixed  Returns an array of profile_id => user_id links
 */
function jrProfile_get_user_linked_profiles($user_id)
{
    if (jrCore_checktype($user_id, 'number_nz')) {
        $uid = (int) $user_id;
        if (!$_rt = jrCore_get_flag("jrprofile_linked_profiles_{$uid}")) {
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            $req = "SELECT * FROM {$tbl} WHERE user_id = '{$uid}'";
            $_rt = jrCore_db_query($req, 'profile_id', false, 'user_id');
            if (!$_rt || !is_array($_rt)) {
                $_rt = array();
            }
            if (jrUser_is_logged_in() && $_SESSION['_user_id'] == $user_id) {
                if (isset($_SESSION['user_temp_linked_profile_ids'])) {
                    $_rt = $_rt + $_SESSION['user_temp_linked_profile_ids'];
                }
            }
            $_rt = jrCore_trigger_event('jrProfile', 'user_linked_profile_ids', $_rt);
            jrCore_set_flag("jrprofile_linked_profiles_{$uid}", $_rt);
        }
        return $_rt;
    }
    return false;
}

/**
 * Change a user's active profile keys to a new profile
 * @param $_profile array New Profile Info
 * @return array
 */
function jrProfile_change_to_profile($_profile)
{
    global $_user;
    if (is_array($_profile)) {
        if (!isset($_user['user_active_profile_id']) || $_user['user_active_profile_id'] != $_profile['_profile_id']) {
            // set all profile info to this profile
            // We need to go through and REMOVE all existing profile entries
            foreach ($_SESSION as $k => $v) {
                if ((strpos($k, '_') === 0 || strpos($k, 'profile_') === 0) && strpos($k, 'profile_home_') !== 0 && !isset($_profile[$k])) {
                    unset($_SESSION[$k]);
                }
            }
            // Merge in new profile data in
            foreach ($_profile as $k => $v) {
                if ((strpos($k, 'profile_') === 0 || strpos($k, 'quota_') === 0) && strpos($k, 'profile_home_') !== 0) {
                    $_SESSION[$k] = $v;
                }
            }
            $_SESSION['_user_id']               = (int) $_user['_user_id'];
            $_SESSION['_profile_id']            = (int) $_profile['_profile_id'];
            $_SESSION['user_active_profile_id'] = (int) $_profile['_profile_id'];
            $_SESSION                           = jrCore_trigger_event('jrProfile', 'change_active_profile', $_SESSION);
            return $_SESSION;
        }
    }
    return $_user;
}

/**
 * Set the active profile tab from a profile view
 * @param $active
 * @return bool
 */
function jrProfile_set_active_profile_tab($active)
{
    return jrCore_set_flag('jrprofile_active_profile_tab', $active);
}

/**
 * Display registered profile tabs
 * @param $profile_id int Profile ID
 * @param $active string Active tab
 * @param $module string Module Name
 * @param $profile_url string Profile URL
 * @param $_tabs array Array of tabs as profile_view => profile_title
 * @return bool
 */
function jrProfile_show_profile_tabs($profile_id, $active, $module, $profile_url, $_tabs)
{
    global $_conf;
    // Core Module URL
    $url   = jrCore_get_module_url($module);
    $_lang = jrUser_load_lang_strings();
    $_temp = array(
        'tabs' => array()
    );

    $show = false;
    foreach ($_tabs as $view => $title) {

        if (is_array($title)) {
            // This tab is defining viewer group
            if (is_numeric($title['label']) && isset($_lang[$module]["{$title['label']}"])) {
                $label = $_lang[$module]["{$title['label']}"];
            }
            else {
                $label = $title['label'];
            }
            if (isset($title['group'])) {
                switch ($title['group']) {
                    case 'master':
                        if (!jrUser_is_master()) {
                            continue 2;
                        }
                        break;
                    case 'admin':
                        if (!jrUser_is_admin()) {
                            continue 2;
                        }
                        break;
                    case 'owner':
                        if (!jrProfile_is_profile_owner($profile_id)) {
                            continue 2;
                        }
                        break;
                    case 'logged_in':
                        if (!jrUser_is_logged_in()) {
                            continue 2;
                        }
                        break;
                }
            }
            if ($view == 'default') {
                $_temp['tabs'][$view] = array(
                    'label' => $label,
                    'url'   => "{$_conf['jrCore_base_url']}/{$profile_url}/{$url}"
                );
            }
            else {
                $_temp['tabs'][$view] = array(
                    'label' => $label,
                    'url'   => "{$_conf['jrCore_base_url']}/{$profile_url}/{$url}/{$view}"
                );
            }
            $_temp['tabs'][$view]['id']    = "t{$view}";
            $_temp['tabs'][$view]['class'] = 'page_tab';
            $show                          = true;
        }
        else {
            if (!jrProfile_is_profile_owner($profile_id)) {
                continue;
            }
            if (is_numeric($title) && isset($_lang[$module][$title])) {
                $title = $_lang[$module][$title];
            }
            if ($view == 'default') {
                $_temp['tabs'][$view] = array(
                    'label' => $title,
                    'url'   => "{$_conf['jrCore_base_url']}/{$profile_url}/{$url}"
                );
            }
            else {
                $_temp['tabs'][$view] = array(
                    'label' => $title,
                    'url'   => "{$_conf['jrCore_base_url']}/{$profile_url}/{$url}/{$view}"
                );
            }
            $_temp['tabs'][$view]['id']    = "t{$view}";
            $_temp['tabs'][$view]['class'] = 'page_tab';
            $show                          = true;
        }
    }
    if (!$show) {
        return false;
    }
    if ($tmp = jrCore_get_flag('jrprofile_active_profile_tab')) {
        if (isset($_temp['tabs'][$tmp])) {
            $_temp['tabs'][$tmp]['active'] = true;
        }
    }
    else {
        if (isset($_temp['tabs'][$active])) {
            $_temp['tabs'][$active]['active'] = true;
        }
    }

    // Let other modules see what we're doing
    $_args = array('module' => $module);
    $_temp = jrCore_trigger_event('jrProfile', 'item_module_tabs', $_temp, $_args);
    return jrCore_parse_template('profile_tabs.tpl', $_temp, 'jrProfile');
}

/**
 * Returns true of current view is a profile view
 * @return mixed
 */
function jrProfile_is_profile_view()
{
    return jrCore_get_flag('jrprofile_view_is_active');
}

/**
 * Displays a profile for a given profile name
 * @param $_post array global $_post (from jrCore_parse_url())
 * @param $_user array Viewing user info
 * @param $_conf array Global Config
 * @return string returns rendered profile page
 */
function jrProfile_show_profile($_post, $_user, $_conf)
{
    global $_urls;

    // Set flag indicating we are in a profile view
    jrCore_set_flag('jrprofile_view_is_active', true);

    // Make sure this is a good module...
    if (!empty($_post['option']) && !isset($_urls["{$_post['option']}"])) {
        jrCore_page_not_found();
    }
    if (!empty($_post['option']) && !jrCore_module_is_active($_urls["{$_post['option']}"])) {
        // Module is not active - not found
        jrCore_page_not_found();
    }

    // Inside our profile call we can handle the following routing:
    // profile_name                     (Skin/profile_index.tpl)
    // profile_name/module_url          (Module/item_index.tpl  => Skin/profile_item_index.tpl)
    // profile_name/module_url/###/...  (Module/item_detail.tpl => Skin/profile_item_detail.tpl)
    // profile_name/module_url/???/...  (Module/item_list.tpl   => Skin/profile_item_list.tpl)
    //
    // <module_url>/<option>/<_1>/...

    // Get profile info for this profile
    if (!isset($_post['module_url']) || mb_strlen($_post['module_url']) === 0) {
        jrCore_page_not_found();
    }
    if (strpos(' ' . $_post['module_url'], 'ltscriptgt')) {
        jrCore_page_not_found();
    }
    if (strpos($_post['module_url'], '_')) {
        $_post['module_url'] = str_replace('_', '-', $_post['module_url']); // JR4 -> JR5 URL change for profiles
    }

    $_rt = jrCore_db_get_item_by_key('jrProfile', 'profile_url', $_post['module_url']);
    if (!$_rt || !is_array($_rt)) {

        // If a profile changes their profile name, pages that were part of their profile before
        // will now get a page not found - check here for this and do a 301 if needed
        if (isset($_post['_1']) && jrCore_checktype($_post['_1'], 'number_nz')) {
            $mod = $_urls["{$_post['option']}"];
            if (!jrCore_module_is_active($mod) || !jrCore_is_datastore_module($mod)) {
                jrCore_page_not_found();
            }
            $_it = jrCore_db_get_item($mod, $_post['_1'], false, true);
            if ($_it && is_array($_it) && isset($_it['profile_url']) && $_it['profile_url'] != $_post['module_url']) {
                $_tm    = explode('/', trim($_post['_uri'], '/'));
                $_tm[0] = $_it['profile_url'];
                header('HTTP/1.1 301 Moved Permanently');
                jrCore_location("{$_conf['jrCore_base_url']}/" . implode('/', $_tm));
            }
        }
        // Fall through - not found
        jrCore_page_not_found();
    }

    // Set flag indicating we are in a profile view
    $_rt = jrCore_trigger_event('jrProfile', 'profile_data', $_rt);
    jrCore_set_flag('jrprofile_view_is_active', $_rt['_profile_id']);
    jrCore_set_flag('jrprofile_active_profile_data', $_rt);

    $_lang = jrUser_load_lang_strings();

    // We global here so it changes at the global level
    global $_post;
    $_post['_profile_id'] = $_rt['_profile_id'];

    // Save our current URL for form cancel purposes
    jrUser_save_url_location();

    // Set our viewing profile for the user
    if (jrProfile_is_profile_owner($_rt['_profile_id'])) {

        // Switch active profile info
        global $_user;
        $_user = jrProfile_change_to_profile($_rt);
    }

    // Check for Profile Privacy - privacy options are bypassed for admin users
    elseif (!jrUser_is_admin()) {

        // See if this profile is even active...
        if (isset($_rt['profile_active']) && $_rt['profile_active'] != '1') {
            jrCore_page_not_found();
        }

        // get profile privacy
        $priv = (isset($_rt['profile_private'])) ? $_rt['profile_private'] : 1;
        if (isset($_rt['quota_jrProfile_privacy_changes']) && $_rt['quota_jrProfile_privacy_changes'] == 'off') {
            $priv = (int) $_rt['quota_jrProfile_default_privacy'];
        }
        switch ($priv) {

            // Fully Private Profile...
            case '0':
                if (!jrUser_is_logged_in()) {
                    jrCore_page_not_found();
                }
                if ($_user['user_active_profile_id'] != $_rt['_profile_id']) {
                    // We are not a profile owner...
                    jrCore_notice_page('error', $_lang['jrProfile'][16]);
                }
                break;

            // "1" is Global
            // Followers only...
            // Followers only (and search)...
            case '2':
            case '3':
                // Make sure we are installed...
                if (jrCore_module_is_active('jrFollower')) {
                    if (jrUser_is_logged_in()) {
                        if (!jrProfile_is_profile_owner($_rt['_profile_id'])) {
                            $_if = jrFollower_is_follower($_user['_user_id'], $_rt['_profile_id']);
                            if (!$_if || $_if['follow_active'] != '1') {

                                $out = jrCore_parse_template('header.tpl', $_rt);
                                $out .= jrCore_parse_template('profile_follow.tpl', $_rt, 'jrProfile');
                                $out .= jrCore_parse_template('footer.tpl', $_rt);
                                return $out;

                            }
                        }
                    }
                    else {
                        // Not logged in - tell them to log in
                        jrUser_session_require_login();
                    }
                }
                break;

        }
    }

    $tmp = '';
    // Module controller - gets everything
    if (isset($_post['option']) && isset($_urls["{$_post['option']}"])) {

        $mod = $_urls["{$_post['option']}"];

        // Make sure Quota is allowed
        if (isset($_rt["quota_{$mod}_allowed"]) && $_rt["quota_{$mod}_allowed"] == 'off') {
            jrCore_page_not_found();
        }

        // Get profile view function
        $active_tab = (!empty($_post['_1'])) ? $_post['_1'] : 'default';
        $function   = "profile_view_{$mod}_{$active_tab}";
        if (!function_exists($function)) {
            if (is_file(APP_DIR . "/modules/{$mod}/profile.php")) {
                require_once APP_DIR . "/modules/{$mod}/profile.php";
            }
            elseif (is_file(APP_DIR . "/modules/{$mod}/profile/{$active_tab}.php")) {
                require_once APP_DIR . "/modules/{$mod}/profile/{$active_tab}.php";
            }
            if (!function_exists($function)) {
                if (is_file(APP_DIR . "/modules/{$mod}/profile/default.php")) {
                    require_once APP_DIR . "/modules/{$mod}/profile/default.php";
                }
                $function   = "profile_view_{$mod}_default";
                $active_tab = 'default';
            }
        }
        if (function_exists($function)) {

            jrCore_page_defer_js_and_css();
            jrCore_page_set_no_header_or_footer();
            $old_module      = $_post['module'];
            $old_option      = $_post['option'];
            $_post['module'] = $mod;
            $_post['option'] = (!empty($_post['_1'])) ? $_post['_1'] : '';

            if ($tmp = $function($_rt, $_post, $_user, $_conf)) {

                // Check for disabled options
                // 0 = disable_header
                // 1 = disable_sidebar
                // 2 = disable_footer
                // 3 = disable_module_tabs
                if (jrCore_get_flag('jrprofile_disable_option_enabled')) {
                    // The parsed template is disabling one of our options - save so we can access these on later requests
                    jrProfile_save_template_disabled_options($mod, $function);
                }
                else {
                    jrProfile_set_template_disabled_options($mod, $function);
                }
                $_rt['profile_disable_header']      = (jrCore_get_flag('jrprofile_disable_header')) ? 1 : 0;
                $_rt['profile_disable_sidebar']     = (jrCore_get_flag('jrprofile_disable_sidebar')) ? 1 : 0;
                $_rt['profile_disable_footer']      = (jrCore_get_flag('jrprofile_disable_footer')) ? 1 : 0;
                $_rt['profile_disable_module_tabs'] = (jrCore_get_flag('jrprofile_disable_module_tabs')) ? 1 : 0;

                if ($active_tab != 'default') {
                    $_rt['profile_option_content'] = $tmp;
                    $tmp                           = jrCore_parse_template('profile_option.tpl', $_rt, 'jrProfile');
                }

                // Add in Profile Tabs if any have been registered for this module
                if ($_rt['profile_disable_module_tabs'] === 0) {
                    $_tab = jrCore_get_registered_module_features('jrProfile', 'profile_tab');
                    if ($_tab && isset($_tab[$mod])) {
                        $tmp = jrProfile_show_profile_tabs($_rt['_profile_id'], $active_tab, $mod, $_rt['profile_url'], $_tab[$mod]) . $tmp;
                    }
                }
                $_rt['profile_item_index_content'] = $tmp;
                $_rt['profile_template_module']    = $mod;
                $tmp                               = jrCore_parse_template('profile_item_index.tpl', $_rt);

                // In our custom profile handlers, we can add in things like
                // custom Javascript and CSS...
                $_pe = jrCore_get_flag('jrcore_page_elements');
                if ($_pe) {
                    $_rt = $_rt + $_pe;
                }
            }
            $_post['module'] = $old_module;
            $_post['option'] = $old_option;
            jrCore_delete_flag('jrcore_page_no_header_or_footer');
        }
    }

    // See if we came out of that with anything...
    if (empty($tmp)) {

        // Item Details (specific item)
        if (isset($_post['_1']) && jrCore_checktype($_post['_1'], 'number_nz')) {

            // It does not matter what comes after _1 - for SEO purposes we should use the <item>_url
            if (!isset($_post['option'])) {
                jrCore_page_not_found();
            }
            $mod = $_urls["{$_post['option']}"];
            if (!jrCore_module_is_active($mod)) {
                jrCore_page_not_found();
            }
            // Must have a detail template
            if (!is_file(APP_DIR . "/modules/{$mod}/templates/item_detail.tpl")) {
                jrCore_page_not_found();
            }
            // Make sure Quota is allowed
            if (isset($_rt["quota_{$mod}_allowed"]) && $_rt["quota_{$mod}_allowed"] == 'off') {
                jrCore_page_not_found();
            }

            // Save the referring URL here - if the user deletes or modifies this
            // item, we need to be able to send them back where they came from.
            jrCore_create_memory_url('item_delete');

            $_it = jrCore_db_get_item($mod, $_post['_1'], false, false, $_rt['_profile_id']);
            if (!$_it || !is_array($_it)) {
                // Redirect to module index for this profile
                jrCore_location("{$_conf['jrCore_base_url']}/{$_rt['profile_url']}/{$_post['option']}");
            }
            // Make sure the item belongs to the profile
            if (isset($_it['_profile_id']) && $_rt['_profile_id'] != $_it['_profile_id']) {
                jrCore_page_not_found();
            }
            // Check to see if this item is pending approval.  If it is, only
            // admins and the profile owner(s) can view it
            $pfx = jrCore_db_get_prefix($mod);
            if (!jrUser_is_logged_in() || (!jrProfile_is_profile_owner($_it['_profile_id']) && $_it['_user_id'] != jrUser_get_session_key('_user_id'))) {
                if (isset($_it["{$pfx}_pending"]) && $_it["{$pfx}_pending"] >= 1) {
                    jrCore_page_not_found();
                }
                // Check if this profile is OVER the allotted number of items
                $cap_type  = jrProfile_get_quota_value($_it, 'jrProfile', 'cap_type', 'hard');
                $max_items = jrProfile_get_quota_value($_it, $mod, 'max_items', 0);
                if ($cap_type == 'hard' && $max_items > 0) {
                    if ($_do = jrCore_db_get_display_order_for_profile_id($mod, $_it['_profile_id'], 'asc')) {
                        $_do = array_flip($_do);
                        $_do = array_slice($_do, 0, $max_items, true);
                        if (!isset($_do["{$_it['_item_id']}"])) {
                            // This item is OVER the limit - not found
                            jrCore_page_not_found();
                        }
                    }
                    else {
                        // No items
                        jrCore_page_not_found();
                    }
                }
            }

            $dir = $mod;

            // Send our item_detail trigger
            $_ag = array(
                'module' => $mod
            );
            $_it = jrCore_trigger_event('jrProfile', 'item_detail_view', $_it, $_ag);

            // See if we have a TITLE field for this item
            $title = $_rt['profile_name'];
            if (isset($_it["{$pfx}_title"])) {
                $title = $_it["{$pfx}_title"] . " - {$_rt['profile_name']}";
            }
            elseif (isset($_lang[$dir]['menu'])) {
                $title = "{$_lang[$dir]['menu']} - {$_rt['profile_name']}";
            }
            jrCore_page_title(jrCore_str_to_lower($title), false);

            $tmp                                = jrCore_parse_template('item_detail.tpl', array('item' => $_it), $dir);
            $_rt['profile_item_detail_content'] = $tmp;

            // Check for disabled options
            // 0 = disable_header
            // 1 = disable_sidebar
            // 2 = disable_footer
            // 3 = disable_module_tabs
            if (jrCore_get_flag('jrprofile_disable_option_enabled')) {
                // The parsed template is disabling one of our options - save so we can access these on later requests
                jrProfile_save_template_disabled_options($dir, 'item_detail.tpl');
            }
            else {
                jrProfile_set_template_disabled_options($dir, 'item_detail.tpl');
            }
            $_rt['profile_disable_header']      = (jrCore_get_flag('jrprofile_disable_header')) ? 1 : 0;
            $_rt['profile_disable_sidebar']     = (jrCore_get_flag('jrprofile_disable_sidebar')) ? 1 : 0;
            $_rt['profile_disable_footer']      = (jrCore_get_flag('jrprofile_disable_footer')) ? 1 : 0;
            $_rt['profile_disable_module_tabs'] = (jrCore_get_flag('jrprofile_disable_module_tabs')) ? 1 : 0;

            $_rt['profile_template_module'] = $mod;
            $tmp                            = jrCore_parse_template('profile_item_detail.tpl', $_rt);
        }

        // Module Item Index
        elseif (isset($_post['option']) && strlen($_post['option']) > 0 && isset($_urls["{$_post['option']}"]) && (!isset($_post['_1']) || strlen($_post['_1']) === 0)) {

            // Make sure Quota is allowed
            $mod = $_urls["{$_post['option']}"];
            if (isset($_rt["quota_{$mod}_allowed"]) && $_rt["quota_{$mod}_allowed"] == 'off') {
                jrCore_page_not_found();
            }
            // Must have an index template
            if (!is_file(APP_DIR . "/modules/{$mod}/templates/item_index.tpl")) {
                jrCore_page_not_found();
            }

            // Send our item_index trigger
            $_ag = array(
                'module' => $mod
            );
            $_rt = jrCore_trigger_event('jrProfile', 'item_index_view', $_rt, $_ag);

            // Add in Profile Tabs if any have been registered for this module
            $tmp = '';
            if (!jrCore_get_flag('jrprofile_disable_module_tabs')) {
                $_tab = jrCore_get_registered_module_features('jrProfile', 'profile_tab');
                if ($_tab && isset($_tab[$mod])) {
                    $tmp = jrProfile_show_profile_tabs($_rt['_profile_id'], 'default', $mod, $_rt['profile_url'], $_tab[$mod]);
                }
            }

            // It's a call to a module index - run our index template
            jrCore_set_flag('jrprofile_item_index_view_data', $_rt);
            $dir = $_urls["{$_post['option']}"];
            $tmp .= jrCore_parse_template('item_index.tpl', $_rt, $dir);
            jrCore_delete_flag('jrprofile_item_index_view_data');
            if ($help = jrCore_get_flag('jrprofile_item_index_help_html')) {
                $tmp .= $help;
                jrCore_delete_flag('jrprofile_item_index_help_html');
            }

            // The item_index template usually has a call to jrCore_list
            // If we get NO ITEMS and the viewing user is the profile owner
            // or a site admin, see if we have instructions for this module

            $_rt['profile_item_index_content'] = $tmp;
            // Set title to module menu entry
            if (isset($_lang[$dir]['menu'])) {
                jrCore_page_title(jrCore_str_to_lower("{$_lang[$dir]['menu']} - {$_rt['profile_name']}"));
            }

            // Check for disabled options
            // 0 = disable_header
            // 1 = disable_sidebar
            // 2 = disable_footer
            // 3 = disable_module_tabs
            if (jrCore_get_flag('jrprofile_disable_option_enabled')) {
                // The parsed template is disabling one of our options - save so we can access these on later requests
                jrProfile_save_template_disabled_options($dir, 'item_index.tpl');
            }
            else {
                jrProfile_set_template_disabled_options($dir, 'item_index.tpl');
            }
            $_rt['profile_disable_header']      = (jrCore_get_flag('jrprofile_disable_header')) ? 1 : 0;
            $_rt['profile_disable_sidebar']     = (jrCore_get_flag('jrprofile_disable_sidebar')) ? 1 : 0;
            $_rt['profile_disable_footer']      = (jrCore_get_flag('jrprofile_disable_footer')) ? 1 : 0;
            $_rt['profile_disable_module_tabs'] = (jrCore_get_flag('jrprofile_disable_module_tabs')) ? 1 : 0;

            $_rt['profile_template_module'] = $mod;
            $tmp                            = jrCore_parse_template('profile_item_index.tpl', $_rt);
        }

        // Module Profile View - check templates
        elseif (isset($_post['_1']) && strlen($_post['_1']) > 0) {

            // Make sure it is a good module...
            if (!isset($_urls["{$_post['option']}"])) {
                jrCore_page_not_found();
            }
            $mod = $_urls["{$_post['option']}"];
            if (isset($_rt["quota_{$mod}_allowed"]) && $_rt["quota_{$mod}_allowed"] == 'off') {
                jrCore_page_not_found();
            }
            $dir = null;

            // Send our item_list trigger
            $_ag = array(
                'module' => $mod
            );
            $_rt = jrCore_trigger_event('jrProfile', 'item_list_view', $_rt, $_ag);

            // Check for Skin template
            if (is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/{$mod}_item_{$_post['_1']}.tpl")) {
                $tpl = "{$mod}_item_{$_post['_1']}.tpl";
            }
            // Check for Module template
            elseif (is_file(APP_DIR . "/modules/{$mod}/templates/item_{$_post['_1']}.tpl")) {
                $tpl = "item_{$_post['_1']}.tpl";
                $dir = $mod;
            }
            if (isset($tpl)) {
                // See if we have a TITLE field for this item (item_album)
                $t = jrCore_strip_html($_post['_1']);
                jrCore_page_title(jrCore_str_to_lower("{$_rt['profile_name']} - $t"));

                $tmp                              = jrCore_parse_template($tpl, $_rt, $dir);
                $_rt['profile_item_list_content'] = $tmp;

                // Check for disabled options
                // 0 = disable_header
                // 1 = disable_sidebar
                // 2 = disable_footer
                // 3 = disable_module_tabs
                if (jrCore_get_flag('jrprofile_disable_option_enabled')) {
                    // The parsed template is disabling one of our options - save so we can access these on later requests
                    jrProfile_save_template_disabled_options($dir, $tpl);
                }
                else {
                    jrProfile_set_template_disabled_options($dir, $tpl);
                }
                $_rt['profile_disable_header']      = (jrCore_get_flag('jrprofile_disable_header')) ? 1 : 0;
                $_rt['profile_disable_sidebar']     = (jrCore_get_flag('jrprofile_disable_sidebar')) ? 1 : 0;
                $_rt['profile_disable_footer']      = (jrCore_get_flag('jrprofile_disable_footer')) ? 1 : 0;
                $_rt['profile_disable_module_tabs'] = (jrCore_get_flag('jrprofile_disable_module_tabs')) ? 1 : 0;

                $_rt['profile_template_module'] = $mod;
                $tmp                            = jrCore_parse_template('profile_item_list.tpl', $_rt);
            }
        }
    }

    // Profile Index (fall through)
    if (strlen($tmp) === 0) {

        // Send out profile index trigger
        $_rt = jrCore_trigger_event('jrProfile', 'profile_index', $_rt);

        // Set title, parse and return
        jrCore_page_title($_rt['profile_name']);

        // Parse profile_index Template
        $tmp = jrCore_parse_template('profile_index.tpl', $_rt);
    }

    // Send out profile view trigger
    $_rt = jrCore_trigger_event('jrProfile', 'profile_view', $_rt);

    // Check for cache (non-logged in users)
    $ckey = false;
    if (!jrUser_is_logged_in()) {
        $ckey = md5(json_encode($_post));
        if ($out = jrCore_is_cached('jrProfile', $ckey)) {
            return $out;
        }
    }

    // Pick up header elements set by plugins
    $_tmp = jrCore_get_flag('jrcore_page_elements');
    if ($_tmp) {
        unset($_tmp['page']);
        $_rt = array_merge($_tmp, $_rt);
    }

    // See if this skin is providing a profile header, or if we are using the standard header
    $hdr = "header.tpl";
    $skn = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if (is_file(APP_DIR . "/skins/{$skn}/profile_header.tpl")) {
        $hdr = "profile_header.tpl";
    }
    $out = jrCore_parse_template($hdr, $_rt, $skn);

    $out .= $tmp;

    $ftr = "footer.tpl";
    if (is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/profile_footer.tpl")) {
        $ftr = "profile_footer.tpl";
    }
    $out .= jrCore_parse_template($ftr, $_rt);

    if (!jrUser_is_logged_in()) {
        jrCore_add_to_cache('jrProfile', $ckey, $out);
    }
    return $out;
}

/**
 * Resets cached info for a given profile ID
 * @param $profile_id integer Profile ID to reset cache for
 * @param $module string Optionally only delete entries for specific module
 * @return bool
 */
function jrProfile_reset_cache($profile_id = null, $module = null)
{
    global $_user;
    if ((is_null($profile_id) || !jrCore_checktype($profile_id, 'number_nz')) && isset($_user['user_active_profile_id'])) {
        $profile_id = (int) $_user['user_active_profile_id'];
    }
    $pid = (int) $profile_id;
    if ($pid > 0) {
        jrCore_delete_profile_cache_entries($pid, $module);
    }
    return true;
}

/**
 * Get a profile's pulse keys and values
 * @param $profile_id int Profile ID
 * @return mixed
 */
function jrProfile_get_pulse_keys($profile_id)
{
    // Get any pulse counts for this profile
    $pid = (int) $profile_id;
    $key = "jrprofile_get_pulse_counts_{$pid}";
    if (!$_pk = jrCore_is_cached('jrProfile', $key, false, false)) {
        $tbl = jrCore_db_table_name('jrProfile', 'pulse');
        $req = "SELECT CONCAT_WS('_', pulse_module, pulse_key) AS k, pulse_count AS c FROM {$tbl} WHERE pulse_profile_id = {$pid}";
        $_pk = jrCore_db_query($req, 'k', false, 'c');
        if (!$_pk || !is_array($_pk)) {
            $_pk = 'no_items';
        }
        jrCore_add_to_cache('jrProfile', $key, $_pk, 0, $pid, false, false);
    }

    // Next - get available pulse keys
    $_ky = array();
    $_mf = jrCore_get_registered_module_features('jrProfile', 'pulse_key');
    if ($_mf && is_array($_mf)) {
        foreach ($_mf as $mod => $_inf) {
            foreach ($_inf as $key) {
                if (isset($_pk["{$mod}_{$key}"])) {
                    $_ky["{$mod}_{$key}"] = (int) $_pk["{$mod}_{$key}"];
                }
                else {
                    $_ky["{$mod}_{$key}"] = 0;
                }
            }
        }
        return jrCore_trigger_event('jrProfile', 'get_pulse_keys', $_ky, $_ky);
    }
    // No pulse keys
    return false;
}

/**
 * Reset a pulse key to 0
 * @param $profile_id int Profile ID
 * @param $module string Module
 * @param $key string Key name
 * @return mixed
 * @deprecated
 */
function jrProfile_reset_pulse_key($profile_id, $module, $key)
{
    if (jrCore_module_is_active($module)) {
        $pid = (int) $profile_id;
        $tbl = jrCore_db_table_name('jrProfile', 'pulse');
        $req = "DELETE FROM {$tbl} WHERE pulse_profile_id = {$pid} AND pulse_module = '{$module}' AND pulse_key = '{$key}'";
        jrCore_db_query($req);
        jrCore_delete_cache('jrProfile', "jrprofile_get_pulse_counts_{$pid}", false, false);
        return jrCore_trigger_event('jrProfile', 'reset_pulse_key', array('profile_id' => $pid, 'key' => $key));
    }
    return false;
}

/**
 * Save the disabled options for a module/template
 * @param string $module
 * @param string $template
 * @return mixed
 */
function jrProfile_save_template_disabled_options($module, $template)
{
    $h = (jrCore_get_flag('jrprofile_disable_header')) ? 1 : 0;
    $s = (jrCore_get_flag('jrprofile_disable_sidebar')) ? 1 : 0;
    $f = (jrCore_get_flag('jrprofile_disable_footer')) ? 1 : 0;
    $t = (jrCore_get_flag('jrprofile_disable_module_tabs')) ? 1 : 0;
    return jrCore_add_to_cache('jrProfile', "{$module}:{$template}", "{$h},{$s},{$f},{$t}", 0, 0, false, false);

}

/**
 * Get the global flags for profile template disabled options
 * @param string $module
 * @param string $template
 * @return true
 */
function jrProfile_set_template_disabled_options($module, $template)
{
    if ($values = jrCore_is_cached('jrProfile', "{$module}:{$template}", false, false)) {
        if ($_tmp = explode(',', $values)) {
            jrCore_set_flag('jrprofile_disable_header', $_tmp[0]);
            jrCore_set_flag('jrprofile_disable_sidebar', $_tmp[1]);
            jrCore_set_flag('jrprofile_disable_footer', $_tmp[2]);
            jrCore_set_flag('jrprofile_disable_module_tabs', $_tmp[3]);
            return true;
        }
    }
    jrCore_set_flag('jrprofile_disable_header', 0);
    jrCore_set_flag('jrprofile_disable_sidebar', 0);
    jrCore_set_flag('jrprofile_disable_footer', 0);
    jrCore_set_flag('jrprofile_disable_module_tabs', 0);
    return true;
}

/**
 * Get modules that are valid for Stats Rebuild
 * @return array|bool
 */
function jrProfile_get_valid_modules()
{
    global $_mods;
    if ($_tmp = jrCore_get_datastore_modules()) {
        $_out = array();
        foreach ($_tmp as $mod => $pfx) {
            switch ($mod) {
                case 'jrUser':
                case 'jrProfile':
                case 'jrCore':
                    break;
                default:
                    $_out[$mod] = $_mods[$mod]['module_name'];
                    break;
            }
        }
        return $_out;
    }
    return false;
}
