<?php
 /**
 * Jamroom Graph Support module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrGraph_meta()
{
    return array(
        'name'        => 'Graph Support',
        'url'         => 'graph',
        'version'     => '1.2.2',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Support for creating and displaying line, point and bar graphs',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/2862/graph-core',
        'category'    => 'core',
        'license'     => 'mpl',
        'priority'    => 10,
        'activate'    => true
    );
}

/**
 * init
 */
function jrGraph_init()
{
    // Our graph module provides the "graph" magic view
    jrCore_register_module_feature('jrCore', 'magic_view', 'jrGraph', 'graph', 'jrGraph_create_graph');

    // Register event trigger
    jrCore_register_event_trigger('jrGraph', 'graph_parameters', 'Fired with parameters when creating a graph');
    jrCore_register_event_trigger('jrGraph', 'graph_data', 'Fired with config data when creating a graph');

    // Register our JS/CSS
    jrCore_register_module_feature('jrCore', 'javascript', 'jrGraph', APP_DIR . '/modules/jrGraph/contrib/strftime/strftime.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrGraph', APP_DIR . '/modules/jrGraph/contrib/flot/jquery.flot.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrGraph', APP_DIR . '/modules/jrGraph/contrib/flot/jquery.flot.time.min.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrGraph', 'jrGraph.js');
    jrCore_register_module_feature('jrCore', 'css', 'jrGraph', 'jrGraph.css');

    return true;
}

//---------------------------------------------------------
// GRAPH FUNCTIONS
//---------------------------------------------------------

/**
 * Display a Graph for a given graph name (MAGIC VIEW)
 * @param $_post array Params from jrCore_parse_url();
 * @param $_user array User information
 * @param $_conf array Global config
 * @return bool Returns true
 */
function jrGraph_create_graph($_post, $_user, $_conf)
{
    // When a graph is requested, it will come in like:
    // <module>/graph/<graph_name>/<width>/<height>/ ... <params>
    // <params> is optional and depends on the <graph_name> to support
    // <graph_name> is registered by the <module> so we know what
    // function is being used to gather the graph data.
    $add = '';
    $nam = false;
    $ttl = false;
    $hrs = false;
    $dys = 10;
    $fil = 1;
    $int = null;
    $uid = jrCore_create_unique_string(6);
    if (!isset($_post['name']) && isset($_post['_1'])) {
        $_post['name'] = $_post['_1'];
    }
    if (strpos($_post['name'], '|ds_items_by_day')) {
        list($mod, $nam) = explode('|', $_post['name']);
        $ttl = 'Items Created by Day';
    }
    else {
        $_tm = jrCore_get_registered_module_features('jrGraph', 'graph_config');
        if ($_tm && is_array($_tm)) {
            foreach ($_tm as $_fnc) {
                foreach ($_fnc as $name => $_inf) {
                    if ($name == $_post['name']) {
                        $nam = $_post['name'];
                        $ttl = $_inf['title'];
                        $fil = (isset($_inf['nofill']) && $_inf['nofill'] === true) ? 0 : 1;
                        $int = (isset($_inf['interval'])) ? intval($_inf['interval']) : null;
                        // Valid group
                        if (isset($_inf['group'])) {
                            switch ($_inf['group']) {
                                case 'master':
                                    if (!jrUser_is_master()) {
                                        jrCore_page_not_found();
                                    }
                                    break;
                                case 'admin':
                                    if (!jrUser_is_admin()) {
                                        jrCore_page_not_found();
                                    }
                                    break;
                                case 'user':
                                    if (!jrUser_is_logged_in()) {
                                        jrCore_page_not_found();
                                    }
                                    break;
                            }
                        }

                        // Hours or Days?
                        if (!empty($_inf['hours']) && jrCore_checktype($_inf['hours'], 'number_nz')) {
                            $hrs          = intval($_inf['hours']);
                            $_inf['days'] = ($_inf['hours'] / 24);
                        }
                        elseif (!empty($_inf['days']) && jrCore_checktype($_inf['days'], 'number_nz')) {
                            $_inf['days'] = intval($_inf['days']);
                        }
                        if (!empty($_post['days']) && jrCore_checktype($_post['days'], 'number_nz')) {
                            $dys = intval($_post['days']);
                        }
                        else {
                            $dys = (isset($_inf['days']) && jrCore_checktype($_inf['days'], 'number_nz')) ? intval($_inf['days']) : 60;
                        }

                        // Are we showing a graph length selector?
                        if (!empty($_inf['selector']) && is_array($_inf['selector'])) {

                            // Add in some extra info needed for load
                            $_inf['module'] = $_post['module'];
                            $_inf['name']   = $_post['name'];
                            $_inf['_post']  = $_post;

                            $dat = base64_encode(json_encode($_inf));
                            $add = jrCore_page_jumper("j-{$uid}", $_inf['selector'], $dys, "var v=$(this).val(); jrGraph_load_graph('{$uid}','{$dat}',v)", 'graph-selector');
                        }
                        break;
                    }
                }
            }
        }
        $mod = $_post['module'];
    }
    jrCore_page_banner($ttl, '<div id="xyval" class="graph-xyval"></div>' . $add);

    $_args = array(
        'module'    => $mod,
        'name'      => $nam,
        'height'    => '280px',
        'width'     => '100%',
        'days'      => $dys,
        'hours'     => $hrs,
        'fill'      => $fil,
        'interval'  => $int,
        'unique_id' => $uid
    );
    if (jrCore_is_ajax_request()) {
        $_args['modal'] = 'modal';
    }
    $_args = jrCore_trigger_event('jrGraph', 'graph_parameters', $_args);

    $temp = new stdClass();
    $html = smarty_function_jrGraph_embed($_args, $temp);
    jrCore_page_custom($html);

    if (jrCore_is_ajax_request()) {
        jrCore_page_close_button('$.modal.close();');
        jrCore_page_set_no_header_or_footer();
    }
    else {
        jrCore_page_set_meta_header_only();
    }
    return jrCore_page_display(true);
}

/**
 * Generate a Graph ( with a timestamp as the X value )
 * @param $module string Module graph is being created for
 * @param $name string Name of Graph
 * @param $params array additional parameters
 * @return string|array
 */
function jrGraph_generate($module, $name, $params = null, $data_only = false)
{
    $_tm = jrCore_get_registered_module_features('jrGraph', 'graph_config');
    if (!$_tm || !isset($_tm[$module]) || !isset($_tm[$module][$name])) {
        jrCore_notice('error', "invalid graph name: {$name} - not registered by module");
    }
    $fnc = $_tm[$module][$name]['function'];
    if (!function_exists($fnc)) {
        jrCore_notice('error', 'invalid graph - graph function registered by module does not exist');
    }
    if (is_null($params)) {
        $params = array();
    }
    if (isset($_tm[$module][$name]['params']) && is_array($_tm[$module][$name]['params'])) {
        foreach ($_tm[$module][$name]['params'] as $k => $v) {
            $params[$k] = $v;
        }
    }
    if (!empty($params['hours'])) {
        // Convert hours to days
        $params['days'] = ($params['hours'] / 24);
    }
    elseif (!isset($params['days'])) {
        $params['days'] = 10;
    }
    $_dt = $fnc($module, $name, $params);

    // our plugin function will return a multidimensional array of data points
    // with time -> value series data - i.e.
    // array(
    //     '_sets' => array(
    //         array( <epoch_time> => <value> )
    //         array( <epoch_time> => <value> )
    //         ...
    //     ),
    //     'params' => array()
    // )

    $_rp = array(
        '_sets'     => array(),
        'clickable' => 0,
        'hoverable' => 0,
        'xaxis'     => ((isset($_dt['xaxis']) && is_array($_dt['xaxis'])) ? $_dt['xaxis'] : array()),
        'yaxis'     => ((isset($_dt['yaxis']) && is_array($_dt['yaxis'])) ? $_dt['yaxis'] : array()),
        'precision' => ((isset($_dt['precision'])) ? intval($_dt['precision']) : 3),
        'unique_id' => (!empty($params['unique_id'])) ? $params['unique_id'] : 't' . jrCore_create_unique_string(6),
        'height'    => ((isset($_dt['height'])) ? $_dt['height'] : ((isset($params['height'])) ? $params['height'] : '350px')),
        'width'     => ((isset($_dt['width'])) ? $_dt['width'] : ((isset($params['width'])) ? $params['width'] : '100%')),
        'days'      => ((isset($_dt['days']) && jrCore_checktype($_dt['days'], 'number_nz')) ? intval($_dt['days']) : $params['days'])
    );
    if (isset($_dt['_sets']) && is_array($_dt['_sets'])) {

        if (isset($_dt['_options']['function'])) {
            // We have a custom click function...
            $_rp['function'] = $_dt['_options']['function'];
            unset($_dt['_options']['function']);
        }
        // SETS
        // Our timestamps must be in Epoch with milliseconds
        foreach ($_dt['_sets'] as $k => $_set) {

            $_rp['_sets'][$k] = array();
            if (isset($_set['_data']) && is_array($_set['_data'])) {
                $_rp['_sets'][$k]['data'] = array();
                if ((isset($params['fill']) && $params['fill'] == 1) || (isset($_set['zerofill']) && $_set['zerofill'] === true)) {
                    if (!isset($params['unit'])) {
                        $params['unit'] = 'days';
                        if (!empty($params['hours'])) {
                            $params['unit'] = 'hours';
                        }
                    }
                    if (!isset($params['interval'])) {
                        $params['interval'] = null;
                    }
                    if (isset($_set['zerofill']) && $_set['zerofill'] === true) {
                        $_set['_data'] = jrGraph_fill_data_gaps($_set['_data'], $params['days'], $params['unit'], $params['interval']);
                    }
                }
                if (!empty($_set['_data'])) {
                    foreach ($_set['_data'] as $epc => $val) {
                        if (strlen($epc) != 13) {
                            $epc = ($epc * 1000);
                        }
                        if ($data_only) {
                            $_rp['_sets'][$k]['data'][] = array($epc, $val);
                        }
                        else {
                            $_rp['_sets'][$k]['data'][] = "[{$epc},{$val}]";
                        }
                    }
                }
            }
            // See if we have set options
            foreach ($_set as $sk => $sv) {
                switch ($sk) {

                    case 'minTickSize':
                        if ($data_only) {
                            $_rp['xaxis']['minTickSize'] = explode(',', str_replace(array('[', ']', ' '), '', $sv));
                            foreach ($_rp['xaxis']['minTickSize'] as $xa => $xv) {
                                if (jrCore_checktype($xv, 'number')) {
                                    $_rp['xaxis']['minTickSize'][$xa] = intval($xv);
                                }
                                else {
                                    $_rp['xaxis']['minTickSize'][$xa] = trim($xv, "\"'");
                                }
                            }
                        }
                        else {
                            $_rp['xaxis']['minTickSize'] = $sv;
                        }
                        break;

                    case 'xticks':
                        $_rp['xaxis']['ticks'] = ($sv - 1);
                        unset($_set[$sk]);
                        break;

                    case 'yticks':
                        $_rp['yaxis']['ticks'] = $sv;
                        unset($_set[$sk]);
                        break;

                    case 'date_format':
                        // If we get the special "date_format" key we know our X-Axis is a Date
                        if ($data_only) {
                            $_rp['xaxis']['mode']  = 'time';
                            $_rp['tooltip_format'] = $sv;
                        }
                        else {
                            $_rp['xaxis']['mode']  = '"time"';
                            $_rp['tooltip_format'] = $sv;
                        }
                        unset($_set[$sk]);
                        break;

                    case 'type':

                        $lw = '';
                        if (isset($_set['lineWidth']) && jrCore_checktype($_set['lineWidth'], 'number_nz')) {
                            $lw = ", lineWidth: {$_set['lineWidth']}";
                        }

                        $fc = '';
                        if (isset($_set['fillColor']) && strlen($_set['fillColor']) > 0) {
                            $fc = ', fill: true, fillColor: "' . trim(str_replace(array("'", '"'), '', $_set['fillColor'])) . '"';
                        }

                        $pr = '';
                        $sp = false;
                        if (isset($_set['pointRadius']) && jrCore_checktype($_set['pointRadius'], 'number_nn')) {
                            $pr = ", radius: {$_set['pointRadius']}";
                            $sp = true;
                        }

                        // We support: line, filled-line, bar and point
                        switch ($sv) {
                            case 'line':
                                if ($data_only) {
                                    $_rp['_sets'][$k]['lines'] = array('show' => true);
                                    if (isset($_set['lineWidth']) && jrCore_checktype($_set['lineWidth'], 'number_nz')) {
                                        $_rp['_sets'][$k]['lines'][0]['lineWidth'] = $_set['lineWidth'];
                                    }
                                    if (isset($_set['fillColor']) && strlen($_set['fillColor']) > 0) {
                                        $_rp['_sets'][$k]['lines'][0]['fill']      = true;
                                        $_rp['_sets'][$k]['lines'][0]['fillColor'] = trim(str_replace(array("'", '"'), '', $_set['fillColor']));
                                    }
                                }
                                else {
                                    $_rp['_sets'][$k]['lines'] = '{ show: true' . $lw . $fc . ' }';
                                }
                                if ($sp) {
                                    if ($data_only) {
                                        $_rp['_sets'][$k]['points'] = array('show' => true, 'radius' => $_set['pointRadius']);
                                        $_dt['_options']['grid']    = array('margin' => 20, 'hoverable' => true);
                                    }
                                    else {
                                        $_rp['_sets'][$k]['points'] = '{ show: true' . $pr . ' }';
                                        $_dt['_options']['grid']    = '{ margin: 20, hoverable: true }';
                                    }
                                    $_rp['hoverable'] = 1;
                                }
                                break;

                            case 'bar':
                                if ($data_only) {
                                    $_rp['_sets'][$k]['bars'] = array('show' => true);
                                    if (isset($_set['fillColor']) && strlen($_set['fillColor']) > 0) {
                                        $_rp['_sets'][$k]['bars'][0]['fill']      = true;
                                        $_rp['_sets'][$k]['bars'][0]['fillColor'] = trim(str_replace(array("'", '"'), '', $_set['fillColor']));
                                    }
                                }
                                else {
                                    $_rp['_sets'][$k]['bars'] = '{ show: true' . $fc . ' }';
                                }
                                break;

                            case 'point':
                                if ($data_only) {
                                    $_rp['_sets'][$k]['points'] = array('show' => true, 'radius' => $_set['pointRadius']);
                                    $_dt['_options']['grid']    = array('margin' => 20, 'hoverable' => true);
                                }
                                else {
                                    $_rp['_sets'][$k]['points'] = '{ show: true' . $pr . ' }';
                                    $_dt['_options']['grid']    = '{ margin: 20, hoverable: true }';
                                }
                                $_rp['hoverable'] = 1;
                                break;
                        }
                        break;

                    case 'xaxis':
                    case 'yaxis':
                    case 'clickable':
                    case 'hoverable':
                    case 'font':
                        $_rp['_sets'][$k][$sk] = $sv;
                        break;

                    case 'color':
                    case 'label':
                    case 'shadowSize':
                    case 'highlightColor':
                        if ($data_only) {
                            $_rp['_sets'][$k][$sk] = trim(str_replace(array("'", '"'), '', $sv));
                        }
                        else {
                            $_rp['_sets'][$k][$sk] = "'" . trim(str_replace(array("'", '"'), '', $sv)) . "'";
                        }
                        break;
                }
            }
        }
        unset($_dt['_sets']);

        // var d = [ [-373597200000, 315.71], [-370918800000, 317.45] ]
        if (!$data_only) {
            foreach ($_rp['_sets'] as $k => $_set) {
                $_st = array();
                foreach ($_set as $sk => $sv) {
                    switch ($sk) {
                        case 'data':
                            $sv = '[' . implode(',', $sv) . "]\n";
                            break;
                    }
                    $_st[] = "{$sk}: {$sv}";
                }
                $_rp['_sets'][$k] = '{' . implode(',', $_st) . '},';
            }
        }

        // X-axis / Y-Axis
        foreach (array('xaxis', 'yaxis') as $axis) {
            // show: null or true/false
            // position: "bottom" or "top" or "left" or "right"
            // mode: null or "time" ("time" requires jquery.flot.time.js plugin)
            // timezone: null, "browser" or timezone (only makes sense for mode: "time")
            // color: null or color spec
            // tickColor: null or color spec
            // font: null or font spec object
            // min: null or number
            // max: null or number
            // autoscaleMargin: null or number
            // transform: null or fn: number -> number
            // inverseTransform: null or fn: number -> number
            // ticks: null or number or ticks array or (fn: axis -> ticks array)
            // tickSize: number or array
            // minTickSize: number or array
            // tickFormatter: (fn: number, object -> string) or string
            // tickDecimals: null or number
            // labelWidth: null or number
            // labelHeight: null or number
            // reserveSpace: null or true
            // tickLength: null or number
            // alignTicksWithAxis: null or number
            if (isset($_rp[$axis]) && count($_rp[$axis]) > 0) {
                if ($data_only) {
                    $_dt['_options'][$axis] = $_rp[$axis];
                }
                else {
                    $ax = '';
                    foreach ($_rp[$axis] as $k => $v) {
                        $ax .= "{$k}: {$v},";
                    }
                    $_dt['_options'][$axis] = '{' . substr($ax, 0, strlen($ax) - 1) . '}';
                    unset($ax);
                }
            }
            unset($_rp[$axis]);
        }

        if (!isset($_dt['legend'])) {
            $_dt['legend'] = array(
                'container' => "#l{$_rp['unique_id']}"
            );
        }

        // Legend
        if (isset($_dt['legend'])) {
            if (!isset($_dt['legend']['container'])) {
                $_dt['legend']['container'] = '#graph-legend';
            }
            // show: boolean
            // labelFormatter: null or (fn: string, series object -> string)
            // labelBoxBorderColor: color
            // noColumns: number
            // position: "ne" or "nw" or "se" or "sw"
            // margin: number of pixels or [x margin, y margin]
            // backgroundColor: null or color
            // backgroundOpacity: number between 0 and 1
            // container: null or jQuery object/DOM element/jQuery expression
            // sorted: null/false, true, "ascending", "descending", "reverse", or a comparator
            if (!$data_only) {
                $lg = '';
                foreach ($_dt['legend'] as $k => $v) {
                    switch ($k) {
                        case 'position':
                        case 'container':
                            $lg .= "{$k}: '{$v}',";
                            break;
                        default:
                            $lg .= "{$k}: {$v},";
                            break;
                    }
                }
                $_dt['_options']['legend'] = '{' . substr($lg, 0, strlen($lg) - 1) . '}';
            }
            else {
                $_dt['_options']['legend'] = $_dt['legend'];
            }
        }

        // Plot options
        if (isset($_dt['_options']) && is_array($_dt['_options'])) {
            if ($data_only) {
                foreach ($_dt['_options'] as $k => $v) {
                    $_rp['options'][$k] = $v;
                }
            }
            else {
                $op = '';
                foreach ($_dt['_options'] as $k => $v) {
                    $op .= "{$k}: {$v},";
                }
                $_rp['options'] = ', {' . substr($op, 0, strlen($op) - 1) . '}';
            }
            unset($_dt['_options']);
        }
    }

    // Let other modules add/remove config options
    $_rp = jrCore_trigger_event('jrGraph', 'graph_data', $_rp);

    if ($data_only) {
        return $_rp;
    }

    $_rp['timezone'] = jrCore_get_config_value('jrCore', 'system_timezone', 'Europe/Dublin');
    $_rp['data_set'] = '';
    foreach ($_rp['_sets'] as $set) {
        $_rp['data_set'] .= " {$set} ";
    }

    $tpl = 'graph.tpl';
    $mod = 'jrGraph';
    if (is_file(APP_DIR . "/modules/{$module}/templates/graph_{$name}.tpl")) {
        $tpl = "graph_{$name}.tpl";
        $mod = $module;
    }
    return jrCore_parse_template($tpl, $_rp, $mod);
}

/**
 * Fill date gaps in the data with zeros
 * @param $_data array incoming data set
 * @param $number number of units to back fill
 * @param string $unit "days" or "hours"
 * @param int $interval number of SECONDS between data values
 * @return array|false
 */
function jrGraph_fill_data_gaps($_data, $number, $unit = 'days', $interval = null)
{
    if ($number == 1) {
        if ($unit == 'days') {
            $unit = 'hours';
        }
        elseif ($unit == 'hours') {
            $unit = 'minutes';
        }
    }
    switch ($unit) {
        case 'days':
            $amt = 86400;
            $now = strtotime('tomorrow midnight');
            break;
        case 'hours':
            $amt = 3600;
            $now = (time() % 3600);
            $now = ((time() - $now) + 3600);
            break;
        case 'minutes':
            $amt = 60;
            $now = (time() % 60);
            $now = ((time() - $now) + 60);
            break;
        default:
            return false;
    }
    if (!is_null($interval) && jrCore_checktype($interval, 'number_nz')) {
        $amt = (int) $interval;
    }

    $_dt = array();
    $beg = ($now - ($number * $amt));
    while ($beg < $now) {
        // Do we have a value in our incoming data that fits this time?
        $end = ($beg + $amt);
        $fnd = false;
        foreach ($_data as $k => $v) {
            if ($k >= $beg && $k < $end) {
                // Yes...
                $_dt[$beg] = intval($v);
                unset($_data[$k]);
                $fnd = true;
            }
        }
        if (!$fnd) {
            $_dt[$beg] = 0;
        }
        $beg += $amt;
    }
    ksort($_dt, SORT_NUMERIC);
    return $_dt;
}

/**
 * Plot a Graph ( without correction for empty timestamp on the  X value )
 * @param $module string Module graph is being created for
 * @param $name string Name of Graph
 * @param $params array additional parameters
 * @return string
 */
function jrGraph_plot($module, $name, $params = null)
{
    global $_mods;
    $_tm = jrCore_get_registered_module_features('jrGraph', 'graph_plot');
    if (!$_tm || !isset($_tm[$module]) || !isset($_tm[$module][$name])) {
        jrCore_notice('error', 'invalid graph - graph plot name not registered by module');
    }
    $fnc = $_tm[$module][$name]['function'];
    if (!function_exists($fnc)) {
        jrCore_notice('error', 'invalid graph - graph plot function registered by module does not exist');
    }

    // include Pie chart JS file
    $crl = jrCore_get_module_url('jrCore');
    $grl = jrCore_get_module_url('jrGraph');
    $_js = array('source' => jrCore_get_base_url() . "/{$crl}/js/{$grl}/contrib/flot/jquery.flot.pie.min.js?v=" . $_mods['jrGraph']['module_updated']);
    jrCore_create_page_element('javascript_href', $_js);

    $_rp = array(
        'flot'      => $fnc($module, $name, $params),
        'unique_id' => 't' . jrCore_create_unique_string(6),
        'height'    => (isset($params['height'])) ? $params['height'] : '350px',
        'width'     => (isset($params['width'])) ? $params['width'] : '100%'
    );
    $tpl = 'plot.tpl';
    $mod = 'jrGraph';
    if (is_file(APP_DIR . "/modules/{$module}/templates/plot_{$name}.tpl")) {
        $tpl = "plot_{$name}.tpl";
        $mod = $module;
    }
    return jrCore_parse_template($tpl, $_rp, $mod);
}

/**
 * Return the JS for the graph resize listener
 * @return string
 */
function jrGraph_get_resize_listener_js()
{
    return <<<'EOD'
(function($,e,t){"$:nomunge";var i=[],n=$.resize=$.extend($.resize,{}),a,r=false,s="setTimeout",u="resize",m=u+"-special-event",o="pendingDelay",l="activeDelay",f="throttleWindow";n[o]=200;n[l]=20;n[f]=true;$.event.special[u]={setup:function(){if(!n[f]&&this[s]){return false}var e=$(this);i.push(this);e.data(m,{w:e.width(),h:e.height()});if(i.length===1){a=t;h()}},teardown:function(){if(!n[f]&&this[s]){return false}var e=$(this);for(var t=i.length-1;t>=0;t--){if(i[t]==this){i.splice(t,1);break}}e.removeData(m);if(!i.length){if(r){cancelAnimationFrame(a)}else{clearTimeout(a)}a=null}},add:function(e){if(!n[f]&&this[s]){return false}var i;function a(e,n,a){var r=$(this),s=r.data(m)||{};s.w=n!==t?n:r.width();s.h=a!==t?a:r.height();i.apply(this,arguments)}if($.isFunction(e)){i=e;return a}else{i=e.handler;e.handler=a}}};function h(t){if(r===true){r=t||1}for(var s=i.length-1;s>=0;s--){var l=$(i[s]);if(l[0]==e||l.is(":visible")){var f=l.width(),c=l.height(),d=l.data(m);if(d&&(f!==d.w||c!==d.h)){l.trigger(u,[d.w=f,d.h=c]);r=t||true}}else{d=l.data(m);d.w=0;d.h=0}}if(a!==null){if(r&&(t==null||t-r<1e3)){a=e.requestAnimationFrame(h)}else{a=setTimeout(h,n[o]);r=false}}}if(!e.requestAnimationFrame){e.requestAnimationFrame=function(){return e.webkitRequestAnimationFrame||e.mozRequestAnimationFrame||e.oRequestAnimationFrame||e.msRequestAnimationFrame||function(t,i){return e.setTimeout(function(){t((new Date).getTime())},n[l])}}()}if(!e.cancelAnimationFrame){e.cancelAnimationFrame=function(){return e.webkitCancelRequestAnimationFrame||e.mozCancelRequestAnimationFrame||e.oCancelRequestAnimationFrame||e.msCancelRequestAnimationFrame||clearTimeout}()}})(jQuery,this);
(function ($) {
    var options = { };
    function init(plot) {
        function onResize() {
            var placeholder = plot.getPlaceholder();
            if (placeholder.width() === 0 || placeholder.height() === 0) return;
            plot.resize();
            plot.setupGrid();
            plot.draw();
        }
        function bindEvents(plot, eventHolder) {
            plot.getPlaceholder().resize(onResize);
        }
        function shutdown(plot, eventHolder) {
            plot.getPlaceholder().unbind("resize", onResize);
        }
        plot.hooks.bindEvents.push(bindEvents);
        plot.hooks.shutdown.push(shutdown);
    }
    $.plot.plugins.push({
        init: init,
        options: options,
        name: 'resize',
        version: '1.0'
    });
})(jQuery);
EOD;
}

//---------------------------------------------------------
// SMARTY FUNCTIONS
//---------------------------------------------------------

/**
 * Embed a Graph in a template
 * @param $params array function parameters
 * @param $smarty object Smarty Object
 * @return string
 */
function smarty_function_jrGraph_embed($params, $smarty = null)
{
    if (empty($params['module'])) {
        return 'jrGraph_embed: module parameter required';
    }
    if (!isset($params['name']) || strlen($params['name']) === 0) {
        return 'jrGraph_embed: graph name parameter required';
    }
    $out = jrGraph_generate($params['module'], $params['name'], $params);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}
