<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Skin
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get info for a skin from the DB skin table
 * @param string $skin
 * @return array|false
 */
function jrCore_get_skin_db_info($skin)
{
    $tbl = jrCore_db_table_name('jrCore', 'skin');
    $skn = jrCore_db_escape($skin);
    $req = "SELECT * FROM {$tbl} WHERE skin_directory = '{$skn}'";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Verify a skin is installed properly
 * @param string $skin Skin to verify
 * @param bool $validate_language set to FALSE to not run lang file validation
 * @return bool
 */
function jrCore_verify_skin($skin, $validate_language = true)
{
    global $_conf;
    // Is this a BRAND NEW skin?
    $ins = false;
    $_sk = jrCore_get_skin_db_info($skin);
    if (!$_sk || !is_array($_sk)) {
        // New Skin
        $tbl = jrCore_db_table_name('jrCore', 'skin');
        $skn = jrCore_db_escape($skin);
        $req = "INSERT INTO {$tbl} (skin_directory, skin_updated, skin_custom_css, skin_direct_css, skin_custom_image)
                VALUES ('{$skn}', UNIX_TIMESTAMP(), '', '', '')
                ON DUPLICATE KEY UPDATE skin_updated = UNIX_TIMESTAMP()";
        $cnt = jrCore_db_query($req, 'COUNT');
        if ($cnt !== 1) {
            // We did NOT install correctly
            return false;
        }
        // This is a NEW install of this skin
        $ins = true;
    }

    if (is_file(APP_DIR . "/skins/{$skin}/include.php")) {
        $func = "{$skin}_skin_init";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin}/include.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }

    // Global Config
    if (is_file(APP_DIR . "/skins/{$skin}/config.php")) {
        $func = "{$skin}_skin_config";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin}/config.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }

    // Quota
    if (is_file(APP_DIR . "/skins/{$skin}/quota.php")) {
        $func = "{$skin}_skin_quota_config";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin}/quota.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }

    // Install Language strings for Skin
    if ($validate_language) {
        jrUser_install_lang_strings('skin', $skin);
    }

    // If this is a NEW install of this skin, run installer
    if ($ins && is_file(APP_DIR . "/skins/{$skin}/install.php")) {
        $func = "{$skin}_skin_install";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin}/install.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }

    // Skin Verify
    if (is_file(APP_DIR . "/skins/{$skin}/include.php") && $_conf['jrCore_active_skin'] == $skin) {
        $func = "{$skin}_skin_verify";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin}/include.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }

    // Let other modules do their work
    jrCore_trigger_event('jrCore', 'verify_skin', jrCore_skin_meta_data($skin));

    return true;
}

/**
 * Fires when the skin is deactivate to allow it to clean itself up
 * @param string $skin Skin to verify
 * @return bool
 */
function jrCore_skin_deactivate($skin)
{
    if (is_file(APP_DIR . "/skins/{$skin}/include.php")) {
        $func = "{$skin}_skin_deactivate";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin}/include.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }
    return true;
}

/**
 * Delete a skin from the system
 * @param string $skin
 * @return bool|string
 */
function jrCore_delete_skin($skin)
{
    // Make sure we are NOT the active skin
    if ($skin == jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2')) {
        return 'error: cannot delete the active skin';
    }

    // Remove from skins
    $tbl = jrCore_db_table_name('jrCore', 'skin');
    $skn = jrCore_db_escape($skin);
    $req = "DELETE FROM {$tbl} WHERE skin_directory = '{$skn}' LIMIT 1";
    jrCore_db_query($req);

    // Remove cache dir
    $cdr = jrCore_get_module_cache_dir($skin);
    jrCore_delete_dir_contents($cdr, true, 0, true);

    // Remove skin directories
    jrCore_start_timer('filesystem');
    $_dirs = glob(APP_DIR . "/skins/{$skin}*");
    jrCore_stop_timer('filesystem');
    if ($_dirs && is_array($_dirs) && count($_dirs) > 0) {
        foreach ($_dirs as $dir) {
            $nam = basename($dir);
            if ($nam == $skin || strpos($nam, "{$skin}-release-") === 0) {
                jrCore_start_timer('filesystem');
                if (is_link($dir)) {
                    unlink($dir);  // OK
                    jrCore_stop_timer('filesystem');
                }
                else {
                    jrCore_stop_timer('filesystem');
                    jrCore_delete_dir_contents($dir, false, 0, true);
                }
            }
        }
    }

    // Remove custom lang entries
    $tbl = jrCore_db_table_name('jrUser', 'language');
    $req = "DELETE FROM {$tbl} WHERE `lang_module` = '{$skn}'";
    jrCore_db_query($req);

    // Remove custom templates
    $tbl = jrCore_db_table_name('jrCore', 'template');
    $req = "DELETE FROM {$tbl} WHERE `template_module` = '{$skn}'";
    jrCore_db_query($req);

    // Remove settings
    $tbl = jrCore_db_table_name('jrCore', 'setting');
    $req = "DELETE FROM {$tbl} WHERE `module` = '{$skn}'";
    jrCore_db_query($req);
    return true;
}

/**
 * Generate a CSS Sprite background image from existing Icon images
 * @param $skin string Skin to use for overriding default icon images
 * @param $color string Icon color set to use (black||white)
 * @param $width int Pixel width for Icons
 * @return bool
 */
function jrCore_create_css_sprite($skin, $color = 'black', $width = 64)
{
    // Our ICON sprites live in jrCore/img/icons, and each can
    // be overridden by the skin with it's own version of the sprite

    // Standard Sprite Icons
    $swidth = 0;

    // Retina Sprite Icons
    $rwidth = 0;

    // 64px is as large as we go on some sprites
    $sval = $width;
    if ($sval > 64) {
        $sval = 64;
    }
    $rval = ($width * 2);
    if ($rval > 64) {
        $rval = 64;
    }

    // Modules
    $cdir = 'white';
    if ($color !== 'white') {
        $cdir = 'black';
    }
    jrCore_start_timer('filesystem');
    $_icons = glob(APP_DIR . "/modules/*/img/icons_{$cdir}/*.png");
    jrCore_stop_timer('filesystem');
    if ($_icons && is_array($_icons)) {
        foreach ($_icons as $k => $v) {
            if (strpos($v, '-release-')) {
                unset($_icons[$k]);
                continue;
            }
            list($module,) = explode('/', str_replace(APP_DIR . '/modules/', '', $v), 2);
            if (!jrCore_module_is_active($module)) {
                unset($_icons[$k]);
                continue;
            }
            $name          = basename($v);
            $_icons[$name] = $v;
            unset($_icons[$k]);
            $swidth += $sval;
            $rwidth += $rval;
        }
    }
    // Override core with skin
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if (is_dir(APP_DIR . "/skins/{$skin}/img/icons_{$cdir}")) {
        jrCore_start_timer('filesystem');
        $_skin = glob(APP_DIR . "/skins/{$skin}/img/icons_{$cdir}/*.png");
        jrCore_stop_timer('filesystem');
        if (is_array($_skin)) {
            foreach ($_skin as $v) {
                $name = basename($v);
                if (!isset($_icons[$name])) {
                    $swidth += $sval;
                    $rwidth += $rval;
                }
                $_icons[$name] = $v;
            }
        }
        unset($_skin);
    }
    asort($_icons);

    // Now create our Sprite image
    $exif = false;
    if (function_exists('exif_imagetype')) {
        $exif = true;
    }

    $ssprite = imagecreatetruecolor($swidth, $sval);
    $rsprite = imagecreatetruecolor($rwidth, $rval);
    imagealphablending($ssprite, false);
    imagealphablending($rsprite, false);
    imagesavealpha($ssprite, true);
    imagesavealpha($rsprite, true);
    $sleft = 0;
    $rleft = 0;

    // Normal
    $curl = jrCore_get_module_url('jrCore');
    $surl = jrCore_get_base_url() . "/{$curl}/icon_sprite/{$sval}/{$color}/sprite.png?_v=" . time();
    $css  = ".sprite_icon_{$color}_{$sval}{display:inline-block;width:{$sval}px;height:{$sval}px;}\n";
    $css  .= ".sprite_icon_{$color}_{$sval}_img{background:url('{$surl}') no-repeat top left; height:100%;width:100%;}";

    // Retina / High Density
    $rurl = jrCore_get_base_url() . "/{$curl}/icon_sprite/{$rval}/{$color}/sprite.png?_v=" . time();
    $rcss = "\n@media only screen and (-webkit-min-device-pixel-ratio: 2),\nonly screen and (-o-min-device-pixel-ratio: 3/2),\nonly screen and (min--moz-device-pixel-ratio: 2),\nonly screen and (min-device-pixel-ratio: 2),\nonly screen and (min-resolution: 192dpi),\nonly screen and (min-resolution: 2dppx) {";
    $rcss .= "\n  .sprite_icon_{$color}_{$sval}_img{background:url('{$rurl}') no-repeat top left; height:100%;width:100%;}";

    $r = false;
    $g = false;
    $b = false;
    if ($color != 'black' && $color != 'white') {
        $r = hexdec(substr($color, 0, 2));
        $g = hexdec(substr($color, 2, 2));
        $b = hexdec(substr($color, 4, 2));
    }

    foreach ($_icons as $name => $image) {

        if ($exif && exif_imagetype($image) === false) {
            continue;
        }
        if ($img = imagecreatefrompng($image)) {
            if ($r) {
                // Changing color
                if (!imagefilter($img, IMG_FILTER_COLORIZE, $r, $g, $b)) {
                    continue;
                }
            }
            if (!imagecopyresampled($ssprite, $img, $sleft, 0, 0, 0, $sval, $sval, 64, 64)) {
                continue;
            }
            if (!imagecopyresampled($rsprite, $img, $rleft, 0, 0, 0, $rval, $rval, 64, 64)) {
                continue;
            }

            // Generate CSS
            $nam = str_replace('.png', '', $name);

            // Standard Resolution
            if ($sleft > 0) {
                $css .= "\n.sprite_icon_{$color}_{$sval}_{$nam}{background-position:-{$sleft}px 0}";
            }
            else {
                $css .= "\n.sprite_icon_{$color}_{$sval}_{$nam}{background-position:0 0}";
            }
            // Retina
            if ($rleft > 0) {
                $rcss .= "\n  .sprite_icon_{$color}_{$sval}_{$nam}{background-position:-{$sleft}px 0;";
            }
            else {
                $rcss .= "\n  .sprite_icon_{$color}_{$sval}_{$nam}{background-position:0 0;";
            }
            $rcss .= "background-size:{$swidth}px {$sval}px}";

            $sleft += $sval;
            $rleft += $rval;
        }
    }
    $rcss .= "\n}";
    $dir  = jrCore_get_module_cache_dir($skin);

    // Standard resolution sprites
    jrCore_write_to_file("{$dir}/sprite_{$color}_{$sval}.css", $css . "\n" . $rcss . "\n");
    imagepng($ssprite, "{$dir}/sprite_{$color}_{$sval}.png");
    imagepng($rsprite, "{$dir}/sprite_{$color}_{$rval}.png");
    imagedestroy($ssprite);
    imagedestroy($rsprite);

    return true;
}

/**
 * Get HTML code for a given CSS icon sprite
 * @param $name string Name of CSS Icon to get
 * @param $size int Size (in pixels) for icon
 * @param $class string Additional icon HTML class
 * @param $color string Icon Color
 * @param $id string Icon DOM ID
 * @return string
 */
function jrCore_get_sprite_html($name, $size = null, $class = null, $color = null, $id = null)
{
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if (is_null($size)) {
        $_tmp = jrCore_get_registered_module_features('jrCore', 'icon_size');
        if (isset($_tmp[$skin])) {
            $size = array_keys($_tmp[$skin]);
            $size = reset($size);
            if (!is_numeric($size)) {
                $size = 32;
            }
        }
        else {
            $size = 32;
        }
    }
    if (is_null($color)) {
        $color = jrCore_get_skin_icon_color();
    }
    else {
        switch (strtolower($color)) {
            case 'ffffff':
                $color = 'white';
                break;
            case '000000':
                $color = 'black';
                break;
        }
    }

    // Let other modules handle icons
    $_data = array(
        'icon'  => $name,
        'size'  => $size,
        'class' => $class,
        'color' => $color,
        'id'    => $id
    );
    $_data = jrCore_trigger_event('jrCore', 'get_icon_html', $_data);
    if (!empty($_data['html'])) {
        return $_data['html'];
    }

    // We have not included this size yet on our page - bring in now
    $dir = jrCore_get_module_cache_dir($skin);
    if (!is_file("{$dir}/sprite_{$color}_{$size}.css")) {
        jrCore_create_css_sprite($skin, $color, $size);
    }
    $out = '';
    $key = "core_get_sprite_html_{$size}_{$color}";
    if (!jrCore_get_flag($key)) {
        $mtm = filemtime("{$dir}/sprite_{$color}_{$size}.css");
        $out = '<link rel="stylesheet" property="stylesheet" href="' . jrCore_get_base_url() . '/' . jrCore_get_module_url('jrCore') . '/icon_css/' . $size . '/' . $color . '/?_v=' . $mtm . '">';
        jrCore_set_flag($key, 1);
    }

    $did = '';
    if (!empty($id)) {
        $did = " id=\"{$id}\"";
    }
    $cls = '';
    if (!empty($class) > 0) {
        $cls = " {$class}";
    }
    // See if we are doing a highlighted icon
    if (strpos($name, '-hilighted')) {
        $name = str_replace('-hilighted', '', $name);
        $out  .= "<span{$did} class=\"sprite_icon sprite_icon_hilighted sprite_icon_{$size} sprite_icon_{$color}_{$size}{$cls}\">";
    }
    // Are we doing a disabled icon
    elseif (strpos($name, '-disabled')) {
        $name = str_replace('-disabled', '', $name);
        $out  .= "<span{$did} class=\"sprite_icon sprite_icon_disabled sprite_icon_{$size} sprite_icon_{$color}_{$size}{$cls}\">";
    }
    else {
        $out .= "<span{$did} class=\"sprite_icon sprite_icon_{$size} sprite_icon_{$color}_{$size}{$cls}\">";
    }
    $out .= "<span class=\"sprite_icon_{$size} sprite_icon_{$color}_{$size} sprite_icon_{$size}_img sprite_icon_{$color}_{$size}_img sprite_icon_{$size}_{$name} sprite_icon_{$color}_{$size}_{$name}\">&nbsp;</span></span>";
    return $out;
}

/**
 * Get HTML for a module icon
 * @param $module string Module name
 * @param $size int Size (width/height)
 * @param null $class
 * @return string
 */
function jrCore_get_module_icon_html($module, $size, $class = null)
{
    global $_mods;
    $siz = (int) $size;
    $cls = '';
    if (!empty($class)) {
        $cls = ' ' . $class;
    }
    $img = false;
    $cat = 'default';
    $ttl = 'default';
    if (isset($_mods[$module])) {
        $cat = (isset($_mods[$module]['module_category'])) ? str_replace(' ', '_', trim(jrCore_str_to_lower($_mods[$module]['module_category']))) : 'default';
        $ttl = (isset($_mods[$module]['module_name'])) ? addslashes($_mods[$module]['module_name']) : 'default';
        // Check for SVG cache
        $c_key = md5("{$module}/{$size}/{$class}");
        if (!$tmp = jrCore_get_local_cache_key($c_key)) {
            if ($tmp = jrCore_file_get_contents(APP_DIR . "/modules/{$module}/icon.svg")) {
                $tmp = base64_encode($tmp);
            }
            else {
                if (is_file(APP_DIR . "/modules/{$module}/icon.png")) {
                    $tmp = 'icon.png';
                }
                else {
                    $tmp = 1;
                }
            }
            jrCore_set_local_cache_key($c_key, $tmp);
        }
        if ($tmp && strlen($tmp) > 10) {
            // We have an SVG Icon
            $img = "data:image/svg+xml;base64,{$tmp}";
            $cls .= ' module_icon_svg';
        }
        elseif ($tmp == 'icon.png') {
            $img = jrCore_get_base_url() . "/modules/{$module}/icon.png?v={$_mods[$module]['module_updated']}";
        }
    }
    if (!$img) {
        // Default
        $c_key = md5("{$module}/{$size}/{$class}/default");
        if (!$tmp = jrCore_get_local_cache_key($c_key)) {
            $tmp = jrCore_file_get_contents(APP_DIR . "/modules/jrCore/img/default_module_icon.svg");
            $tmp = base64_encode($tmp);
            jrCore_set_local_cache_key($c_key, $tmp);
        }
        $img = "data:image/svg+xml;base64,{$tmp}";
    }
    return "<span class=\"module_icon module_icon_{$cat} module_icon_{$module}{$cls}\"><img src=\"{$img}\" alt=\"{$ttl}\" title=\"{$ttl}\" width=\"{$siz}\" height=\"{$siz}\"></span>";
}

/**
 * Get HTML for a skin icon
 * @param $skin string Skin name
 * @param $size int Size (width/height)
 * @param null $class
 * @return string
 */
function jrCore_get_skin_icon_html($skin, $size, $class = null)
{
    global $_mods;
    $_md = jrCore_skin_meta_data($skin);
    $siz = (int) $size;
    $cls = '';
    if (!is_null($class) && strlen($class) > 0) {
        $cls = ' ' . $class;
    }
    if (!$_md || !is_array($_md)) {
        $ttl = 'default';
        $url = jrCore_get_module_url('jrImage');
        $img = jrCore_get_base_url() . "/{$url}/img/module/jrCore/default_skin_icon.png?v={$_mods['jrCore']['module_updated']}";
    }
    else {
        $ttl = addslashes($_md['title']);
        $url = jrCore_get_module_url('jrCore');
        $tmp = @filemtime(APP_DIR . "/skins/{$skin}/icon.png");
        $img = jrCore_get_base_url() . "/{$url}/icon/skin/{$skin}?v={$tmp}";
    }
    return "<span class=\"module_icon skin_icon{$cls}\" style=\"width:{$siz}px\"><img src=\"{$img}\" alt=\"{$ttl}\" title=\"{$ttl}\" width=\"{$siz}\" height=\"{$siz}\"></span>";
}

/**
 * Get the configured icon size for a skin
 * @param int $default Size to use if unable to determine size
 * @return int
 */
function jrCore_get_skin_icon_size($default = 32)
{
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $_tmp = jrCore_get_registered_module_features('jrCore', 'icon_size');
    if ($_tmp && isset($_tmp[$skin])) {
        $size = array_keys($_tmp[$skin]);
        $size = reset($size);
        if (is_numeric($size)) {
            $default = (int) $size;
        }
    }
    return $default;
}

/**
 * Get the skin icon color
 * @param string $default
 * @return string
 */
function jrCore_get_skin_icon_color($default = 'black')
{
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $_tmp = jrCore_get_registered_module_features('jrCore', 'icon_color');
    if (!empty($_tmp[$skin])) {
        $color = array_keys($_tmp[$skin]);
        return reset($color);
    }
    return $default;
}

/**
 * Get the activity indicator color
 * @return string|false
 */
function jrCore_get_skin_spinner_color()
{
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $_tmp = jrCore_get_registered_module_features('jrCore', 'spinner_color');
    if (!empty($_tmp[$skin])) {
        $color = array_keys($_tmp[$skin]);
        return reset($color);
    }
    return false;
}

/**
 * Get activity indicator HTML
 * @param string $id DOM ID of indicator
 * @param int $size Width/Height in pixel or Percent
 * @param string $color HEX color
 * @param string $width Width of border in pixels
 * @param bool $display by default indicator is hidden - set to TRUE to display on page load
 * @return mixed
 */
function jrCore_get_activity_indicator($id, $size = null, $color = null, $width = '3px', $display = false)
{
    if (is_null($size)) {
        $size = jrCore_get_skin_icon_size();
    }
    if (is_null($color)) {
        if (!$color = jrCore_get_skin_spinner_color()) {
            $color = jrCore_get_skin_icon_color();
            if (strlen($color) === 6) {
                // Check if this is a hex color without the #
                if (ctype_xdigit("{$color}")) {
                    $color = '#' . $color;
                }
            }
        }
    }
    $_rp = array(
        'id'      => $id,
        'size'    => (int) $size . 'px',
        'color'   => $color,
        'width'   => $width,
        'display' => ($display) ? '' : 'display:none'
    );
    return jrCore_parse_template('activity_indicator.tpl', $_rp, 'jrCore');
}

/**
 * jrCore_skin_meta_data - get meta data for a skin
 * @param string $skin skin string skin name
 * @return array|false
 */
function jrCore_skin_meta_data($skin)
{
    $func = "{$skin}_skin_meta";
    if (!function_exists($func) && is_file(APP_DIR . "/skins/{$skin}/include.php")) {
        require_once APP_DIR . "/skins/{$skin}/include.php";
    }
    if (!function_exists($func)) {
        return false;
    }
    $_tmp = $func();
    if ($_tmp && is_array($_tmp)) {
        return $_tmp;
    }
    return false;
}

/**
 * Add a skin's global config to $_conf
 * @param $skin string Skin name to load
 * @param $_conf array Global Config
 * @return array
 */
function jrCore_load_skin_config($skin, $_conf)
{
    $key = "loaded_skin_config_{$skin}";
    if (!jrCore_get_flag($key)) {
        $tbl = jrCore_db_table_name('jrCore', 'setting');
        $req = "SELECT CONCAT_WS('_', `module`, `name`) AS k, `value` AS v FROM {$tbl} WHERE `module` = '" . jrCore_db_escape($skin) . "'";
        $_cf = jrCore_db_query($req, 'k', false, 'v');
        if ($_cf && is_array($_cf)) {
            foreach ($_cf as $k => $v) {
                $_conf[$k] = $v;
            }
        }
        jrCore_set_flag($key, 1);
    }
    return $_conf;
}

/**
 * Get installed skins
 * @param bool $include_skin_name
 * @return array
 */
function jrCore_get_skins($include_skin_name = false)
{
    $key = 'jrcore_get_skins_' . intval($include_skin_name);
    if ($tmp = jrCore_get_flag($key)) {
        return $tmp;
    }
    $_sk = array();
    jrCore_start_timer('filesystem');
    if ($h = opendir(APP_DIR . '/skins')) {
        while (($file = readdir($h)) !== false) {
            if ($file == '.' || $file == '..' || strpos($file, '-release-')) {
                continue;
            }
            elseif (is_file(APP_DIR . "/skins/{$file}/include.php")) {
                if ($include_skin_name) {
                    if ($_mta = jrCore_skin_meta_data($file)) {
                        $_sk[$file] = $_mta['title'];
                    }
                }
                else {
                    $_sk[$file] = $file;
                }
            }
        }
        closedir($h);
    }
    jrCore_stop_timer('filesystem');
    if ($include_skin_name) {
        natcasesort($_sk);
    }
    else {
        ksort($_sk);
    }
    jrCore_set_flag($key, $_sk);
    return $_sk;
}

/**
 * Delete an existing Skin Menu Item
 * @param $module string Module Name that created the Skin Menu Item
 * @param $unique string Unique name/tag for the Skin Menu Item
 * @return int|false
 */
function jrCore_delete_skin_menu_item($module, $unique)
{
    $tbl = jrCore_db_table_name('jrCore', 'menu');
    $req = "DELETE FROM {$tbl} WHERE menu_module = '" . jrCore_db_escape($module) . "' AND menu_unique = '" . jrCore_db_escape($unique) . "' LIMIT 1";
    return jrCore_db_query($req, 'COUNT');
}

/**
 * Get a cached template or parse a template
 * @param string $directory Module or Skin directory
 * @param string $template Template file name
 * @param array $_replace Template Replacement variables
 * @return false|mixed
 */
function jrCore_get_parsed_or_cached_template($directory, $template, $_replace)
{
    $key = "{$directory}-{$template}-" . json_encode($_replace);
    if (!$out = jrCore_is_cached('jrCore', $key)) {
        $out = jrCore_parse_template($template, $_replace, $directory);
        jrCore_add_to_cache('jrCore', $key, $out);
    }
    return $out;
}

/**
 * Parses a template and returns the result
 *
 * <br><p>
 * This is one of the main functions used to move data from php out to the smarty templates.
 * anything you put in the $_rep array becomes a template variable.  so if you have $_rep['foo'] = 'bar'; then you can call
 *  &#123;$foo} in the template to produce 'bar' output.
 * </p><br>
 * <p>
 *  If you have a template in your module, the system will look for it in the /templates directory.  so call it with <br>
 *  <i>$out = jrCore_parse_template('embed.tpl',null,'jrDisqus');</i>  will call /modules/jrDisqus/templates/embed.tpl
 * </p><br>
 * <p>
 *  Skins can over-ride the modules template by defining their own version of it. so
 * /module/jrDisqus/templates/embed.tpl can be over-ridden by the skin by defining:
 * /skin/jrElastic/jrDisqus_embed.tpl
 * </p>
 * @param string $template Name of template
 * @param array $_rep (Optional) replacement variables for use in template.
 * @param string $directory default active skin directory, module directory for module/templates
 * @param bool $disable_override - set to TRUE to disable skin template override
 * @return mixed
 */
function jrCore_parse_template($template, $_rep = null, $directory = null, $disable_override = false)
{
    global $_conf, $_post, $_user, $_mods;

    // make sure we get smarty included
    jrCore_start_timer('template');
    if (!isset($GLOBALS['smarty_object']) || !is_object($GLOBALS['smarty_object'])) {

        $dir = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');

        // create Smarty global object
        $GLOBALS['smarty_object'] = new Smarty;
        $GLOBALS['smarty_object']->setCompileDir(APP_DIR . '/data/cache/' . $dir);
        $GLOBALS['smarty_object']->merge_compiled_includes = true;
        $GLOBALS['smarty_object']->compile_check           = false;

        // Get plugin directories
        $_dir = array(APP_DIR . '/modules/jrCore/contrib/smarty4/libs/plugins');
        $GLOBALS['smarty_object']->setPluginsDir($_dir);
        $GLOBALS['smarty_object']->error_reporting = (E_ALL ^ E_NOTICE ^ E_WARNING ^ E_DEPRECATED);

        // If we are running in developer mode, make sure compiled template is removed on every call
        if (jrCore_is_developer_mode()) {
            $GLOBALS['smarty_object']->force_compile = true;
        }
    }
    else {
        $GLOBALS['smarty_object']->clearAllAssign();
    }

    // Our template directory
    if (is_null($directory)) {
        $directory = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    }

    // Our Data
    $_data = array();
    if ($_rep && is_array($_rep)) {
        $_data = $_rep;
    }
    $_data['page_title']            = jrCore_get_flag('jrcore_html_page_title');
    $_data['jamroom_dir']           = APP_DIR;
    $_data['jamroom_url']           = $_conf['jrCore_base_url'];
    $_data['current_url']           = jrCore_get_current_url();
    $_data['_conf']                 = $_conf;
    $_data['_post']                 = $_post;
    $_data['_mods']                 = $_mods;
    $_data['_user']                 = (isset($_SESSION)) ? $_SESSION : $_user;
    $_data['jr_template']           = $template;
    $_data['jr_template_type']      = 'string';
    $_data['jr_template_directory'] = $directory;
    $_data['jr_template_no_ext']    = str_replace('.tpl', '', $template);

    jrCore_set_flag('jrcore_parse_template_active_template', $template);
    jrCore_set_flag('jrcore_parse_template_active_directory', $directory);

    // Remove User and MySQL info - we don't want this to ever leak into a template
    unset($_data['_user']['user_password'], $_data['_user']['user_old_password'], $_data['_user']['user_forgot_key']);
    unset($_data['_conf']['jrCore_db_host'], $_data['_conf']['jrCore_db_user'], $_data['_conf']['jrCore_db_pass'], $_data['_conf']['jrCore_db_name'], $_data['_conf']['jrCore_db_port']);

    $t_id = "{$directory}^{$template}";
    if (strpos($template, '.tpl') && jrCore_checktype($template, 'file_name')) {
        if (jrCore_local_cache_is_enabled()) {
            $file = 'string: ' . jrCore_get_apc_template_file($template, $directory, false, $disable_override);
        }
        else {
            // No APCu - fall back to file system based template cached (slower)
            $file                      = jrCore_get_template_file($template, $directory, false, $disable_override);
            $_data['jr_template_type'] = 'file';
        }
        $tkey = $t_id;
    }
    else {
        $file = 'string:' . $template;
        $tkey = md5($template);
        $t_id = false;
    }

    // Lastly, see if we have already shown this template in this process
    $_data['template_unique_key']    = $tkey;
    $_data['template_already_shown'] = '1';
    if (!jrCore_get_flag("template_shown_{$tkey}")) {
        jrCore_set_flag("template_shown_{$tkey}", 1);
        $_data['template_already_shown'] = '0';
    }

    // Trigger for additional replacement vars
    $_data = jrCore_trigger_event('jrCore', 'template_variables', $_data, $_rep);

    // Take care of additional page elements in meta/footer
    switch ($template) {
        case 'meta.tpl':
        case 'footer.tpl':
            if ($_tmp = jrCore_get_flag('jrcore_page_elements')) {
                $_data = array_merge($_data, $_tmp);
            }
            break;
    }
    $GLOBALS['smarty_object']->assign($_data);
    if ($t_id) {
        $GLOBALS['smarty_object']->compile_id = $t_id;
    }

    try {
        ob_start();
        $GLOBALS['smarty_object']->display($file); // DO NOT USE fetch() here!
        $out = ob_get_clean();
    }
    catch (Exception $e) {

        // No corruption on string: templates
        jrCore_record_event('template_error');
        if (strpos($file, 'string:') !== 0) {

            // Rebuild this template
            $GLOBALS['smarty_object']->force_compile = true;
            $GLOBALS['smarty_object']->clearCache($file);
            $GLOBALS['smarty_object']->clearCompiledTemplate($file);

            // Delete the existing template
            $cdr = jrCore_get_module_cache_dir('jrCore');
            $tpl = basename($file);
            jrCore_unlink("{$cdr}/{$tpl}");

            // Rebuild template
            $file = jrCore_get_template_file($template, $directory, false, $disable_override);

            try {
                ob_start();
                $GLOBALS['smarty_object']->display($file);
                $out = ob_get_clean();
            }
            catch (Exception $e) {
                jrCore_logger('MAJ', "core: error rebuilding corrupt template file: {$tpl}");
                $out = '';
            }
            if (strlen($out) > 0) {
                if (!jrCore_get_flag("jrcore_corrupt_{$tpl}")) {
                    jrCore_logger('MAJ', "core: deleted and rebuilt corrupt template file: {$tpl}");
                    jrCore_set_flag("jrcore_corrupt_{$tpl}", 1);
                }
            }
            $GLOBALS['smarty_object']->force_compile = false;

        }
        else {
            $out = '';
        }
    }
    // Remove old media play keys
    if (strpos($out, '[jrCore_media_play_key]')) {
        $out = str_replace('[jrCore_media_play_key]', 1, $out);
    }
    jrCore_stop_timer('template');
    return jrCore_trigger_event('jrCore', 'parsed_template', $out, $_data);
}

/**
 * Returns the proper template to use for display.  Will also create/maintain the template cache
 * @note: This function is slower will only be used on systems without APCu
 * @param string $template Template file to get
 * @param string $directory Name of module or skin that the template belongs to
 * @param bool $reset Set to TRUE to reset the template cache
 * @param bool $disable_override Set to TRUE to disable Skin template override of module template
 * @return string
 */
function jrCore_get_template_file($template, $directory, $reset = false, $disable_override = false)
{
    global $_conf;
    // Check for skin override
    if (!$disable_override && is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/{$directory}_{$template}")) {
        $template  = "{$directory}_{$template}";
        $directory = $_conf['jrCore_active_skin'];
    }
    if (is_null($directory) || $directory === false || strlen($directory) === 0) {
        $directory = $_conf['jrCore_active_skin'];
    }

    // Trigger template event
    $_tmp = array(
        'template'  => $template,
        'directory' => $directory
    );
    $_tmp = jrCore_trigger_event('jrCore', 'template_file', $_tmp);
    if (!empty($_tmp['template']) && $_tmp['template'] != $template || isset($_tmp['directory']) && $_tmp['directory'] != $directory) {
        $template  = $_tmp['template'];
        $directory = $_tmp['directory'];
    }

    // We check for our "cached" template, as that will be the proper one to display
    // depending on if the admin has customized the template or not.  If we do NOT
    // have the template in our cache, we have to go get it.
    $cdir = jrCore_get_module_cache_dir('jrCore');
    $hash = md5($_conf['jrCore_active_skin'] . '-' . $directory . '-' . $template);
    $file = "{$cdir}/{$hash}^{$directory}^{$template}";
    if (!is_file($file) || $reset || $_conf['jrCore_default_cache_seconds'] == '0' || jrCore_is_developer_mode()) {

        $_rt = false;
        if (jrCore_get_config_value('jrCore', 'disable_db_templates', 'off') == 'off') {
            $_rt = jrCore_get_flag("jrcore_get_template_cache");
            if (!$_rt) {
                // We need to check for a customized version of this template
                $tbl = jrCore_db_table_name('jrCore', 'template');
                $req = "SELECT CONCAT_WS('_',template_module,template_name) AS template_name, template_body FROM {$tbl} WHERE template_active = '1'";
                $_rt = jrCore_db_query($req, 'template_name');
                if ($_rt && is_array($_rt)) {
                    jrCore_set_flag('jrcore_get_template_cache', $_rt);
                }
                else {
                    jrCore_set_flag('jrcore_get_template_cache', 1);
                }
            }
        }
        $key = "{$directory}_{$template}";
        if ($_rt && is_array($_rt) && isset($_rt[$key]) && !empty($_rt[$key]['template_body'])) {
            if (!jrCore_write_to_file($file, $_rt[$key]['template_body'])) {
                jrCore_notice('Error', "unable to write to template cache directory: data/cache/jrCore", false);
            }
        }
        // Check for skin template
        elseif (is_file(APP_DIR . "/skins/{$directory}/{$template}")) {
            if (!copy(APP_DIR . "/skins/{$directory}/{$template}", $file)) {
                jrCore_notice('Error', "unable to copy skins/{$directory}/{$template} to template cache directory: data/cache/jrCore", false);
            }
        }
        // Module template
        elseif (is_file(APP_DIR . "/modules/{$directory}/templates/{$template}")) {
            if (!copy(APP_DIR . "/modules/{$directory}/templates/{$template}", $file)) {
                jrCore_notice('Error', "unable to copy modules/{$directory}/templates/{$template} to template cache directory: data/cache/jrCore", false);
            }
        }
        else {
            $_tmp  = array(
                'template'  => $template,
                'directory' => $directory
            );
            $_data = jrCore_trigger_event('jrCore', 'tpl_404', $_tmp);
            if (empty($_data['file'])) {
                jrCore_notice('Error', "invalid template: {$template}, or template directory: {$directory}", false);
            }
            if (!copy($_data['file'], $file)) {
                jrCore_notice('Error', "unable to copy " . str_replace(APP_DIR . '/', '', $_data['file']) . " to template cache directory: data/cache/jrCore", false);
            }
        }
    }
    return $file;
}

/**
 * Reset the APC template cache for a specific module/template
 * @param string $template Template name
 * @param string $directory Module | Skin directory
 * @return bool
 */
function jrCore_reset_apc_template_file($template, $directory)
{
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $key1 = 't:' . md5("{$skin}/{$template}/{$directory}/1");
    $key2 = 't:' . md5("{$skin}/{$template}/{$directory}/0");
    jrCore_delete_local_cache_key($key1);
    jrCore_delete_local_cache_key($key2);
    return true;
}

/**
 * Returns the proper template to use for display - uses APC for caching
 * @param string $template Template file to get
 * @param string $directory Name of module or skin that the template belongs to
 * @param bool $reset Set to TRUE to reset the template cache
 * @param bool $disable_override Set to TRUE to disable Skin template override of module template
 * @return mixed Returns full file path on success, bool false on failure
 */
function jrCore_get_apc_template_file($template, $directory, $reset = false, $disable_override = false)
{
    // Are we already cached in APC?
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $ckey = 't:' . md5("{$skin}/{$template}/{$directory}/" . intval($disable_override));
    if (!$reset) {
        if ($file = jrCore_get_local_cache_key($ckey)) {
            return $file;
        }
    }

    // Check for skin override
    if (!$disable_override && $directory != $skin && is_file(APP_DIR . "/skins/{$skin}/{$directory}_{$template}")) {
        $template  = "{$directory}_{$template}";
        $directory = $skin;
    }

    // Trigger template event
    $_tmp = array(
        'template'  => $template,
        'directory' => $directory
    );
    $_tmp = jrCore_trigger_event('jrCore', 'template_file', $_tmp);
    if (!empty($_tmp['template']) && $_tmp['template'] != $template) {
        // We have been changed by a listener
        $template = $_tmp['template'];
    }
    if (!empty($_tmp['directory']) && $_tmp['directory'] != $directory) {
        // We have been changed by a listener
        $directory = $_tmp['directory'];
    }
    if (empty($directory)) {
        $directory = $skin;  // Must have a good directory
    }
    $_rt = array();
    $sec = jrCore_get_config_value('jrCore', 'default_cache_seconds', 300);
    if (jrCore_get_config_value('jrCore', 'disable_db_templates', 'off') == 'off') {
        // We need to check for a customized version of this template
        $key = 'jrcore_get_template_cache';
        if (!$_rt = jrCore_get_local_cache_key($key)) {
            // @note: this get_flag call helps when running in Developer Mode since templates will not change mid-process
            if (!$_rt = jrCore_get_flag($key)) {
                $tbl = jrCore_db_table_name('jrCore', 'template');
                $req = "SELECT CONCAT_WS('_', template_module, template_name) AS template_name, template_body FROM {$tbl} WHERE template_active = 1";
                $_rt = jrCore_db_query($req, 'template_name');
                if (!$_rt || !is_array($_rt)) {
                    $_rt = 'no_templates';
                }
                jrCore_set_flag($key, $_rt);
            }
            jrCore_set_local_cache_key($key, $_rt, $sec);
        }
        if (!is_array($_rt)) {
            $_rt = array();
        }
    }
    // Check for Database Override
    if (!empty($_rt["{$directory}_{$template}"]['template_body'])) {
        $file = $_rt["{$directory}_{$template}"]['template_body'];
    }
    // Check for skin template
    elseif (is_file(APP_DIR . "/skins/{$directory}/{$template}")) {
        $file = jrCore_file_get_contents(APP_DIR . "/skins/{$directory}/{$template}");
    }
    // Module template
    elseif (is_file(APP_DIR . "/modules/{$directory}/templates/{$template}")) {
        $file = jrCore_file_get_contents(APP_DIR . "/modules/{$directory}/templates/{$template}");
    }
    else {
        $_tmp  = array(
            'template'  => $template,
            'directory' => $directory
        );
        $_data = jrCore_trigger_event('jrCore', 'tpl_404', $_tmp);
        if (empty($_data['file']) || !is_file($_data['file'])) {
            jrCore_notice('Error', "invalid template: {$template}, or template directory: {$directory}", false);
        }
        $file = jrCore_file_get_contents($_data['file']);
    }
    jrCore_set_local_cache_key($ckey, $file, $sec);
    return $file;
}

/**
 * Get an array of cached skin templates
 * @return array
 */
function jrCore_get_active_skin_cached_template_array()
{
    global $_mods;
    $_out = array();
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if ($_tpl = glob(APP_DIR . "/data/cache/{$skin}/*.php")) {
        $skin_name = jrCore_skin_meta_data($skin);
        $skin_name = $skin_name['title'];
        foreach ($_tpl as $file) {
            // jrCore_activity_indicator_tpl^5bc8cb50f44f13bfebf2d1fc80b5819b740051f3_1.string.php
            $name = basename($file);
            list($tpl,) = explode('^', $name, 2);
            list($mod, $tpl) = explode('_', $tpl, 2);
            if (isset($_mods[$mod]) || $mod == $skin) {
                $mod_name = (isset($_mods[$mod])) ? $_mods[$mod]['module_name'] : $skin_name;
                if (!isset($_out[$mod_name])) {
                    $_out[$mod_name] = array();
                }
                $_out[$mod_name]["{$mod}_{$tpl}"] = str_replace('_tpl', '.tpl', $tpl);
            }
        }
    }
    if (!empty($_out)) {
        foreach ($_out as $k => $v) {
            natcasesort($_out[$k]);
        }
        ksort($_out, SORT_NATURAL);
    }
    return $_out;
}

/**
 * Returns a 404 page not found
 */
function jrCore_page_not_found()
{
    global $_post;
    jrCore_trigger_event('jrCore', '404_not_found', $_post);

    $_ln = jrUser_load_lang_strings();
    jrCore_page_title($_ln['jrCore'][84]);

    if (!$out = jrCore_is_cached('jrCore', '404_not_found_template', false, false, true, false)) {
        $out = jrCore_parse_template('404.tpl', array());
        jrCore_add_to_cache('jrCore', '404_not_found_template', $out, 0, 0, false, false, false);
    }
    $out = jrCore_trigger_event('jrCore', 'view_results', $out);

    jrCore_set_custom_header('HTTP/1.0 404 Not Found');
    jrCore_send_response_and_detach($out);
}

/**
 * Format Custom Skin CSS
 * @param $_custom array CSS Rules to format
 * @param $pretty bool set to TRUE to prettify
 * @return string
 */
function jrCore_format_custom_css($_custom, $pretty = false)
{
    $out = '';
    $chr = '';
    $spc = '';
    if ($pretty) {
        $chr = "\n";
        $spc = '    ';
    }
    if ($_custom && is_array($_custom)) {
        foreach ($_custom as $sel => $_rules) {
            if (strpos($sel, '@media') === 0) {
                $out .= $sel . " {{$chr}";
                foreach ($_rules as $ms => $_mr) {
                    $out .= $spc . $ms . " {{$chr}";
                    $_cr = array();
                    foreach ($_mr as $k => $v) {
                        if (!strpos($v, '!important')) {
                            $_cr[] = $spc . $spc . $k . ':' . $v . " !important;{$chr}";
                        }
                        else {
                            $_cr[] = $spc . $spc . $k . ':' . $v . $chr;
                        }
                    }
                    $out .= implode('', $_cr) . $spc . "}\n";
                }
                $out .= "}\n{$chr}";
            }
            else {
                $out .= $sel . " {{$chr}";
                $_cr = array();
                foreach ($_rules as $k => $v) {
                    if (!strpos($v, '!important')) {
                        $_cr[] = $spc . $k . ':' . $v . " !important;{$chr}";
                    }
                    else {
                        $_cr[] = $spc . $k . ':' . $v . $chr;
                    }
                }
                $out .= implode('', $_cr) . "}\n{$chr}";
            }
        }
    }
    return $out;
}

/**
 * Create a new master CSS files from module and skin CSS files
 * @param string $skin Skin to create CSS file for
 * @param bool $skip_trigger
 * @return string Returns MD5 checksum of CSS contents
 */
function jrCore_create_master_css($skin, $skip_trigger = false)
{
    global $_conf, $_mods;

    // Make sure we get a good skin
    if (empty($skin) || !is_dir(APP_DIR . "/skins/{$skin}")) {
        return false;
    }

    // We only rebuild MAX once every 5 seconds
    $key = md5(APP_DIR . $skin . 'rebuild_css');
    if (jrCore_get_local_cache_key($key)) {
        // Built within last 5 seconds
        if ($sum = jrCore_get_config_value('jrCore', "{$skin}_css_version", false)) {
            return $sum;
        }
    }
    jrCore_set_local_cache_key($key, 1, 5);
    jrCore_record_event('rebuild_css');

    $out = '';
    // First - round up any custom CSS from modules
    $_tm = jrCore_get_registered_module_features('jrCore', 'css');
    if ($_tm && is_array($_tm)) {
        $_tm = jrCore_trigger_event('jrCore', 'included_css', $_tm);
        ksort($_tm);
        foreach ($_tm as $mod => $_entries) {
            if (!jrCore_module_is_active($mod) || !is_dir(APP_DIR . "/modules/{$mod}")) {
                // Skin gets added below so it can override everything it needs
                continue;
            }
            foreach ($_entries as $script => $ignore) {
                if (strpos($script, 'http') === 0 || strpos($script, '//') === 0) {
                    continue;
                }
                if (strpos($script, APP_DIR) !== 0) {
                    $script = APP_DIR . "/modules/{$mod}/css/{$script}";
                }
                if (is_file(APP_DIR . "/skins/{$skin}/css/{$mod}_{$script}")) {
                    $script = APP_DIR . "/skins/{$skin}/css/{$mod}_{$script}";
                }
                // Developer mode OR already minimized
                if (strpos($script, '.min') || jrCore_is_developer_mode()) {
                    $out .= "\n/* " . str_replace(APP_DIR . '/', '', $script) . " */\n";
                    $out .= "\n\n" . jrCore_file_get_contents($script);
                }
                else {
                    $o    = false;
                    $_tmp = @file($script);
                    if ($_tmp && is_array($_tmp)) {
                        foreach ($_tmp as $line) {
                            $line = trim($line);
                            // check for start of comment
                            if (strpos($line, '/*') === 0 && !$o) {
                                if (!strpos(' ' . $line, '*/')) {
                                    // start of multi-line comment
                                    $o = true;
                                }
                                continue;
                            }
                            if ($o) {
                                // We're still in a comment - see if we are closing
                                if (strpos(' ' . $line, '*/')) {
                                    // Closed - continue
                                    $o = false;
                                }
                                continue;
                            }
                            if (strlen($line) > 0) {
                                $out .= $line;
                            }
                        }
                    }
                }
                $out .= "\n";
            }
        }
    }

    // Skin last (so it can override modules if needed)
    if (isset($_tm[$skin]) && is_array($_tm[$skin])) {
        ksort($_tm);
        foreach ($_tm[$skin] as $script => $ignore) {

            if (strpos($script, 'http') === 0 || strpos($script, '//') === 0) {
                // full URLs to external sources are handled at registration time
                continue;
            }
            if (strpos($script, APP_DIR) !== 0) {
                $script = APP_DIR . "/skins/{$skin}/css/{$script}";
            }
            if (jrCore_is_developer_mode()) {
                $out .= "\n/* " . str_replace(APP_DIR . '/', '', $script) . " */\n";
                $out .= "\n\n" . jrCore_file_get_contents($script);
            }
            else {
                $o    = false;
                $_tmp = @file($script);
                if ($_tmp && is_array($_tmp)) {
                    foreach ($_tmp as $line) {
                        $line = trim($line);
                        // check for start of comment
                        if (strpos($line, '/*') === 0 && !$o) {
                            if (!strpos(' ' . $line, '*/')) {
                                // start of multi-line comment
                                $o = true;
                            }
                            continue;
                        }
                        if ($o) {
                            // We're still in a comment - see if we are closing
                            if (strpos(' ' . $line, '*/')) {
                                // Closed - continue
                                $o = false;
                            }
                            continue;
                        }
                        if (strlen($line) > 0) {
                            $out .= $line;
                        }
                    }
                }
            }
            $out .= "\n";
        }
    }

    // Next, get our customized style from the database
    $tbl = jrCore_db_table_name('jrCore', 'skin');
    $req = "SELECT skin_custom_css, skin_direct_css FROM {$tbl} WHERE skin_directory = '" . jrCore_db_escape($skin) . "'";
    $_rt = jrCore_db_query($req, 'SINGLE', false, null, false);
    if (!empty($_rt['skin_custom_css'])) {
        $_custom = json_decode($_rt['skin_custom_css'], true);
        if ($_custom && is_array($_custom)) {
            $out .= jrCore_format_custom_css($_custom);
        }
    }
    // Custom raw CSS
    if (!empty($_rt['skin_direct_css'])) {
        $out .= "\n{$_rt['skin_direct_css']}";
    }

    $url = jrCore_get_base_url();
    $prt = jrCore_get_server_protocol();
    if ($prt === 'https') {
        $url = str_replace('http:', 'https:', $url);
    }
    // Save file
    $sum = md5($out);
    $_rp = array(
        '{$jamroom_url}' => $url,
        '/* @ignore */'  => '',
        ': '             => ':',
        ', '             => ','
    );
    $crl = jrCore_get_module_url('jrImage');
    foreach ($_tm as $mod => $_entries) {
        if (isset($_mods[$mod]['module_url'])) {
            $_rp['{$' . $mod . '_img_url}'] = "{$url}/{$crl}/img/module/{$mod}";
        }
        else {
            $_rp['{$' . $mod . '_img_url}'] = "{$url}/{$crl}/img/skin/{$mod}";
        }
    }
    $out = "/* {$_conf['jrCore_system_name']} css " . date('r') . " */\n" . str_replace(array_keys($_rp), $_rp, $out);
    $cdr = jrCore_get_module_cache_dir($skin);

    // Our SSL version of the CSS file is prefixed with an "S".
    if ($prt === 'https') {
        jrCore_write_to_file("{$cdr}/S{$sum}.css", $out, true);
    }
    else {
        jrCore_write_to_file("{$cdr}/{$sum}.css", $out, true);
    }

    // Trigger our CSS event
    if (!$skip_trigger) {
        jrCore_trigger_event('jrCore', 'create_master_css', $_tm);
    }

    // We need to store the MD5 of this file in the settings table - thus
    // we don't have to look it up on each page load, and we can then set
    // a VERSION on the css so our visitors will immediately see any CSS
    // changes without having to worry about a cached old version
    $_field = array(
        'name'     => "{$skin}_css_version",
        'type'     => 'hidden',
        'validate' => 'md5',
        'value'    => $sum,
        'default'  => $sum
    );
    jrCore_update_setting('jrCore', $_field);

    $_conf["jrCore_{$skin}_css_version"] = $sum;
    return $sum;
}

/**
 * Create the site level User and Admin javascript
 * @param string $skin Skin to create Javascript file for
 * @param bool $skip_trigger
 * @return string Returns MD5 checksum of Javascript contents
 */
function jrCore_create_master_javascript($skin, $skip_trigger = false)
{
    global $_conf, $_urls;

    // Make sure we get a good skin
    if (empty($skin) || !is_dir(APP_DIR . "/skins/{$skin}")) {
        return false;
    }

    // We only rebuild MAX once every 5 seconds
    $key = md5(APP_DIR . $skin . 'rebuild_js');
    if (jrCore_get_local_cache_key($key)) {
        // Built within last 5 seconds
        if ($sum = jrCore_get_config_value('jrCore', "{$skin}_javascript_version", false)) {
            return $sum;
        }
    }
    jrCore_set_local_cache_key($key, 1, 5);
    jrCore_record_event('rebuild_js');

    // Create our output
    $kurl = jrCore_get_base_url();
    $kprt = jrCore_get_server_protocol();
    if ($kprt === 'https') {
        $kurl = str_replace('http:', 'https:', $kurl);
    }
    $top = "var core_cookie_id = '" . jrUser_unique_install_id() . "';\nvar jrImage_url='" . jrCore_get_module_url('jrImage') . "';\n";
    $out = '';
    $min = '';
    $adm = '';

    // We keep track of the MP5 hash of every JS script we include - this
    // keeps us from including the same JS from different modules
    $_hs = array();
    $_as = array();

    // First - round up any custom JS from modules
    $_tm = jrCore_get_registered_module_features('jrCore', 'javascript');
    // Add in custom module javascript
    if ($_tm && is_array($_tm)) {
        $_tm = jrCore_trigger_event('jrCore', 'included_javascript', $_tm);
        $_ur = array_flip($_urls);
        $_dn = array();
        foreach ($_tm as $mod => $_entries) {
            if ($mod == $skin || !jrCore_module_is_active($mod)) {
                continue;
            }
            if (isset($_ur[$mod])) {
                $url = $_ur[$mod];
                if (!isset($_dn[$url])) {
                    $top       .= "var {$mod}_url='{$url}';\n";
                    $_dn[$url] = 1;
                }
            }
        }
        foreach ($_tm as $mod => $_entries) {
            if ($mod == $skin || !jrCore_module_is_active($mod)) {
                continue;
            }
            foreach ($_entries as $script => $group) {
                // @note Javascript that is external the JR system is loaded in the jrCore_enable_external_javascript() function
                if (strpos($script, 'http') === 0 || strpos($script, '//') === 0 || intval($script) === 1) {
                    continue;
                }
                if (strpos($script, APP_DIR) !== 0) {
                    $script = APP_DIR . "/modules/{$mod}/js/{$script}";
                }
                if (is_file(APP_DIR . "/skins/{$skin}/js/{$mod}_{$script}")) {
                    $script = APP_DIR . "/skins/{$skin}/js/{$mod}_{$script}";
                }
                $tmp = jrCore_file_get_contents($script);
                // This MD5 check ensures we don't include the same JS script 2 times from different modules
                $key = md5($tmp);

                if (!strpos($script, '.min')) {
                    if ($group === 'admin') {
                        if (!isset($_as[$key]) && !isset($_hs[$key])) {
                            $adm       .= $tmp . "\n";
                            $_as[$key] = 1;
                        }
                    }
                    else {
                        if (!isset($_hs[$key])) {
                            $out       .= $tmp . "\n";
                            $_hs[$key] = 1;
                        }
                    }
                }
                else {
                    if ($group === 'admin') {
                        if (!isset($_as[$key]) && !isset($_hs[$key])) {
                            $adm       .= $tmp . "\n";
                            $_as[$key] = 1;
                        }
                    }
                    else {
                        if (!isset($_hs[$key])) {
                            $min       .= $tmp . "\n";
                            $_hs[$key] = 1;
                        }
                    }
                }

            }
        }
    }

    // Skin last (so it can override modules if needed)
    if (isset($_tm[$skin]) && is_array($_tm[$skin])) {
        ksort($_tm);
        foreach ($_tm[$skin] as $script => $group) {

            // @brian we have to skip some JS here no longer used by old skins to
            // prevent the site from breaking when upgrading to Core 6.3.1+
            switch ($script) {
                case 'jquery.mobile.min.js':
                case 'jquery.scrollTo.min.js':
                    continue 2;
            }

            if (strpos($script, 'http') === 0 || strpos($script, '//') === 0) {
                continue;
            }
            if (strpos($script, APP_DIR) !== 0) {
                $script = APP_DIR . "/skins/{$skin}/js/{$script}";
            }
            $tmp = jrCore_file_get_contents($script);
            $key = md5($tmp);
            if (!strpos($script, '.min')) {
                if ($group === 'admin') {
                    if (!isset($_as[$key]) && !isset($_hs[$key])) {
                        $adm       .= $tmp . "\n";
                        $_as[$key] = 1;
                    }
                }
                else {
                    if (!isset($_hs[$key])) {
                        $out       .= $tmp . "\n";
                        $_hs[$key] = 1;
                    }
                }
            }
            else {
                if ($group === 'admin') {
                    if (!isset($_as[$key]) && !isset($_hs[$key])) {
                        $adm       .= $tmp . "\n";
                        $_as[$key] = 1;
                    }
                }
                else {
                    if (!isset($_hs[$key])) {
                        $min       .= $tmp . "\n";
                        $_hs[$key] = 1;
                    }
                }
            }
        }
    }

    // Save file
    $cdr = jrCore_get_module_cache_dir($skin);
    $sum = md5($top . $min . $adm . $out);

    if (!jrCore_is_developer_mode()) {
        // Compress $out
        require_once APP_DIR . '/modules/jrCore/contrib/jsmin/jsmin.php';
        $out = JSMin::minify($out);
        if (strlen($adm) > 0) {
            $adm = JSMin::minify($adm);
        }
    }
    $out = "/* {$_conf['jrCore_system_name']} */\nvar core_system_url='{$kurl}';\nvar core_active_skin='{$skin}';\n{$top}\n{$min}\n{$out}";

    if ($kprt === 'https') {
        jrCore_write_to_file("{$cdr}/S{$sum}.js", $out, true);
    }
    else {
        jrCore_write_to_file("{$cdr}/{$sum}.js", $out, true);
    }

    if (strlen($adm) > 0) {
        if ($kprt === 'https') {
            jrCore_write_to_file("{$cdr}/S{$sum}-admin.js", $adm, true);
        }
        else {
            jrCore_write_to_file("{$cdr}/{$sum}-admin.js", $adm, true);
        }
    }

    // Trigger our Javascript event
    if (!$skip_trigger) {
        jrCore_trigger_event('jrCore', 'create_master_javascript', $_tm);
    }

    // We need to store the MD5 of this file in the settings table - thus
    // we don't have to look it up on each page load, and we can then set
    // a VERSION on the js so our visitors will immediately see any JS
    // changes without having to worry about a cached old version
    $_field = array(
        'name'     => "{$skin}_javascript_version",
        'type'     => 'hidden',
        'validate' => 'md5',
        'value'    => $sum,
        'default'  => $sum
    );
    jrCore_update_setting('jrCore', $_field);

    $_conf["jrCore_{$skin}_javascript_version"] = $sum;
    return $sum;
}

/**
 * Return "order" button for item index
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_item_order_button($module, $_item, $_args, $smarty, $test_only = false)
{
    // See if this module has registered for item order support
    $_tm = jrCore_get_registered_module_features('jrCore', 'item_order_support');
    if (!isset($_tm[$module])) {
        return false;
    }
    if ($test_only) {
        return true;
    }
    $_args['module'] = $module;
    return smarty_function_jrCore_item_order_button($_args, $smarty);
}

/**
 * Return "create" button for an item
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_item_create_button($module, $_item, $_args, $smarty, $test_only = false)
{
    if ($test_only) {
        return true;
    }
    $_args['module'] = $module;
    if (!isset($_args['profile_id'])) {
        $_args['profile_id'] = $_item['_profile_id'];
    }
    return smarty_function_jrCore_item_create_button($_args, $smarty);
}

/**
 * Return "create" button for a bundle
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_bundle_create_button($module, $_item, $_args, $smarty, $test_only = false)
{
    if ($test_only) {
        return true;
    }
    $_args['module'] = $module;
    return smarty_function_jrCore_item_create_button($_args, $smarty);
}

/**
 * Return "update" button for the item
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_item_update_button($module, $_item, $_args, $smarty, $test_only = false)
{
    if ($test_only) {
        return true;
    }
    $_args['module']     = $module;
    $_args['profile_id'] = $_item['_profile_id'];
    $_args['item_id']    = $_item['_item_id'];
    return smarty_function_jrCore_item_update_button($_args, $smarty);
}

/**
 * Return "update" button for a bundle
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_bundle_update_button($module, $_item, $_args, $smarty, $test_only = false)
{
    if ($test_only) {
        return true;
    }
    $_args['module'] = $module;
    return smarty_function_jrCore_item_update_button($_args, $smarty);
}

/**
 * Return "delete" button for the item
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_item_delete_button($module, $_item, $_args, $smarty, $test_only = false)
{
    if ($test_only) {
        return true;
    }
    $_args['module']     = $module;
    $_args['profile_id'] = $_item['_profile_id'];
    $_args['item_id']    = $_item['_item_id'];
    return smarty_function_jrCore_item_delete_button($_args, $smarty);
}

/**
 * Return "delete" button for a bundle
 * @param $module string Module name
 * @param $_item array Item Array
 * @param $_args array Smarty function parameters
 * @param $smarty object Smarty Object
 * @param $test_only bool check if button WOULD be shown for given module
 * @return string
 */
function jrCore_bundle_delete_button($module, $_item, $_args, $smarty, $test_only = false)
{
    if ($test_only) {
        return true;
    }
    $_args['module'] = $module;
    return smarty_function_jrCore_item_delete_button($_args, $smarty);
}

/**
 * Test a smarty template for errors
 * @param string $module
 * @param string $content
 * @return bool|string
 */
function jrCore_test_template_for_errors($module, $content)
{
    global $_conf;
    // We need to test this template and make sure it does not cause any Smarty errors
    $key = jrCore_create_unique_string(8);
    jrCore_set_temp_value('jrCore', "{$key}_template", $content);
    $url = jrCore_get_module_url('jrCore');
    $out = jrCore_load_url("{$_conf['jrCore_base_url']}/{$url}/test_template/{$key}/{$module}");
    jrCore_delete_temp_value('jrCore', "{$key}_template");
    if ($out && strlen($out) > 1 && (strpos($out, 'error:') === 0 || stristr($out, 'fatal error') || stristr($out, 'Smarty Compiler:'))) {
        // SmartyCompilerException: Syntax error in template "file:/.../1480710660.tpl"  on line 181 "" unclosed {if} tag in /modules/jrCore/contrib/smarty/libs/sysplugins/smarty_internal_templatecompilerbase.php on line 181
        $_ad = array();
        $_tm = explode("\n", $out);
        if ($_tm && is_array($_tm)) {
            foreach ($_tm as $line) {
                if (strpos($line, $key)) {
                    $_rp  = array(
                        'error: syntax: ',
                        'SmartyCompilerException:',
                        'file:' . APP_DIR . '/',
                        '""'
                    );
                    $line = str_replace($_rp, '', $line);
                    list($line,) = explode(APP_DIR, $line);
                    $_ad[] = rtrim(trim($line), 'in');
                }
            }
        }
        if (count($_ad) > 0) {
            return 'error: There are syntax error(s) in your template - please fix and try again:<br>' . jrCore_strip_html(implode('<br>', $_ad));
        }
        elseif (strpos($out, 'error:') === 0) {
            return $out;
        }
    }
    return true;
}

/**
 * Get a cached SVG image
 * @param string $module Module where svg file is located
 * @param string $file SVG file name
 * @param int $size Size in pixels
 * @param string $class Icon Class
 * @param string $version Version to bypass cache
 * @return string
 */
function jrCore_get_cached_svg($module, $file, $size, $class = 'svg_icon form_icon_button', $version = 0)
{
    $key = md5("{$module}/{$file}/{$version}");
    if (!$svg = jrCore_get_local_cache_key($key)) {
        $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
        $dir  = APP_DIR . "/skins/{$skin}/svg";
        if ($module != $skin && !is_file("{$dir}/{$module}_{$file}")) {
            $dir = APP_DIR . "/modules/{$module}/svg";
        }
        if ($svg = jrCore_file_get_contents("{$dir}/{$file}")) {
            jrCore_set_local_cache_key($key, $svg, 3600);
        }
        else {
            return '';
        }
    }
    $sid = 'svg-' . jrCore_create_unique_string(6);
    return "<style>#{$sid} svg { display:block;width:100%;height:100% }</style><div id=\"{$sid}\" class=\"{$class}\" style=\"width:{$size}px;height:{$size}px\">{$svg}</div>";
}
