<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package DataStore
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

// Constants
const SKIP_TRIGGERS = true;
const NO_CACHE      = true;

/**
 * @return array
 * @ignore
 * jrCore_get_datastore_plugins
 */
function jrCore_get_datastore_plugins()
{
    return jrCore_get_system_plugins('datastore');
}

/**
 * Get the active datastore function
 * @param $module string DS module
 * @param $function string DS function to run
 * @return string
 */
function jrCore_get_active_datastore_function($module, $function)
{
    if ($plugin = jrCore_get_config_value('jrCore', "active_datastore_system_{$module}", false)) {
        if ($plugin != 'jrCore') {
            $func = "_{$plugin}_{$function}";
            if (function_exists($func)) {
                return $func;
            }
        }
    }
    if ($plugin = jrCore_get_config_value('jrCore', 'active_datastore_system', false)) {
        if ($plugin != 'jrCore') {
            $func = "_{$plugin}_{$function}";
            if (function_exists($func)) {
                return $func;
            }
        }
    }
    return '_jrCore_' . $function;
}

/**
 * An array of modules that have a datastore enabled
 */
function jrCore_get_datastore_modules()
{
    global $_mods;
    $_out = array();
    foreach ($_mods as $module => $_inf) {
        if (isset($_inf['module_prefix']) && strlen($_inf['module_prefix']) > 0) {
            $_out[$module] = $_inf['module_prefix'];
        }
    }
    return $_out;
}

/**
 * Return TRUE if given module is a DS module
 * @param $module string module
 * @return bool
 */
function jrCore_is_datastore_module($module)
{
    global $_mods;
    return isset($_mods[$module]['module_prefix']) && strlen($_mods[$module]['module_prefix']) > 0;
}

/**
 * Returns DataStore Prefix for a module
 * @param string $module Module to return prefix for
 * @return mixed
 */
function jrCore_db_get_prefix($module)
{
    global $_mods;
    if (!empty($_mods[$module]['module_prefix'])) {
        return $_mods[$module]['module_prefix'];
    }
    // Could be we JUST created this DS
    elseif ($_tmp = jrCore_get_flag('jrcore_db_create_datastore_prefixes')) {
        if (!empty($_tmp[$module])) {
            return $_tmp[$module];
        }
    }
    return false;
}

/**
 * Get all index tables for a module
 * @param string $module
 * @return array|bool
 */
function jrCore_db_get_all_index_tables_for_module($module)
{
    global $_conf;
    if (jrCore_get_config_value('jrCore', 'index_keys', false) !== false) {
        if ($_tm = explode(',', trim(trim($_conf['jrCore_index_keys'], ',')))) {
            $_tb = array();
            foreach ($_tm as $v) {
                if (strpos($v, "{$module}:") === 0) {
                    list(, $k) = explode(':', $v);
                    $k       = trim($k);
                    $_tb[$k] = $k;
                }
            }
            return $_tb;
        }
    }
    return false;
}

/**
 * Returns true if a key has an index table
 * @param string $module
 * @param string $key
 * @return bool
 */
function jrCore_db_key_has_index_table($module, $key)
{
    if ($key == '_item_id') {
        return false;
    }
    if ($keys = jrCore_get_config_value('jrCore', 'index_keys', false)) {
        if (strpos(" {$keys}", ",{$module}:{$key},")) {
            return true;
        }
    }
    return false;
}

/**
 * Get a table name for an index table
 * @param string $module
 * @param string $key
 * @param bool $key_name_only
 * @return bool|string
 */
function jrCore_db_get_index_table_name($module, $key, $key_name_only = false)
{
    // Name of table will be: jr_<module>_item_key_<key_name_without_prefix>
    // @note: Max length of table name is 64 characters
    if (!$pfx = jrCore_db_get_prefix($module)) {
        return false;
    }
    $name = $key;
    if (strpos($key, $pfx) === 0) {
        $name = substr($key, strlen($pfx) + 1);
    }
    if (strlen($name) > 54) {
        $name = substr($name, 0, 55);
    }
    if ($key_name_only) {
        return "item_key_{$name}";
    }
    return jrCore_db_table_name($module, "item_key_{$name}");
}

/**
 * Creates a new module DataStore
 * @param string $module Module to create DataStore for
 * @param string $prefix Key Prefix in DataStore
 * @return bool
 */
function jrCore_db_create_datastore($module, $prefix)
{
    if (strlen($prefix) === 0) {
        jrCore_logger('CRI', "core: invalid datastore module_prefix for module: {$module}");
        return false;
    }
    $func = jrCore_get_active_datastore_function($module, 'db_create_datastore');
    if (function_exists($func)) {
        if ($func($module, $prefix)) {

            // Make sure our DataStore prefix is stored with the module info
            $efx = jrCore_db_get_prefix($module);
            if (!$efx || $efx != $prefix || strlen($efx) === 0) {
                $tbl = jrCore_db_table_name('jrCore', 'module');
                $req = "UPDATE {$tbl} SET module_prefix = '" . jrCore_db_escape($prefix) . "' WHERE module_directory = '" . jrCore_db_escape($module) . "' LIMIT 1";
                jrCore_db_query($req, 'COUNT');
            }

            // Lastly, if this DS is being created in a jrCore_verify_module, and the
            // module has an install.php script, the prefix won't be available in $_mods
            // until cache is reset and the page reloaded, so put it in a tmp place
            if ($_tmp = jrCore_get_flag('jrcore_db_create_datastore_prefixes')) {
                $_tmp[$module] = $prefix;
            }
            else {
                $_tmp = array($module => $prefix);
            }
            jrCore_set_flag('jrcore_db_create_datastore_prefixes', $_tmp);

            // Let modules know we are creating/validating a DataStore
            $_args = array(
                'module' => $module,
                'prefix' => $prefix
            );
            jrCore_trigger_event('jrCore', 'db_create_datastore', array(), $_args);
            return true;
        }
    }
    return false;
}

/**
 * Deletes an existing module datastore
 * @param string $module Module to delete DataStore for
 * @return bool
 */
function jrCore_db_delete_datastore($module)
{
    $func = jrCore_get_active_datastore_function($module, 'db_delete_datastore');
    if (function_exists($func)) {
        if ($func($module)) {
            jrCore_trigger_event('jrCore', 'db_delete_datastore', array('module' => $module));
            return true;
        }
    }
    return true;
}

/**
 * Migrate profile_ids in datastore
 * @param string $module DS Module
 * @param bool $modal_updates Set to TRUE to add modal updates (i.e. during integrity check)
 * @return int
 */
function jrCore_db_sync_datastore_profile_ids($module, $modal_updates = false)
{
    if (jrCore_is_datastore_module($module)) {
        $func = jrCore_get_active_datastore_function($module, 'db_sync_datastore_profile_ids');
        if (function_exists($func)) {
            if ($modal_updates) {
                jrCore_form_modal_notice('update', "syncing profile_ids for {$module} - this could take a bit");
            }
            if ($cnt = $func($module)) {
                jrCore_logger('INF', "core: updated " . jrCore_number_format($cnt) . " {$module} datastore items with correct _profile_id");
                return $cnt;
            }
        }
    }
    return 0;
}

/**
 * Repair datastore items for a module
 * @param string $module module to repair
 * @return bool
 * @deprecated
 */
function jrCore_db_repair_datastore_items($module)
{
    return true;
}

/**
 * Run repair operations on a DataStore
 * @param string $module
 * @return bool
 */
function jrCore_db_repair_datastore($module)
{
    if (jrCore_is_datastore_module($module)) {
        $func = jrCore_get_active_datastore_function($module, 'db_repair_datastore');
        if (function_exists($func)) {
            return $func($module);
        }
    }
    return true;
}

/**
 * Truncate and reset a module DataStore
 * @param $module string Module DataStore to truncate
 * @return bool
 */
function jrCore_db_truncate_datastore($module)
{
    $func = jrCore_get_active_datastore_function($module, 'db_truncate_datastore');
    if ($func($module)) {
        jrCore_trigger_event('jrCore', 'db_truncate_datastore', array('module' => $module));
        return true;
    }
    return false;
}

/**
 * Get number of items in a module DataStore
 * @param $module string Module to get number of items for
 * @return int
 */
function jrCore_db_get_datastore_item_count($module)
{
    $func = jrCore_get_active_datastore_function($module, 'db_get_datastore_item_count');
    return $func($module);
}

/**
 * Run a key "function" in matching values
 * @param $module string Module DataStore
 * @param $key string Key to match
 * @param $match string Value to match - '*' for all
 * @param $function string function to run on key values (sum, avg, min, max, std, count)
 * @return mixed
 */
function jrCore_db_run_key_function($module, $key, $match, $function)
{
    $func = jrCore_get_active_datastore_function($module, 'db_run_key_function');
    return $func($module, $key, $match, $function);
}

/**
 * Set the special "display_order" keys for items in a DataStore
 * @param $module string Module DataStore to set values in
 * @param $_ids array Array of id => value entries
 * @return bool
 */
function jrCore_db_set_display_order($module, $_ids)
{
    if (!is_array($_ids)) {
        return false;
    }
    $func = jrCore_get_active_datastore_function($module, 'db_set_display_order');
    return $func($module, $_ids);
}

/**
 * Create a new key for all entries in a DataStore and set it to a default value
 * @param $module string Module DataStore to create new key in
 * @param $key string Key to create
 * @param $value mixed initial value
 * @return bool
 */
function jrCore_db_create_default_key($module, $key, $value)
{
    $func = jrCore_get_active_datastore_function($module, 'db_create_default_key');
    return $func($module, $key, $value);
}

/**
 * Create a new key for all entries in a DataStore and set it to a default value
 * @param $module string Module DataStore to create new key in
 * @param $key string Key to create
 * @param $value mixed value to set keys to
 * @param $default mixed if a value is set set to $default, it will be changed to $value
 * @return int
 */
function jrCore_db_update_default_key($module, $key, $value, $default)
{
    $func = jrCore_get_active_datastore_function($module, 'db_update_default_key');
    return $func($module, $key, $value, $default);
}

/**
 * Increment a DataStore key for an Item ID or Array of Item IDs by a given value
 * @param $module string Module Name
 * @param $id mixed Unique Item ID OR Array of Item IDs
 * @param $key string Key to increment
 * @param $value number Integer/Float to increment by
 * @param $update bool set to TRUE to update updated timed
 * @param $cache_reset bool set to FALSE to prevent cache reset
 * @param array $profile_ids item_id => profile_id array for profile ID matching
 * @return bool
 */
function jrCore_db_increment_key($module, $id, $key, $value, $update = false, $cache_reset = true, $profile_ids = null)
{
    if (!is_numeric($value)) {
        return false;
    }
    if (!is_array($id)) {
        $id = array(intval($id));
    }
    else {
        foreach ($id as $k => $iid) {
            $id[$k] = (int) $iid;
        }
    }
    $_arg = array(
        'module'      => $module,
        'id'          => $id,
        'key'         => $key,
        'value'       => $value,
        'update'      => $update,
        'cache_reset' => $cache_reset,
        'profile_ids' => $profile_ids
    );
    $_arg = jrCore_trigger_event('jrCore', 'db_increment_key', $_arg);
    $func = jrCore_get_active_datastore_function($module, 'db_increment_key');
    if ($func($_arg['module'], $_arg['id'], $_arg['key'], $_arg['value'], $_arg['update'], $profile_ids)) {
        if ($cache_reset) {
            // Reset cache for these items
            $_ch = array();
            foreach ($id as $uid) {
                $_ch[] = array($module, "{$module}-{$uid}-0");
                $_ch[] = array($module, "{$module}-{$uid}-1");
            }
            jrCore_delete_multiple_cache_entries($_ch);
        }
        return true;
    }
    return false;
}

/**
 * Decrement a DataStore key for an Item ID or Array of Item IDs by a given value
 * @param $module string Module Name
 * @param $id mixed Unique Item ID OR Array of Item IDs
 * @param $key string Key to decrement
 * @param $value number Integer/Float to decrement by
 * @param $min_value number Lowest Value allowed for Key (default 0)
 * @param $update bool set to TRUE to update updated timed
 * @param $cache_reset bool set to FALSE to prevent cache reset
 * @param array $profile_ids item_id => profile_id array for profile ID matching
 * @return bool
 */
function jrCore_db_decrement_key($module, $id, $key, $value, $min_value = null, $update = false, $cache_reset = true, $profile_ids = null)
{
    if (!is_numeric($value)) {
        return false;
    }
    if (!is_numeric($min_value)) {
        $min_value = 0;
    }
    if (!is_array($id)) {
        $id = array(intval($id));
    }
    else {
        foreach ($id as $k => $iid) {
            $id[$k] = (int) $iid;
        }
    }
    $_arg = array(
        'module'      => $module,
        'id'          => $id,
        'key'         => $key,
        'value'       => $value,
        'min_value'   => $min_value,
        'update'      => $update,
        'cache_reset' => $cache_reset,
        'profile_ids' => $profile_ids
    );
    $_arg = jrCore_trigger_event('jrCore', 'db_decrement_key', $_arg);
    $func = jrCore_get_active_datastore_function($module, 'db_decrement_key');
    if ($func($_arg['module'], $_arg['id'], $_arg['key'], $_arg['value'], $_arg['min_value'], $_arg['update'], $profile_ids)) {
        if ($cache_reset) {
            // Reset cache for these items
            $_ch = array();
            foreach ($id as $uid) {
                $_ch[] = array($module, "{$module}-{$uid}-0");
                $_ch[] = array($module, "{$module}-{$uid}-1");
            }
            jrCore_delete_multiple_cache_entries($_ch);
        }
        return true;
    }
    return false;
}

/**
 * Return an array of _item_id's that do NOT have a specified key set
 * @param $module string Module DataStore to search through
 * @param $key string Key Name that should not be set
 * @param $limit int Limit number of results to $limit
 * @return array|bool
 */
function jrCore_db_get_items_missing_key($module, $key, $limit = 0)
{
    $func = jrCore_get_active_datastore_function($module, 'db_get_items_missing_key');
    return $func($module, $key, $limit);
}

/**
 * Check if a specific KEY name exists in a DataStore
 * @param $module string Module DataStore
 * @param $key string Key to check for
 * @return bool
 */
function jrCore_db_item_key_exists($module, $key)
{
    $func = jrCore_get_active_datastore_function($module, 'db_item_key_exists');
    return $func($module, $key);
}

/**
 * Get all unique Key Names from a module DS
 * @param $module string Module
 * @return array|bool
 */
function jrCore_db_get_unique_keys($module)
{
    $func = jrCore_get_active_datastore_function($module, 'db_get_unique_keys');
    return $func($module);
}

/**
 * Get all values for a key
 * @param $module string Module DataStore
 * @param $key string Key to get value for
 * @return bool|mixed
 */
function jrCore_db_get_all_key_values($module, $key)
{
    $func = jrCore_get_active_datastore_function($module, 'db_get_all_key_values');
    return $func($module, $key);
}

/**
 * Get all values for a key for a Profile ID
 * @param int $profile_id Profile ID
 * @param string $module Module DataStore
 * @param string $key Key to get value for
 * @return bool|mixed
 */
function jrCore_db_get_all_values_for_key_by_profile_id($profile_id, $module, $key)
{
    $func = jrCore_get_active_datastore_function($module, 'db_get_all_values_for_key_by_profile_id');
    return $func($profile_id, $module, $key);
}

/**
 * Deletes multiple keys from an item
 * @param string $module Module the DataStore belongs to
 * @param int $id Item ID
 * @param array $_keys Keys to delete
 * @param bool $core_check by default you cannot delete keys that begin with _
 * @param bool $cache_reset by default cache is reset
 * @param bool $update set to FALSE to prevent updating of _updated key
 * @return bool
 */
function jrCore_db_delete_multiple_item_keys($module, $id, $_keys, $core_check = true, $cache_reset = true, $update = true)
{
    global $_user;
    if (!is_array($_keys) || count($_keys) === 0) {
        return false;
    }

    // Some things we cannot remove
    foreach ($_keys as $k => $key) {
        if ($core_check && strpos($key, '_') === 0) {
            // internally used - cannot remove
            unset($_keys[$k]);
        }
    }
    if (empty($_keys)) {
        return false;
    }

    $uid   = intval($id);
    $_args = array('module' => $module, '_item_id' => $uid);
    $_keys = jrCore_trigger_event('jrCore', 'db_delete_keys', $_keys, $_args);

    $func = jrCore_get_active_datastore_function($module, 'db_delete_multiple_item_keys');
    if ($func($module, $id, $_keys, $core_check, $cache_reset, $update)) {
        // reset cache for this item
        if ($cache_reset) {
            $_ch = array(
                array($module, "{$module}-{$id}-0"),
                array($module, "{$module}-{$id}-1")
            );
            jrCore_delete_multiple_cache_entries($_ch);
        }
        // If this is a delete for the USER or PROFILE modules, and the viewing
        // user is the id we are being deleted from, we have to also remove those
        // keys from the active session otherwise they will not be removed
        if (jrUser_is_logged_in()) {
            foreach ($_keys as $key) {
                if ($module == 'jrUser' && $_user['_user_id'] == $id) {
                    jrUser_delete_session_key($key);
                }
                elseif ($module == 'jrProfile' && $_user['user_active_profile_id'] == $id) {
                    jrUser_delete_session_key($key);
                }
            }
        }
        // Set our delete flag for user/profile key deletes - this will be
        // used in jrUser_set_session_sync() to flag which keys need to
        // be REMOVED from the active session
        if ($module == 'jrUser' || $module == 'jrProfile') {
            if (!$_rs = jrCore_get_flag('session_sync_delete_keys')) {
                $_rs = array();
            }
            if (!isset($_rs[$module])) {
                $_rs[$module] = array();
            }
            $_rs[$module][$id] = $_keys;
            jrCore_set_flag('session_sync_delete_keys', $_rs);
        }

        // Event Trigger
        $_data = array(
            'module'   => $module,
            '_item_id' => $id,
            '_keys'    => $_keys
        );
        jrCore_trigger_event('jrCore', 'db_delete_multiple_item_keys', $_data);

        return true;
    }
    return false;
}

/**
 * Deletes a single key from an item
 * @param string $module Module the DataStore belongs to
 * @param int $id Item ID
 * @param string $key Key to delete
 * @param bool $core_check by default you cannot delete keys that begin with _
 * @param bool $cache_reset by default cache is reset
 * @param bool $update set to FALSE to prevent updating of _updated key
 * @return mixed INSERT_ID on success, false on error
 */
function jrCore_db_delete_item_key($module, $id, $key, $core_check = true, $cache_reset = true, $update = true)
{
    $func = jrCore_get_active_datastore_function($module, 'db_delete_item_key');
    return $func($module, $id, $key, $core_check, $cache_reset, $update);
}

/**
 * Delete DataStore Key(s) from Multiple Items
 * @param string $module Module the DataStore belongs to
 * @param array $_ids IDs of items to delete keys from
 * @param mixed $key key name or array of key names
 * @param bool $cache_reset by default cache is reset
 * @param bool $update set to FALSE to prevent updating of _updated key
 * @return bool
 */
function jrCore_db_delete_key_from_multiple_items($module, $_ids, $key, $cache_reset = true, $update = true)
{
    global $_user;
    $func = jrCore_get_active_datastore_function($module, 'db_delete_key_from_multiple_items');
    if ($func($module, $_ids, $key, $update)) {
        if ($cache_reset) {
            $_cch = array();
            foreach ($_ids as $id) {
                $_cch[] = array($module, "{$module}-{$id}-0");
                $_cch[] = array($module, "{$module}-{$id}-1");
            }
            jrCore_delete_multiple_cache_entries($_cch);
        }
        // If this is a delete for the USER or PROFILE modules, and the viewing
        // user is the id we are being deleted from, we have to also remove those
        // keys from the active session otherwise they will not be removed
        if (jrUser_is_logged_in()) {
            foreach ($_ids as $id) {
                if ($module == 'jrUser' && $_user['_user_id'] == $id) {
                    jrUser_delete_session_key($key);
                }
                elseif ($module == 'jrProfile' && $_user['user_active_profile_id'] == $id) {
                    jrUser_delete_session_key($key);
                }
            }
        }

        // Event Trigger
        $_data = array(
            'module'    => $module,
            '_item_ids' => $_ids,
            'key'       => $key
        );
        jrCore_trigger_event('jrCore', 'db_delete_key_from_multiple_items', $_data);

        return true;
    }
    return false;
}

/**
 * Delete DataStore Key(s) from All Items
 * @note This function does NOT set a new value for _updated on all items!
 * @param string $module Module the DataStore belongs to
 * @param string $key key name
 * @return bool
 */
function jrCore_db_delete_key_from_all_items($module, $key)
{
    $func = jrCore_get_active_datastore_function($module, 'db_delete_key_from_all_items');
    if ($count = $func($module, $key)) {
        $_rp = array(
            'delete_count' => $count,
            'module'       => $module,
            'key'          => $key
        );
        jrCore_trigger_event('jrCore', 'db_delete_key_from_all_items', $_rp);
        return true;
    }
    return false;
}

/**
 * Validates DataStore key names are allowed and correct
 * @param string $module Module the DataStore belongs to
 * @param array $_data Array of Key => Value pairs to check
 * @return array|false
 */
function jrCore_db_get_allowed_item_keys($module, $_data)
{
    if (!$_data || !is_array($_data) || count($_data) === 0) {
        return false;
    }
    if (!$pfx = jrCore_db_get_prefix($module)) {
        return false;
    }
    $_rt = array();
    foreach ($_data as $k => $v) {
        if (strpos($k, $pfx) !== 0) {
            jrCore_notice_page('CRI', "invalid key name: {$k} - key name must begin with module prefix: {$pfx}_");
        }
        elseif (@preg_match('/[^A-Za-z\d_-]/', $k) > 0) {
            jrCore_notice_page('CRI', "invalid key name: {$k} - key can only contain letters, numbers, underscores and dashes");
        }
        $_rt[$k] = $v;
    }
    return $_rt;
}

/**
 * Create a new Unique Item ID for an item
 * @param $module string module
 * @param $count int number of IDs to create
 * @return mixed
 */
function jrCore_db_create_unique_item_id($module, $count = 1)
{
    $func = jrCore_get_active_datastore_function($module, 'db_create_unique_item_id');
    return $func($module, $count);
}

/**
 * Update module item count for a profile
 * @param $module string
 * @param $profile_id int
 * @return mixed
 */
function jrCore_db_update_profile_item_count($module, $profile_id)
{
    $func = jrCore_get_active_datastore_function($module, 'db_update_profile_item_count');
    return $func($module, $profile_id);
}

/**
 * Update module item count for a user
 * @param $module string
 * @param $profile_id int
 * @param $user_id int
 * @return mixed
 */
function jrCore_db_update_user_item_count($module, $profile_id, $user_id)
{
    $func = jrCore_get_active_datastore_function($module, 'db_update_user_item_count');
    return $func($module, $profile_id, $user_id);
}

/**
 * Creates a new item in a module datastore
 * @param string $module Module the DataStore belongs to
 * @param array $_data Array of Key => Value pairs for insertion
 * @param array $_core Array of Key => Value pairs for insertion - skips jrCore_db_get_allowed_item_keys()
 * @param bool $profile_count If set to true, profile_count will be incremented for given _profile_id
 * @param bool $skip_trigger Set to TRUE to skip sending out create_item trigger
 * @return mixed INSERT_ID on success, false on error
 */
function jrCore_db_create_item($module, $_data, $_core = null, $profile_count = true, $skip_trigger = false)
{
    global $_user;

    // See if we are limiting the number of items that can be created by a profile in this quota
    if (!jrUser_is_admin()) {
        if (isset($_user["quota_{$module}_max_items"]) && $_user["quota_{$module}_max_items"] > 0) {
            if ($p_cnt = jrCore_db_get_item_key('jrProfile', $_user['user_active_profile_id'], "profile_{$module}_item_count")) {
                if ($p_cnt >= $_user["quota_{$module}_max_items"]) {
                    // We've hit the limit for this quota
                    $_ln = jrUser_load_lang_strings();
                    jrCore_set_flag("max_{$module}_items_reached", $_ln['jrCore'][70]);
                    return false;
                }
            }
        }
    }

    // Validate incoming data
    if (!$_data = jrCore_db_get_allowed_item_keys($module, $_data)) {
        $_data = array();
    }

    // Check for additional core fields being added in
    if ($_core && is_array($_core)) {
        foreach ($_core as $k => $v) {
            if (strpos($k, '_') === 0) {
                $_data[$k] = $v;
            }
        }
    }
    $_core = null;

    // Internal defaults
    $_check = array(
        '_created'    => 'UNIX_TIMESTAMP()',
        '_updated'    => 'UNIX_TIMESTAMP()',
        '_profile_id' => 0,
        '_user_id'    => 0
    );

    if (jrUser_is_logged_in()) {
        switch ($module) {
            // Do not set _user_id or _profile_id on User/Profile module creates
            case 'jrUser':
            case 'jrProfile':
                break;
            default:
                // If user is logged in, defaults to their account
                $_check['_profile_id'] = (isset($_user['user_active_profile_id'])) ? intval($_user['user_active_profile_id']) : jrUser_get_profile_home_key('_profile_id');
                $_check['_user_id']    = (int) $_user['_user_id'];
                break;
        }
    }

    foreach ($_check as $k => $v) {
        // Any of our _check values can be removed by setting it to false
        if (isset($_data[$k]) && $_data[$k] === false) {
            unset($_data[$k]);
        }
        elseif (!isset($_data[$k])) {
            $_data[$k] = $v;
        }
    }

    // Our module DS prefix
    $pfx = jrCore_db_get_prefix($module);

    // Check for item_order_support
    $_pn = jrCore_get_registered_module_features('jrCore', 'item_order_support');
    if ($_pn && isset($_pn[$module]) && !isset($_data["{$pfx}_display_order"])) {
        // New entries at top
        $_data["{$pfx}_display_order"] = 0;
    }

    // Let listeners add/remove data or prevent item creation altogether
    if (!$skip_trigger) {
        $_args = array(
            'module' => $module
        );
        $_data = jrCore_trigger_event('jrCore', 'db_create_item_data', $_data, $_args);
        // Our listeners can tell us to NOT create the item
        if (isset($_data['db_create_item']) && jrCore_checktype($_data['db_create_item'], 'is_false')) {
            // We've been short circuited by a listener
            return false;
        }
    }

    // Generate unique ID for this item
    $iid = jrCore_db_create_unique_item_id($module);
    if (!$iid) {
        return false;
    }

    // Trigger create event
    if (!$skip_trigger) {
        $_args['_item_id'] = $iid;
        $_data             = jrCore_trigger_event('jrCore', 'db_create_item', $_data, $_args);
    }

    // Check for Pending Support for this module
    // Items created by master/admin users bypass pending
    $eml = true;
    $lid = 0;
    $lmd = '';
    if (!isset($_data["{$pfx}_pending"])) {
        $_pn = jrCore_get_registered_module_features('jrCore', 'pending_support');
        if ($_pn && isset($_pn[$module])) {
            $_data["{$pfx}_pending"] = 0;

            // Pending support is on for this module - check quota setting:
            // 0 = immediately active
            // 1 = review needed on CREATE
            // 2 = review needed on CREATE and UPDATE
            if (!jrUser_is_admin() && isset($_user["quota_{$module}_pending"]) && intval($_user["quota_{$module}_pending"]) > 0) {
                $_data["{$pfx}_pending"] = 1;
            }

        }
    }
    else {
        // See if this item was set pending by a db_create_item event listener
        if ($_data["{$pfx}_pending"] == 1) {

            jrCore_set_flag("jrcore_created_pending_item_{$module}_{$iid}", 1);

            // Check for actions that are linking to pending items
            // Important: This part must be BEFORE the active DS create call so we can remove the extra keys
            if (isset($_data['action_pending_linked_item_id']) && jrCore_checktype($_data['action_pending_linked_item_id'], 'number_nz')) {
                $lid = (int) $_data['action_pending_linked_item_id'];
                $lmd = jrCore_db_escape($_data['action_pending_linked_item_module']);
                unset($_data['action_pending_linked_item_id'], $_data['action_pending_linked_item_module']);
                $eml = false;
            }

        }
    }

    // Create item
    $func = jrCore_get_active_datastore_function($module, 'db_create_item');
    if ($func($module, $iid, $_data, $_core, $profile_count, $skip_trigger)) {

        // Add pending entry to Pending table...
        if (isset($_data["{$pfx}_pending"]) && $_data["{$pfx}_pending"] == 1) {
            $uid = (int) $_data['_user_id'];
            $pid = (int) $_data['_profile_id'];
            $pnd = jrCore_db_table_name('jrCore', 'pending');
            $req = "INSERT INTO {$pnd} (pending_created, pending_module, pending_profile_id, pending_user_id, pending_item_id, pending_linked_item_module, pending_linked_item_id)
                    VALUES (UNIX_TIMESTAMP(), '" . jrCore_db_escape($module) . "', {$pid}, {$uid}, '{$iid}', '{$lmd}', '{$lid}')
                    ON DUPLICATE KEY UPDATE pending_created = UNIX_TIMESTAMP()";
            jrCore_db_query($req, null, false, null, false);

            // Notify admins of new pending item
            if ($eml && $lid == 0) {
                $_rt = jrUser_get_admin_user_ids();
                $_mi = jrCore_get_module_info($module);
                $_us = ($uid == $_user['_user_id']) ? $_user : jrCore_db_get_item('jrUser', $uid);
                if ($_rt && is_array($_rt)) {
                    $_temp                = array_merge($_us, $_data);
                    $_temp['system_name'] = jrCore_get_config_value('jrCore', 'system_name', '');
                    $_temp['_item_id']    = $iid;
                    $_temp['_created']    = time();
                    $_temp['_updated']    = time();
                    $_temp['module']      = $module;
                    $_temp['module_name'] = $_mi['module_name'];
                    $_temp['item_title']  = jrCore_get_item_title($module, $_data);
                    $_temp['item_url']    = jrCore_get_item_url('detail', $module, $_temp);
                    $_temp['_user']       = $_us;
                    list($sub, $msg) = jrCore_parse_email_templates('jrCore', 'pending_item', $_temp);
                    jrCore_db_notify_admins_of_pending_item($module, $_rt, $sub, $msg);
                }
            }

            // Reset pending item_id's cache for this module
            jrCore_db_set_pending_reset_for_module($module);

        }

        // Increment profile counts for this item
        if ($profile_count) {
            switch ($module) {

                // Some modules we do not store counts for
                case 'jrCore':
                case 'jrProfile':
                case 'jrUser':
                    break;

                default:
                    if (isset($_data['_profile_id'])) {
                        $pid = intval($_data['_profile_id']);
                        if ($pid > 0) {
                            jrCore_db_increment_key('jrProfile', $pid, "profile_{$module}_item_count", 1);
                        }
                    }
                    if (isset($_data['_user_id'])) {
                        $uid = intval($_data['_user_id']);
                        if ($uid > 0) {
                            jrCore_db_increment_key('jrUser', $uid, "user_{$module}_item_count", 1);
                        }
                    }
                    break;
            }
        }

        // Trigger create_item_exit event
        if (!$skip_trigger) {
            $_args = array(
                '_item_id' => $iid,
                'module'   => $module
            );
            jrCore_trigger_event('jrCore', 'db_create_item_exit', $_data, $_args);
        }
        return $iid;
    }
    return false;
}

/**
 * Create multiple items in a module datastore
 * @param string $module Module the DataStore belongs to
 * @param array $_data Array of Key => Value pairs for insertion
 * @param array $_core Array of Key => Value pairs for insertion - skips jrCore_db_get_allowed_item_keys()
 * @param bool $skip_trigger bool Set to TRUE to skip sending out create_item trigger
 * @return array|false array of INSERT_ID's on success, false on error
 */
function jrCore_db_create_multiple_items($module, $_data, $_core = null, $skip_trigger = false)
{
    global $_user;

    // Our module DS prefix
    $pfx = jrCore_db_get_prefix($module);

    // Check for item_order_support
    $_pn = jrCore_get_registered_module_features('jrCore', 'item_order_support');

    // Validate incoming data
    foreach ($_data as $k => $_dt) {
        if (!is_array($_dt)) {
            // bad data
            return false;
        }
        $_data[$k] = jrCore_db_get_allowed_item_keys($module, $_dt);
        if ($_pn && isset($_pn[$module]) && !isset($_data[$k]["{$pfx}_display_order"])) {
            // New entries at top
            $_data[$k]["{$pfx}_display_order"] = 0;
        }
    }

    // Check for additional core fields being added in
    if (is_array($_core)) {
        foreach ($_core as $ck => $_cr) {
            foreach ($_cr as $k => $v) {
                if (strpos($k, '_') === 0) {
                    $_data[$ck][$k] = $v;
                }
            }
        }
        $_core = null;
    }

    // Internal defaults
    $_check = array(
        '_created'    => 'UNIX_TIMESTAMP()',
        '_updated'    => 'UNIX_TIMESTAMP()',
        '_profile_id' => 0,
        '_user_id'    => 0
    );
    // If user is logged in, defaults to their account
    if (jrUser_is_logged_in()) {
        $_check['_profile_id'] = (isset($_user['user_active_profile_id'])) ? intval($_user['user_active_profile_id']) : jrUser_get_profile_home_key('_profile_id');
        $_check['_user_id']    = (int) $_user['_user_id'];
    }
    foreach ($_data as $k => $_dt) {
        foreach ($_check as $ck => $v) {
            // Any of our _check values can be removed by setting it to false
            if (isset($_data[$k][$ck]) && $_data[$k][$ck] === false) {
                unset($_data[$k][$ck]);
            }
            elseif (!isset($_data[$k][$ck])) {
                $_data[$k][$ck] = $v;
            }
        }
    }

    // Let listeners add/remove data or prevent item creation altogether
    if (!$skip_trigger) {
        $_args = array(
            'module' => $module
        );
        foreach ($_data as $k => $_dt) {
            $_dt = jrCore_trigger_event('jrCore', 'db_create_item_data', $_dt, $_args);
            // Our listeners can tell us to NOT create the item
            if (isset($_dt['db_create_item']) && jrCore_checktype($_dt['db_create_item'], 'is_false')) {
                // We've been short circuited by a listener
                unset($_data[$k]);
            }
        }
        if (count($_data) === 0) {
            // Nothing to create
            return false;
        }
    }

    // Generate unique ID for these items - note that we only get
    // the FIRST ID in the set - the rest are incremental from that
    $iid = jrCore_db_create_unique_item_id($module, count($_data));
    if (!$iid) {
        return false;
    }

    // Trigger create event
    if (!$skip_trigger) {
        $uid = $iid;
        foreach ($_data as $k => $_dt) {
            $_args     = array(
                '_item_id' => $uid,
                'module'   => $module
            );
            $_data[$k] = jrCore_trigger_event('jrCore', 'db_create_item', $_dt, $_args);
            $uid++;
        }
    }

    $func = jrCore_get_active_datastore_function($module, 'db_create_multiple_items');
    if ($func($module, $iid, $_data, $_core, $skip_trigger)) {
        $_id = array();
        $uid = $iid;
        foreach ($_data as $_dt) {
            // Trigger create_item_exit event
            if (!$skip_trigger) {
                $_args = array(
                    '_item_id' => $uid,
                    'module'   => $module
                );
                jrCore_trigger_event('jrCore', 'db_create_item_exit', $_dt, $_args);
            }
            $_id[] = $uid++;
        }
        return $_id;
    }
    return false;
}

/**
 * Gets all items from a module datastore matching a key and value
 * @param string $module Module the item belongs to
 * @param string $key Key name to match
 * @param mixed $value Value to find in matched key (can be array of key => values)
 * @param bool $item_id_array if set to TRUE returns array of id's
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @param bool $index_key set to key name to use that key value as array index
 * @return array
 */
function jrCore_db_get_multiple_items_by_key($module, $key, $value, $item_id_array = false, $skip_caching = false, $index_key = false)
{
    $func = jrCore_get_active_datastore_function($module, 'db_get_multiple_items_by_key');
    $_tmp = $func($module, $key, $value, $item_id_array, $skip_caching);
    if (!$item_id_array && $index_key && is_array($_tmp)) {
        $_tmp = jrCore_db_create_item_id_index_array($_tmp, $index_key);
    }
    return $_tmp;
}

/**
 * Gets all items from a module datastore where a key value matches the search condition
 * @param string $module Module the item belongs to
 * @param string $key Key name to match
 * @param string $match Search condition - i.e. '<=', 'LIKE', etc.
 * @param string|array $value Value to find in matched key (can be array of key => values)
 * @param bool $item_id_array if set to TRUE returns array of id's
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @param bool $index_key set to key name to use that key value as array index
 * @param int $cache_profile_id Profile ID for caching
 * @return array
 */
function jrCore_db_search_multiple_items_by_key($module, $key, $match, $value, $item_id_array = false, $skip_caching = false, $index_key = false, $cache_profile_id = 0)
{
    $ckey = 'db_search_multiple_items_by_key' . json_encode(func_get_args());
    $_tmp = array(
        'module'       => $module,
        'key'          => $key,
        'match'        => $match,
        'skip_caching' => $skip_caching,
        'index_key'    => $index_key,
        '_profile_id'  => $cache_profile_id
    );
    $_tmp = jrCore_trigger_event('jrCore', 'db_search_multiple_item', $_tmp);
    if (isset($_tmp['_items'])) {
        // Our result set has been handled by a listener
        if (!$skip_caching) {
            $pid = ($cache_profile_id > 0) ? intval($cache_profile_id) : 0;
            jrCore_add_to_cache($module, $ckey, $_tmp['_items'], 0, $pid, false, false, false);
        }
        return $_tmp['_items'];
    }

    if (!$skip_caching) {
        if ($_rt = jrCore_is_cached($module, $ckey, false, false, true, false)) {
            return $_rt;
        }
    }

    $func = jrCore_get_active_datastore_function($module, 'db_search_multiple_items_by_key');
    $_tmp = $func($module, $key, $match, $value, $item_id_array, $skip_caching);
    if (!$item_id_array && $index_key && is_array($_tmp)) {
        $_tmp = jrCore_db_create_item_id_index_array($_tmp, $index_key);
    }
    if (!$skip_caching) {
        $pid = ($cache_profile_id > 0) ? intval($cache_profile_id) : 0;
        jrCore_add_to_cache($module, $ckey, $_tmp, 0, $pid, false, false, false);
    }
    return $_tmp;
}

/**
 * Gets a single item from a module datastore by key name and value
 * @param string $module Module the item belongs to
 * @param string $key Key name to find
 * @param mixed $value Value to find
 * @param bool $skip_trigger By default the db_get_item event trigger is sent out to allow additional modules to add data to the item.  Set to TRUE to just return the item from the item datastore.
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @return array|false array on success, bool false on failure
 */
function jrCore_db_get_item_by_key($module, $key, $value, $skip_trigger = false, $skip_caching = false)
{
    // See if we are cached - this is a GLOBAL cache
    // since it will be the same for any viewing user
    $ckey = "{$module}-{$key}-{$value}-" . intval($skip_trigger);
    if (!$skip_caching) {
        if ($_rt = jrCore_is_cached($module, $ckey, false, false, true, false)) {
            return (is_array($_rt)) ? $_rt : false;
        }
    }
    $func = jrCore_get_active_datastore_function($module, 'db_get_item_by_key');
    if (!$_rt = $func($module, $key, $value, $skip_trigger, $skip_caching)) {
        $_rt = 'no_data';
    }
    // Save to cache
    if (!$skip_caching) {
        $pid = 0;
        if (is_array($_rt) && !empty($_rt['_profile_id'])) {
            $pid = $_rt['_profile_id'];
        }
        jrCore_add_to_cache($module, $ckey, $_rt, 0, $pid, false, false, false);
    }
    return (is_array($_rt)) ? $_rt : false;
}

/**
 * Gets an item from a module datastore
 * @param string $module Module the item belongs to
 * @param int $id Item ID to retrieve
 * @param bool $skip_trigger By default the db_get_item event trigger is sent out to allow additional modules to add data to the item.  Set to TRUE to just return the item from the item datastore.
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @param int $cache_profile_id By default the item will be cached using the item's _profile_id value - override by providing a different profile_id
 * @return array|false
 */
function jrCore_db_get_item($module, $id, $skip_trigger = false, $skip_caching = false, $cache_profile_id = 0)
{
    if (!is_numeric($id)) {
        return false;
    }

    // See if we are cached - this is a GLOBAL cache
    // since it will be the same for any viewing user
    $key = ($skip_trigger) ? 1 : 0;
    $key = "{$module}-{$id}-{$key}";
    if (!$skip_caching) {
        if ($_rt = jrCore_is_cached($module, $key, false, false, true, false)) {
            return $_rt;
        }
    }

    $func = jrCore_get_active_datastore_function($module, 'db_get_item');
    if ($_itm = $func($module, $id, $skip_trigger, $skip_caching)) {

        // $skip_trigger can be:
        // true  = only the item data will be included
        // false = event trigger is fired and will include User, Profile and Quota info
        if ($skip_trigger === false) {

            switch ($module) {

                case 'jrProfile':
                    // We only add in Quota info (below)
                    break;

                // For Users we always add in their ACTIVE profile info
                case 'jrUser':
                    if (isset($_itm['_profile_id']) && $_itm['_profile_id'] > 0) {
                        if ($_tm = jrCore_db_get_item('jrProfile', $_itm['_profile_id'], true)) {
                            unset($_tm['_item_id']);
                            $_itm = $_itm + $_tm;
                        }
                        unset($_tm);
                    }
                    break;

                // Everything else gets BOTH User and Profile
                default:
                    if (isset($_itm['_user_id']) && $_itm['_user_id'] > 0) {
                        // Add in User Info
                        if ($_tm = jrCore_db_get_item('jrUser', $_itm['_user_id'], true)) {
                            // We do not return passwords
                            unset($_tm['_item_id'], $_tm['user_password'], $_tm['user_old_password']);
                            $_itm = $_itm + $_tm;
                        }
                        unset($_tm);
                    }
                    if (isset($_itm['_profile_id']) && $_itm['_profile_id'] > 0) {
                        // Add in Profile Info
                        if ($_tm = jrCore_db_get_item('jrProfile', $_itm['_profile_id'], true)) {
                            unset($_tm['_item_id']);
                            $_itm = $_itm + $_tm;
                        }
                        unset($_tm);
                    }
                    break;
            }

            // Add in Quota info to item
            if (isset($_itm['profile_quota_id'])) {
                if ($_tm = jrProfile_get_quota($_itm['profile_quota_id'])) {
                    unset($_tm['_item_id']);
                    $_itm = $_itm + $_tm;
                }
                unset($_tm);
            }

            $_itm = jrCore_trigger_event('jrCore', 'db_get_item', $_itm, array('module' => $module));
            // Make sure listeners did not change our _item_id
            $_itm['_item_id'] = intval($id);
        }

        // Save to cache
        if (!$skip_caching) {
            $pid = 0;
            if ($cache_profile_id > 0) {
                $pid = (int) $cache_profile_id;
            }
            elseif (isset($_itm['_profile_id'])) {
                $pid = (int) $_itm['_profile_id'];
            }
            jrCore_add_to_cache($module, $key, $_itm, 0, $pid, false, false, false);
        }

        return $_itm;
    }
    return false;
}

/**
 * Get multiple items by _item_id from a module datastore
 *
 * This function does NOT send out a trigger to add User/Profile information.  If you need
 * User and Profile information in the returned array of items, make sure and use jrCore_db_search_items
 * With an "in" search for your items ids - i.e. _item_id IN 1,5,7,9,12
 *
 * @param string $module Module the item belongs to
 * @param array $_ids array array of _item_id's to get
 * @param array $_keys Array of key names to get, default is all keys for each item
 * @param bool $skip_caching Set to TRUE to force item reload (skip caching)
 * @param bool $index_key set to a key to use that key value as the array index
 * @param int $cache_profile_id By default the item will be cached using the item's _profile_id value - override by providing a different profile_id
 * @return array|false
 */
function jrCore_db_get_multiple_items($module, $_ids, $_keys = null, $skip_caching = false, $index_key = false, $cache_profile_id = 0)
{
    if (!$_ids || !is_array($_ids)) {
        return false;
    }
    // validate id's
    $_id = array();
    foreach ($_ids as $k => $id) {
        if (!jrCore_checktype($id, 'number_nz')) {
            unset($_ids[$k]);
            continue;
        }
        $_id[$id] = $id;
    }
    if (count($_id) === 0) {
        return false;
    }

    $_tmp = array(
        'module'       => $module,
        '_ids'         => $_ids,
        '_keys'        => $_keys,
        'skip_caching' => $skip_caching,
        'index_key'    => $index_key,
        '_profile_id'  => $cache_profile_id
    );
    $_tmp = jrCore_trigger_event('jrCore', 'db_get_multiple_item', $_tmp);
    if (isset($_tmp['_items'])) {
        // Our result set has been handled by a listener
        if (!$skip_caching) {
            $key = json_encode(func_get_args());
            $pid = 0;
            if ($cache_profile_id > 0) {
                $pid = (int) $cache_profile_id;
            }
            jrCore_add_to_cache($module, $key, $_tmp['_items'], 0, $pid, false, false, false);
        }
        return $_tmp['_items'];
    }

    if (!$skip_caching) {
        $key = json_encode(func_get_args());
        if ($_rt = jrCore_is_cached($module, $key, false, false, true, false)) {
            return $_rt;
        }
    }

    $func = jrCore_get_active_datastore_function($module, 'db_get_multiple_items');
    if ($_tmp = $func($module, $_id, $_keys, $skip_caching)) {
        if ($index_key) {
            $_tmp = jrCore_db_create_item_id_index_array($_tmp, $index_key);
        }
        if (!$skip_caching) {
            $key = json_encode(func_get_args());
            $pid = 0;
            if ($cache_profile_id > 0) {
                $pid = (int) $cache_profile_id;
            }
            jrCore_add_to_cache($module, $key, $_tmp, 0, $pid, false, false, false);
        }
        return $_tmp;
    }
    return false;
}

/**
 * Gets a single item attribute from a module datastore
 * @param string $module Module the item belongs to
 * @param int $id Item ID to retrieve
 * @param string $key Key value to return
 * @return string|false
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function jrCore_db_get_item_key($module, $id, $key)
{
    if (!jrCore_checktype($id, 'number_nz')) {
        return false;
    }
    $func = jrCore_get_active_datastore_function($module, 'db_get_item_key');
    return $func($module, $id, $key);
}

/**
 * Updates multiple Item in a module datastore
 * @param string $module Module the DataStore belongs to
 * @param array $_data Array of Key => Value pairs for insertion
 * @param array $_core Array of Key => Value pairs for insertion - skips jrCore_db_get_allowed_item_keys()
 * @param bool $update set to FALSE to prevent _updated key from being set to UNIX_TIMESTAMP
 * @param bool $cache_reset set to FALSE to prevent cache reset
 * @param bool $exist_check set to FALSE to prevent checking if item exists before updating
 * @param bool $skip_triggers set to TRUE to prevent db_update_item event
 * @return bool true on success, false on error
 */
function jrCore_db_update_multiple_items($module, $_data = null, $_core = null, $update = true, $cache_reset = true, $exist_check = true, $skip_triggers = false)
{
    global $_post, $_user;
    $check = true;
    if (!$_data || !is_array($_data)) {
        if (!is_array($_core)) {
            // Nothing to do
            return false;
        }
        $_data = $_core;
        $_core = null;
        $check = false;
    }

    $pfx = jrCore_db_get_prefix($module);
    foreach ($_data as $id => $_up) {

        // Must be valid ID
        if (!jrCore_checktype($id, 'number_nz')) {
            return false;
        }
        // Keys must come in as array
        if (!is_array($_up)) {
            $_data[$id] = array();
        }
        elseif ($check) {
            if (!$_data[$id] = jrCore_db_get_allowed_item_keys($module, $_up)) {
                $_data[$id] = array();
            }
        }

        // We're being updated
        if ($update) {
            $_data[$id]['_updated'] = 'UNIX_TIMESTAMP()';
        }

        // Check for additional core fields being overridden
        if (!is_null($_core) && isset($_core[$id]) && is_array($_core[$id])) {
            foreach ($_core[$id] as $k => $v) {
                if (strpos($k, '_') === 0) {
                    $_data[$id][$k] = $v;
                }
            }
        }

        // Check for Pending Support for this module
        // We must check for this function being called as part of another (usually save)
        // routine - we don't want to change the value if this is an update that is part of a create process
        // and we don't want to change it if the update is being done by a different module (rating, comment, etc.)
        if (!jrUser_is_admin() && isset($_post['module']) && $_post['module'] == $module && !jrCore_is_magic_view()) {
            if (!jrCore_get_flag("jrcore_created_pending_item_{$module}_{$id}")) {
                $_pnd = jrCore_get_registered_module_features('jrCore', 'pending_support');
                if ($_pnd && isset($_pnd[$module])) {
                    // Pending support is on for this module - check quota
                    // 0 = immediately active
                    // 1 = review needed on CREATE
                    // 2 = review needed on CREATE and UPDATE
                    if (isset($_user["quota_{$module}_pending"]) && $_user["quota_{$module}_pending"] == '2') {
                        $_data[$id]["{$pfx}_pending"] = 1;
                    }
                }
            }
        }
    }

    // Trigger update event
    $_li = array();
    $_lm = array();
    foreach ($_data as $id => $_v) {

        if (!$skip_triggers) {
            $_args      = array(
                '_item_id' => $id,
                'module'   => $module
            );
            $_data[$id] = jrCore_trigger_event('jrCore', 'db_update_item', $_v, $_args);
        }

        // Check for actions that are linking to pending items
        $_li[$id] = 0;
        $_lm[$id] = '';
        if (isset($_v['action_pending_linked_item_id']) && jrCore_checktype($_v['action_pending_linked_item_id'], 'number_nz')) {
            $_li[$id] = (int) $_v['action_pending_linked_item_id'];
            $_lm[$id] = jrCore_db_escape($_v['action_pending_linked_item_module']);
            unset($_data[$id]['action_pending_linked_item_id']);
            unset($_data[$id]['action_pending_linked_item_module']);
        }
    }

    $func = jrCore_get_active_datastore_function($module, 'db_update_multiple_items');
    if ($func($module, $_data, $exist_check)) {

        // Check for pending
        $_rq = array();
        foreach ($_data as $id => $_vals) {
            if (!jrCore_get_flag("jrcore_created_pending_item_{$module}_{$id}")) {
                if (isset($_vals["{$pfx}_pending"]) && $_vals["{$pfx}_pending"] == '1') {
                    // @note: These values are not likely to actually be used due to the use of ON DUPLICATE KEY UPDATE
                    $uid   = (int) $_user['_user_id'];
                    $pid   = (int) $_user['user_active_profile_id'];
                    $_rq[] = "(UNIX_TIMESTAMP(), '" . jrCore_db_escape($module) . "', {$pid}, {$uid}, '{$id}', '{$_lm[$id]}', '{$_li[$id]}')";
                }
            }
        }
        if (count($_rq) > 0) {
            $pnd = jrCore_db_table_name('jrCore', 'pending');
            $req = "INSERT INTO {$pnd} (pending_created, pending_module, pending_profile_id, pending_user_id, pending_item_id, pending_linked_item_module, pending_linked_item_id)
                    VALUES " . implode(',', $_rq) . "
                    ON DUPLICATE KEY UPDATE pending_created = UNIX_TIMESTAMP()";
            $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
            if ($cnt === 1) {
                // Notify admins of new pending item
                if ($_rt = jrUser_get_admin_user_ids()) {
                    $_mi                  = jrCore_get_module_info($module);
                    $_rp                  = reset($_data);
                    $_temp                = $_rp;
                    $_temp['system_name'] = jrCore_get_config_value('jrCore', 'system_name', '');
                    $_temp['_updated']    = time();
                    $_temp['module']      = $module;
                    $_temp['module_name'] = $_mi['module_name'];
                    $_temp['item_title']  = jrCore_get_item_title($module, $_rp);
                    $_temp['item_url']    = jrCore_get_item_url('detail', $module, $_rp);
                    $_temp['_user']       = ($_rp['_user_id'] == $_user['_user_id']) ? $_user : jrCore_db_get_item('jrUser', $_rp['_user_id']);
                    list($sub, $msg) = jrCore_parse_email_templates('jrCore', 'pending_item', $_temp);
                    jrCore_db_notify_admins_of_pending_item($module, $_rt, $sub, $msg);
                }
                // Flag to reset cached pending item_id's on exit
                jrCore_db_set_pending_reset_for_module($module);
            }
        }

        if ($cache_reset) {
            $_ch = array();
            foreach ($_data as $id => $_vals) {
                // module-item_id-skip_triggers
                $_ch[] = array($module, "{$module}-{$id}-0");
                $_ch[] = array($module, "{$module}-{$id}-1");
            }
            jrCore_delete_multiple_cache_entries($_ch);
        }

        // Let other modules know we have updated
        if (!$skip_triggers) {
            jrCore_trigger_event('jrCore', 'db_update_multiple_item', $_data, array('module' => $module));
        }

        return true;
    }
    return false;
}

/**
 * Updates an Item in a module datastore
 * @param string $module Module the DataStore belongs to
 * @param int $id Unique ID to update
 * @param array $_data Array of Key => Value pairs for insertion
 * @param array $_core Array of Key => Value pairs for insertion - skips jrCore_db_get_allowed_item_keys()
 * @param bool $update set to FALSE to prevent _updated key from being set to UNIX_TIMESTAMP
 * @param bool $cache_reset set to FALSE to prevent cache reset
 * @param bool $exist_check set to prevent checking if item exists before updating
 * @param bool $skip_triggers set to TRUE to prevent db_update_item event
 * @return bool true on success, false on error
 */
function jrCore_db_update_item($module, $id, $_data = null, $_core = null, $update = true, $cache_reset = true, $exist_check = true, $skip_triggers = false)
{
    $_dt = array(
        $id => $_data
    );
    $_cr = null;
    if (!is_null($_core)) {
        $_cr = array(
            $id => $_core
        );
    }
    return jrCore_db_update_multiple_items($module, $_dt, $_cr, $update, $cache_reset, $exist_check, $skip_triggers);
}

/**
 * Deletes an Item in the module DataStore
 * By default this function will also delete any media files that are associated with the item id!
 * @param string $module Module the DataStore belongs to
 * @param int $id Item ID to delete
 * @param bool $delete_media Set to false to NOT delete associated media files
 * @param mixed $profile_count If set to true, profile_count will be decremented by 1 for given _profile_id.  If set to an integer, it will be used as the profile_id for the counts
 * @param bool $cache_reset set to FALSE to skip resetting caching for item
 * @param bool $recycle_bin set to FALSE to skip adding item to recycle bin
 * @param bool $skip_triggers set to TRUE to skip sending db_delete_item event
 * @return bool
 */
function jrCore_db_delete_item($module, $id, $delete_media = true, $profile_count = true, $cache_reset = true, $recycle_bin = true, $skip_triggers = false)
{
    $id = array(intval($id));
    return jrCore_db_delete_multiple_items($module, $id, $delete_media, $profile_count, $cache_reset, $recycle_bin, $skip_triggers);
}

/**
 * Delete multiple items from a module DataStore
 * @param $module string Module DataStore belongs to
 * @param $_ids array Array of _item_id's to delete
 * @param bool $delete_media Set to false to NOT delete associated media files
 * @param mixed $profile_count If set to true, profile counts for the deleted items will be decremented
 * @param bool $cache_reset set to FALSE to prevent cache reset after deletion
 * @param bool $recycle_bin set to FALSE to prevent items being added to recycle bin
 * @param bool $skip_triggers set to TRUE to skip sending db_delete_item triggers
 * @return bool
 */
function jrCore_db_delete_multiple_items($module, $_ids, $delete_media = true, $profile_count = true, $cache_reset = true, $recycle_bin = true, $skip_triggers = false)
{
    @ini_set('max_execution_time', 1800); // 30 minutes max
    if (!is_array($_ids) || count($_ids) === 0) {
        return false;
    }
    // validate id's
    foreach ($_ids as $id) {
        if (!jrCore_checktype($id, 'number_nz')) {
            return false;
        }
    }
    // Get all items so we can check for attached media
    // NOTE: Do not use $_keys here - we need all keys for the Recycle Bin
    $_it = jrCore_db_get_multiple_items($module, $_ids);
    if (!$_it || !is_array($_it)) {
        // no items matching
        return true;
    }

    $func = jrCore_get_active_datastore_function($module, 'db_delete_multiple_items');
    if ($func($module, $_ids, $delete_media, $profile_count)) {

        // At this point the items have been removed from the DB
        // Remove from Cache and Pending
        $_ch = array();
        $_pn = array();
        $_ui = array();
        $_pi = array();
        $_uc = array();
        $_pc = array();
        $pfx = jrCore_db_get_prefix($module);
        foreach ($_it as $_item) {

            $iid = (int) $_item['_item_id'];
            $uid = (int) $_item['_user_id'];
            $pid = (int) $_item['_profile_id'];

            $_ui[$uid] = $pid;
            $_pi[$pid] = $pid;

            // User Counts
            if (!isset($_uc[$uid])) {
                $_uc[$uid] = 0;
            }
            $_uc[$uid]++;

            // Profile Counts
            if (!isset($_pc[$pid])) {
                $_pc[$pid] = 0;
            }
            $_pc[$pid]++;

            // reset caches
            if ($cache_reset) {
                // module, key, (user logged in)
                $_ch[] = array($module, "{$module}-{$iid}-0");
                $_ch[] = array($module, "{$module}-{$iid}-1");
                if (count($_ch) >= 100) {
                    jrCore_delete_multiple_cache_entries($_ch);
                    $_ch = array();
                }
            }

            // If this item was pending, remove it from pending
            if (isset($_item["{$pfx}_pending"]) && $_item["{$pfx}_pending"] >= 1) {
                $_pn[] = $iid;
            }
        }
        // Handle any left overs
        if (count($_ch) > 0) {
            jrCore_delete_multiple_cache_entries($_ch);
        }

        $_qe = array(
            'module' => $module,
            '_files' => array()
        );
        // Is the Recycle Bin turned on?
        if ($recycle_bin && jrCore_get_config_value('jrCore', 'recycle_bin', 'on') == 'on') {

            // Yes - recycle bin is on
            $_qe['recycle_bin'] = 1;

            $_rb = array();
            $mod = jrCore_db_escape($module);
            foreach ($_it as $_item) {

                switch ($module) {
                    case 'jrProfile':
                        $tag = 'profile_id';
                        $iid = (int) $_item['_profile_id'];
                        $ttl = (isset($_item['profile_name'])) ? jrCore_strip_non_utf8($_item['profile_name']) : 'unknown';
                        break;
                    case 'jrUser':
                        $tag = 'user_id';
                        $iid = (int) $_item['_user_id'];
                        $ttl = (isset($_item['user_name'])) ? jrCore_strip_non_utf8($_item['user_name']) : 'unknown';
                        break;
                    default:
                        $tag = 'item_id';
                        $iid = (int) $_item['_item_id'];
                        if (!$ttl = jrCore_get_item_title($module, $_item)) {
                            $ttl = '?';
                        }
                        break;
                }

                // When we first come into db_delete_multiple_items, it will be for the item
                // being removed (profile, audio, etc).  There are listeners that are
                // going to delete other items related to this item (i.e. ratings, comments,
                // etc.) - each of those will need to be part of our group id.
                $gid = jrCore_get_flag('jrCore_db_delete_item_group_id');
                if (!$gid || strlen($gid) < 2) {
                    // We are the FIRST item in our delete set
                    $gid = 1;
                    jrCore_set_flag('jrCore_db_delete_item_group_id', "{$module}:{$iid}");
                }

                // Handle this item's media
                if ($delete_media) {
                    // Rename associated media files with rb_ prefix
                    foreach ($_item as $k => $v) {
                        if (strpos($k, '_extension')) {
                            $field                  = str_replace('_extension', '', $k);
                            $_qe['_files'][]        = array($_item['_profile_id'], $_item['_item_id'], $field);
                            $_item['rb_item_media'] = 1;
                        }
                    }
                }

                // Gather any additional custom data about this item.  A module can register
                // for RB support on additional tables by giving the table and column to match
                $_rbt = jrCore_get_registered_module_features('jrCore', "recycle_bin_{$tag}_table");

                // We have extra recycle bin tables to handle
                if ($_rbt && is_array($_rbt)) {
                    foreach ($_rbt as $rb_module => $_rb_tables) {
                        foreach ($_rb_tables as $rb_table => $rb_column) {

                            $rbl = jrCore_db_table_name($rb_module, $rb_table);
                            $sql = false;
                            $_rs = false;
                            if ($tag == 'item_id') {
                                // For item_id RB tables, the $rb_column value can contain some extra info. $rb_column can be in 1 of 2 formats:
                                // 1) module_column,item_id_column - will run for ALL modules and match the module and _item_id
                                // 2) module_dir:item_id_column - if prefaced with a MODULE directory name, the delete only happens for that specific module
                                if (strpos($rb_column, ':')) {
                                    // We have to be deleting an item from the MATCHING module in order to run
                                    if (strpos($rb_column, "{$module}:") === 0) {
                                        list(, $rbc_column) = explode(':', $rb_column, 2);
                                        $rbc_column = trim($rbc_column);
                                        if (!empty($rbc_column)) {
                                            $sql = "SELECT * FROM {$rbl} WHERE `{$rbc_column}` = '{$iid}'";
                                            $_rs = jrCore_db_query($sql, 'NUMERIC', false, null, false);
                                        }
                                    }
                                }
                                else {
                                    // This module stores entries by MODULE => ITEM_ID, so we are getting the module column + the item_id column
                                    list($rbc_module, $rbc_column) = explode(',', $rb_column, 2);
                                    $rbc_module = trim($rbc_module);
                                    $rbc_column = trim($rbc_column);
                                    if (!empty($rbc_module) && !empty($rbc_column)) {
                                        $sql = "SELECT * FROM {$rbl} WHERE `{$rbc_module}` = '" . jrCore_db_escape($module) . "' AND `{$rbc_column}` = '{$iid}'";
                                        $_rs = jrCore_db_query($sql, 'NUMERIC', false, null, false);
                                    }
                                }
                            }
                            else {
                                $sql = "SELECT * FROM {$rbl} WHERE `{$rb_column}` = '{$iid}'";
                                $_rs = jrCore_db_query($sql, 'NUMERIC', false, null, false);
                            }

                            if ($sql && $_rs && is_array($_rs)) {
                                if (!isset($_item['_tables'])) {
                                    $_item['_tables'] = array();
                                }
                                if (!isset($_item['_tables'][$rb_module])) {
                                    $_item['_tables'][$rb_module] = array();
                                }
                                if (!isset($_item['_tables'][$rb_module][$rb_table])) {
                                    $_item['_tables'][$rb_module][$rb_table] = $_rs;
                                }
                                else {
                                    $_item['_tables'][$rb_module][$rb_table] = array_merge($_item['_tables'][$rb_module][$rb_table], $_rs);
                                }
                                // Cleanup
                                $sql = str_replace('SELECT *', 'DELETE', $sql);
                                jrCore_db_query($sql, null, false, null, false);
                            }

                        }
                    }
                }

                // Store it's data in our recycle bin
                $_rb[] = "('{$gid}',UNIX_TIMESTAMP(),'{$mod}','" . intval($_item['_profile_id']) . "','{$iid}','" . jrCore_db_escape(mb_substr($ttl, 0, 254)) . "','" . jrCore_db_escape(json_encode($_item)) . "')";

                // Trigger Delete Event
                // NOTE: This must stay in this location!
                if (!$skip_triggers) {
                    $_args = array(
                        'module'   => $module,
                        '_item_id' => $iid
                    );
                    jrCore_trigger_event('jrCore', 'db_delete_item', $_item, $_args);
                }

                if ($gid == 1) {
                    jrCore_delete_flag('jrCore_db_delete_item_group_id');
                }

                // Insert into Recycle Bin every 100 entries
                if (count($_rb) >= 100) {

                    $rbl = jrCore_db_table_name('jrCore', 'recycle');
                    $req = "INSERT INTO {$rbl} (r_group_id,r_time,r_module,r_profile_id,r_item_id,r_title,r_data) VALUES " . implode(',', $_rb) . "
                            ON DUPLICATE KEY UPDATE r_time = UNIX_TIMESTAMP(), r_title = VALUES(r_title), r_data = VALUES(r_data)";
                    jrCore_db_query($req, null, false, null, false);
                    $_rb = array();

                    // Handle media files
                    if (count($_qe['_files']) > 0) {
                        jrCore_queue_create('jrCore', 'db_delete_item_media', $_qe);
                        $_qe['_files'] = array();
                    }
                }

            }

            // Catch last deleted entries and insert in to Recycle bin
            if (count($_rb) > 0) {
                $rbl = jrCore_db_table_name('jrCore', 'recycle');
                $req = "INSERT INTO {$rbl} (r_group_id,r_time,r_module,r_profile_id,r_item_id,r_title,r_data) VALUES " . implode(',', $_rb) . "
                        ON DUPLICATE KEY UPDATE r_time = UNIX_TIMESTAMP(), r_title = VALUES(r_title), r_data = VALUES(r_data)";
                jrCore_db_query($req, null, false, null, false);
            }

        }
        else {

            // Recycle Bin is turned off
            $_qe['recycle_bin'] = 0;

            foreach ($_it as $_item) {

                switch ($module) {
                    case 'jrProfile':
                        $tag = 'profile_id';
                        $iid = (int) $_item['_profile_id'];
                        break;
                    case 'jrUser':
                        $tag = 'user_id';
                        $iid = (int) $_item['_user_id'];
                        break;
                    default:
                        $tag = 'item_id';
                        $iid = (int) $_item['_item_id'];
                        break;
                }

                // Delete associated media files via queue
                if ($delete_media) {
                    foreach ($_item as $k => $v) {
                        if (strpos($k, '_extension')) {
                            $field           = str_replace('_extension', '', $k);
                            $_qe['_files'][] = array($_item['_profile_id'], $_item['_item_id'], $field);
                        }
                    }
                }

                // Gather additional tables and clean up
                $_rbt = jrCore_get_registered_module_features('jrCore', "recycle_bin_{$tag}_table");

                // We have extra recycle bin tables to handle
                // @note: We are skipping recycle bin here so we just delete
                if ($_rbt && is_array($_rbt)) {
                    $_rq = array();
                    foreach ($_rbt as $rb_module => $_rb_tables) {
                        foreach ($_rb_tables as $rb_table => $rb_column) {
                            $rbl = jrCore_db_table_name($rb_module, $rb_table);

                            // For item_id RB tables, the $rb_column value can contain some extra info. $rb_column can be in 1 of 2 formats:
                            // 1) module_column,item_id_column - will run for ALL modules and match the module and _item_id
                            // 2) module_dir:item_id_column - if prefaced with a MODULE directory name, the delete only happens for that specific module
                            if (strpos($rb_column, ':')) {
                                // We have to be deleting an item from the MATCHING module in order to run
                                if (strpos($rb_column, "{$module}:") === 0) {
                                    list(, $rbc_column) = explode(':', $rb_column, 2);
                                    $rbc_column = trim($rbc_column);
                                    if (!empty($rbc_column)) {
                                        $_rq[] = "DELETE FROM {$rbl} WHERE `{$rbc_column}` = '{$iid}'";
                                    }
                                }
                            }
                            elseif (strpos($rb_column, ',')) {
                                // This module stores entries by MODULE => ITEM_ID, so we are
                                // getting the module column + the item_id column
                                list($rbc_module, $rbc_column) = explode(',', $rb_column, 2);
                                $rbc_module = trim($rbc_module);
                                $rbc_column = trim($rbc_column);
                                $_rq[]      = "DELETE FROM {$rbl} WHERE `{$rbc_module}` = '" . jrCore_db_escape($module) . "' AND `{$rbc_column}` = '{$iid}'";
                            }
                            else {
                                $_rq[] = "DELETE FROM {$rbl} WHERE `{$rb_column}` = '{$iid}'";
                            }
                        }
                    }
                    if (count($_rq) > 0) {
                        jrCore_db_multi_query($_rq, false, false);
                    }
                }

                // Trigger delete event
                if (!$skip_triggers) {
                    $_args = array(
                        'module'   => $module,
                        '_item_id' => $_item['_item_id']
                    );
                    jrCore_trigger_event('jrCore', 'db_delete_item', $_item, $_args);
                }

            }
        }

        // Handle media files
        if ($delete_media && count($_qe['_files']) > 0) {
            jrCore_queue_create('jrCore', 'db_delete_item_media', $_qe);
        }

        // Take care of profile counts
        if ($profile_count) {
            switch ($module) {

                // We do not maintain counts for some modules
                case 'jrProfile':
                case 'jrUser':
                case 'jrCore':
                    break;

                default:
                    // Profile Counts
                    if (count($_pc) > 0) {
                        foreach ($_pc as $pid => $cnt) {
                            if ($pid > 0) {
                                jrCore_db_decrement_key('jrProfile', $pid, "profile_{$module}_item_count", $cnt, 0, false, true, $_pi);
                            }
                        }
                    }
                    // User Counts
                    if (count($_uc) > 0) {
                        foreach ($_uc as $uid => $cnt) {
                            if ($uid > 0) {
                                jrCore_db_decrement_key('jrUser', $uid, "user_{$module}_item_count", $cnt, 0, false, true, $_ui);
                            }
                        }
                    }
                    break;
            }
        }

        // Some of these items were pending - cleanup pending table
        if (count($_pn) > 0) {
            $tbl = jrCore_db_table_name('jrCore', 'pending');
            $req = "DELETE FROM {$tbl} WHERE (`pending_module` = '{$module}' AND `pending_item_id` IN(" . implode(',', $_pn) . ")) OR (`pending_linked_item_module` = '{$module}' AND `pending_linked_item_id` IN(" . implode(',', $_pn) . "))";
            jrCore_db_query($req, null, false, null, false);
            jrCore_db_set_pending_reset_for_module($module);
        }

        // Trigger delete multiple event
        if (!$skip_triggers) {
            $_data = array(
                'module'    => $module,
                '_item_ids' => $_ids
            );
            jrCore_trigger_event('jrCore', 'db_delete_multiple_item', $_data);
        }

    }
    return true;
}

/**
 * Get an item order array for a given module and profile_id
 * @param string $module
 * @param int $profile_id
 * @param string $order_dir
 * @return array|bool
 */
function jrCore_db_get_display_order_for_profile_id($module, $profile_id, $order_dir)
{
    $pid = (int) $profile_id;
    $key = "display_order_for_profile_id_{$pid}";
    if (!$_rt = jrCore_is_cached($module, $key, false, false, true, false)) {
        if (jrCore_is_datastore_module($module)) {
            $func = jrCore_get_active_datastore_function($module, 'db_get_display_order_for_profile_id');
            if (function_exists($func)) {
                $_rt = $func($module, $profile_id);
                if (!is_array($_rt)) {
                    $_rt = 'no_items';
                }
                else {
                    // With display_order we order by <pfx>_display_order ASC, _item_id DESC
                    krsort($_rt, SORT_NUMERIC);
                    switch (strtolower($order_dir)) {
                        case 'asc':
                        case 'numerical_asc':
                            asort($_rt, SORT_NUMERIC);
                            break;
                        case 'desc':
                        case 'numerical_desc':
                            arsort($_rt, SORT_NUMERIC);
                            break;
                    }
                    $_rt = array_keys($_rt);
                }
            }
            else {
                $_rt = 'no_items';
            }
        }
        else {
            $_rt = 'no_items';
        }
        jrCore_add_to_cache($module, $key, $_rt, 0, $pid, false, false, false);
    }
    return (is_array($_rt)) ? $_rt : false;
}

/**
 * Get Private Profiles
 * @return array|bool
 */
function jrCore_db_get_private_profiles()
{
    // Get profiles that are NOT public and are allowed to change their profile privacy
    $key = 'db_get_private_profiles';
    if (!$_pp = jrCore_is_cached('jrCore', $key, false, false, true, false)) {
        $func = jrCore_get_active_datastore_function('jrProfile', 'db_get_private_profiles');
        if (function_exists($func)) {
            $_pp = $func();
            if (!$_pp || !is_array($_pp)) {
                $_pp = 'no_items';
            }
            jrCore_add_to_cache('jrCore', $key, $_pp, 0, 0, false, false, false);
        }
    }
    return (is_array($_pp)) ? $_pp : false;
}

/**
 * Get an array of pending item_ids for a module
 * @param string $module
 * @return array|bool
 */
function jrCore_db_get_pending_item_ids($module)
{
    $key = "db_get_pending_item_ids_{$module}";
    if (!$_pp = jrCore_is_cached('jrCore', $key, false, false, true, false)) {
        $func = jrCore_get_active_datastore_function($module, 'db_get_pending_item_ids');
        if (function_exists($func)) {
            $_pp = $func($module);
            if (!$_pp || !is_array($_pp)) {
                $_pp = 'no_items';
            }
            else {
                $_pp = array_keys($_pp);
            }
            jrCore_add_to_cache('jrCore', $key, $_pp, 0, 0, false, false, false);
        }
    }
    return (is_array($_pp)) ? $_pp : false;
}

/**
 * Search a module DataStore and return matching items
 *
 * $_params is an array that contains all the function parameters - i.e.:
 *
 * <code>
 * $_params = array(
 *     'search' => array(
 *         'user_name = brian',
 *         'user_height > 72'
 *     ),
 *     'order_by' => array(
 *         'user_name' => 'asc',
 *         'user_height' => 'desc'
 *     ),
 *     'group_by' => '_user_id',
 *     'return_keys' => array(
 *         'user_email',
 *         'username'
 *      ),
 *     'index_key' => true,
 *     'return_count' => true|false,
 *     'limit' => 50
 * );
 *
 * wildcard searches use a % in the key name:
 * 'search' => array(
 *     'user_% = brian',
 *     '% like brian%'
 * );
 * </code>
 *
 * "no_cache" - by default search results are cached - this will disable caching if set to true
 *
 * "cache_seconds" - set length of time result set is cached
 *
 * "return_keys" - only return the matching keys
 *
 * "return_count" - If the "return_count" parameter is set to TRUE, then only the COUNT of matching
 * entries will be returned.
 *
 * "index_key" - Set to the name of a DS key, the value of the key will be used as the index in the result set
 *
 * "privacy_check" - by default only items that are viewable to the calling user will be returned -
 * set "privacy_check" to FALSE to disable privacy settings checking.
 *
 * "ignore_pending" - by default only items that are NOT pending are shown - set ignore_pending to
 * TRUE to skip the pending item check
 *
 * "exclude_(module)_keys" - some modules (such as jrUser and jrProfile) add extra keys into the returned
 * results - you can skip adding these extra keys in by disable the module(s) you do not want keys for.
 *
 * "skip_triggers" - don't run the db_search_params or db_search_items event triggers
 *
 * Valid Search conditions are:
 * <code>
 *  =           - "equals"
 *  !=          - "not equals"
 *  >           - greater than
 *  >=          - greater than or equal to
 *  <           - less than
 *  <=          - less than or equal to
 *  between     - between and including a low,high value - i.e. "profile_latitude between 1.50,1.60
 *  not_between - not between anf including low,high value - i.e. "profile_latitude not_between 1.50,1.60
 *  like        - wildcard text search - i.e. "user_name like %ob%" would find "robert" and "bob". % is wildcard character.
 *  not_like    - wildcard text negated search - same format as "like"
 *  in          - "in list" of values - i.e. "user_name in brian,douglas,paul,michael" would find all 4 matches
 *  not_in      - negated "in least" search - same format as "in"
 *  regexp      - MySQL regular expression match
 * </code>
 * @param string $module Module the DataStore belongs to
 * @param array $_params Search Parameters
 * @return array|bool
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function jrCore_db_search_items($module, $_params)
{
    if (!$_params || !is_array($_params) || count($_params) === 0) {
        return false;
    }
    if (isset($_params['skip_all_checks']) && jrCore_checktype($_params['skip_all_checks'], 'is_true')) {
        // If skip_all_checks is enabled, we skip all triggers, pending, missing and privacy checks
        $_params['skip_triggers']  = true;
        $_params['ignore_pending'] = true;
        $_params['ignore_missing'] = true;
        $_params['privacy_check']  = false;
        $_params['quota_check']    = false;
    }
    $func = jrCore_get_active_datastore_function($module, 'db_search_items');
    return $func($module, $_params);
}

/**
 * Log jrCore_db_search_items to the fdebug log
 * @param array $_backup
 * @param array $_params
 * @param string $d_query
 * @param string $d_query_time
 * @param array $_results
 * @param string $c_query
 * @param string $c_query_time
 * @return bool
 */
function jrCore_db_search_fdebug($_backup, $_params, $d_query, $d_query_time, $_results, $c_query = null, $c_query_time = null)
{
    if (isset($_params['fdebug']) && jrCore_checktype($_params['fdebug'], 'is_true')) {
        $_remove = array(
            'cache_key'
        );
        foreach ($_remove as $key) {
            if (isset($_backup[$key])) {
                unset($_backup[$key]);
            }
            if (isset($_params[$key])) {
                unset($_params[$key]);
            }
        }
        if (!is_null($c_query)) {
            fdebug("\nOriginal Parameters:\n" . print_r($_backup, true) . "\nModified Parameters:\n" . print_r($_params, true) . "\nPagination Query:\n{$c_query}\n\nPagination Query Time: {$c_query_time}\n\nData Query:\n{$d_query}\n\nData Query Time: {$d_query_time}\n\n" . "Result Set:\n" . print_r($_results, true)); // OK
        }
        else {
            fdebug("\nOriginal Parameters:\n" . print_r($_backup, true) . "\nModified Parameters:\n" . print_r($_params, true) . "\nData Query:\n{$d_query}\n\nData Query Time: {$d_query_time}\n\n" . "Result Set:\n" . print_r($_results, true)); // OK
        }
    }
    return true;
}

/**
 * Log an internal search error
 * @param string $error
 * @param array $_backup
 * @param array $_params
 */
function jrCore_db_search_fdebug_error($error, $_backup, $_params)
{
    if (isset($_params['fdebug']) && jrCore_checktype($_params['fdebug'], 'is_true')) {
        fdebug("\njrCore_db_search_items error: {$error}\n\nOriginal Parameters:\n" . print_r($_backup, true) . "\nModified Parameters:\n" . print_r($_params, true)); // OK
    }
}

/**
 * Check if a given search operator is valid
 * @param $search string Search Condition
 * @return array
 */
function jrCore_db_check_for_supported_operator($search)
{
    $cd = false;
    @list($key, $opt, $val) = explode(' ', trim($search), 3);
    switch (jrCore_str_to_lower($opt)) {
        case '>':
        case '>=':
        case '<':
        case '<=':
            if (strpos($val, '.')) {
                $cd = array($key, $opt, floatval($val));
            }
            else {
                $cd = array($key, $opt, intval($val));
            }
            break;
        case '!=':
        case '=':
        case 'like':
        case 'regexp':
            $cd = array($key, $opt, "'" . jrCore_db_escape($val) . "'");
            break;
        case 'not_like':
            $cd = array($key, 'not like', "'" . jrCore_db_escape($val) . "'");
            break;
        case 'in':
            $_vl = array();
            foreach (explode(',', $val) as $iv) {
                if (ctype_digit("{$iv}")) {
                    $_vl[] = (int) $iv;
                }
                else {
                    $_vl[] = "'" . jrCore_db_escape($iv) . "'";
                }
            }
            $val = "(" . implode(',', $_vl) . ") ";
            $cd  = array($key, 'IN', $val);
            break;
        case 'not_in':
            $_vl = array();
            foreach (explode(',', $val) as $iv) {
                if (ctype_digit("{$iv}")) {
                    $_vl[] = (int) $iv;
                }
                else {
                    $_vl[] = "'" . jrCore_db_escape($iv) . "'";
                }
            }
            $val = "(" . implode(',', $_vl) . ") ";
            $cd  = array($key, 'NOT IN', $val);
            break;
        case 'between':
        case 'not_between':
            if (strpos($val, ',')) {
                list($vl1, $vl2) = explode(',', $val);
                $vl1 = trim($vl1);
                $vl2 = trim($vl2);
                if (is_numeric($vl1) && is_numeric($vl2)) {
                    if (strpos(' ' . $vl1, '.')) {
                        $vl1 = floatval($vl1);
                    }
                    else {
                        $vl1 = intval($vl1);
                    }
                    if (strpos(' ' . $vl2, '.')) {
                        $vl2 = floatval($vl2);
                    }
                    else {
                        $vl2 = intval($vl2);
                    }
                    if ($vl2 < $vl1) {
                        $cd = array($key, jrCore_str_to_lower($opt), $vl2, $vl1);
                    }
                    else {
                        $cd = array($key, jrCore_str_to_lower($opt), $vl1, $vl2);
                    }
                }
            }
            break;
    }
    return $cd;
}

/**
 * Return TRUE if key is a key that is found on all items in the DS
 * @param string $key
 * @return bool
 */
function jrCore_db_key_found_on_all_items($key)
{
    switch ($key) {
        case '_item_id':
        case '_user_id':
        case '_profile_id':
        case '_created':
        case '_updated':
            return true;
        default:
            if (strpos($key, '_title')) {
                return true;
            }
            if (strpos($key, '_count')) {
                return true;
            }
            if (strpos($key, '_order')) {
                return true;
            }
            if (strpos($key, '_pending')) {
                return true;
            }
            break;
    }
    return false;
}

/**
 * Some key searches do NOT need an index offset check
 * @param $key string DS key name
 * @return bool
 */
function jrCore_is_ds_index_needed($key)
{
    switch ($key) {
        case '_created':
        case '_updated':
        case 'user_active':
        case 'user_birthdate':
        case 'user_email':
        case 'user_group':
        case 'user_last_login':
        case 'user_language':
        case 'user_name':
        case 'user_validate':
        case 'user_validated':
        case 'profile_active':
        case 'profile_disk_usage':
        case 'profile_location':
        case 'profile_name':
        case 'profile_private':
        case 'profile_quota_id':
        case 'profile_url':
        case 'ut_increment_key':
        case 'ut_decrement_key':
            return false;
    }
    $key = ' ' . $key;
    if (strpos($key, '_image_')) {
        return false;
    }
    elseif (strpos($key, 'date')) {
        return false;
    }
    elseif (strpos($key, 'rating')) {
        return false;
    }
    elseif (strpos($key, 'type')) {
        return false;
    }
    elseif (strpos($key, 'profile_id')) {
        return false;
    }
    elseif (strpos($key, 'user_id')) {
        return false;
    }
    elseif (strpos($key, 'item_id')) {
        return false;
    }
    elseif (strpos($key, 'group_id')) {
        return false;
    }
    elseif (strpos($key, '_number')) {
        return false;
    }
    elseif (strpos($key, '_version')) {
        return false;
    }
    elseif (strpos($key, '_email')) {
        return false;
    }
    elseif (strpos($key, '_genre')) {
        return false;
    }
    elseif (strpos($key, '_album')) {
        return false;
    }
    elseif (strpos($key, '_active')) {
        return false;
    }
    elseif (strpos($key, '_size')) {
        return false;
    }
    elseif (strpos($key, '_color')) {
        return false;
    }
    elseif (strpos($key, '_breed')) {
        return false;
    }
    elseif (strpos($key, 'latitude')) {
        return false;
    }
    elseif (strpos($key, 'longitude')) {
        return false;
    }
    elseif (strpos($key, 'geo_lat')) {
        return false;
    }
    elseif (strpos($key, 'geo_long')) {
        return false;
    }
    elseif (strpos($key, 'ckey')) {
        return false;
    }
    elseif (strpos($key, 'pkey')) {
        return false;
    }
    elseif (strpos($key, '_approve')) {
        return false;
    }
    elseif (strpos($key, '_enabled')) {
        return false;
    }
    elseif (strlen($key) > 3) {
        if (strrpos($key, '_count', -6)) {
            return false;
        }
        elseif (strrpos($key, '_name', -5)) {
            return false;
        }
        elseif (strrpos($key, '_title', -6)) {
            return false;
        }
        elseif (strrpos($key, '_url', -4)) {
            return false;
        }
    }
    return true;
}

/**
 * Delete the Private Profile cache
 * @return bool
 */
function jrCore_db_delete_private_profile_cache()
{
    $key = 'db_get_private_profiles';
    jrCore_delete_flag($key);
    return jrCore_delete_cache('jrCore', $key, false, false, false);
}

/**
 * Set flag for modules needed pending item_id cache reset
 * @param string $module
 * @return bool
 */
function jrCore_db_set_pending_reset_for_module($module)
{
    $key = "core_db_set_pending_reset_for_module";
    if (!$_pm = jrCore_get_flag($key)) {
        $_pm = array();
    }
    $_pm[$module] = 1;
    return jrCore_set_flag($key, $_pm);
}

/**
 * Get flags for cache resets needed for pending modules
 * @return mixed
 */
function jrCore_db_get_pending_item_id_resets()
{
    return jrCore_get_flag('core_db_set_pending_reset_for_module');
}

/**
 * Delete all pending item ids for a given module
 * @param string $module Module
 * @return bool
 */
function jrCore_db_delete_pending_item_ids_cache($module)
{
    $key = "db_get_pending_item_ids_{$module}";
    jrCore_delete_flag($key);
    return jrCore_delete_cache('jrCore', $key, false, false, false);
}

/**
 * Delete pending items cache on process exit
 */
function jrCore_process_exit_delete_pending_items_cache()
{
    if ($_pi = jrCore_db_get_pending_item_id_resets()) {
        foreach ($_pi as $m => $i) {
            jrCore_db_delete_pending_item_ids_cache($m);
        }
    }
    return true;
}

/**
 * Notify admin users of pending item (wrapper)
 * @param $module string Module
 * @param $_admins array Admin User_id's
 * @param $subject string
 * @param $message string
 * @return bool
 */
function jrCore_db_notify_admins_of_pending_item($module, $_admins, $subject, $message)
{
    $key = 'jrcore_db_pending_notified';
    if (!$_tm = jrCore_get_flag($key)) {
        $_tm = array();
    }
    if (isset($_tm[$module])) {
        // We already notified in this process for this module - not again
        return true;
    }
    jrUser_notify($_admins, 0, 'jrCore', 'pending_item', $subject, $message);
    $_tm[$module] = 1;
    jrCore_set_flag($key, $_tm);
    return true;
}

/**
 * Take 2 arrays and return only values in first array that do not exist in the second
 * @param array $_include
 * @param array $_exclude
 * @return array
 */
function jrCore_create_combined_equal_array($_include, $_exclude)
{
    foreach ($_exclude as $i) {
        if (isset($_include[$i])) {
            unset($_include[$i]);
        }
    }
    return $_include;
}

/**
 * Create a new array where the _item_id is the array index
 * @param array $_array
 * @param string $index_key
 * @return array|false
 */
function jrCore_db_create_item_id_index_array($_array, $index_key = '_item_id')
{
    if (!empty($_array)) {
        $_tmp = array();
        foreach ($_array as $v) {
            $_tmp["{$v[$index_key]}"] = $v;
        }
        return $_tmp;
    }
    return false;
}

/**
 * Get list of items to be used in an IN() or NOT_IN() list
 * @param array $_ids
 * @param array $_params
 * @return array
 */
function jrCore_db_get_item_id_in_list($_ids, $_params)
{
    if (isset($_params['simplepagebreak']) && (!isset($_params['limit']) || $_params['limit'] > $_params['simplepagebreak'])) {
        if (!isset($_params['page']) || !jrCore_checktype($_params['page'], 'number_nz')) {
            $_params['page'] = 1;
        }
        $_ids = array_slice($_ids, (($_params['page'] - 1) * $_params['simplepagebreak']), ($_params['simplepagebreak'] + 1));
    }
    elseif (isset($_params['pagebreak']) && (!isset($_params['limit']) || $_params['limit'] > $_params['pagebreak'])) {
        if (!isset($_params['page']) || !jrCore_checktype($_params['page'], 'number_nz')) {
            $_params['page'] = 1;
        }
        $_ids = array_slice($_ids, (($_params['page'] - 1) * $_params['pagebreak']), $_params['pagebreak']);
    }
    else {
        $field_limit = 10;
        if (!empty($_params['limit'])) {
            $field_limit = (int) $_params['limit'];
        }
        $_ids = array_slice($_ids, 0, $field_limit);
    }
    return $_ids;
}

/**
 * Update core DS row counts
 * @return array|false
 */
function jrCore_db_update_row_counts()
{
    if ($_ds = jrCore_get_datastore_modules()) {

        // We are going to save the CURRENT counts to the temp value
        // table so every 10 minutes we can increment our DS created counters
        $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
        $req = "SELECT temp_module, temp_value FROM {$tbl} WHERE temp_module IN('" . implode("','", array_keys($_ds)) . "') AND temp_key = 'core_ds_new_count'";
        $_rt = jrCore_db_query($req, 'temp_module', false, 'temp_value');

        $_nw = array();
        $_cn = array();
        foreach ($_ds as $m => $p) {
            $_cn[$m] = jrCore_db_number_rows($m, 'item', true);
            if (isset($_rt[$m])) {
                $diff = ($_cn[$m] - intval($_rt[$m]));
                if ($diff < 0) {
                    $diff = 0;
                }
                $_nw[$m] = $diff;
            }
            else {
                $_nw[$m] = $_cn[$m];
            }
        }

        // Save TOTAL row counters - these are used to determine pagination
        jrCore_add_to_cache('jrCore', 'core_ds_row_counts', $_cn, 0, 0, false, false);
        jrCore_set_temp_value('jrCore', 'core_ds_row_counts', $_cn);

        // Save DS NEW stats
        if (!empty($_nw)) {
            $_rq = array();
            foreach ($_nw as $m => $c) {
                jrCore_create_stat_entry($m, 'ds_counts', 'created', 0, 0, null, false, $c);
                $_rq[] = "('{$m}', 'core_ds_new_count', UNIX_TIMESTAMP(), '" . $_cn[$m] . "')";
            }
            $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
            $req = "INSERT INTO {$tbl} (temp_module, temp_key, temp_updated, temp_value) VALUES " . implode(',', $_rq) . "
                    ON DUPLICATE KEY UPDATE temp_updated = VALUES(temp_updated), temp_value = VALUES(temp_value)";
            jrCore_db_query($req);
        }

        return $_cn;
    }
    return false;
}

/**
 * Get all DS ROW counts
 * @return array
 */
function jrCore_db_get_all_ds_row_counts()
{
    // Get total DS row counts
    if (!$_rt = jrCore_is_cached('jrCore', 'core_ds_row_counts', false, false, true, false)) {
        if (!$_rt = jrCore_get_temp_value('jrCore', 'core_ds_row_counts')) {
            $_rt = jrCore_db_update_row_counts();
        }
        else {
            jrCore_add_to_cache('jrCore', 'core_ds_row_counts', $_rt, 0, 0, false, false, false);
        }
    }
    return $_rt;
}

/**
 * Get the DS Row count for a module
 * @note: This value is CACHED use this in place of jrCore_db_get_datastore_item_count() when possible
 * @param string $module
 * @return int
 */
function jrCore_db_get_ds_row_count($module)
{
    if ($_counts = jrCore_is_cached('jrCore', 'core_ds_row_counts', false, false, true, false)) {
        if (isset($_counts[$module])) {
            return $_counts[$module];
        }
    }
    elseif ($_counts = jrCore_get_temp_value('jrCore', 'core_ds_row_counts')) {
        jrCore_add_to_cache('jrCore', 'core_ds_row_counts', $_counts, 0, 0, false, false, false);
        if (isset($_counts[$module])) {
            return $_counts[$module];
        }
    }
    // Fall through - no caching - get counts
    if (jrCore_is_datastore_module($module)) {
        return jrCore_db_number_rows($module, 'item');
    }
    return 0;
}

/**
 * Update DS Row counts worker
 * @param array $_queue
 * @return bool
 */
function jrCore_update_ds_counts_worker($_queue)
{
    // Update Row Counts
    $now = explode(' ', microtime());
    $now = $now[1] + $now[0];

    jrCore_db_update_row_counts();

    $end = explode(' ', microtime());
    $end = $end[1] + $end[0];
    $end = round(($end - $now), 2);
    if ($end > 5) {
        jrCore_logger('DBG', "core: update_ds_counts_worker took {$end} seconds to generate row counts");
    }
    return true;
}

/**
 * Get a mapped operator
 * @param string $op
 * @return string|false
 */
function jrCore_get_ds_mapped_operator($op)
{
    switch (strtolower($op)) {
        case '=':
        case 'eq':
            return '=';
        case '!=':
        case 'neq':
            return '!=';
        case '>':
        case 'gt':
            return '>';
        case '>=':
        case 'gte':
            return '>=';
        case '<':
        case 'lt':
            return '<';
        case '<=':
        case 'lte':
            return '<=';
        case 'like':
        case 'not_like':
        case 'in':
        case 'not_in':
        case 'between':
        case 'not_between':
        case 'regexp':
            return $op;
    }
    return false;
}

/**
 * Get a search condition for use in jrCore_db_search_items()
 * @param string $ss search string
 * @param array $_fields array of fields to search - uses '%' if none provided
 * @return string
 */
function jrCore_get_ds_search_condition($ss, $_fields = null)
{
    global $_post;
    // Are we looking for an email?
    if ($_post['module'] == 'jrUser' && jrCore_checktype($ss, 'email')) {
        return "user_email = {$ss}";
    }
    // Check for support operand as second field
    if (strpos($ss, ' ')) {
        if ($op = jrCore_string_field($ss, 2)) {
            if ($sm = jrCore_get_ds_mapped_operator($op)) {
                list($sf, , $ss) = explode(' ', $ss, 3);
                return "{$sf} {$sm} {$ss}";
            }
        }
    }
    // Check for passing in a specific key name for search
    if (strpos($ss, ':')) {
        list($sf, $ss) = explode(':', $ss, 2);
        $sf = trim($sf);
        $ss = trim($ss);
        if (!strpos(' ' . $ss, '%')) {
            return "{$sf} = {$ss}";
        }
        return "{$sf} like {$ss}";
    }
    // We are NOT searching a specific field
    if (!strpos(' ' . $ss, '%')) {
        $ss = "%{$ss}%";
    }
    if (is_array($_fields)) {
        $_mt = array();
        foreach ($_fields as $f) {
            $_mt[] = "{$f} like {$ss}";
        }
        $match = implode(' || ', $_mt);
    }
    else {
        $match = "% like {$ss}"; // all fields
    }
    return $match;
}
