/*
 * Mentions Input
 * Version 1.0.1
 * Written by: Kenneth Auchenberg (Podio)
 * Using underscore.js
 * License: MIT License - http://www.opensource.org/licenses/mit-license.php
 */

(function($, _, undefined) {

    // Settings
    var ky = {BACKSPACE: 8, TAB: 9, RETURN: 13, ESC: 27, LEFT: 37, UP: 38, RIGHT: 39, DOWN: 40, COMMA: 188, SPACE: 32, HOME: 36, END: 35}; // Keys "enum"
    var ds = {
        triggerChar: '@',
        defaultClass: 'mentions-input-box',
        onDataRequest: $.noop,
        minChars: 3,
        showAvatars: true,
        classes: {
            autoCompleteItemActive: "active"
        },
        templates: {
            wrapper: _.template('<div class="<%= defaultclass %>"></div>'),
            autocompleteList: _.template('<div class="item mentions-autocomplete-list"></div>'),
            autocompleteListItem: _.template('<li class="mention_item" data-ref-id="<%= id %>" data-ref-type="<%= type %>" data-display="<%= display %>"><%= content %></li>'),
            autocompleteListItemAvatar: _.template('<img alt="img" src="<%= avatar %>" />'),
            autocompleteListItemIcon: _.template('<div class="icon <%= icon %>"></div>'),
            mentionsOverlay: _.template('<div class="mentions"><div></div></div>'),
            mentionItemSyntax: _.template('@[<%= value %>](<%= type %>:<%= id %>)'),
            mentionItemHighlight: _.template('<strong><span><%= value %></span></strong>')
        }
    };

    var ut = {
        htmlEncode: function(s) {
            return _.escape(s);
        },
        highlightTerm: function(v, t) {
            if (!t && !t.length) {
                return v;
            }
            return v.replace(new RegExp("(?![^&;]+;)(?!<[^<>]*)(" + t + ")(?![^<>]*>)(?![^&;]+;)", "gi"), "<b>$1</b>");
        },
        rtrim: function(s) {
            return s.replace(/\s+$/, "");
        }
    };

    var MentionsInput = function(input) {

        // st = settings
        // ib = elmInputBox
        // iw = elmInputWrapper
        // al = elmAutocompleteList
        // wb = elmWrapperBox
        // mo = elmMentionsOverlay
        // ai = elmActiveAutoCompleteItem
        // dq = ?? direct query (probably)
        // mc = mentionsCollection
        // iu = inputBuffer
        var st, ib, iw, al, wb, mo, ai, dq, mc = [], iu = [];

        function initTextarea()
        {
            ib = $(input);

            if (ib.attr('data-mentions-input') === 'true') {
                return;
            }

            iw = ib.parent();
            wb = $(st.templates.wrapper({'defaultclass': ds.defaultClass}));
            ib.wrapAll(wb);
            wb = iw.find('> div');

            ib.attr('data-mentions-input', 'true');
            ib.bind('keydown', onInputBoxKeyDown);
            ib.bind('input', onInputBoxInput);
            ib.bind('click', onInputBoxClick);

        }

        function initAutocomplete()
        {
            al = $(st.templates.autocompleteList());
            al.appendTo(wb);
            al.delegate('li', 'click', onAutoCompleteItemClick);
        }

        function initMentionsOverlay()
        {
            mo = $(st.templates.mentionsOverlay());
            mo.prependTo(wb);
        }

        function updateValues()
        {
            var sm = getInputBoxValue();

            _.each(mc, function(m) {
                var ts = st.templates.mentionItemSyntax({value: m.value, type: m.type, id: m.id});
                sm = sm.replace(m.value, ts);
            });

            var mt = ut.htmlEncode(sm);

            _.each(mc, function(m) {
                var ts = st.templates.mentionItemSyntax({value: ut.htmlEncode(m.value), type: m.type, id: m.id});
                var th = st.templates.mentionItemHighlight({value: ut.htmlEncode(m.value)});
                mt = mt.replace(ts, th);
            });

            mt = mt.replace(/\n/g, '<br />').replace(/ {2}/g, '&nbsp; ');

            ib.data('messageText', sm);
            mo.find('div').html(mt);
        }

        function resetBuffer()
        {
            iu = [];
        }

        function updateMentionsCollection()
        {
            var it = getInputBoxValue();
            mc = _.reject(mc, function(m) {
                return !m.value || it.indexOf(m.value) === -1;
            });
            mc = _.compact(mc);
        }

        function addMention(value, id, type)
        {
            var r, u, c = ' ' + getInputBoxValue() + ' ';
            var s = st.triggerChar.split(',');
            _.each(s, function(i, x) {
                r = new RegExp(dq + ' ', 'gi');
                u = $.trim(c.replace(r, ' ' + value + ' '));
            });

            mc.push({
                id: id,
                type: type,
                value: value
            });

            // Cleaning before inserting the value, otherwise auto-complete would be triggered with "old" inputbuffer
            resetBuffer();
            dq = '';
            hideAutoComplete();

            // Mentions & syntax message
            ib.val(u + ' ');

            // Set correct focus and selection
            ib.focus();
        }

        function getInputBoxValue()
        {
            return $.trim(ib.val());
        }

        function onAutoCompleteItemClick()
        {
            var elmTarget = $(this);
            addMention(elmTarget.attr('data-display'), elmTarget.attr('data-ref-id'), elmTarget.attr('data-ref-type'));
            return false;
        }

        function onInputBoxClick()
        {
            resetBuffer();
        }

        function onInputBoxInput(e)
        {
            iu = ib.val().split('');

            updateValues();
            updateMentionsCollection();

            if (iu.length > 0) {
                if (st.triggerChar.indexOf(',') > -1) {
                    var t = st.triggerChar.split(',');
                    _.each(t, function(i, x) {
                        var ti = _.lastIndexOf(iu, i);
                        if (ti > -1) {
                            var tmp;
                            tmp = iu.slice(ti).join('');
                            tmp = ut.rtrim(tmp);
                            if (/\s/.test(tmp)) {
                                // It has any kind of whitespace
                                return;
                            }
                            dq = tmp;
                            if (dq.length > 2) {
                                _.defer(_.bind(doSearch, this, dq));
                            }
                            else {
                                hideAutoComplete()
                            }
                        }
                    });
                }
                else {
                    var ti = _.lastIndexOf(iu, st.triggerChar);
                    if (ti > -1) {
                        dq = iu.slice(ti).join('');
                        dq = ut.rtrim(dq);
                        if (dq.length > 2) {
                            _.defer(_.bind(doSearch, this, dq));
                        }
                        else {
                            hideAutoComplete();
                        }
                    }
                }
            }
        }

        function onInputBoxKeyDown(e)
        {
            // This also matches HOME/END on OSX which is CMD+LEFT, CMD+RIGHT
            if (e.keyCode === ky.LEFT || e.keyCode === ky.RIGHT || e.keyCode === ky.HOME || e.keyCode === ky.END) {
                // Defer execution to ensure carat pos has changed after HOME/END keys
                _.defer(resetBuffer);
                return;
            }

            if (e.keyCode === ky.BACKSPACE) {
                iu.pop();
                return;
            }

            if (!al.is(':visible')) {
                return true;
            }

            switch (e.keyCode) {
                case ky.UP:
                case ky.DOWN:
                    var elmCurrentAutoCompleteItem = null;
                    if (e.keyCode === ky.DOWN) {
                        if (ai && ai.length) {
                            elmCurrentAutoCompleteItem = ai.next();
                        }
                        else {
                            elmCurrentAutoCompleteItem = al.find('li').first();
                        }
                    }
                    else {
                        elmCurrentAutoCompleteItem = $(ai).prev();
                    }
                    if (elmCurrentAutoCompleteItem.length) {
                        selectAutoCompleteItem(elmCurrentAutoCompleteItem);
                    }
                    return false;

                case ky.RETURN:
                case ky.TAB:
                    if (ai && ai.length) {
                        ai.click();
                        return false;
                    }
                    break;
            }
            return true;
        }

        function hideAutoComplete()
        {
            ai = null;
            al.empty().hide();
        }

        function selectAutoCompleteItem(e)
        {
            e.addClass(st.classes.autoCompleteItemActive);
            e.siblings().removeClass(st.classes.autoCompleteItemActive);
            ai = e;
        }

        // q = query
        // r = results
        function populateDropdown(q, r)
        {
            // if the query has a space in it, skip
            if (/\s/.test(q)) {
                // It has any kind of whitespace
                return;
            }

            al.show();
            // Filter items that has already been mentioned
            var mv = _.pluck(mc, 'value');
            r = _.reject(r, function(i) {
                return _.include(mv, i.name);
            });

            if (!r.length) {
                hideAutoComplete();
                return;
            }

            al.empty();
            var dd = $("<ul>").appendTo(al).hide();

            // i = item
            // x = index
            _.each(r, function(i, x) {
                if (typeof i.title != "undefined") {
                    var li = $(st.templates.autocompleteListItem({
                        'id': ut.htmlEncode(i.id),
                        'display': ut.htmlEncode(i.name),
                        'title': ut.htmlEncode(i.title),
                        'type': ut.htmlEncode(i.type),
                        'content': ut.highlightTerm(ut.htmlEncode((i.title)), q)
                    }));
                }
                else {
                    var li = $(st.templates.autocompleteListItem({
                        'id': ut.htmlEncode(i.id),
                        'display': ut.htmlEncode(i.name),
                        'type': ut.htmlEncode(i.type),
                        'content': ut.highlightTerm(ut.htmlEncode((i.name)), q)
                    }));
                }

                if (x === 0) {
                    selectAutoCompleteItem(li);
                }

                if (st.showAvatars) {
                    var io;
                    if (i.avatar) {
                        io = $(st.templates.autocompleteListItemAvatar({avatar: i.avatar}));
                    }
                    else {
                        io = $(st.templates.autocompleteListItemIcon({icon: i.icon}));
                    }
                    io.prependTo(li);
                }
                li.appendTo(dd);
            });

            al.show();
            dd.show();
        }

        function doSearch(q)
        {
            // if the query has a space in it, skip
            if (/\s/.test(q)) {
                // It has any kind of whitespace
                hideAutoComplete();
                return;
            }

            if (q && q.length && q.length >= st.minChars) {
                st.onDataRequest.call(this, 'search', q, function(r) {
                    populateDropdown(q, r);
                });
            }
            else {
                hideAutoComplete();
            }
        }

        // Public methods
        return {
            init: function(o) {
                st = o;
                initTextarea();
                initAutocomplete();
                initMentionsOverlay();
            },
            val: function(c) {
                if (!_.isFunction(c)) {
                    return;
                }

                var v = mc.length ? ib.data('messageText') : getInputBoxValue();
                c.call(this, v);
            },
            reset: function() {
                ib.val('');
                mc = [];
                updateValues();
            }
        };
    };

    $.fn.mentionsInput = function(m, st) {

        if (typeof m === 'object' || !m) {
            st = $.extend(true, {}, ds, m);
        }

        var oa = arguments;
        return this.each(function() {
            var ic = $.data(this, 'mentionsInput') || $.data(this, 'mentionsInput', new MentionsInput(this));
            if (_.isFunction(ic[m])) {
                return ic[m].apply(this, Array.prototype.slice.call(oa, 1));
            }
            else if (typeof m === 'object' || !m) {
                return ic.init.call(this, st);
            }
            else {
                $.error('Method ' + m + ' does not exist');
            }
        });
    };

})(jQuery, _);
